/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.writer;

import com.codahale.metrics.Timer;
import com.github.rholder.retry.RetryerBuilder;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.concurrent.TimeUnit;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.instrumented.Instrumented;
import org.apache.gobblin.metrics.GobblinMetrics;
import org.apache.gobblin.stream.RecordEnvelope;
import org.apache.gobblin.util.Decorator;
import org.apache.gobblin.util.FinalState;
import org.apache.gobblin.util.limiter.Limiter;
import org.apache.gobblin.util.limiter.RateBasedLimiter;
import org.apache.gobblin.writer.DataWriter;
import org.apache.gobblin.writer.Retriable;
import org.apache.gobblin.writer.RetryWriter;
import org.apache.gobblin.writer.WriterWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThrottleWriter<D>
extends WriterWrapper<D>
implements Decorator,
FinalState,
Retriable {
    private static final Logger LOG = LoggerFactory.getLogger(ThrottleWriter.class);
    public static final String WRITER_THROTTLE_TYPE_KEY = "gobblin.writer.throttle_type";
    public static final String WRITER_LIMIT_RATE_LIMIT_KEY = "gobblin.writer.throttle_rate";
    public static final String WRITES_THROTTLED_TIMER = "gobblin.writer.throttled_time";
    public static final String THROTTLED_TIME_KEY = "ThrottledTime";
    private static final String LOCAL_JOB_LAUNCHER_TYPE = "LOCAL";
    private final State state;
    private final DataWriter<D> writer;
    private final Limiter limiter;
    private final ThrottleType type;
    private final Optional<Timer> throttledTimer;
    private long throttledTime;

    public ThrottleWriter(DataWriter<D> writer, State state) {
        Preconditions.checkNotNull(writer, (Object)"DataWriter is required.");
        Preconditions.checkNotNull((Object)state, (Object)"State is required.");
        this.state = state;
        this.writer = writer;
        this.type = ThrottleType.valueOf(state.getProp(WRITER_THROTTLE_TYPE_KEY));
        int rateLimit = this.computeRateLimit(state);
        LOG.info("Rate limit for each writer: " + rateLimit + " " + (Object)((Object)this.type));
        this.limiter = new RateBasedLimiter((double)this.computeRateLimit(state));
        this.throttledTimer = GobblinMetrics.isEnabled((State)state) ? Optional.of((Object)Instrumented.getMetricContext((State)state, this.getClass()).timer(WRITES_THROTTLED_TIMER)) : Optional.absent();
    }

    public Object getDecoratedObject() {
        return this.writer;
    }

    private int computeRateLimit(State state) {
        String jobLauncherType = state.getProp("launcher.type", LOCAL_JOB_LAUNCHER_TYPE);
        int parallelism = 1;
        parallelism = LOCAL_JOB_LAUNCHER_TYPE.equals(jobLauncherType) ? state.getPropAsInt("taskexecutor.threadpool.size", 2) : state.getPropAsInt("mr.job.max.mappers", 100);
        parallelism = Math.min(parallelism, state.getPropAsInt("source.max.number.of.partitions", 20));
        parallelism = Math.max(parallelism, 1);
        int rateLimit = state.getPropAsInt(WRITER_LIMIT_RATE_LIMIT_KEY) / parallelism;
        rateLimit = Math.max(rateLimit, 1);
        return rateLimit;
    }

    public void writeEnvelope(RecordEnvelope<D> record) throws IOException {
        try {
            if (ThrottleType.QPS.equals((Object)this.type)) {
                this.acquirePermits(1L);
            }
            long beforeWrittenBytes = this.writer.bytesWritten();
            this.writer.writeEnvelope(record);
            if (ThrottleType.Bytes.equals((Object)this.type)) {
                long delta = this.writer.bytesWritten() - beforeWrittenBytes;
                if (delta < 0L) {
                    throw new UnsupportedOperationException("Cannot throttle on bytes because " + this.writer.getClass().getSimpleName() + " does not supports bytesWritten");
                }
                if (delta > 0L) {
                    this.acquirePermits(delta);
                }
            }
        }
        catch (InterruptedException e) {
            throw new IOException("Failed while acquiring permits.", e);
        }
    }

    private void acquirePermits(long permits) throws InterruptedException {
        long startMs = System.currentTimeMillis();
        this.limiter.acquirePermits(permits);
        long permitAcquisitionTime = System.currentTimeMillis() - startMs;
        if (this.throttledTimer.isPresent()) {
            Instrumented.updateTimer(this.throttledTimer, (long)permitAcquisitionTime, (TimeUnit)TimeUnit.MILLISECONDS);
        }
        this.throttledTime += permitAcquisitionTime;
    }

    public void close() throws IOException {
        this.writer.close();
    }

    public void commit() throws IOException {
        this.writer.commit();
    }

    public void cleanup() throws IOException {
        this.writer.cleanup();
    }

    public long recordsWritten() {
        return this.writer.recordsWritten();
    }

    public long bytesWritten() throws IOException {
        return this.writer.bytesWritten();
    }

    @Override
    public RetryerBuilder<Void> getRetryerBuilder() {
        if (this.writer instanceof Retriable) {
            return ((Retriable)this.writer).getRetryerBuilder();
        }
        return RetryWriter.createRetryBuilder(this.state);
    }

    public State getFinalState() {
        State state = new State();
        if (this.writer instanceof FinalState) {
            state.addAll(((FinalState)this.writer).getFinalState());
        } else {
            LOG.warn("Wrapped writer does not implement FinalState: " + this.writer.getClass());
        }
        state.setProp(THROTTLED_TIME_KEY, (Object)this.throttledTime);
        return state;
    }

    public void flush() throws IOException {
        this.writer.flush();
    }

    public static enum ThrottleType {
        QPS,
        Bytes;

    }
}

