/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.planner.plan.rules.logical;

import org.apache.flink.table.planner.plan.utils.InputRefVisitor;

import org.apache.calcite.plan.RelOptRule;
import org.apache.calcite.plan.RelOptRuleCall;
import org.apache.calcite.rel.logical.LogicalCalc;
import org.apache.calcite.rel.logical.LogicalFilter;
import org.apache.calcite.rel.rules.FilterCalcMergeRule;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.rex.RexUtil;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Extends calcite's FilterCalcMergeRule for streaming scenario, modification: does not merge the
 * filter references field which generated by non-deterministic function.
 */
public class FlinkFilterCalcMergeRule extends FilterCalcMergeRule {

    public static final RelOptRule INSTANCE = new FlinkFilterCalcMergeRule(Config.DEFAULT);

    protected FlinkFilterCalcMergeRule(Config config) {
        super(config);
    }

    @Override
    public void onMatch(RelOptRuleCall call) {
        LogicalFilter filter = call.rel(0);
        LogicalCalc calc = call.rel(1);

        List<RexNode> expandProjects =
                calc.getProgram().getProjectList().stream()
                        .map(p -> calc.getProgram().expandLocalRef(p))
                        .collect(Collectors.toList());
        InputRefVisitor inputRefVisitor = new InputRefVisitor();
        filter.getCondition().accept(inputRefVisitor);
        boolean existNonDeterministicRef =
                Arrays.stream(inputRefVisitor.getFields())
                        .anyMatch(i -> !RexUtil.isDeterministic(expandProjects.get(i)));

        if (!existNonDeterministicRef) {
            super.onMatch(call);
        }
    }
}
