/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#include <catch2/catch.hpp>
#include <fstream>
#include <frequent_items_sketch.hpp>

namespace datasketches {

// assume the binary sketches for this test have been generated by datasketches-java code
// in the subdirectory called "java" in the root directory of this project
static std::string testBinaryInputPath = std::string(TEST_BINARY_INPUT_PATH) + "../../java/";

TEST_CASE("frequent longs", "[serde_compat]") {
  const unsigned n_arr[] = {0, 1, 10, 100, 1000, 10000, 100000, 1000000};
  for (const unsigned n: n_arr) {
    std::ifstream is;
    is.exceptions(std::ios::failbit | std::ios::badbit);
    is.open(testBinaryInputPath + "frequent_long_n" + std::to_string(n) + "_java.sk", std::ios::binary);
    const auto sketch = frequent_items_sketch<int64_t>::deserialize(is);
    REQUIRE(sketch.is_empty() == (n == 0));
    if (n > 10) {
      REQUIRE(sketch.get_maximum_error() > 0);
    } else {
      REQUIRE(sketch.get_maximum_error() == 0);
    }
    REQUIRE(sketch.get_total_weight() == n);
  }
}

TEST_CASE("frequent strings", "[serde_compat]") {
  const unsigned n_arr[] = {0, 1, 10, 100, 1000, 10000, 100000, 1000000};
  for (const unsigned n: n_arr) {
    std::ifstream is;
    is.exceptions(std::ios::failbit | std::ios::badbit);
    is.open(testBinaryInputPath + "frequent_string_n" + std::to_string(n) + "_java.sk", std::ios::binary);
    const auto sketch = frequent_items_sketch<std::string>::deserialize(is);
    REQUIRE(sketch.is_empty() == (n == 0));
    if (n > 10) {
      REQUIRE(sketch.get_maximum_error() > 0);
    } else {
      REQUIRE(sketch.get_maximum_error() == 0);
    }
    REQUIRE(sketch.get_total_weight() == n);
  }
}

TEST_CASE("frequent strings ascii", "[serde_compat]") {
  std::ifstream is;
  is.exceptions(std::ios::failbit | std::ios::badbit);
  is.open(testBinaryInputPath + "frequent_string_ascii_java.sk", std::ios::binary);
  const auto sketch = frequent_items_sketch<std::string>::deserialize(is);
  REQUIRE_FALSE(sketch.is_empty());
  REQUIRE(sketch.get_maximum_error() == 0);
  REQUIRE(sketch.get_total_weight() == 10);
  REQUIRE(sketch.get_estimate("aaaaaaaaaaaaaaaaaaaaaaaaaaaaa") == 1);
  REQUIRE(sketch.get_estimate("bbbbbbbbbbbbbbbbbbbbbbbbbbbbb") == 2);
  REQUIRE(sketch.get_estimate("ccccccccccccccccccccccccccccc") == 3);
  REQUIRE(sketch.get_estimate("ddddddddddddddddddddddddddddd") == 4);
}

TEST_CASE("frequent strings utf8", "[serde_compat]") {
  std::ifstream is;
  is.exceptions(std::ios::failbit | std::ios::badbit);
  is.open(testBinaryInputPath + "frequent_string_utf8_java.sk", std::ios::binary);
  const auto sketch = frequent_items_sketch<std::string>::deserialize(is);
  REQUIRE_FALSE(sketch.is_empty());
  REQUIRE(sketch.get_maximum_error() == 0);
  REQUIRE(sketch.get_total_weight() == 28);
  REQUIRE(sketch.get_estimate("абвгд") == 1);
  REQUIRE(sketch.get_estimate("еёжзи") == 2);
  REQUIRE(sketch.get_estimate("йклмн") == 3);
  REQUIRE(sketch.get_estimate("опрст") == 4);
  REQUIRE(sketch.get_estimate("уфхцч") == 5);
  REQUIRE(sketch.get_estimate("шщъыь") == 6);
  REQUIRE(sketch.get_estimate("эюя") == 7);
}

} /* namespace datasketches */
