/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.security;

import com.google.common.collect.ImmutableSet;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PEMReader {
    public static final Set<String> SUPPORTED_PRIVATE_KEY_ALGORITHMS = ImmutableSet.of((Object)"RSA", (Object)"DSA", (Object)"EC");
    private static final Logger logger = LoggerFactory.getLogger(PEMReader.class);
    private static final Pattern CERT_PATTERN = Pattern.compile("-+BEGIN\\s+.*CERTIFICATE[^-]*-+(?:\\s|\\r|\\n)+([a-z0-9+/=\\r\\n]+)-+END\\s+.*CERTIFICATE[^-]*-+", 2);
    private static final Pattern KEY_PATTERN = Pattern.compile("-+BEGIN\\s+.*PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+([a-z0-9+/=\\r\\n]+)-+END\\s+.*PRIVATE\\s+KEY[^-]*-+", 2);

    public static PrivateKey extractPrivateKey(String unencryptedPEMKey) throws IOException, GeneralSecurityException {
        return PEMReader.extractPrivateKey(unencryptedPEMKey, null);
    }

    public static PrivateKey extractPrivateKey(String pemKey, String keyPassword) throws IOException, GeneralSecurityException {
        PKCS8EncodedKeySpec keySpec;
        String base64EncodedKey = PEMReader.extractBase64EncodedKey(pemKey);
        byte[] derKeyBytes = PEMReader.decodeBase64(base64EncodedKey);
        if (!StringUtils.isEmpty((CharSequence)keyPassword)) {
            byte[] rawKeyBytes;
            logger.debug("Encrypted key's length: {}, key's password length: {}", (Object)derKeyBytes.length, (Object)keyPassword.length());
            EncryptedPrivateKeyInfo epki = new EncryptedPrivateKeyInfo(derKeyBytes);
            logger.debug("Encrypted private key info's algorithm name: {}", (Object)epki.getAlgName());
            AlgorithmParameters params = epki.getAlgParameters();
            PBEKeySpec pbeKeySpec = new PBEKeySpec(keyPassword.toCharArray());
            SecretKey encryptionKey = SecretKeyFactory.getInstance(epki.getAlgName()).generateSecret(pbeKeySpec);
            pbeKeySpec.clearPassword();
            logger.debug("Key algorithm: {}, key format: {}", (Object)encryptionKey.getAlgorithm(), (Object)encryptionKey.getFormat());
            Cipher cipher = Cipher.getInstance(epki.getAlgName());
            cipher.init(2, (Key)encryptionKey, params);
            try {
                rawKeyBytes = cipher.doFinal(epki.getEncryptedData());
            }
            catch (BadPaddingException e) {
                throw new GeneralSecurityException("Failed to decrypt the private key data. Either the password provided for the key is wrong or the private key data is corrupted. msg=" + e.getMessage(), e);
            }
            logger.debug("Decrypted private key's length: {}", (Object)rawKeyBytes.length);
            keySpec = new PKCS8EncodedKeySpec(rawKeyBytes);
        } else {
            logger.debug("Key length: {}", (Object)derKeyBytes.length);
            keySpec = new PKCS8EncodedKeySpec(derKeyBytes);
        }
        PrivateKey privateKey = null;
        for (String privateKeyAlgorithm : SUPPORTED_PRIVATE_KEY_ALGORITHMS) {
            try {
                privateKey = KeyFactory.getInstance(privateKeyAlgorithm).generatePrivate(keySpec);
                logger.info("Parsing for the private key finished with {} algorithm.", (Object)privateKeyAlgorithm);
                return privateKey;
            }
            catch (Exception e) {
                logger.debug("Failed to parse the private key with {} algorithm. Will try the other supported algorithms.", (Object)privateKeyAlgorithm);
            }
        }
        throw new GeneralSecurityException("The given private key could not be parsed with any of the supported algorithms. Please see PEMReader#SUPPORTED_PRIVATE_KEY_ALGORITHMS.");
    }

    public static Certificate[] extractCertificates(String pemCerts) throws GeneralSecurityException {
        ArrayList<Certificate> certificateList = new ArrayList<Certificate>();
        List<String> base64EncodedCerts = PEMReader.extractBase64EncodedCerts(pemCerts);
        for (String base64EncodedCertificate : base64EncodedCerts) {
            certificateList.add(PEMReader.generateCertificate(base64EncodedCertificate));
        }
        Certificate[] certificates = certificateList.toArray(new Certificate[0]);
        return certificates;
    }

    private static Certificate generateCertificate(String base64Certificate) throws GeneralSecurityException {
        byte[] decodedCertificateBytes = PEMReader.decodeBase64(base64Certificate);
        CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
        X509Certificate certificate = (X509Certificate)certificateFactory.generateCertificate(new ByteArrayInputStream(decodedCertificateBytes));
        PEMReader.logCertificateDetails(certificate);
        return certificate;
    }

    private static void logCertificateDetails(X509Certificate certificate) {
        assert (certificate != null);
        logger.info("*********** Certificate Details *****************");
        logger.info("Subject DN: {}", (Object)certificate.getSubjectDN());
        logger.info("Issuer DN: {}", (Object)certificate.getIssuerDN());
        logger.info("Serial Number: {}", (Object)certificate.getSerialNumber());
        logger.info("Expiry: {}", (Object)certificate.getNotAfter());
    }

    private static String extractBase64EncodedKey(String pemKey) throws GeneralSecurityException {
        Matcher matcher = KEY_PATTERN.matcher(pemKey);
        if (matcher.find()) {
            return matcher.group(1).replaceAll("\\s", "");
        }
        throw new GeneralSecurityException("Invalid private key format");
    }

    private static List<String> extractBase64EncodedCerts(String pemCerts) throws GeneralSecurityException {
        ArrayList<String> certificateList = new ArrayList<String>();
        Matcher matcher = CERT_PATTERN.matcher(pemCerts);
        if (!matcher.find()) {
            throw new GeneralSecurityException("Invalid certificate format");
        }
        int start = 0;
        while (matcher.find(start)) {
            String certificate = matcher.group(1).replaceAll("\\s", "");
            certificateList.add(certificate);
            start = matcher.end();
        }
        return certificateList;
    }

    private static byte[] decodeBase64(String base64Input) throws GeneralSecurityException {
        try {
            return Base64.getDecoder().decode(base64Input);
        }
        catch (IllegalArgumentException e) {
            throw new GeneralSecurityException("Failed to decode given base64 input. msg=" + e.getMessage(), e);
        }
    }
}

