/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.core.mgmt.persist;

import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.brooklyn.api.mgmt.ManagementContext;
import org.apache.brooklyn.api.mgmt.ha.HighAvailabilityMode;
import org.apache.brooklyn.core.mgmt.persist.FileBasedStoreObjectAccessor;
import org.apache.brooklyn.core.mgmt.persist.PersistMode;
import org.apache.brooklyn.core.mgmt.persist.PersistenceObjectStore;
import org.apache.brooklyn.core.server.BrooklynServerConfig;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.exceptions.FatalConfigurationRuntimeException;
import org.apache.brooklyn.util.io.FileUtil;
import org.apache.brooklyn.util.os.Os;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedObjectStore
implements PersistenceObjectStore {
    private static final Logger log = LoggerFactory.getLogger(FileBasedObjectStore.class);
    private static final int SHUTDOWN_TIMEOUT_MS = 10000;
    private static boolean WARNED_ON_NON_ATOMIC_FILE_UPDATES = false;
    private final File basedir;
    private final ListeningExecutorService executor;
    private ManagementContext mgmt;
    private boolean prepared = false;
    private boolean deferredBackupNeeded = false;
    private boolean doneFirstContentiousWrite = false;

    public FileBasedObjectStore(File basedir) {
        this.basedir = this.checkPersistenceDirPlausible(basedir);
        this.executor = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool());
        log.debug("File-based objectStore will use directory {}", (Object)basedir);
    }

    @Override
    public String getSummaryName() {
        return this.getBaseDir().getAbsolutePath();
    }

    public File getBaseDir() {
        return this.basedir;
    }

    @Override
    public synchronized void prepareForMasterUse() {
        if (this.doneFirstContentiousWrite) {
            return;
        }
        try {
            if (this.deferredBackupNeeded) {
                File backup = this.backupDirByCopying(this.basedir);
                log.info("Persistence deferred backup, directory " + this.basedir + " backed up to " + backup.getAbsolutePath());
                this.deferredBackupNeeded = false;
            }
            this.doneFirstContentiousWrite = true;
        }
        catch (Exception e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    @Override
    public void createSubPath(String subPath) {
        if (!this.prepared) {
            throw new IllegalStateException("Not yet prepared: " + this);
        }
        File dir = new File(this.getBaseDir(), subPath);
        if (dir.mkdir()) {
            try {
                FileUtil.setFilePermissionsTo700((File)dir);
            }
            catch (IOException e) {
                log.warn("Unable to set sub-directory permissions to 700 (continuing): " + dir);
            }
        } else if (!dir.exists()) {
            throw new IllegalStateException("Cannot create " + dir + "; call returned false");
        }
        this.checkPersistenceDirAccessible(dir);
    }

    @Override
    public PersistenceObjectStore.StoreObjectAccessor newAccessor(String path) {
        if (!this.prepared) {
            throw new IllegalStateException("Not yet prepared: " + this);
        }
        String tmpExt = ".tmp";
        if (this.mgmt != null && this.mgmt.getManagementNodeId() != null) {
            tmpExt = "." + this.mgmt.getManagementNodeId() + tmpExt;
        }
        return new FileBasedStoreObjectAccessor(new File(Os.mergePaths((String[])new String[]{this.getBaseDir().getAbsolutePath(), path})), tmpExt);
    }

    @Override
    public List<String> listContentsWithSubPath(final String parentSubPath) {
        if (!this.prepared) {
            throw new IllegalStateException("Not yet prepared: " + this);
        }
        Preconditions.checkNotNull((Object)parentSubPath);
        File subPathDir = new File(this.basedir, parentSubPath);
        FileFilter fileFilter = new FileFilter(){

            @Override
            public boolean accept(File file) {
                return !file.getName().endsWith(".tmp") && !file.getName().endsWith(".swp");
            }
        };
        File[] subPathDirFiles = subPathDir.listFiles(fileFilter);
        if (subPathDirFiles == null) {
            return ImmutableList.of();
        }
        return FluentIterable.from(Arrays.asList(subPathDirFiles)).transform((Function)new Function<File, String>(){

            @Nullable
            public String apply(@Nullable File input) {
                return String.format("%s/%s", parentSubPath, input.getName());
            }
        }).toList();
    }

    @Override
    public void close() {
        this.executor.shutdown();
        try {
            this.executor.awaitTermination(10000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("basedir", (Object)this.basedir).toString();
    }

    @Override
    public void injectManagementContext(ManagementContext mgmt) {
        if (this.mgmt != null && !this.mgmt.equals(mgmt)) {
            throw new IllegalStateException("Cannot change mgmt context of " + this);
        }
        this.mgmt = mgmt;
    }

    @Override
    public void prepareForSharedUse(@Nullable PersistMode persistMode, HighAvailabilityMode haMode) {
        block24: {
            if (this.mgmt == null) {
                throw new NullPointerException("Must inject ManagementContext before preparing " + this);
            }
            if (persistMode == null || persistMode == PersistMode.DISABLED) {
                this.prepared = true;
                return;
            }
            Boolean backups = (Boolean)this.mgmt.getConfig().getConfig(BrooklynServerConfig.PERSISTENCE_BACKUPS_REQUIRED);
            if (Boolean.TRUE.equals(backups)) {
                log.warn("Using legacy backup for " + this + "; functionality will be removed in future versions, in favor of promotion/demotion-specific backups to a configurable backup location.");
            }
            if (backups == null) {
                backups = false;
            }
            File dir = this.getBaseDir();
            try {
                String persistencePath = dir.getAbsolutePath();
                switch (persistMode) {
                    case CLEAN: {
                        if (dir.exists()) {
                            this.checkPersistenceDirAccessible(dir);
                            try {
                                if (backups.booleanValue()) {
                                    File old = this.backupDirByMoving(dir);
                                    log.info("Persistence mode CLEAN, directory " + persistencePath + " backed up to " + old.getAbsolutePath());
                                    break;
                                }
                                this.deleteCompletely();
                                log.info("Persistence mode CLEAN, directory " + persistencePath + " deleted");
                                break;
                            }
                            catch (IOException e) {
                                throw new FatalConfigurationRuntimeException("Error using existing persistence directory " + dir.getAbsolutePath(), (Throwable)e);
                            }
                        }
                        log.debug("Persistence mode CLEAN, directory " + persistencePath + ", no previous state");
                        break;
                    }
                    case REBIND: {
                        File backup;
                        this.checkPersistenceDirAccessible(dir);
                        this.checkPersistenceDirNonEmpty(dir);
                        try {
                            if (!backups.booleanValue()) break;
                            if (haMode == HighAvailabilityMode.MASTER) {
                                backup = this.backupDirByCopying(dir);
                                log.info("Persistence mode REBIND, directory " + persistencePath + " backed up to " + backup.getAbsolutePath());
                                break;
                            }
                            this.deferredBackupNeeded = true;
                            break;
                        }
                        catch (IOException e) {
                            throw new FatalConfigurationRuntimeException("Error backing up persistence directory " + dir.getAbsolutePath(), (Throwable)e);
                        }
                    }
                    case AUTO: {
                        File backup;
                        if (dir.exists()) {
                            this.checkPersistenceDirAccessible(dir);
                        }
                        if (dir.exists() && !FileBasedObjectStore.isMementoDirExistButEmpty(dir)) {
                            try {
                                if (!backups.booleanValue()) break;
                                if (haMode == HighAvailabilityMode.MASTER) {
                                    backup = this.backupDirByCopying(dir);
                                    log.info("Persistence mode REBIND, directory " + persistencePath + " backed up to " + backup.getAbsolutePath());
                                    break;
                                }
                                this.deferredBackupNeeded = true;
                                break;
                            }
                            catch (IOException e) {
                                throw new FatalConfigurationRuntimeException("Error backing up persistence directory " + dir.getAbsolutePath(), (Throwable)e);
                            }
                        }
                        log.debug("Persistence mode AUTO, directory " + persistencePath + ", no previous state");
                        break;
                    }
                    default: {
                        throw new FatalConfigurationRuntimeException("Unexpected persist mode " + (Object)((Object)persistMode) + "; modified during initialization?!");
                    }
                }
                if (dir.exists()) break block24;
                boolean success = dir.mkdirs();
                if (success) {
                    FileUtil.setFilePermissionsTo700((File)dir);
                    break block24;
                }
                throw new FatalConfigurationRuntimeException("Failed to create persistence directory " + dir);
            }
            catch (Exception e) {
                throw Exceptions.propagate((Throwable)e);
            }
        }
        this.prepared = true;
    }

    protected File checkPersistenceDirPlausible(File dir) {
        Preconditions.checkNotNull((Object)dir, (Object)"directory");
        if (!dir.exists()) {
            return dir;
        }
        if (dir.isFile()) {
            throw new FatalConfigurationRuntimeException("Invalid persistence directory" + dir + ": must not be a file");
        }
        if (!dir.canRead() || !dir.canWrite()) {
            throw new FatalConfigurationRuntimeException("Invalid persistence directory" + dir + ": " + (!dir.canRead() ? "not readable" : (!dir.canWrite() ? "not writable" : "unknown reason")));
        }
        return dir;
    }

    protected void checkPersistenceDirAccessible(File dir) {
        if (!(dir.exists() && dir.isDirectory() && dir.canRead() && dir.canWrite())) {
            FatalConfigurationRuntimeException problem = new FatalConfigurationRuntimeException("Invalid persistence directory " + dir + ": " + (!dir.exists() ? "does not exist" : (!dir.isDirectory() ? "not a directory" : (!dir.canRead() ? "not readable" : (!dir.canWrite() ? "not writable" : "unknown reason")))));
            log.debug("Invalid persistence directory " + dir + " (rethrowing): " + problem, (Throwable)problem);
        } else {
            log.debug("Created dir {} for {}", (Object)dir, (Object)this);
        }
    }

    protected void checkPersistenceDirNonEmpty(File persistenceDir) {
        if (!persistenceDir.exists()) {
            FatalConfigurationRuntimeException problem = new FatalConfigurationRuntimeException("Invalid persistence directory " + persistenceDir + " because directory does not exist");
            log.debug("Invalid persistence directory " + persistenceDir + " (rethrowing): " + problem, (Throwable)problem);
            throw problem;
        }
        if (FileBasedObjectStore.isMementoDirExistButEmpty(persistenceDir)) {
            FatalConfigurationRuntimeException problem = new FatalConfigurationRuntimeException("Invalid persistence directory " + persistenceDir + " because directory is empty");
            log.debug("Invalid persistence directory " + persistenceDir + " (rethrowing): " + problem, (Throwable)problem);
            throw problem;
        }
    }

    protected File backupDirByCopying(File dir) throws IOException, InterruptedException {
        File parentDir = dir.getParentFile();
        String simpleName = dir.getName();
        String timestamp = new SimpleDateFormat("yyyyMMdd-hhmmssSSS").format(new Date());
        File backupDir = new File(parentDir, simpleName + "." + timestamp + ".bak");
        FileUtil.copyDir((File)dir, (File)backupDir);
        FileUtil.setFilePermissionsTo700((File)backupDir);
        return backupDir;
    }

    protected File backupDirByMoving(File dir) throws InterruptedException, IOException {
        File parentDir = dir.getParentFile();
        String simpleName = dir.getName();
        String timestamp = new SimpleDateFormat("yyyyMMdd-hhmmssSSS").format(new Date());
        File newDir = new File(parentDir, simpleName + "." + timestamp + ".bak");
        FileUtil.moveDir((File)dir, (File)newDir);
        return newDir;
    }

    static void moveFile(File srcFile, File destFile) throws IOException, InterruptedException {
        if (destFile.isDirectory()) {
            FileBasedObjectStore.deleteCompletely(destFile);
        }
        try {
            Files.move(srcFile.toPath(), destFile.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            if (!WARNED_ON_NON_ATOMIC_FILE_UPDATES) {
                WARNED_ON_NON_ATOMIC_FILE_UPDATES = true;
                log.warn("Unable to perform atomic file update (" + srcFile + " to " + destFile + "); file system not recommended for production HA/DR");
            }
            Files.move(srcFile.toPath(), destFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        if (log.isTraceEnabled()) {
            log.trace("Completly moved from {} to {} completed", new Object[]{srcFile, destFile});
        }
    }

    static boolean isMementoDirExistButEmpty(String dir) {
        return FileBasedObjectStore.isMementoDirExistButEmpty(new File(dir));
    }

    static boolean isMementoDirExistButEmpty(File dir) {
        if (!dir.exists()) {
            return false;
        }
        File[] contents = dir.listFiles();
        if (contents == null) {
            return false;
        }
        for (File sub : contents) {
            if (sub.isFile()) {
                return false;
            }
            if (!sub.isDirectory() || sub.listFiles().length <= 0) continue;
            return false;
        }
        return true;
    }

    @Override
    public void deleteCompletely() {
        FileBasedObjectStore.deleteCompletely(this.getBaseDir());
    }

    public static void deleteCompletely(File d) {
        Os.DeletionResult result = Os.deleteRecursively((File)d);
        if (!result.wasSuccessful()) {
            log.warn("Unable to delete persistence dir " + d);
        }
    }
}

