/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.GarbageCollector;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.api.LedgerMetadata;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.meta.ZkLedgerUnderreplicationManager;
import org.apache.bookkeeper.meta.zk.ZKMetadataDriverBase;
import org.apache.bookkeeper.net.BookieId;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScanAndCompareGarbageCollector
implements GarbageCollector {
    static final Logger LOG = LoggerFactory.getLogger(ScanAndCompareGarbageCollector.class);
    static final int MAX_CONCURRENT_ZK_REQUESTS = 1000;
    private final LedgerManager ledgerManager;
    private final CompactableLedgerStorage ledgerStorage;
    private final ServerConfiguration conf;
    private final BookieId selfBookieAddress;
    private ZooKeeper zk = null;
    private boolean enableGcOverReplicatedLedger;
    private final long gcOverReplicatedLedgerIntervalMillis;
    private long lastOverReplicatedLedgerGcTimeMillis;
    private final String zkServers;
    private final String zkLedgersRootPath;
    private final boolean verifyMetadataOnGc;
    private int activeLedgerCounter;

    public ScanAndCompareGarbageCollector(LedgerManager ledgerManager, CompactableLedgerStorage ledgerStorage, ServerConfiguration conf, StatsLogger statsLogger) throws IOException {
        this.ledgerManager = ledgerManager;
        this.ledgerStorage = ledgerStorage;
        this.conf = conf;
        this.selfBookieAddress = Bookie.getBookieId(conf);
        this.gcOverReplicatedLedgerIntervalMillis = conf.getGcOverreplicatedLedgerWaitTimeMillis();
        this.lastOverReplicatedLedgerGcTimeMillis = System.currentTimeMillis();
        if (this.gcOverReplicatedLedgerIntervalMillis > 0L) {
            this.enableGcOverReplicatedLedger = true;
        }
        this.zkServers = ZKMetadataDriverBase.resolveZkServers(conf);
        this.zkLedgersRootPath = ZKMetadataDriverBase.resolveZkLedgersRootPath(conf);
        LOG.info("Over Replicated Ledger Deletion : enabled=" + this.enableGcOverReplicatedLedger + ", interval=" + this.gcOverReplicatedLedgerIntervalMillis);
        this.verifyMetadataOnGc = conf.getVerifyMetadataOnGC();
        this.activeLedgerCounter = 0;
    }

    public int getNumActiveLedgers() {
        return this.activeLedgerCounter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void gc(GarbageCollector.GarbageCleaner garbageCleaner) {
        if (null == this.ledgerManager) {
            return;
        }
        try {
            boolean checkOverreplicatedLedgers;
            TreeSet bkActiveLedgers = Sets.newTreeSet(this.ledgerStorage.getActiveLedgersInRange(0L, Long.MAX_VALUE));
            this.activeLedgerCounter = bkActiveLedgers.size();
            long curTime = System.currentTimeMillis();
            boolean bl = checkOverreplicatedLedgers = this.enableGcOverReplicatedLedger && curTime - this.lastOverReplicatedLedgerGcTimeMillis > this.gcOverReplicatedLedgerIntervalMillis;
            if (checkOverreplicatedLedgers) {
                this.zk = ZooKeeperClient.newBuilder().connectString(this.zkServers).sessionTimeoutMs(this.conf.getZkTimeout()).build();
                Set<Long> overReplicatedLedgers = this.removeOverReplicatedledgers(bkActiveLedgers, garbageCleaner);
                if (overReplicatedLedgers.isEmpty()) {
                    LOG.info("No over-replicated ledgers found.");
                } else {
                    LOG.info("Removed over-replicated ledgers: {}", overReplicatedLedgers);
                }
                this.lastOverReplicatedLedgerGcTimeMillis = System.currentTimeMillis();
            }
            long zkOpTimeoutMs = this.conf.getZkTimeout() * 2;
            LedgerManager.LedgerRangeIterator ledgerRangeIterator = this.ledgerManager.getLedgerRanges(zkOpTimeoutMs);
            Set<Object> ledgersInMetadata = null;
            long end = -1L;
            boolean done = false;
            AtomicBoolean isBookieInEnsembles = new AtomicBoolean(false);
            Versioned metadata = null;
            while (!done) {
                long start = end + 1L;
                if (ledgerRangeIterator.hasNext()) {
                    LedgerManager.LedgerRange lRange = ledgerRangeIterator.next();
                    ledgersInMetadata = lRange.getLedgers();
                    end = lRange.end();
                } else {
                    ledgersInMetadata = new TreeSet();
                    end = Long.MAX_VALUE;
                    done = true;
                }
                NavigableSet<Long> subBkActiveLedgers = bkActiveLedgers.subSet(start, true, end, true);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Active in metadata {}, Active in bookie {}", ledgersInMetadata, subBkActiveLedgers);
                }
                for (Long bkLid : subBkActiveLedgers) {
                    if (ledgersInMetadata.contains(bkLid)) continue;
                    if (this.verifyMetadataOnGc) {
                        isBookieInEnsembles.set(false);
                        metadata = null;
                        int rc = 0;
                        try {
                            metadata = (Versioned)FutureUtils.result(this.ledgerManager.readLedgerMetadata(bkLid), (long)zkOpTimeoutMs, (TimeUnit)TimeUnit.MILLISECONDS);
                        }
                        catch (TimeoutException | BKException e) {
                            if (e instanceof BKException) {
                                rc = ((BKException)e).getCode();
                            }
                            LOG.warn("Time-out while fetching metadata for Ledger {} : {}.", (Object)bkLid, (Object)e.getMessage());
                            continue;
                        }
                        if (metadata != null && metadata.getValue() != null) {
                            ((LedgerMetadata)metadata.getValue()).getAllEnsembles().forEach((entryId, ensembles) -> {
                                if (ensembles != null && ensembles.contains(this.selfBookieAddress)) {
                                    isBookieInEnsembles.set(true);
                                }
                            });
                            if (isBookieInEnsembles.get()) {
                                continue;
                            }
                        } else if (rc != -25) {
                            LOG.warn("Ledger {} Missing in metadata list, but ledgerManager returned rc: {}.", (Object)bkLid, (Object)rc);
                            continue;
                        }
                    }
                    garbageCleaner.clean(bkLid);
                }
            }
        }
        catch (Throwable t) {
            LOG.warn("Exception when iterating over the metadata", t);
        }
        finally {
            if (this.zk != null) {
                try {
                    this.zk.close();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    LOG.error("Error closing zk session", (Throwable)e);
                }
                this.zk = null;
            }
        }
    }

    private Set<Long> removeOverReplicatedledgers(Set<Long> bkActiveledgers, GarbageCollector.GarbageCleaner garbageCleaner) throws InterruptedException, KeeperException {
        List<ACL> zkAcls = ZkUtils.getACLs(this.conf);
        HashSet overReplicatedLedgers = Sets.newHashSet();
        Semaphore semaphore = new Semaphore(1000);
        CountDownLatch latch = new CountDownLatch(bkActiveledgers.size());
        for (Long ledgerId : bkActiveledgers) {
            try {
                if (ZkLedgerUnderreplicationManager.isLedgerBeingReplicated(this.zk, this.zkLedgersRootPath, ledgerId)) {
                    latch.countDown();
                    continue;
                }
                ZkLedgerUnderreplicationManager.acquireUnderreplicatedLedgerLock(this.zk, this.zkLedgersRootPath, ledgerId, zkAcls);
                semaphore.acquire();
                this.ledgerManager.readLedgerMetadata(ledgerId).whenComplete((metadata, exception) -> {
                    block15: {
                        try {
                            if (exception != null) break block15;
                            if (!((LedgerMetadata)metadata.getValue()).isClosed()) {
                                return;
                            }
                            NavigableMap<Long, ? extends List<BookieId>> ensembles = ((LedgerMetadata)metadata.getValue()).getAllEnsembles();
                            for (List ensemble : ensembles.values()) {
                                if (!ensemble.contains(this.selfBookieAddress)) continue;
                                return;
                            }
                            overReplicatedLedgers.add(ledgerId);
                            garbageCleaner.clean(ledgerId);
                        }
                        finally {
                            semaphore.release();
                            latch.countDown();
                            try {
                                ZkLedgerUnderreplicationManager.releaseUnderreplicatedLedgerLock(this.zk, this.zkLedgersRootPath, ledgerId);
                            }
                            catch (Throwable t) {
                                LOG.error("Exception when removing underreplicated lock for ledger {}", (Object)ledgerId, (Object)t);
                            }
                        }
                    }
                });
            }
            catch (Throwable t) {
                LOG.error("Exception when iterating through the ledgers to check for over-replication", t);
                latch.countDown();
            }
        }
        latch.await();
        bkActiveledgers.removeAll(overReplicatedLedgers);
        return overReplicatedLedgers;
    }
}

