/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.proto;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.handler.ssl.SslHandler;
import io.netty.util.HashedWheelTimer;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.apache.bookkeeper.auth.AuthToken;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.common.util.MathUtils;
import org.apache.bookkeeper.common.util.OrderedExecutor;
import org.apache.bookkeeper.common.util.SafeRunnable;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.processor.RequestProcessor;
import org.apache.bookkeeper.proto.AuthHandler;
import org.apache.bookkeeper.proto.BKStats;
import org.apache.bookkeeper.proto.BookieProtocol;
import org.apache.bookkeeper.proto.BookkeeperProtocol;
import org.apache.bookkeeper.proto.ForceLedgerProcessorV3;
import org.apache.bookkeeper.proto.GetBookieInfoProcessorV3;
import org.apache.bookkeeper.proto.GetListOfEntriesOfLedgerProcessorV3;
import org.apache.bookkeeper.proto.LongPollReadEntryProcessorV3;
import org.apache.bookkeeper.proto.ReadEntryProcessor;
import org.apache.bookkeeper.proto.ReadEntryProcessorV3;
import org.apache.bookkeeper.proto.ReadLacProcessorV3;
import org.apache.bookkeeper.proto.RequestStats;
import org.apache.bookkeeper.proto.RequestUtils;
import org.apache.bookkeeper.proto.ResponseBuilder;
import org.apache.bookkeeper.proto.WriteEntryProcessor;
import org.apache.bookkeeper.proto.WriteEntryProcessorV3;
import org.apache.bookkeeper.proto.WriteLacProcessorV3;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.tls.SecurityException;
import org.apache.bookkeeper.tls.SecurityHandlerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class BookieRequestProcessor
implements RequestProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(BookieRequestProcessor.class);
    private final ServerConfiguration serverCfg;
    private final long waitTimeoutOnBackpressureMillis;
    private final boolean preserveMdcForTaskExecution;
    final Bookie bookie;
    private final OrderedExecutor readThreadPool;
    private final OrderedExecutor writeThreadPool;
    private final SecurityHandlerFactory shFactory;
    private final OrderedExecutor longPollThreadPool;
    private final OrderedExecutor highPriorityThreadPool;
    private final HashedWheelTimer requestTimer;
    private final BKStats bkStats = BKStats.getInstance();
    private final boolean statsEnabled;
    private final RequestStats requestStats;
    final Semaphore addsSemaphore;
    final Semaphore readsSemaphore;
    final Optional<Cache<Channel, Boolean>> blacklistedChannels;
    final Consumer<Channel> onResponseTimeout;
    private final ByteBufAllocator allocator;

    public BookieRequestProcessor(ServerConfiguration serverCfg, Bookie bookie, StatsLogger statsLogger, SecurityHandlerFactory shFactory, ByteBufAllocator allocator) throws SecurityException {
        this.serverCfg = serverCfg;
        this.allocator = allocator;
        this.waitTimeoutOnBackpressureMillis = serverCfg.getWaitTimeoutOnResponseBackpressureMillis();
        this.preserveMdcForTaskExecution = serverCfg.getPreserveMdcForTaskExecution();
        this.bookie = bookie;
        this.readThreadPool = this.createExecutor(this.serverCfg.getNumReadWorkerThreads(), "BookieReadThreadPool", serverCfg.getMaxPendingReadRequestPerThread(), statsLogger);
        this.writeThreadPool = this.createExecutor(this.serverCfg.getNumAddWorkerThreads(), "BookieWriteThreadPool", serverCfg.getMaxPendingAddRequestPerThread(), statsLogger);
        if (serverCfg.getNumLongPollWorkerThreads() <= 0 && this.readThreadPool != null) {
            this.longPollThreadPool = this.readThreadPool;
        } else {
            int numThreads = this.serverCfg.getNumLongPollWorkerThreads();
            if (numThreads <= 0) {
                numThreads = Runtime.getRuntime().availableProcessors();
            }
            this.longPollThreadPool = this.createExecutor(numThreads, "BookieLongPollThread-" + serverCfg.getBookiePort(), -1, statsLogger);
        }
        this.highPriorityThreadPool = this.createExecutor(this.serverCfg.getNumHighPriorityWorkerThreads(), "BookieHighPriorityThread-" + serverCfg.getBookiePort(), -1, statsLogger);
        this.shFactory = shFactory;
        if (shFactory != null) {
            shFactory.init(SecurityHandlerFactory.NodeType.Server, serverCfg, allocator);
        }
        this.requestTimer = new HashedWheelTimer(new ThreadFactoryBuilder().setNameFormat("BookieRequestTimer-%d").build(), (long)this.serverCfg.getRequestTimerTickDurationMs(), TimeUnit.MILLISECONDS, this.serverCfg.getRequestTimerNumTicks());
        this.blacklistedChannels = this.waitTimeoutOnBackpressureMillis > 0L ? Optional.of(CacheBuilder.newBuilder().expireAfterWrite(this.waitTimeoutOnBackpressureMillis, TimeUnit.MILLISECONDS).build()) : Optional.empty();
        this.onResponseTimeout = serverCfg.getCloseChannelOnResponseTimeout() ? ch -> {
            LOG.warn("closing channel {} because it was non-writable for longer than {} ms", ch, (Object)this.waitTimeoutOnBackpressureMillis);
            ch.close();
        } : ch -> {};
        this.statsEnabled = serverCfg.isStatisticsEnabled();
        this.requestStats = new RequestStats(statsLogger);
        int maxAdds = serverCfg.getMaxAddsInProgressLimit();
        this.addsSemaphore = maxAdds > 0 ? new Semaphore(maxAdds, true) : null;
        int maxReads = serverCfg.getMaxReadsInProgressLimit();
        this.readsSemaphore = maxReads > 0 ? new Semaphore(maxReads, true) : null;
    }

    protected void onAddRequestStart(Channel channel) {
        if (this.addsSemaphore != null && !this.addsSemaphore.tryAcquire()) {
            long throttlingStartTimeNanos = MathUtils.nowInNano();
            channel.config().setAutoRead(false);
            LOG.info("Too many add requests in progress, disabling autoread on channel {}", (Object)channel);
            this.requestStats.blockAddRequest();
            this.addsSemaphore.acquireUninterruptibly();
            channel.config().setAutoRead(true);
            long delayNanos = MathUtils.elapsedNanos((long)throttlingStartTimeNanos);
            LOG.info("Re-enabled autoread on channel {} after AddRequest delay of {} nanos", (Object)channel, (Object)delayNanos);
            this.requestStats.unblockAddRequest(delayNanos);
        }
        this.requestStats.trackAddRequest();
    }

    protected void onAddRequestFinish() {
        this.requestStats.untrackAddRequest();
        if (this.addsSemaphore != null) {
            this.addsSemaphore.release();
        }
    }

    protected void onReadRequestStart(Channel channel) {
        if (this.readsSemaphore != null && !this.readsSemaphore.tryAcquire()) {
            long throttlingStartTimeNanos = MathUtils.nowInNano();
            channel.config().setAutoRead(false);
            LOG.info("Too many read requests in progress, disabling autoread on channel {}", (Object)channel);
            this.requestStats.blockReadRequest();
            this.readsSemaphore.acquireUninterruptibly();
            channel.config().setAutoRead(true);
            long delayNanos = MathUtils.elapsedNanos((long)throttlingStartTimeNanos);
            LOG.info("Re-enabled autoread on channel {} after ReadRequest delay of {} nanos", (Object)channel, (Object)delayNanos);
            this.requestStats.unblockReadRequest(delayNanos);
        }
        this.requestStats.trackReadRequest();
    }

    protected void onReadRequestFinish() {
        this.requestStats.untrackReadRequest();
        if (this.readsSemaphore != null) {
            this.readsSemaphore.release();
        }
    }

    @VisibleForTesting
    int maxAddsInProgressCount() {
        return this.requestStats.maxAddsInProgressCount();
    }

    @VisibleForTesting
    int maxReadsInProgressCount() {
        return this.requestStats.maxReadsInProgressCount();
    }

    @Override
    public void close() {
        this.shutdownExecutor(this.writeThreadPool);
        this.shutdownExecutor(this.readThreadPool);
        if (this.serverCfg.getNumLongPollWorkerThreads() > 0 || this.readThreadPool == null) {
            this.shutdownExecutor(this.longPollThreadPool);
        }
        this.shutdownExecutor(this.highPriorityThreadPool);
        this.requestTimer.stop();
    }

    private OrderedExecutor createExecutor(int numThreads, String nameFormat, int maxTasksInQueue, StatsLogger statsLogger) {
        if (numThreads <= 0) {
            return null;
        }
        return OrderedExecutor.newBuilder().numThreads(numThreads).name(nameFormat).traceTaskExecution(this.serverCfg.getEnableTaskExecutionStats()).preserveMdcForTaskExecution(this.serverCfg.getPreserveMdcForTaskExecution()).statsLogger(statsLogger).maxTasksInQueue(maxTasksInQueue).build();
    }

    private void shutdownExecutor(OrderedExecutor service) {
        if (null != service) {
            service.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void processRequest(Object msg, Channel c) {
        if (msg instanceof BookkeeperProtocol.Request) {
            BookkeeperProtocol.Request r = (BookkeeperProtocol.Request)msg;
            this.restoreMdcContextFromRequest(r);
            try {
                BookkeeperProtocol.BKPacketHeader header = r.getHeader();
                switch (header.getOperation()) {
                    case ADD_ENTRY: {
                        this.processAddRequestV3(r, c);
                        return;
                    }
                    case READ_ENTRY: {
                        this.processReadRequestV3(r, c);
                        return;
                    }
                    case FORCE_LEDGER: {
                        this.processForceLedgerRequestV3(r, c);
                        return;
                    }
                    case AUTH: {
                        LOG.info("Ignoring auth operation from client {}", (Object)c.remoteAddress());
                        BookkeeperProtocol.AuthMessage message = BookkeeperProtocol.AuthMessage.newBuilder().setAuthPluginName("AuthDisabledPlugin").setPayload(ByteString.copyFrom((byte[])AuthToken.NULL.getData())).build();
                        BookkeeperProtocol.Response.Builder authResponse = BookkeeperProtocol.Response.newBuilder().setHeader(r.getHeader()).setStatus(BookkeeperProtocol.StatusCode.EOK).setAuthResponse(message);
                        c.writeAndFlush((Object)authResponse.build());
                        return;
                    }
                    case WRITE_LAC: {
                        this.processWriteLacRequestV3(r, c);
                        return;
                    }
                    case READ_LAC: {
                        this.processReadLacRequestV3(r, c);
                        return;
                    }
                    case GET_BOOKIE_INFO: {
                        this.processGetBookieInfoRequestV3(r, c);
                        return;
                    }
                    case START_TLS: {
                        this.processStartTLSRequestV3(r, c);
                        return;
                    }
                    case GET_LIST_OF_ENTRIES_OF_LEDGER: {
                        this.processGetListOfEntriesOfLedgerProcessorV3(r, c);
                        return;
                    }
                    default: {
                        LOG.info("Unknown operation type {}", (Object)header.getOperation());
                        BookkeeperProtocol.Response.Builder response = BookkeeperProtocol.Response.newBuilder().setHeader(r.getHeader()).setStatus(BookkeeperProtocol.StatusCode.EBADREQ);
                        c.writeAndFlush((Object)response.build());
                        if (!this.statsEnabled) return;
                        this.bkStats.getOpStats(2).incrementFailedOps();
                        return;
                    }
                }
            }
            finally {
                MDC.clear();
            }
        } else {
            BookieProtocol.Request r = (BookieProtocol.Request)msg;
            switch (r.getOpCode()) {
                case 1: {
                    Preconditions.checkArgument((boolean)(r instanceof BookieProtocol.ParsedAddRequest));
                    this.processAddRequest((BookieProtocol.ParsedAddRequest)r, c);
                    return;
                }
                case 2: {
                    Preconditions.checkArgument((boolean)(r instanceof BookieProtocol.ReadRequest));
                    this.processReadRequest((BookieProtocol.ReadRequest)r, c);
                    return;
                }
                case 3: {
                    LOG.info("Ignoring auth operation from client {}", (Object)c.remoteAddress());
                    BookkeeperProtocol.AuthMessage message = BookkeeperProtocol.AuthMessage.newBuilder().setAuthPluginName("AuthDisabledPlugin").setPayload(ByteString.copyFrom((byte[])AuthToken.NULL.getData())).build();
                    c.writeAndFlush((Object)new BookieProtocol.AuthResponse(2, message));
                    return;
                }
                default: {
                    LOG.error("Unknown op type {}, sending error", (Object)r.getOpCode());
                    c.writeAndFlush((Object)ResponseBuilder.buildErrorResponse(100, r));
                    if (!this.statsEnabled) return;
                    this.bkStats.getOpStats(2).incrementFailedOps();
                }
            }
        }
    }

    private void restoreMdcContextFromRequest(BookkeeperProtocol.Request req) {
        if (this.preserveMdcForTaskExecution) {
            MDC.clear();
            for (BookkeeperProtocol.ContextPair pair : req.getRequestContextList()) {
                MDC.put((String)pair.getKey(), (String)pair.getValue());
            }
        }
    }

    private void processWriteLacRequestV3(BookkeeperProtocol.Request r, Channel c) {
        WriteLacProcessorV3 writeLac = new WriteLacProcessorV3(r, c, this);
        if (null == this.writeThreadPool) {
            writeLac.run();
        } else {
            this.writeThreadPool.executeOrdered(r.getAddRequest().getLedgerId(), (SafeRunnable)writeLac);
        }
    }

    private void processReadLacRequestV3(BookkeeperProtocol.Request r, Channel c) {
        ReadLacProcessorV3 readLac = new ReadLacProcessorV3(r, c, this);
        if (null == this.readThreadPool) {
            readLac.run();
        } else {
            this.readThreadPool.executeOrdered(r.getAddRequest().getLedgerId(), (SafeRunnable)readLac);
        }
    }

    private void processAddRequestV3(BookkeeperProtocol.Request r, Channel c) {
        WriteEntryProcessorV3 write = new WriteEntryProcessorV3(r, c, this);
        OrderedExecutor threadPool = RequestUtils.isHighPriority(r) ? this.highPriorityThreadPool : this.writeThreadPool;
        if (null == threadPool) {
            write.run();
        } else {
            try {
                threadPool.executeOrdered(r.getAddRequest().getLedgerId(), (SafeRunnable)write);
            }
            catch (RejectedExecutionException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failed to process request to add entry at {}:{}. Too many pending requests", (Object)r.getAddRequest().getLedgerId(), (Object)r.getAddRequest().getEntryId());
                }
                BookkeeperProtocol.AddResponse.Builder addResponse = BookkeeperProtocol.AddResponse.newBuilder().setLedgerId(r.getAddRequest().getLedgerId()).setEntryId(r.getAddRequest().getEntryId()).setStatus(BookkeeperProtocol.StatusCode.ETOOMANYREQUESTS);
                BookkeeperProtocol.Response.Builder response = BookkeeperProtocol.Response.newBuilder().setHeader(write.getHeader()).setStatus(addResponse.getStatus()).setAddResponse(addResponse);
                BookkeeperProtocol.Response resp = response.build();
                write.sendResponse(addResponse.getStatus(), resp, this.requestStats.getAddRequestStats());
            }
        }
    }

    private void processForceLedgerRequestV3(BookkeeperProtocol.Request r, Channel c) {
        ForceLedgerProcessorV3 forceLedger = new ForceLedgerProcessorV3(r, c, this);
        OrderedExecutor threadPool = RequestUtils.isHighPriority(r) ? this.highPriorityThreadPool : this.writeThreadPool;
        if (null == threadPool) {
            forceLedger.run();
        } else {
            try {
                threadPool.executeOrdered(r.getForceLedgerRequest().getLedgerId(), (SafeRunnable)forceLedger);
            }
            catch (RejectedExecutionException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failed to process request to force ledger {}. Too many pending requests", (Object)r.getForceLedgerRequest().getLedgerId());
                }
                BookkeeperProtocol.ForceLedgerResponse.Builder forceLedgerResponse = BookkeeperProtocol.ForceLedgerResponse.newBuilder().setLedgerId(r.getForceLedgerRequest().getLedgerId()).setStatus(BookkeeperProtocol.StatusCode.ETOOMANYREQUESTS);
                BookkeeperProtocol.Response.Builder response = BookkeeperProtocol.Response.newBuilder().setHeader(forceLedger.getHeader()).setStatus(forceLedgerResponse.getStatus()).setForceLedgerResponse(forceLedgerResponse);
                BookkeeperProtocol.Response resp = response.build();
                forceLedger.sendResponse(forceLedgerResponse.getStatus(), resp, this.requestStats.getForceLedgerRequestStats());
            }
        }
    }

    private void processReadRequestV3(BookkeeperProtocol.Request r, Channel c) {
        OrderedExecutor threadPool;
        ReadEntryProcessorV3 read;
        ExecutorService fenceThread;
        ExecutorService executorService = fenceThread = null == this.highPriorityThreadPool ? null : this.highPriorityThreadPool.chooseThread((Object)c);
        if (RequestUtils.isLongPollReadRequest(r.getReadRequest())) {
            ExecutorService lpThread = this.longPollThreadPool.chooseThread((Object)c);
            read = new LongPollReadEntryProcessorV3(r, c, this, fenceThread, lpThread, this.requestTimer);
            threadPool = this.longPollThreadPool;
        } else {
            read = new ReadEntryProcessorV3(r, c, this, fenceThread);
            boolean isHighPriority = RequestUtils.isHighPriority(r) || RequestUtils.hasFlag(r.getReadRequest(), BookkeeperProtocol.ReadRequest.Flag.FENCE_LEDGER);
            threadPool = isHighPriority ? this.highPriorityThreadPool : this.readThreadPool;
        }
        if (null == threadPool) {
            read.run();
        } else {
            try {
                threadPool.executeOrdered(r.getReadRequest().getLedgerId(), (SafeRunnable)read);
            }
            catch (RejectedExecutionException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failed to process request to read entry at {}:{}. Too many pending requests", (Object)r.getReadRequest().getLedgerId(), (Object)r.getReadRequest().getEntryId());
                }
                BookkeeperProtocol.ReadResponse.Builder readResponse = BookkeeperProtocol.ReadResponse.newBuilder().setLedgerId(r.getReadRequest().getLedgerId()).setEntryId(r.getReadRequest().getEntryId()).setStatus(BookkeeperProtocol.StatusCode.ETOOMANYREQUESTS);
                BookkeeperProtocol.Response.Builder response = BookkeeperProtocol.Response.newBuilder().setHeader(read.getHeader()).setStatus(readResponse.getStatus()).setReadResponse(readResponse);
                BookkeeperProtocol.Response resp = response.build();
                read.sendResponse(readResponse.getStatus(), resp, this.requestStats.getReadRequestStats());
            }
        }
    }

    private void processStartTLSRequestV3(final BookkeeperProtocol.Request r, final Channel c) {
        BookkeeperProtocol.Response.Builder response = BookkeeperProtocol.Response.newBuilder();
        BookkeeperProtocol.BKPacketHeader.Builder header = BookkeeperProtocol.BKPacketHeader.newBuilder();
        header.setVersion(BookkeeperProtocol.ProtocolVersion.VERSION_THREE);
        header.setOperation(r.getHeader().getOperation());
        header.setTxnId(r.getHeader().getTxnId());
        response.setHeader(header.build());
        if (this.shFactory == null) {
            LOG.error("Got StartTLS request but TLS not configured");
            response.setStatus(BookkeeperProtocol.StatusCode.EBADREQ);
            c.writeAndFlush((Object)response.build());
        } else {
            final SslHandler sslHandler = this.shFactory.newTLSHandler();
            c.pipeline().addFirst("tls", (ChannelHandler)sslHandler);
            response.setStatus(BookkeeperProtocol.StatusCode.EOK);
            BookkeeperProtocol.StartTLSResponse.Builder builder = BookkeeperProtocol.StartTLSResponse.newBuilder();
            response.setStartTLSResponse(builder.build());
            sslHandler.handshakeFuture().addListener((GenericFutureListener)new GenericFutureListener<Future<Channel>>(){

                public void operationComplete(Future<Channel> future) throws Exception {
                    AuthHandler.ServerSideHandler authHandler = (AuthHandler.ServerSideHandler)c.pipeline().get(AuthHandler.ServerSideHandler.class);
                    authHandler.authProvider.onProtocolUpgrade();
                    if (future.isSuccess()) {
                        LOG.info("Session is protected by: {}", (Object)sslHandler.engine().getSession().getCipherSuite());
                    } else {
                        LOG.error("TLS Handshake failure.", future.cause());
                        BookkeeperProtocol.Response.Builder errResponse = BookkeeperProtocol.Response.newBuilder().setHeader(r.getHeader()).setStatus(BookkeeperProtocol.StatusCode.EIO);
                        c.writeAndFlush((Object)errResponse.build());
                        if (BookieRequestProcessor.this.statsEnabled) {
                            BookieRequestProcessor.this.bkStats.getOpStats(2).incrementFailedOps();
                        }
                    }
                }
            });
            c.writeAndFlush((Object)response.build());
        }
    }

    private void processGetBookieInfoRequestV3(BookkeeperProtocol.Request r, Channel c) {
        GetBookieInfoProcessorV3 getBookieInfo = new GetBookieInfoProcessorV3(r, c, this);
        if (null == this.readThreadPool) {
            getBookieInfo.run();
        } else {
            this.readThreadPool.submit((Runnable)getBookieInfo);
        }
    }

    private void processGetListOfEntriesOfLedgerProcessorV3(BookkeeperProtocol.Request r, Channel c) {
        GetListOfEntriesOfLedgerProcessorV3 getListOfEntriesOfLedger = new GetListOfEntriesOfLedgerProcessorV3(r, c, this);
        if (null == this.readThreadPool) {
            getListOfEntriesOfLedger.run();
        } else {
            this.readThreadPool.submit((Runnable)getListOfEntriesOfLedger);
        }
    }

    private void processAddRequest(BookieProtocol.ParsedAddRequest r, Channel c) {
        WriteEntryProcessor write = WriteEntryProcessor.create(r, c, this);
        OrderedExecutor threadPool = r.isHighPriority() ? this.highPriorityThreadPool : this.writeThreadPool;
        if (null == threadPool) {
            write.run();
        } else {
            try {
                threadPool.executeOrdered(r.getLedgerId(), (SafeRunnable)write);
            }
            catch (RejectedExecutionException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failed to process request to add entry at {}:{}. Too many pending requests", (Object)r.ledgerId, (Object)r.entryId);
                }
                write.sendResponse(106, ResponseBuilder.buildErrorResponse(106, r), this.requestStats.getAddRequestStats());
            }
        }
    }

    private void processReadRequest(BookieProtocol.ReadRequest r, Channel c) {
        ExecutorService fenceThreadPool = null == this.highPriorityThreadPool ? null : this.highPriorityThreadPool.chooseThread((Object)c);
        ReadEntryProcessor read = ReadEntryProcessor.create(r, c, this, fenceThreadPool);
        OrderedExecutor threadPool = r.isHighPriority() || r.isFencing() ? this.highPriorityThreadPool : this.readThreadPool;
        if (null == threadPool) {
            read.run();
        } else {
            try {
                threadPool.executeOrdered(r.getLedgerId(), (SafeRunnable)read);
            }
            catch (RejectedExecutionException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failed to process request to read entry at {}:{}. Too many pending requests", (Object)r.ledgerId, (Object)r.entryId);
                }
                read.sendResponse(106, ResponseBuilder.buildErrorResponse(106, r), this.requestStats.getReadRequestStats());
            }
        }
    }

    public long getWaitTimeoutOnBackpressureMillis() {
        return this.waitTimeoutOnBackpressureMillis;
    }

    public void blacklistChannel(Channel channel) {
        this.blacklistedChannels.ifPresent(x -> x.put((Object)channel, (Object)true));
    }

    public void invalidateBlacklist(Channel channel) {
        this.blacklistedChannels.ifPresent(x -> x.invalidate((Object)channel));
    }

    public boolean isBlacklisted(Channel channel) {
        return this.blacklistedChannels.map(x -> (Boolean)x.getIfPresent((Object)channel)).orElse(false);
    }

    public void handleNonWritableChannel(Channel channel) {
        this.onResponseTimeout.accept(channel);
    }

    ServerConfiguration getServerCfg() {
        return this.serverCfg;
    }

    boolean isPreserveMdcForTaskExecution() {
        return this.preserveMdcForTaskExecution;
    }

    Bookie getBookie() {
        return this.bookie;
    }

    OrderedExecutor getReadThreadPool() {
        return this.readThreadPool;
    }

    OrderedExecutor getWriteThreadPool() {
        return this.writeThreadPool;
    }

    SecurityHandlerFactory getShFactory() {
        return this.shFactory;
    }

    OrderedExecutor getLongPollThreadPool() {
        return this.longPollThreadPool;
    }

    OrderedExecutor getHighPriorityThreadPool() {
        return this.highPriorityThreadPool;
    }

    HashedWheelTimer getRequestTimer() {
        return this.requestTimer;
    }

    BKStats getBkStats() {
        return this.bkStats;
    }

    boolean isStatsEnabled() {
        return this.statsEnabled;
    }

    RequestStats getRequestStats() {
        return this.requestStats;
    }

    Semaphore getAddsSemaphore() {
        return this.addsSemaphore;
    }

    Semaphore getReadsSemaphore() {
        return this.readsSemaphore;
    }

    Optional<Cache<Channel, Boolean>> getBlacklistedChannels() {
        return this.blacklistedChannels;
    }

    Consumer<Channel> getOnResponseTimeout() {
        return this.onResponseTimeout;
    }

    ByteBufAllocator getAllocator() {
        return this.allocator;
    }
}

