/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bigtop.manager.stack.core.utils;

import org.apache.bigtop.manager.stack.core.exception.StackException;

import lombok.extern.slf4j.Slf4j;

import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Properties;

@Slf4j
public class PropertiesUtils {

    /**
     * readProperties from file
     *
     * @param fileName fileName
     * @return Map<Object, Object>
     * @throws IOException
     */
    public static Map<Object, Object> readProperties(String fileName) {
        Properties properties = new Properties();
        try {
            properties.load(new FileReader(fileName, StandardCharsets.UTF_8));
        } catch (IOException e) {
            log.error("readProperties error", e);
            throw new StackException(e);
        }

        return properties;
    }

    /**
     * writeProperties to file
     *
     * @param fileName  fileName
     * @param configMap configMap
     * @throws IOException
     */
    public static void writeProperties(String fileName, Map<String, Object> configMap) {
        try {
            OutputStream os = new FileOutputStream(fileName);
            for (Map.Entry<String, Object> entry : configMap.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                String s = key + "=" + value + "\n";
                os.write(s.getBytes());
            }
            os.flush();
            os.close();
        } catch (Exception e) {
            log.error("writeProperties error", e);
        }
    }

    /**
     * writeProperties to file
     *
     * @param fileName   fileName
     * @param configList configList
     * @throws IOException
     */
    public static void writeProperties(String fileName, List<Map<String, Object>> configList) {
        Properties properties = new Properties();

        for (Map<String, Object> map : configList) {
            properties.setProperty(String.valueOf(map.get("name")), String.valueOf(map.get("value")));
        }

        log.info("writeProperties: {}", properties);
        try {
            properties.store(new FileWriter(fileName, StandardCharsets.UTF_8), "Generated by Apache BigTop Manager.");
        } catch (IOException e) {
            log.error("writeProperties error", e);
            throw new StackException(e);
        }
    }
}
