/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapred;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.mortbay.jetty.nio.SelectChannelConnector;

class JettyBugMonitor
extends Thread {
    private static final Log LOG = LogFactory.getLog(JettyBugMonitor.class);
    private static final ThreadMXBean threadBean = ManagementFactory.getThreadMXBean();
    private static final String CHECK_ENABLED_KEY = "mapred.tasktracker.jetty.cpu.check.enabled";
    private static final boolean CHECK_ENABLED_DEFAULT = true;
    static final String CHECK_INTERVAL_KEY = "mapred.tasktracker.jetty.cpu.check.interval";
    private static final long CHECK_INTERVAL_DEFAULT = 15000L;
    private long checkInterval;
    private static final String WARN_THRESHOLD_KEY = "mapred.tasktracker.jetty.cpu.threshold.warn";
    private static final float WARN_THRESHOLD_DEFAULT = 0.5f;
    private float warnThreshold;
    private static final String FATAL_THRESHOLD_KEY = "mapred.tasktracker.jetty.cpu.threshold.fatal";
    private static final float FATAL_THRESHOLD_DEFAULT = 0.9f;
    private float fatalThreshold;
    private boolean stopping = false;

    public static JettyBugMonitor create(Configuration conf) {
        if (!conf.getBoolean(CHECK_ENABLED_KEY, true)) {
            return null;
        }
        if (!threadBean.isThreadCpuTimeSupported()) {
            LOG.warn("Not starting monitor for Jetty bug since thread CPU time measurement is not supported by this JVM");
            return null;
        }
        return new JettyBugMonitor(conf);
    }

    JettyBugMonitor(Configuration conf) {
        this.setName("Monitor for Jetty bugs");
        this.setDaemon(true);
        this.warnThreshold = conf.getFloat(WARN_THRESHOLD_KEY, 0.5f);
        this.fatalThreshold = conf.getFloat(FATAL_THRESHOLD_KEY, 0.9f);
        this.checkInterval = conf.getLong(CHECK_INTERVAL_KEY, 15000L);
    }

    @Override
    public void run() {
        try {
            this.doRun();
        }
        catch (InterruptedException ie) {
            if (!this.stopping) {
                LOG.warn("Jetty monitor unexpectedly interrupted", ie);
            }
        }
        catch (Throwable t) {
            LOG.error("Jetty bug monitor failed", t);
        }
        LOG.debug("JettyBugMonitor shutting down");
    }

    private void doRun() throws InterruptedException {
        List<Long> tids = this.waitForJettyThreads();
        if (tids.isEmpty()) {
            LOG.warn("Could not locate Jetty selector threads");
            return;
        }
        try {
            while (true) {
                this.monitorThreads(tids);
            }
        }
        catch (ThreadNotRunningException tnre) {
            return;
        }
    }

    private void monitorThreads(List<Long> tids) throws InterruptedException, ThreadNotRunningException {
        long timeBefore = System.nanoTime();
        long usageBefore = JettyBugMonitor.getCpuUsageNanos(tids);
        while (true) {
            Thread.sleep(this.checkInterval);
            long usageAfter = JettyBugMonitor.getCpuUsageNanos(tids);
            long timeAfter = System.nanoTime();
            long delta = usageAfter - usageBefore;
            double percentCpu = (double)delta / (double)(timeAfter - timeBefore);
            String msg = String.format("Jetty CPU usage: %.1f%%", percentCpu * 100.0);
            if (percentCpu > (double)this.fatalThreshold) {
                LOG.fatal("************************************************************\n" + msg + ". This is greater than the fatal threshold " + FATAL_THRESHOLD_KEY + ". Aborting JVM.\n" + "************************************************************");
                this.doAbort();
            } else if (percentCpu > (double)this.warnThreshold) {
                LOG.warn(msg);
            } else if (LOG.isDebugEnabled()) {
                LOG.debug(msg);
            }
            usageBefore = usageAfter;
            timeBefore = timeAfter;
        }
    }

    protected void doAbort() {
        Runtime.getRuntime().exit(1);
    }

    protected List<Long> waitForJettyThreads() throws InterruptedException {
        List<Long> tids = new ArrayList<Long>();
        int i = 0;
        while (tids.isEmpty() & i++ < 30) {
            Thread.sleep(1000L);
            tids = JettyBugMonitor.getJettyThreadIds();
        }
        return tids;
    }

    private static long getCpuUsageNanos(List<Long> tids) throws ThreadNotRunningException {
        long total = 0L;
        for (long tid : tids) {
            long time = threadBean.getThreadCpuTime(tid);
            if (time == -1L) {
                LOG.warn("Unable to monitor CPU usage for thread: " + tid);
                throw new ThreadNotRunningException();
            }
            total += time;
        }
        return total;
    }

    static List<Long> getJettyThreadIds() {
        long[] threadIds;
        ArrayList<Long> tids = new ArrayList<Long>();
        for (long tid : threadIds = threadBean.getAllThreadIds()) {
            if (!JettyBugMonitor.isJettySelectorThread(tid)) continue;
            tids.add(tid);
        }
        return tids;
    }

    private static boolean isJettySelectorThread(long tid) {
        ThreadInfo info = threadBean.getThreadInfo(tid, 20);
        for (StackTraceElement stack : info.getStackTrace()) {
            if (!SelectChannelConnector.class.getName().equals(stack.getClassName())) continue;
            LOG.debug("Thread #" + tid + " (" + info.getThreadName() + ") " + "is a Jetty selector thread.");
            return true;
        }
        LOG.debug("Thread #" + tid + " (" + info.getThreadName() + ") " + "is not a jetty thread");
        return false;
    }

    public void shutdown() {
        this.stopping = true;
        this.interrupt();
    }

    private static class ThreadNotRunningException
    extends Exception {
        private static final long serialVersionUID = 1L;

        private ThreadNotRunningException() {
        }
    }
}

