/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.data;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.UnsignedBytes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedSet;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.rfile.RFile;
import org.apache.accumulo.core.clientImpl.bulk.BulkImport;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.BinaryComparable;
import org.apache.hadoop.io.Text;

public class LoadPlan {
    private final List<Destination> destinations;
    private static final Gson gson = new GsonBuilder().disableHtmlEscaping().disableJdkUnsafe().serializeNulls().create();
    private static final Set<String> EXPECTED_FIELDS = Arrays.stream(JsonAll.class.getDeclaredFields()).map(Field::getName).collect(Collectors.toSet());
    private static final Set<String> EXPECTED_DEST_FIELDS = Arrays.stream(JsonDestination.class.getDeclaredFields()).map(Field::getName).collect(Collectors.toSet());
    private static final TableId FAKE_ID = TableId.of("999");

    private static byte[] copy(byte[] data) {
        return data == null ? null : Arrays.copyOf(data, data.length);
    }

    private static byte[] copy(Text data) {
        return data == null ? null : data.copyBytes();
    }

    private static byte[] copy(CharSequence data) {
        return data == null ? null : data.toString().getBytes(StandardCharsets.UTF_8);
    }

    @SuppressFBWarnings(value={"PATH_TRAVERSAL_IN"}, justification="this code is validating the input")
    private static String checkFileName(String fileName) {
        Preconditions.checkArgument((Paths.get(fileName, new String[0]).getNameCount() == 1 ? 1 : 0) != 0, (String)"Expected only filename, but got %s", (Object)fileName);
        return fileName;
    }

    private LoadPlan(List<Destination> destinations) {
        this.destinations = destinations;
    }

    public Collection<Destination> getDestinations() {
        return this.destinations;
    }

    public static Builder builder() {
        return new Builder(){
            final ImmutableList.Builder<Destination> fmb = ImmutableList.builder();

            @Override
            public Builder loadFileTo(String fileName, RangeType rangeType, Text startRow, Text endRow) {
                this.fmb.add((Object)new Destination(fileName, rangeType, LoadPlan.copy(startRow), LoadPlan.copy(endRow)));
                return this;
            }

            @Override
            public Builder loadFileTo(String fileName, RangeType rangeType, byte[] startRow, byte[] endRow) {
                this.fmb.add((Object)new Destination(fileName, rangeType, LoadPlan.copy(startRow), LoadPlan.copy(endRow)));
                return this;
            }

            @Override
            public Builder loadFileTo(String fileName, RangeType rangeType, CharSequence startRow, CharSequence endRow) {
                this.fmb.add((Object)new Destination(fileName, rangeType, LoadPlan.copy(startRow), LoadPlan.copy(endRow)));
                return this;
            }

            @Override
            public Builder addPlan(LoadPlan plan) {
                this.fmb.addAll(plan.getDestinations());
                return this;
            }

            @Override
            public LoadPlan build() {
                return new LoadPlan((List<Destination>)this.fmb.build());
            }
        };
    }

    public String toJson() {
        return gson.toJson((Object)new JsonAll(this.destinations));
    }

    public static LoadPlan fromJson(String json) {
        try {
            Objects.requireNonNull(json);
            Preconditions.checkArgument((!json.isBlank() ? 1 : 0) != 0, (Object)"Empty json is not accepted.");
            JsonObject jsonObj = (JsonObject)gson.fromJson(json, JsonObject.class);
            Preconditions.checkArgument((boolean)jsonObj.keySet().equals(EXPECTED_FIELDS), (String)"Expected fields %s and saw %s", EXPECTED_FIELDS, (Object)jsonObj.keySet());
            Preconditions.checkArgument((boolean)(jsonObj.get("destinations") instanceof JsonArray), (Object)"Expected value of destinations field to be array");
            JsonArray destinations = jsonObj.getAsJsonArray("destinations");
            destinations.forEach(dest -> {
                Set keySet = dest.getAsJsonObject().keySet();
                Preconditions.checkArgument((boolean)keySet.equals(EXPECTED_DEST_FIELDS), (String)"Expected fields %s and saw %s", EXPECTED_DEST_FIELDS, (Object)keySet);
            });
            List<Destination> dests = ((JsonAll)LoadPlan.gson.fromJson((String)json, JsonAll.class)).destinations.stream().map(JsonDestination::toDestination).collect(Collectors.toUnmodifiableList());
            return new LoadPlan(dests);
        }
        catch (IllegalArgumentException | NullPointerException e) {
            throw e;
        }
        catch (RuntimeException e) {
            throw new IllegalArgumentException(e);
        }
    }

    public static LoadPlan compute(URI file, SplitResolver splitResolver) throws IOException {
        return LoadPlan.compute(file, Map.of(), splitResolver);
    }

    public static LoadPlan compute(URI file, Map<String, String> properties, SplitResolver splitResolver) throws IOException {
        try (Scanner scanner = RFile.newScanner().from(file.toString()).withoutSystemIterators().withTableProperties(properties).withIndexCache(10000000L).build();){
            BulkImport.NextRowFunction nextRowFunction = row -> {
                scanner.setRange(new Range(row, null));
                Iterator<Map.Entry<Key, Value>> iter = scanner.iterator();
                if (iter.hasNext()) {
                    return iter.next().getKey().getRow();
                }
                return null;
            };
            Function<Text, KeyExtent> rowToExtentResolver = row -> {
                TableSplits tabletRange = splitResolver.apply((Text)row);
                KeyExtent extent = new KeyExtent(FAKE_ID, tabletRange.endRow, tabletRange.prevRow);
                Preconditions.checkState((boolean)extent.contains((BinaryComparable)row), (String)"%s does not contain %s", (Object)tabletRange, (Object)row);
                return extent;
            };
            List<KeyExtent> overlapping = BulkImport.findOverlappingTablets(rowToExtentResolver, nextRowFunction);
            Path path = new Path(file);
            Builder builder = LoadPlan.builder();
            for (KeyExtent extent : overlapping) {
                builder.loadFileTo(path.getName(), RangeType.TABLE, extent.prevEndRow(), extent.endRow());
            }
            LoadPlan loadPlan = builder.build();
            return loadPlan;
        }
    }

    private static final class JsonAll {
        List<JsonDestination> destinations;

        JsonAll() {
        }

        JsonAll(List<Destination> destinations) {
            this.destinations = destinations.stream().map(JsonDestination::new).collect(Collectors.toList());
        }
    }

    public static interface SplitResolver
    extends Function<Text, TableSplits> {
        public static SplitResolver from(SortedSet<Text> splits) {
            return row -> {
                SortedSet<Text> headSet = splits.headSet(row);
                Text prevRow = headSet.isEmpty() ? null : headSet.last();
                SortedSet<Text> tailSet = splits.tailSet(row);
                Text endRow = tailSet.isEmpty() ? null : tailSet.first();
                return new TableSplits(prevRow, endRow);
            };
        }

        @Override
        public TableSplits apply(Text var1);
    }

    public static interface Builder {
        public Builder loadFileTo(String var1, RangeType var2, Text var3, Text var4);

        public Builder loadFileTo(String var1, RangeType var2, byte[] var3, byte[] var4);

        public Builder loadFileTo(String var1, RangeType var2, CharSequence var3, CharSequence var4);

        public Builder addPlan(LoadPlan var1);

        public LoadPlan build();
    }

    public static enum RangeType {
        TABLE,
        FILE;

    }

    public static class TableSplits {
        private final Text prevRow;
        private final Text endRow;

        public TableSplits(Text prevRow, Text endRow) {
            Preconditions.checkArgument((prevRow == null || endRow == null || prevRow.compareTo((BinaryComparable)endRow) < 0 ? 1 : 0) != 0, (String)"%s >= %s", (Object)prevRow, (Object)endRow);
            this.prevRow = prevRow;
            this.endRow = endRow;
        }

        public Text getPrevRow() {
            return this.prevRow;
        }

        public Text getEndRow() {
            return this.endRow;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TableSplits that = (TableSplits)o;
            return Objects.equals(this.prevRow, that.prevRow) && Objects.equals(this.endRow, that.endRow);
        }

        public int hashCode() {
            return Objects.hash(this.prevRow, this.endRow);
        }

        public String toString() {
            return "(" + String.valueOf(this.prevRow) + "," + String.valueOf(this.endRow) + "]";
        }
    }

    private static class JsonDestination {
        String fileName;
        String startRow;
        String endRow;
        RangeType rangeType;

        JsonDestination() {
        }

        JsonDestination(Destination destination) {
            this.fileName = destination.getFileName();
            this.startRow = destination.getStartRow() == null ? null : Base64.getUrlEncoder().encodeToString(destination.getStartRow());
            this.endRow = destination.getEndRow() == null ? null : Base64.getUrlEncoder().encodeToString(destination.getEndRow());
            this.rangeType = destination.getRangeType();
        }

        Destination toDestination() {
            return new Destination(this.fileName, this.rangeType, this.startRow == null ? null : Base64.getUrlDecoder().decode(this.startRow), this.endRow == null ? null : Base64.getUrlDecoder().decode(this.endRow));
        }
    }

    public static class Destination {
        private final String fileName;
        private final byte[] startRow;
        private final byte[] endRow;
        private final RangeType rangeType;

        private byte[] checkRow(RangeType type, byte[] row) {
            if (type == RangeType.FILE && row == null) {
                throw new IllegalArgumentException("Row can not be null when range type is " + String.valueOf((Object)RangeType.FILE));
            }
            return row;
        }

        private Destination(String fileName, RangeType rangeType, byte[] startRow, byte[] endRow) {
            this.fileName = LoadPlan.checkFileName(fileName);
            this.rangeType = rangeType;
            this.startRow = this.checkRow(rangeType, startRow);
            this.endRow = this.checkRow(rangeType, endRow);
            if (rangeType == RangeType.FILE) {
                if (UnsignedBytes.lexicographicalComparator().compare(startRow, endRow) > 0) {
                    String srs = new String(startRow, StandardCharsets.UTF_8);
                    String ers = new String(endRow, StandardCharsets.UTF_8);
                    throw new IllegalArgumentException("Start row is greater than end row : " + srs + " " + ers);
                }
            } else if (rangeType == RangeType.TABLE) {
                if (startRow != null && endRow != null && UnsignedBytes.lexicographicalComparator().compare(startRow, endRow) >= 0) {
                    String srs = new String(startRow, StandardCharsets.UTF_8);
                    String ers = new String(endRow, StandardCharsets.UTF_8);
                    throw new IllegalArgumentException("Start row is greater than or equal to end row : " + srs + " " + ers);
                }
            } else {
                throw new IllegalStateException("Unknown range type : " + String.valueOf((Object)rangeType));
            }
        }

        public String getFileName() {
            return this.fileName;
        }

        public byte[] getStartRow() {
            return LoadPlan.copy(this.startRow);
        }

        public byte[] getEndRow() {
            return LoadPlan.copy(this.endRow);
        }

        public RangeType getRangeType() {
            return this.rangeType;
        }
    }
}

