/*
 * Copyright (c) 1997 Erez Zadok
 * Copyright (c) 1990 Jan-Simon Pendry
 * Copyright (c) 1990 Imperial College of Science, Technology & Medicine
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry at Imperial College, London.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      %W% (Berkeley) %G%
 *
 * $Id: mount_linux.c,v 5.2.1.1 90/10/21 22:30:59 jsp Exp $
 */

/*
 * Linux mount helper
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <am_defs.h>
#include <amu.h>

struct opt_map {
  const char *opt;		/* option name */
  int inv;			/* true if flag value should be inverted */
  int mask;			/* flag mask value */
};

const struct opt_map opt_map[] =
{
  {"defaults", 0, 0},
  {MNTTAB_OPT_RO, 0, MNT2_GEN_OPT_RDONLY},
  {MNTTAB_OPT_RW, 1, MNT2_GEN_OPT_RDONLY},
  {MNTTAB_OPT_EXEC, 1, MNT2_GEN_OPT_NOEXEC},
  {MNTTAB_OPT_NOEXEC, 0, MNT2_GEN_OPT_NOEXEC},
  {MNTTAB_OPT_SUID, 1, MNT2_GEN_OPT_NOSUID},
  {MNTTAB_OPT_NOSUID, 0, MNT2_GEN_OPT_NOSUID},
#ifdef MNT2_GEN_OPT_NODEV
  {MNTTAB_OPT_DEV, 1, MNT2_GEN_OPT_NODEV},
  {MNTTAB_OPT_NODEV, 0, MNT2_GEN_OPT_NODEV},
#endif /* MNT2_GEN_OPT_NODEV */
#ifdef MNT2_GEN_OPT_SYNC
  {MNTTAB_OPT_SYNC, 0, MNT2_GEN_OPT_SYNC},
  {MNTTAB_OPT_ASYNC, 1, MNT2_GEN_OPT_SYNC},
#endif /* MNT2_GEN_OPT_SYNC */
#ifdef MNT2_GEN_OPT_NOSUB
  {MNTTAB_OPT_SUB, 1, MNT2_GEN_OPT_NOSUB},
  {MNTTAB_OPT_NOSUB, 0, MNT2_GEN_OPT_NOSUB},
#endif /* MNT2_GEN_OPT_NOSUB */
  {NULL, 0, 0}
};


static inline void
parse_opt(const char *opt, int *mask, char *extra_opts)
{
  const struct opt_map *om;

  for (om = opt_map; om->opt != NULL; ++om)
    if (STREQ(opt, om->opt)) {
      if (om->inv)
	*mask &= ~om->mask;
      else
	*mask |= om->mask;
      return;
    }
  strcat(extra_opts, ",");
  strcat(extra_opts, opt);
}


/*
 * Take -o options list and compute 4th and 5th args to mount(2).  Flags
 * gets the standard options and extra_opts anything we don't recognize.
 */
static void
parse_opts(char *opts, int *flags, char **extra_opts, int *noauto)
{
  char *opt;
  int readonly = 0, readwrite = 0;

  *noauto = 0;
  if (opts != NULL) {
    *extra_opts = xmalloc(strlen(opts) + 2);
    **extra_opts = '\0';

    for (opt = strtok(opts, ",");
	 opt != NULL;
	 opt = strtok(NULL, ","))
      if (!((readwrite = (STREQ(opt, MNTTAB_OPT_RW))) ||
	    (readonly = (STREQ(opt, MNTTAB_OPT_RO))) ||
	    (*noauto = (STREQ(opt, MNTTAB_OPT_NOAUTO))) ||
	    (strncmp(opt, "type", 4) == 0)))
	parse_opt(opt, flags, *extra_opts);
  }
  if (readonly)
    *flags |= MNT2_GEN_OPT_RDONLY;
  if (readwrite)
    *flags &= ~MNT2_GEN_OPT_RDONLY;
}


int
mount_linux(MTYPE_TYPE type, mntent_t *mnt, int flags, caddr_t data)
{
  char *extra_opts = NULL;
  char *tmp_opts = NULL;
  char *sub_type = NULL;
  int noauto = 0;
  int errorcode;
  nfs_args_t *mnt_data = (nfs_args_t *) data;

  if (mnt->mnt_opts && STREQ(mnt->mnt_opts, "defaults"))
    mnt->mnt_opts = NULL;

  if (type == NULL)
    type = index(mnt->mnt_fsname, ':') ? MOUNT_TYPE_NFS : MOUNT_TYPE_UFS;

  if (STREQ(type, MOUNT_TYPE_NFS)) {
    /* Fake some values for linux */
    mnt_data->version = NFS_MOUNT_VERSION;
    if (!mnt_data->timeo)
      mnt_data->timeo = 7;
    if (!mnt_data->retrans)
      mnt_data->retrans = 3;

    /* These are the only two reliable values currently */
    if (!mnt_data->rsize)
      mnt_data->rsize = 1024;
    if (!mnt_data->wsize)
      mnt_data->wsize = 1024;
    if (((mnt_data->fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) ||
	(bindresvport(mnt_data->fd, 0) < 0) ||
	(connect(mnt_data->fd, (struct sockaddr *) &mnt_data->addr,
		 sizeof(mnt_data->addr)) < 0)) {
      perror("Can't create socket for kernel");
      errorcode = 1;
      goto fail;
    }

#ifdef DEBUG
    plog(XLOG_INFO, "linux mount: type %s\n", type);
    plog(XLOG_INFO, "linux mount: version %d\n", mnt_data->version);
    plog(XLOG_INFO, "linux mount: fd %d\n", mnt_data->fd);
    plog(XLOG_INFO, "linux mount: hostname %s\n", \
	 inet_ntoa(mnt_data->addr.sin_addr));
    plog(XLOG_INFO, "linux mount: port %d\n", \
	 htons(mnt_data->addr.sin_port));
#endif /* DEBUG */

  } else {

    /* Non nfs mounts */
    if ((sub_type = hasmntopt(mnt, "type")) &&
	(sub_type = index(sub_type, '=')) &&
	(sub_type = strdup(sub_type + 1))) {
      type = strpbrk(sub_type, ",:;\n\t");
      if (type == NULL)
	type = MOUNT_TYPE_UFS;
      else {
	*type = '\0';
	type = sub_type;
      }
    }

    if (!hasmntopt(mnt, "type"))
      mnt->mnt_type = MOUNT_TYPE_UFS;
    /* We only parse opts if non nfs drive */
    parse_opts(tmp_opts = strdup(mnt->mnt_opts), &flags, &extra_opts, &noauto);

#ifdef DEBUG
    plog(XLOG_INFO, "linux mount: type %s\n", type);
    plog(XLOG_INFO, "linux mount: xopts %s\n", extra_opts);
#endif /* DEBUG */
  }

#ifdef DEBUG
  plog(XLOG_INFO, "linux mount: fsname %s\n", mnt->mnt_fsname);
  plog(XLOG_INFO, "linux mount: type (mntent) %s\n", mnt->mnt_type);
  plog(XLOG_INFO, "linux mount: opts %s\n", mnt->mnt_opts);
  plog(XLOG_INFO, "linux mount: dir %s\n", mnt->mnt_dir);
#endif /* DEBUG */

/*
 * If we have an nfs mount, the 5th argument to system mount() must be the
 * nfs_mount_data structure, otherwise it is the return from parse_opts()
 * XXX: fix this ugly 0xC0ED0000 in the call below.
 */
  errorcode = mount(mnt->mnt_fsname, mnt->mnt_dir, type,
		    MS_MGC_VAL | flags,
		    STREQ(type, MOUNT_TYPE_NFS) ? (char *) mnt_data : extra_opts);

/*
 * If we failed, (i.e. errorcode != 0), then close the socket if its is open
 */
  if (errorcode)
    if (mnt_data->fd != -1)
      close(mnt_data->fd);

/*
 * Free all allocated space and return errorcode.
 */
fail:
  if (extra_opts != NULL)
    free(extra_opts);
  if (tmp_opts != NULL)
    free(tmp_opts);
  if (sub_type != NULL)
    free(sub_type);
  return (errorcode);
}
