/*
 * Copyright (c) 1997 Erez Zadok
 * Copyright (c) 1990 Jan-Simon Pendry
 * Copyright (c) 1990 Imperial College of Science, Technology & Medicine
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry at Imperial College, London.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      %W% (Berkeley) %G%
 *
 * $Id: ops_autofs.c,v 5.2.2.3 1992/08/02 10:42:21 jsp Exp $
 *
 */

/*
 * Automounter filesystem
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <am_defs.h>
#include <amd.h>

/*
 * CLUDGE: wrap whole file in HAVE_FS_AUTOFS, becasue
 * not all systems with an automounter file system are supported
 * by am-utils yet...
 */

#ifdef HAVE_FS_AUTOFS

/*
 * MACROS:
 */
#ifndef AUTOFS_NULL
# define AUTOFS_NULL	((u_long)0)
#endif /* not AUTOFS_NULL */

/*
 * VARIABLES:
 */

/* forward declarations */
static int mount_autofs(char *dir, char *opts);
static int autofs_mount_1_svc(struct mntrequest *mr, struct mntres *result, struct authunix_parms *cred);
static int autofs_unmount_1_svc(struct umntrequest *ur, struct umntres *result, struct authunix_parms *cred);

/* externam declarations */
extern bool_t xdr_mntrequest(XDR *, mntrequest *);
extern bool_t xdr_mntres(XDR *, mntres *);
extern bool_t xdr_umntrequest(XDR *, umntrequest *);
extern bool_t xdr_umntres(XDR *, umntres *);


/*
 * Mount the top-level using autofs
 */
int
autofs_mount(am_node *mp)
{
  mntfs *mf = mp->am_mnt;
  struct stat stb;
  char opts[256], preopts[256];
  int error;
  char *mnttype;

  /*
   * Mounting the automounter.
   * Make sure the mount directory exists, construct
   * the mount options and call the mount_autofs routine.
   */

  if (stat(mp->am_path, &stb) < 0) {
    return errno;
  } else if ((stb.st_mode & S_IFMT) != S_IFDIR) {
    plog(XLOG_WARNING, "%s is not a directory", mp->am_path);
    return ENOTDIR;
  }
  if (mf->mf_ops == &autofs_ops)
    mnttype = "indirect";
  else if (mf->mf_ops == &dfs_ops)
    mnttype = "direct";
#ifdef HAVE_AM_FS_UNION
  else if (mf->mf_ops == &union_ops)
    mnttype = "union";
#endif /* HAVE_AM_FS_UNION */
  else
    mnttype = "auto";

  /*
   * Construct some mount options:
   *
   * Tack on magic map=<mapname> option in mtab to emulate
   * SunOS automounter behavior.
   */
  preopts[0] = '\0';
#ifdef MNTTAB_OPT_INTR
  strcat(preopts, MNTTAB_OPT_INTR);
  strcat(preopts, ",");
#endif /* MNTTAB_OPT_INTR */
#ifdef MNTTAB_OPT_IGNORE
  strcat(preopts, MNTTAB_OPT_IGNORE);
  strcat(preopts, ",");
#endif /* MNTTAB_OPT_IGNORE */
  sprintf(opts, "%s%s,%s=%d,%s=%d,%s=%d,%s,map=%s",
	  preopts,
	  MNTTAB_OPT_RW,
	  MNTTAB_OPT_PORT, nfs_port,
	  MNTTAB_OPT_TIMEO, gopt.afs_timeo,
	  MNTTAB_OPT_RETRANS, gopt.afs_retrans,
	  mnttype, mf->mf_info);

  /* now do the mount */
  error = mount_autofs(mf->mf_mount, opts);
  if (error) {
    errno = error;
    plog(XLOG_FATAL, "mount_autofs: %m");
    return error;
  }
  return 0;
}


void
autofs_mounted(mntfs *mf)
{
  afs_mkcacheref(mf);
}


/*
 * Unmount a top-level automount node
 */
int
autofs_umount(am_node *mp)
{
  int error;
  struct stat stb;

  /*
   * The lstat is needed if this mount is type=direct.  When that happens,
   * the kernel cache gets confused between the underlying type (dir) and
   * the mounted type (link) and so needs to be re-synced before the
   * unmount.  This is all because the unmount system call follows links and
   * so can't actually unmount a link (stupid!).  It was noted that doing an
   * ls -ld of the mount point to see why things were not working actually
   * fixed the problem - so simulate an ls -ld here.
   */
  if (lstat(mp->am_path, &stb) < 0) {
#ifdef DEBUG
    dlog("lstat(%s): %m", mp->am_path);
#endif /* DEBUG */
  }
  error = UMOUNT_FS(mp->am_path, mnttab_file_name);
  if (error == EBUSY && mp->am_flags & AMF_AUTOFS) {
    plog(XLOG_WARNING, "autofs_unmount of %s busy (autofs). exit", mp->am_path);
    error = 0;			/* fake unmount was ok */
  }
  return error;
}


/*
 * Mount an automounter directory.
 * The automounter is connected into the system
 * as a user-level NFS server.  mount_autofs constructs
 * the necessary NFS parameters to be given to the
 * kernel so that it will talk back to us.
 */
static int
mount_autofs(char *dir, char *opts)
{
  char fs_hostname[MAXHOSTNAMELEN + MAXPATHLEN + 1];
  char *map_opt, buf[MAXHOSTNAMELEN];
  int retry, error, flags;
  struct utsname utsname;
  mntent_t mnt;
  autofs_args_t autofs_args;
  MTYPE_TYPE type = MOUNT_TYPE_AUTOFS;

  memset((voidp) &autofs_args, 0, sizeof(autofs_args)); /* Paranoid */

  memset((voidp) &mnt, 0, sizeof(mnt));
  mnt.mnt_dir = dir;
  mnt.mnt_fsname = pid_fsname;
  mnt.mnt_opts = opts;
  mnt.mnt_type = type;

  retry = hasmntval(&mnt, "retry");
  if (retry <= 0)
    retry = 2;			/* XXX */

  /*
   * SET MOUNT ARGS
   */
  if (uname(&utsname) < 0) {
    strcpy(buf, "localhost.autofs");
  } else {
    strcpy(buf, utsname.nodename);
    strcat(buf, ".autofs");
  }
#ifdef HAVE_FIELD_AUTOFS_ARGS_T_ADDR
  autofs_args.addr.buf = buf;
  autofs_args.addr.len = strlen(autofs_args.addr.buf);
  autofs_args.addr.maxlen = autofs_args.addr.len;
#endif /* HAVE_FIELD_AUTOFS_ARGS_T_ADDR */

  autofs_args.path = dir;
  autofs_args.opts = opts;

  map_opt = hasmntopt(&mnt, "map");
  if (map_opt) {
    map_opt += sizeof("map="); /* skip the "map=" */
    if (map_opt == NULL) {
      plog(XLOG_WARNING, "map= has a null map name. reset to amd.unknown");
      map_opt = "amd.unknown";
    }
  }
  autofs_args.map = map_opt;

  /* XXX: these I set arbitrarily... */
  autofs_args.mount_to = 300;
  autofs_args.rpc_to = 60;
  autofs_args.direct = 0;

  /*
   * Make a ``hostname'' string for the kernel
   */
  sprintf(fs_hostname, "pid%ld@%s:%s", foreground ? mypid : getppid(),
	  hostname, dir);

  /*
   * Most kernels have a name length restriction.
   */
  if (strlen(fs_hostname) >= MAXHOSTNAMELEN)
    strcpy(fs_hostname + MAXHOSTNAMELEN - 3, "..");

  /*
   * Finally we can compute the mount flags set above.
   */
  flags = compute_mount_flags(&mnt);

  /*
   * This is it!  Here we try to mount amd on its mount points.
   */
  error = mount_fs(&mnt, flags, (caddr_t) &autofs_args, retry, type, 0, NULL, mnttab_file_name);
  return error;
}


/****************************************************************************/
/* autofs program dispatcher */
void
autofs_program_1(struct svc_req *rqstp, SVCXPRT *transp)
{
  int ret;
  union {
    mntrequest autofs_mount_1_arg;
    umntrequest autofs_umount_1_arg;
  } argument;
  union {
    mntres mount_res;
    umntres umount_res;
  } result;

  bool_t (*xdr_argument)(), (*xdr_result)();
  int (*local)();

  switch (rqstp->rq_proc) {

  case AUTOFS_NULL:
    svc_sendreply(transp,
		  (XDRPROC_T_TYPE) xdr_void,
		  (SVC_IN_ARG_TYPE) NULL);
    return;

  case AUTOFS_MOUNT:
    xdr_argument = xdr_mntrequest;
    xdr_result = xdr_mntres;
    local = (int (*)()) autofs_mount_1_svc;
    break;

  case AUTOFS_UNMOUNT:
    xdr_argument = xdr_umntrequest;
    xdr_result = xdr_umntres;
    local = (int (*)()) autofs_unmount_1_svc;
    break;

  default:
    svcerr_noproc(transp);
    return;
  }

  memset((char *) &argument, 0, sizeof(argument));
  if (!svc_getargs(transp,
		   (XDRPROC_T_TYPE) xdr_argument,
		   (SVC_IN_ARG_TYPE) &argument)) {
    plog(XLOG_ERROR,
	 "AUTOFS xdr decode failed for %d %d %d",
	 rqstp->rq_prog, rqstp->rq_vers, rqstp->rq_proc);
    svcerr_decode(transp);
    return;
  }

  ret = (*local) (&argument, &result, rqstp);
  if (!svc_sendreply(transp,
		     (XDRPROC_T_TYPE) xdr_result,
		     (SVC_IN_ARG_TYPE) &result)) {
    svcerr_systemerr(transp);
  }

  if (!svc_freeargs(transp,
		    (XDRPROC_T_TYPE) xdr_argument,
		    (SVC_IN_ARG_TYPE) &argument)) {
    plog(XLOG_FATAL, "unable to free rpc arguments in autofs_program_1");
    going_down(1);
  }
}


static int
autofs_mount_1_svc(struct mntrequest *mr, struct mntres *result, struct authunix_parms *cred)
{
  int err = 0;
  am_node *anp, *anp2;

  plog(XLOG_INFO, "XXX: autofs_mount_1_svc: %s:%s:%s:%s",
       mr->map, mr->name, mr->opts, mr->path);

  /* look for map (eg. "/home") */
  anp = find_ap(mr->path);
  if (!anp) {
    plog(XLOG_ERROR, "map %s not found", mr->path);
    err = ENOENT;
    goto out;
  }
  /* turn on autofs in map flags */
  if (!(anp->am_flags & AMF_AUTOFS)) {
    plog(XLOG_INFO, "turning on AMF_AUTOFS for node %s", mr->path);
    anp->am_flags |= AMF_AUTOFS;
  }

  /*
   * Look for (and create if needed) the new node.
   *
   * If an error occurred, return it.  If a -1 was returned, that indicates
   * that a mount is in progress, so sleep a while (while the backgrounded
   * mount is happening), and then signal the autofs to retry the mount.
   *
   * There's something I don't understand.  I was thinking that this code
   * here is the one which will succeed eventually and will send an RPC
   * reply to the kernel, but apparently that happens somewhere else, not
   * here.  It works though, just that I don't know how.  Arg. -Erez.
   * */
  err = 0;
  anp2 = autofs_lookuppn(anp, mr->name, &err, VLOOK_CREATE);
  if (!anp2) {
    if (err == -1) {		/* then tell autofs to retry */
      sleep(1);
      err = EAGAIN;
    }
    goto out;
  }

out:
  result->status = err;
  return err;
}


static int
autofs_unmount_1_svc(struct umntrequest *ur, struct umntres *result, struct authunix_parms *cred)
{
  int err = 0;

#ifdef HAVE_FIELD_UMNTREQUEST_RDEVID
  plog(XLOG_INFO, "XXX: autofs_unmount_1_svc: %d:%u:%lu:0x%x",
       ur->isdirect, ur->devid, ur->rdevid, ur->next);
#else /* HAVE_FIELD_UMNTREQUEST_RDEVID */
  plog(XLOG_INFO, "XXX: autofs_unmount_1_svc: %d:%u:0x%x",
       ur->isdirect, ur->devid, ur->next);
#endif /* HAVE_FIELD_UMNTREQUEST_RDEVID */

  err = EINVAL;			/* XXX: not implemented yet */
  goto out;

out:
  result->status = err;
  return err;
}

#endif /* HAVE_FS_AUTOFS */
