
/******************************************************
 *                                                    *
 * FPAC project.            FPAC NODE                 *
 *                                                    *
 * Parts of code from different sources of ax25-utils *
 *                                                    *
 * F6FBB 05-1997                                      *
 *                                                    *
 ******************************************************/

/******************************************************
 * 12/05/97 1.00 F6FBB First draft !
 *
 ******************************************************/
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <ctype.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>

#include <linux/netdevice.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/sockios.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "axutils.h"
#include "axconfig.h"

#include "node.h"

#define FPACWP "/var/ax25/fpac/fpacwp.dat"

static long *wp_head;

static int init_done = 0;
static int semid = -1;

/*** Prototypes *******************/
static long crc32(ax25_address *addr);
static void sem_lock(void);
static void sem_unlock(void);

int wp_init(void)
{
	int shmid;

	wp_head = NULL;

	shmid = shmget(IPC_KEY, 0, 0666);
	if (shmid == -1)
	{
		perror("shmget");
		return(0);
	}

	wp_head = (long *)shmat(shmid, NULL, 0);
	if ((int)wp_head == -1)
	{
		perror("shmat");
		wp_head = NULL;
		return(0);
	}

	/* Create or get semaphore */
	semid = semget(IPC_KEY, 1, 0644);
	if (semid == -1)
	{
		perror("semget");
		return(0);
	}

	init_done = 1;

	return(1);
}

ax25_address *call_clean(ax25_address *call)
{
	/* Cleans the callsign */
	char *ptr = (char *)call;
	
	ptr[0] &= 0xfe;
	ptr[1] &= 0xfe;
	ptr[2] &= 0xfe;
	ptr[3] &= 0xfe;
	ptr[4] &= 0xfe;
	ptr[5] &= 0xfe;
	ptr[6] &= 0x1e;

	return(call);
}

int add_wp(struct sockaddr_rose *addr)
{
	int pos;
	int record;
	wp_t wp;
	FILE *fptr;

	if (wp_head == NULL)
		return(1);

	/* dump_rose("add_wp", addr); */

	sem_lock();

	memset(&wp, 0, sizeof(wp_t));
	wp.date = time(NULL);
	wp.address = *addr;

	call_clean(&wp.address.srose_call);
	if (wp.address.srose_ndigis)
		call_clean(&wp.address.srose_digi);

	/* Check if not already known */
	pos = search_wp(&wp.address.srose_call, NULL);
	if (pos == 0)
		record = wp_head[0];
	else
		record = pos;

	fptr = fopen(FPACWP, "r+");
	if (fptr == NULL)
	{
		sem_unlock();
		return(0);
	}

	if (fseek(fptr, record * sizeof(wp_t), SEEK_SET) == -1)
	{
		fclose(fptr);
		sem_unlock();
		return(0);
	}
	if (fwrite(&wp, sizeof(wp_t), 1, fptr) == 0)
	{
		fclose(fptr);
		sem_unlock();
		return(0);
	}

	fclose(fptr);

	if (pos == 0)
	{
/* fprintf(stderr, "Ajoute %s en %d\n", ax2asc(&wp.address.srose_call), record); */
		/* New record */
		wp_head[record++] = crc32(&wp.address.srose_call);
		wp_head[0] = record;
	}

	sem_unlock();
	return(1);
}

long nb_wp(void)
{
	long nb;

	if (wp_head == NULL)
		return(-1);

	sem_lock();
	nb = wp_head[0];
	sem_unlock();

	return(nb);
}

int search_wp(ax25_address *call, wp_t *wp)
{
	int nb;
	long hash;
	long *ptr;
	FILE *fptr;
	wp_t wptmp;
	ax25_address scall = *call;

	if (wp_head == NULL)
		return(0);

	sem_lock();

	call_clean(&scall);

	hash = crc32(&scall);

	for (ptr = wp_head, nb = 0 ; nb < wp_head[0] ; ptr++, nb++)
	{
		if (*ptr == hash)
		{
			/* possibly found */

			fptr = fopen(FPACWP, "r");
			if (fptr == NULL)
			{
				sem_unlock();
				return(0);
			}

			if (fseek(fptr, nb * sizeof(wp_t), SEEK_SET) == -1)
			{
				fclose(fptr);
				sem_unlock();
				return(0);
			}

			if (fread(&wptmp, sizeof(wp_t), 1, fptr) == 0)
			{
				fclose(fptr);
				sem_unlock();
				return(0);
			}

			fclose(fptr);

			if (ax25cmp(&scall, &wptmp.address.srose_call) == 0)
			{
				if (wp)
					*wp = wptmp;
				sem_unlock();
				return(nb);
			}
			
		}
	}

	/* not found ... */

	sem_unlock();

	return(0);
}

/*
  Finds the CRC32 of a set of bytes.
  from Peter Cammaert's code. 
*/
static long crc32(ax25_address *addr) 
{ 
	/* indices */
	int perByte;
	int perBit;
	/* crc polynomial for Ethernet */
	const unsigned long poly = 0xedb88320;
	/* crc value - preinitialized to all 1's */
	unsigned long crc_value = 0xffffffff; 
	char *s = (char *)addr;
	int length = sizeof(ax25_address);

	for ( perByte = 0; perByte < length; perByte ++ ) 
	{
		unsigned char	c;
	
		c = *(s++);
		for ( perBit = 0; perBit < 8; perBit++ ) 
		{
			crc_value = (crc_value>>1)^
				(((crc_value^c)&0x01)?poly:0);
			c >>= 1;
		}
	}
	return(crc_value);
} 

static int lock = 0;

static void sem_lock(void)
{
	int pid;
	int val;
	struct sembuf sops;

	/* If already locked, do nothing */
	if (lock)
	{
		return;
	}

	sops.sem_num = 0;
	sops.sem_op =  -1;
	sops.sem_flg = IPC_NOWAIT;

	for (;;)
	{
		val = semop(semid, &sops, 1);
		if (val == -1)
		{
			if (errno != EAGAIN)
			{
				perror("semop lock");
				return;
			}
		}
		else
		{
			/* Semaphone available and now locked */
			break;
		}

		/* Semaphone is locked. Is the process still existing ? */
		pid = semctl(semid, 0, GETPID, 0);
		if (pid == -1)
		{
			perror("semctl GETPID");
			return;
		}

		val = kill(pid, 0);
		if (val == -1)
		{
			/* The PID does not exist. Cleaning the semaphore */
			lock = 1;
			sem_unlock();
		}

		usleep(100000L);
	}
	lock = 1;
	return;	
}

static void sem_unlock(void)
{
	int val;
	struct sembuf sops;

	/* If already unlocked, do nothing */
	if (!lock)
	{
		return;
	}

	sops.sem_num = 0;
	sops.sem_op =  1;
	sops.sem_flg = IPC_NOWAIT;

	val = semop(semid, &sops, 1);
	if ((val == -1) && (errno != EAGAIN))
	{
		perror("semop unlock");
		return;
	}
	lock = 0;
}
