/******************************************************************************/
/*                                                                            */
/*  Title       : maketlex.c                                                  */
/*  Author      : Manfred Bester                                              */
/*  Date        : 13Feb94                                                     */
/*  Last change : 15Mar95                                                     */
/*                                                                            */
/*  Synopsis    : This program generates the data file 'tlex.dat' which is    */
/*                a concatenation of various other files with two-line        */
/*                Keplerian element sets in the order listed below in the     */
/*                source code.                                                */
/*                                                                            */
/*  Input files : tlelist.dat                                                 */
/*  Output files: index.dat, satlist_all.dat, tlex.dat                        */
/*                                                                            */
/*                                                                            */
/*  SatTrack is Copyright (c) 1992, 1993, 1994, 1995 by Manfred Bester.       */
/*  All Rights Reserved.                                                      */
/*                                                                            */
/*  Permission to use, copy, and distribute SatTrack and its documentation    */
/*  in its entirety for educational, research and non-profit purposes,        */
/*  without fee, and without a written agreement is hereby granted, provided  */
/*  that the above copyright notice and the following three paragraphs appear */
/*  in all copies. SatTrack may be modified for personal purposes, but        */
/*  modified versions may NOT be distributed without prior consent of the     */
/*  author.                                                                   */
/*                                                                            */
/*  Permission to incorporate this software into commercial products may be   */
/*  obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,    */
/*  Berkeley, CA 94709, USA. Note that distributing SatTrack 'bundled' in     */
/*  with ANY product is considered to be a 'commercial purpose'.              */
/*                                                                            */
/*  IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, */
/*  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF   */
/*  THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED  */
/*  OF THE POSSIBILITY OF SUCH DAMAGE.                                        */
/*                                                                            */
/*  THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT      */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A   */
/*  PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"      */
/*  BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, */
/*  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                  */
/*                                                                            */
/******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef STDLIB
#include <stdlib.h>
#endif

#include "sattrack.h"

#ifdef HPTERM
#include "hpterm.h"
#else
#include "vt100.h"
#endif

/******************************************************************************/
/*                                                                            */
/* external functions                                                         */
/*                                                                            */
/******************************************************************************/

extern double getElement();

/******************************************************************************/
/*                                                                            */
/* structures                                                                 */
/*                                                                            */
/******************************************************************************/

#define MAXFILES 100

typedef struct {
    char tleFile[80];
    } listType;

listType list[MAXFILES];

/******************************************************************************/
/*                                                                            */
/* main program                                                               */
/*                                                                            */
/******************************************************************************/

void main(argc,argv)

int  argc;
char *argv[];

{
    long satNum, satNumList[MAXSATS];

    int  i, k, K, duplFlag, verboseFlag, numTleFiles;

    char data[100], tle[100], tleLine0[100], tleLine1[100], tleLine2[100];
    char str[100];
    char inputFile[100], outputFile[100], indexFile[100], listFile[100];
    char satlistFile[100];
    char *strpHome, *getenv();

    FILE *InputFile, *OutputFile, *ListFile, *IndexFile, *SatlistFile;

    verboseFlag = (argc == 2 && !strncmp(argv[1],"-v",2)) ? TRUE : FALSE;

#ifdef HOMEDIR
    strpHome = getenv("HOME");
#else
    strpHome = SATDIR;
#endif

    sprintf(data,"%s/%s",strpHome,DATA);
    sprintf(indexFile,"%s/index.dat",data);
    sprintf(listFile,"%s/tlelist.dat",data);
    sprintf(satlistFile,"%s/satlist_all.dat",data);

    sprintf(tle,"%s/%s",strpHome,TLE);
    sprintf(outputFile,"%s/tlex.dat",tle);

    if ((OutputFile = fopen(outputFile,"w")) == NULL)
    {
        doBeep(); nl(); reverseBlink();
        printf("can't write %s\n",outputFile);
        normal(); nl();
        exit(-1);
    }

    else
        fclose(OutputFile);

    if ((IndexFile = fopen(indexFile,"w")) == NULL)
    {
        doBeep(); nl(); reverseBlink();
        printf("can't write %s\n",indexFile);
        normal(); nl();
        exit(-1);
    }

    else
        fclose(IndexFile);

    if ((SatlistFile = fopen(satlistFile,"w")) == NULL)
    {
        doBeep(); nl(); reverseBlink();
        printf("can't write %s\n",satlistFile);
        normal(); nl();
        exit(-1);
    }

    else
        fclose(SatlistFile);

    if ((ListFile = fopen(listFile,"r")) == NULL)
    {
        doBeep(); nl(); reverseBlink();
        printf("'%s' not found\n",listFile);
        normal(); nl();
        exit(-1);
    }

    numTleFiles = 0;

    while (fgets(str,80,ListFile))
    {
        str[strlen(str) - 1] = '\0';
        strcpy(list[numTleFiles].tleFile,str);
        numTleFiles++;
    }

    if (verboseFlag)
    {
        printf("\ngot %d file names with tle data from tlex.dat\n\n",
            numTleFiles);

        for (i = 0; i < numTleFiles; i++)
        {
            printf("%2d:  %s\n",i+1,list[i].tleFile);
        }

        printf("\n");
    }

    K = 0;

    for (i = 0; i < numTleFiles; i++)
    {
        sprintf(inputFile,"%s/%s",tle,list[i].tleFile);

        if (verboseFlag)
            printf("\nreading from '%s'\n",list[i].tleFile);

        if ((InputFile = fopen(inputFile,"r")) == NULL)
        {
            doBeep(); nl(); reverseBlink();
            printf("'%s' not found\n",inputFile);
            normal(); nl();
            exit(-1);
        }

        else
            fclose(InputFile);

        InputFile   = fopen(inputFile, "r");
        OutputFile  = fopen(outputFile,"a");
        IndexFile   = fopen(indexFile,"a");
        SatlistFile = fopen(satlistFile,"a");

        while (fgets(tleLine0,80,InputFile) &&  K < MAXSATS)
        {
            if (!strncmp(tleLine0,TLEHEADER,10))       /* discard header line */
                fgets(tleLine0,80,InputFile);

            fgets(tleLine1,80,InputFile);
            fgets(tleLine2,80,InputFile);

            satNum   = (long) (getElement(tleLine1,3,7) + ONEPPM);
            duplFlag = FALSE;

            for (k = 0; k < K; k++)
            {
                if (satNum == satNumList[k])
                    duplFlag = TRUE;
            }

            if (!duplFlag)
            {
                fprintf(OutputFile,"%s",tleLine0);
                fprintf(OutputFile,"%s",tleLine1);
                fprintf(OutputFile,"%s",tleLine2);
                fprintf(IndexFile,"%-6ld  %s",satNum,tleLine0);
                fprintf(SatlistFile,"%ld\n",satNum);

                satNumList[K] = satNum;
                K++;
            }

            else
            {
                if (verboseFlag)
                    printf("  duplicate entry: %-5ld  %s",satNum,tleLine0);
            }
        }

        fclose(InputFile);
        fclose(OutputFile);
        fclose(IndexFile);
        fclose(SatlistFile);
    }

    if (verboseFlag)
        printf("\n'tlex.dat' contains %d two-line element sets.\n\n",K);
}

/******************************************************************************/
/*                                                                            */
/* getElement: returns double of orbital element out of ASCII string          */
/*                                                                            */
/******************************************************************************/

double getElement(gstr,gstart,gstop)

int  gstart, gstop;
char gstr[80];

{
    int  k, glength;
    char gestr[80];

    glength = gstop - gstart + 1;

    for (k = 0; k <= glength; k++)
        gestr[k] = gstr[gstart+k-1];

    gestr[glength] = '\0';

    return((double) atof(gestr));
}

/******************************************************************************/
/*                                                                            */
/* End of program maketlex.c                                                  */
/*                                                                            */
/******************************************************************************/
