/* beepAdLib.c -- Jacek M. Holeczek 3/2000 */
/* A lot of code taken from the sccw Linux Morse program by */
/* Steven J. Merrifield  VK3ESM  sjm@ee.latrobe.edu.au */

/*
  Implementation of beep for the AdLib/SoundBlaster FM chip.
  
  Warning: All low-level access is achieved by writing directly to the
  IO ports of the soundcard, hence this program must be compiled using
  at least -O2 and run setuid root. Hence, this program must be installed
  by root, but can then be run by a normal user.
  
  Must compile with -O2 -lm and optionally -DDEBUG_FREQ
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <sys/io.h>

#include "alarm.h"
#include "beep.h"

/* Define it if you want to reset the AdLib/SoundBlaster FM chip. */
#undef RESET_ADLIB

/* The port addresses for the FM chip is always the same. */
#define reg_port 0x0388
#define data_port reg_port+1

void wr_register(int index, int data) /* See the PC Game Programmers Encyclopedia */
{
  int i,temp;
  outb(index,reg_port);
  for (i=1;i<7;i++)
    temp = inb(reg_port);
  outb(data,data_port);
  for (i=1;i<36;i++)
    temp = inb(reg_port);
}

int AdLib_found(void)	/* See the PC Game Programmers Encyclopedia */
{
  long temp1,temp2;
  wr_register(4,0x60);
  wr_register(4,0x80);
  temp1 = inb(reg_port);
  wr_register(2,0xFF);
  wr_register(4,0x21);
  sleep(1);
  temp2 = inb(reg_port);
  wr_register(4,0x60);
  wr_register(4,0x80);
  if (((temp1 & 0xE0) == 0) && ((temp2 & 0xE0) == 0xC0))
    {
      return 0; /* card found and no errors */
    } else {
      return 1; /* card not found or errors */
    }
}

#ifdef RESET_ADLIB
void Reset_AdLib(void)	/* Set all registers to 0 */
{
  int i;
  for (i=1;i<245;i++)
    wr_register(i,0);
}
#endif

int BeepInit(void)
{
  if (geteuid())
    {
      fprintf(stderr,"This program must run as root (setuid root).\n");
      return 1;
    }
  ioperm(reg_port,17,1);
  ioperm(data_port,17,1);
  if (AdLib_found())
    {
      fprintf(stderr,"The AdLib/SoundBlaster card not found.\n");
      ioperm(reg_port,17,0);
      ioperm(data_port,17,0);
      return 2;
    }
#ifdef RESET_ADLIB
  Reset_AdLib();
#else
  wr_register(0xB0,0x00);		/* Turn off sound, was 0x11 */
#endif
  return 0;
}

int Beep(int time, int volume, int pitch)
{
  int blk = -1;
  int f_num;
  int A0_reg,B0_reg,B0_reg10,B0_reg432;
  
  BeepWait();
  AlarmSet(time);
  
  if (volume > 0  &&  pitch > 0)
    {
      wr_register(0x20,1);	/* Play sound at fundamental freq. */
      wr_register(0x23,1);	/* Carrier at specified freq. */
      wr_register(0x43,(64*(100-volume)/100));  /* Volume 0-64 */
      
      wr_register(0x63,0xF0);	/* Carrier attack = fast, decay = slow */
      wr_register(0x83,0xFF);	/* Carrier sustain = soft, release = fast */
      
      /* Would you belive it took three days to work out how to do this ? */
      do
	{
	  blk++;
	  f_num = (int)(((float)pitch / 50000.0) / (1/(pow(2,abs(blk-20)))));
	} while (f_num > 1023);
      A0_reg = f_num & 0x00FF;
      B0_reg10 = (f_num & 0x0300) >> 8;
      B0_reg432 = (blk << 2) & 0x00FF;   /* Do I really need to & 0x00ff ?? */
      B0_reg = (32 | B0_reg432 | B0_reg10) & 0x00FF;
      
      wr_register(0xA0,A0_reg);
      wr_register(0xB0,B0_reg);
      
#ifdef DEBUG_FREQ
      printf("blk = %d    f_num = %d\n",blk,f_num);
      printf("B0_reg432 = %2x\n",B0_reg432);  /* bits 4 3 2 */
      printf("B0_reg10 = %2x\n",B0_reg10);    /* bits 1 0 */
      printf("B0_reg = %2x   A0_reg = %2x\n\n\n",B0_reg,A0_reg);
#endif
    }
  
  return 0;
}

int BeepWait(void)
{
  AlarmWait();
  wr_register(0xB0,0x00);		/* Turn off sound, was 0x11 */
  return 0;
}

int BeepCleanup(void)
{
#ifdef RESET_ADLIB
  Reset_AdLib();
#else
  wr_register(0xB0,0x00);		/* Turn off sound, was 0x11 */
#endif
  ioperm(reg_port,17,0);
  ioperm(data_port,17,0);
  return 0;
}

int BeepResume(void)
{
  ioperm(reg_port,17,1);
  ioperm(data_port,17,1);
#ifdef RESET_ADLIB
  Reset_AdLib();
#else
  wr_register(0xB0,0x00);		/* Turn off sound, was 0x11 */
#endif
  return 0;
}
