#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <linux/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pwd.h>
#include <linux/ax25.h>			/* axutils.h needs this...	*/

#include "config.h"
#include "axutils.h"
#include "axconfig.h"
#include "node.h"
#include "io.h"

long IdleTimeout	= 900L;		/* default to 15 mins */
long ConnTimeout	= 3600L;	/* default to 60 mins */
char *HostName		= NULL;
int ReConnectTo		= 0;
char *CallServer	= "localhost";

struct alias *Aliases	= NULL;
struct ecmd *Extcmds	= NULL;

static unsigned long Permissions	= 0L;
static unsigned long LocalNet		= 0L;
static unsigned long LocalMask		= ~0L;
static char *HiddenPorts[32]		= {0};

static int do_idletimeout(int, char **);
static int do_conntimeout(int, char **);
static int do_reconnect(int, char **);
static int do_hostname(int, char **);
static int do_localnet(int, char **);
static int do_callserver(int, char **);
static int do_alias(int, char **);
static int do_hiddenports(int, char **);
static int do_extcmd(int, char **);

static struct cmd cfg_cmds[] = {
	{ "alias",		do_alias	},
	{ "idletimeout",	do_idletimeout	},
	{ "conntimeout",	do_conntimeout	},
	{ "reconnect",		do_reconnect	},
	{ "hostname",		do_hostname	},
	{ "localnet",		do_localnet	},
	{ "callserver",		do_callserver	},
	{ "hiddenports",	do_hiddenports	},
	{ "extcmd",		do_extcmd	},
	{ NULL,			NULL		}
};

/*
 * Return non-zero if `port' is a hidden port.
 */
int is_hidden(const char *port)
{
	int i;

	for (i = 0; HiddenPorts[i] != NULL && i < 31; i++)
		if (!strcmp(port, HiddenPorts[i]))
			return 1;
	return 0;
}

/*
 * Return non-zero if peer is on "local" or loopback network.
 */
static int is_local(unsigned long peer)
{
	return ((peer & LocalMask) == LocalNet) || ((peer & 0xff) == 127);
}

/*
 * Return non-zero if peer is on amprnet.
 */
static int is_ampr(unsigned long peer)
{
	return ((peer & 0xff) == 44);
}

/*
 * Convert NOS style width to a netmask in network byte order.
 */
static unsigned long bits_to_mask(int bits)
{
	return htonl(~0L << (32 - bits));
}

int check_perms(int what, unsigned long peer)
{
	if (what == PERM_TELNET) {
		if (is_local(peer)) {
			if (Permissions & PERM_TELNET_LOCAL)
				return 0;
		} else if (is_ampr(peer)) {
			if (Permissions & PERM_TELNET_AMPR)
				return 0;
		} else {
			if (Permissions & PERM_TELNET_INET)
				return 0;
		}
		return -1;
	}
	if ((Permissions & what) == 0) {
		return -1;
	}
	return 0;
}

/*
 * Read permissions
 */
char *read_perms(char *user, int family, unsigned long peer, char *port)
{
	FILE *fp;
	char line[256], *argv[32], *cp;
	int argc, n = 0;

	if ((fp = fopen(CONF_NODE_PERMS_FILE, "r")) == NULL) {
		node_perror(CONF_NODE_PERMS_FILE, errno);
		return NULL;
	}
	if ((cp = strchr(user, '-')) != NULL)
		*cp = 0;
	while (fgets(line, 256, fp) != NULL) {
		n++;
		argc = parse_args(argv, line);
		if (argc == 0 || *argv[0] == '#')
			continue;
		if (argc != 5) {
			tprintf("%s} Syntax error in permission file at line %d\n", NrId, n);
			break;
		}
		if (strcmp(argv[0], "*") && strcasecmp(argv[0], user))
			continue;
		switch (family) {
		case AF_AX25:
			if (!strcmp(argv[1], "*"))
				break;
			if (!strcasecmp(argv[1], "ax25"))
				break;
			continue;
		case AF_NETROM:
			if (!strcmp(argv[1], "*"))
				break;
			if (!strcasecmp(argv[1], "netrom"))
				break;
			continue;
		case AF_INET:
			if (!strcmp(argv[1], "*"))
				break;
			if (!strcasecmp(argv[1], "local") && is_local(peer))
				break;
			if (!strcasecmp(argv[1], "ampr") && is_ampr(peer))
				break;
			if (!strcasecmp(argv[1], "inet") && !is_local(peer) && !is_ampr(peer))
				break;
			continue;
		case AF_UNSPEC:
			if (!strcmp(argv[1], "*"))
				break;
			if (!strcasecmp(argv[1], "host"))
				break;
			continue;
		}
		if (family == AF_AX25) {
			if (strcmp(argv[2], "*") && strcmp(argv[2], port))
				continue;
		}
		if (cp != NULL)
			*cp = '-';
		if ((Permissions = strtoul(argv[4], NULL, 10)) == 0)
			return NULL;
		else
			return strdup(argv[3]);
	}
	return NULL;
}

static int do_alias(int argc, char **argv)
{
	struct alias *ap;

	if (Aliases == NULL) {
		if ((Aliases = malloc(sizeof(struct alias))) == NULL) {
			node_perror("do_alias: malloc", errno);
			return -2;
		}
		Aliases->name = strdup(argv[1]);
		Aliases->command = strdup(argv[2]);
		Aliases->next = NULL;
	} else {
		ap = Aliases;
		while (ap->next != NULL)
			ap = ap->next;
		if ((ap->next = malloc(sizeof(struct alias))) == NULL) {
			node_perror("do_alias: malloc", errno);
			return -2;
		}
		ap = ap->next;
		ap->name = strdup(argv[1]);
		ap->command = strdup(argv[2]);
		ap->next = NULL;
	}
	return 0;
}

static int do_idletimeout(int argc, char **argv)
{
	if (argc < 2)
		return -1;
	IdleTimeout = atol(argv[1]);
	return 0;
}

static int do_conntimeout(int argc, char **argv)
{
	if (argc < 2)
		return -1;
	ConnTimeout = atol(argv[1]);
	return 0;
}

static int do_hostname(int argc, char **argv)
{
	if (argc < 2)
		return -1;
	HostName = strdup(argv[1]);
	return 0;
}

static int do_callserver(int argc, char **argv)
{
	if (argc < 2)
		return -1;
	CallServer = strdup(argv[1]);
	return 0;
}

static int do_localnet(int argc, char **argv)
{
	char *cp;

	if (argc < 2)
		return -1;
	if ((cp = strchr(argv[1], '/')) != NULL) {
		*cp = 0;
		LocalMask = bits_to_mask(atoi(++cp));
	}
	LocalNet = inet_addr(argv[1]);
	LocalNet &= LocalMask;
	return 0;
}

static int do_reconnect(int argc, char **argv)
{
	if (argc < 2)
		return -1;
	if (!strcasecmp(argv[1], "on"))
		ReConnectTo = 1;
	else
		ReConnectTo = 0;
	return 0;
}

static int do_hiddenports(int argc, char **argv)
{
	int i;

	if (argc < 2)
		return -1;
	for (i = 1; i < argc && i < 31; i++) {
		if (ax25_config_get_dev(argv[i]) == NULL) {
			tprintf("%s} do_hiddenports(): invalid port %s\n", NrId, argv[i]);
			return -2;
		}
		HiddenPorts[i - 1] = strdup(argv[i]);
	}
	HiddenPorts[i - 1] = NULL;
	return 0;
}

static int do_extcmd(int argc, char **argv)
{
	struct ecmd *ep;
	struct passwd *pw;
	int i;

	if (argc < 6)
		return -1;
	if (Extcmds == NULL) {
		if ((Extcmds = malloc(sizeof(struct ecmd))) == NULL) {
			node_perror("do_extcmd: malloc", errno);
			return -2;
		}
		ep = Extcmds;
	} else {
		ep = Extcmds;
		while (ep->next != NULL)
			ep = ep->next;
		if ((ep->next = malloc(sizeof(struct ecmd))) == NULL) {
			node_perror("do_extcmd: malloc", errno);
			return -2;
		}
		ep = ep->next;
	}
	ep->name = strdup(argv[1]);
	ep->flags = atoi(argv[2]);
	if ((pw = getpwnam(argv[3])) == NULL) {
		tprintf("%s} Unknown user %s\n", NrId, argv[3]);
		return -2;
	}
	ep->uid = pw->pw_uid;
	ep->gid = pw->pw_gid;
	ep->path = strdup(argv[4]);
	for (i = 0; argv[i + 5] != NULL; i++)
		ep->argv[i] = strdup(argv[i + 5]);
	ep->ok = 0;
	ep->next = NULL;
	return 0;
}

int read_config(void)
{
	FILE *fp;
	char line[256];
	int ret, n = 0;

	if ((fp = fopen(CONF_NODE_FILE, "r")) == NULL) {
		node_perror(CONF_NODE_FILE, errno);
		return -1;
	}
	while (fgets(line, 256, fp) != NULL) {
		n++;
		ret = cmdparse(cfg_cmds, line);
		if (ret == -1)
			tprintf("%s} Syntax error in config file at line %d: %s\n", NrId, n, line);
		if (ret < 0) {
			fclose(fp);
			return -1;
		}
	}
	fclose(fp);
	return 0;
}
