/*****************************************************************************/
/*
 *      linuxio.c  --  Linux realtime audio output.
 *
 *      Copyright (C) 1996-1998  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *        Swiss Federal Institute of Technology (ETH), Electronics Lab
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *  This is the Linux realtime sound output driver
 */

/*****************************************************************************/
      
#include "config.h"

#include <sys/time.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#ifdef HAVE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#endif
#ifdef HAVE_SYS_AUDIOIO_H
#include <sys/audioio.h>
#endif /* HAVE_SYS_AUDIOIO_H */
#include <net/if.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <grp.h>
#include <errno.h>
#include <termios.h>
#include <dlfcn.h>
#include <syslog.h>
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif /* HAVE_SCHED_H */

#include "sm.h"

/* --------------------------------------------------------------------- */

#define PIDFILE LOCALSTATEDIR "/run/soundmodem.pid"

/* --------------------------------------------------------------------- */

static int logging = 0;
static struct soundmodem_state *state = NULL;  /* globally needed currently for sig_usr1 */

/* --------------------------------------------------------------------- */
/*
 * Logging functions
 */

void errprintf(int severity, const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	if (logging) {
		char tmp[512];
		vsnprintf(tmp, sizeof(tmp), fmt, args);
		syslog(severity, tmp);
	} else {
		fprintf(stderr, "soundmodem[%u]: ", getpid());
		vfprintf(stderr, fmt, args);
	}
	va_end(args);
	if (severity <= SEV_FATAL)
		exit(1);
}

/* --------------------------------------------------------------------- */

void errstr(int severity, const char *st)
{
	errprintf(severity, "soundmodem: %s: %s\n", st, strerror(errno));
}

/* --------------------------------------------------------------------- */

#ifdef HAVE_MKISS

#ifdef HAVE_PTY_H

#include <pty.h>

static void find_pty(int fds[2])
{
	char name[32];

	if (openpty(fds, fds+1, name, NULL, NULL) == -1)
		errstr(SEV_FATAL, "openpty");
}

#else /* HAVE_PTY_H */

static void find_pty(int fds[2])
{
	static const char ptych1[] = "pqrstuvwxyzabcde";
	static const char ptych2[] = "0123456789abcdef";
	char ptyname[] = "/dev/pty00";
        struct group *gr = getgrnam("tty");
	int i, j;

	for (i = 0; i < strlen(ptych1); i++) {
		ptyname[8] = ptych1[i];
		for (j = 0; j < strlen(ptych2); j++) {
			ptyname[9] = ptych2[j];
			if ((fds[0] = open(ptyname, O_RDWR)) >= 0) {
                                chown(ptyname, getuid(), gr ? gr->gr_gid : -1);
                                chmod(ptyname, S_IRUSR|S_IWUSR|S_IWGRP);
				ptyname[5] = 't';
				if ((fds[1] = open(ptyname, O_RDWR)) >= 0) 
					return;
				close(fds[0]);
			}
		}
	}
	errprintf(SEV_FATAL, "Unable to find free pty/tty pair\n");
	exit(1);
}

#endif /* HAVE_PTY_H */

/* --------------------------------------------------------------------- */

static void set_termios(int fds[2], char *devname) 
{
	struct termios tm;
	int disc = N_AX25;
        int encap = 4;

	memset(&tm, 0, sizeof(tm));
	tm.c_cflag = CS8 | CREAD | CLOCAL;
	if (tcsetattr(fds[0], TCSANOW, &tm))
		errstr(SEV_INFO, "master: tcsetattr");
	memset(&tm, 0, sizeof(tm));
	tm.c_cflag = CS8 | CREAD | CLOCAL;
	if (tcsetattr(fds[1], TCSANOW, &tm))
		errstr(SEV_INFO, "slave: tcsetattr");
        if (ioctl(fds[0], TIOCSETD, &disc) == -1)
                errstr(SEV_FATAL, "ioctl: TIOCSETD");
	if (ioctl(fds[0], SIOCGIFNAME, devname) == -1)
                errstr(SEV_FATAL, "ioctl: SIOCGIFNAME");
        if (ioctl(fds[0], SIOCSIFENCAP, &encap) == -1)
                errstr(SEV_FATAL, "ioctl: SIOCSIFENCAP");
}

/* --------------------------------------------------------------------- */

static void startif(char *devname)
{
        struct ifreq ifr;
        int fd;
        
        if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
                errstr(SEV_FATAL, "socket");
        strcpy(ifr.ifr_name, devname);
	ifr.ifr_mtu = 256;
        if (ioctl(fd, SIOCSIFMTU, &ifr) < 0)
                errstr(SEV_FATAL, "ioctl: SIOCSIFMTU");
        if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0)
                errstr(SEV_FATAL, "ioctl: SIOCGIFFLAGS");
        ifr.ifr_flags &= IFF_NOARP;
        ifr.ifr_flags |= IFF_UP;
        ifr.ifr_flags |= IFF_RUNNING;
        if (ioctl(fd, SIOCSIFFLAGS, &ifr) < 0)
                errstr(SEV_FATAL, "ioctl: SIOCSIFFLAGS");
        close(fd);
}

#endif /* HAVE_MKISS */

/* --------------------------------------------------------------------- */

#define ADDRXCH(x)                        \
do {                                      \
	s->kissout.b[wr] = (x);           \
	wr = (wr + 1) % KISSOUTBUF_SIZE;  \
	if (wr == rd)                     \
		goto errout;              \
        total++;                          \
} while (0);

static size_t rxpacket(struct soundmodem_state *s, const struct iovec *vector, size_t count)
{
	u_int8_t *bp;
	size_t len;
	size_t total = 0;
	unsigned wr = s->kissout.wr;
	unsigned rd = s->kissout.rd;

	ADDRXCH(KISS_FEND);
	for (; count > 0; count--, vector++) {
		for (bp = vector->iov_base, len = vector->iov_len; len > 0; len--, bp++) {
			if (*bp == KISS_FEND) {
				ADDRXCH(KISS_FESC);
				ADDRXCH(KISS_TFEND);
			} else if (*bp == KISS_FESC) {
				ADDRXCH(KISS_FESC);
				ADDRXCH(KISS_TFESC);
			} else {
				ADDRXCH(*bp);
			}
		}
	}
	if (total <= 1)
		return 0;
	ADDRXCH(KISS_FEND);
	s->kissout.wr = wr;
	s->stat.kiss_out++;
	return total;
       
  errout:
	s->stat.kiss_outerr++;
	return 0;
}

#undef ADDRXCH

/* --------------------------------------------------------------------- */

extern inline void simple_rxpacket(struct soundmodem_state *sm, void *pkt, unsigned len)
{
	struct iovec iov[1];

	iov[0].iov_base = pkt;
	iov[0].iov_len = len;
	rxpacket(sm, iov, 1);
}

/* --------------------------------------------------------------------- */

#define GETSTATEPTR(x) ((struct soundmodem_state *)(((long)(x))&(~0xff)))
#define GETNR(x) (((long)(x)/(2*sizeof(void *)))%MAXMODEM)

void sm_rxpacket(void **state, u_int8_t *pkt, unsigned len)
{
	struct soundmodem_state *sm = GETSTATEPTR(state);
	unsigned nr = GETNR(state);
	u_int8_t cmd = KISS_CMD_DATA | (nr << 4);
	struct iovec iov[2];
	char buf[512];

	if (len < 7)
		return;
	iov[0].iov_base = &cmd;
	iov[0].iov_len = 1;
	iov[1].iov_base = pkt;
	iov[1].iov_len = len;
	rxpacket(sm, iov, 2);

	if (snprintpkt(buf, sizeof(buf), pkt, len) > 0)
		printf("rx: %s", buf);
}

void sm_diag(void **state, unsigned nr, unsigned mode, int sync, int16_t *samp, int numsamp)
{
	struct soundmodem_state *sm = GETSTATEPTR(state);
	unsigned demnr = GETNR(state);
	struct sm_message msg;
	struct iovec iov[2];
	int cnt;

	if (!(sm->demodulator[demnr].diagmode & (0x001 << nr)))
		return;
	memcpy(msg.hdr.signature, SOUNDMODEM_SIGNATURE, sizeof(msg.hdr.signature));
	msg.hdr.signature[0] |= (demnr << 4);
	msg.hdr.cmd = SOUNDMODEM_CMD_DIAGDATA + nr;
	msg.d.diagsamp.flags = 0;
	msg.d.diagsamp.sperbit = 0;
	iov[0].iov_base = &msg;
	iov[0].iov_len = ((char *)&msg.d.diagsamp.sample) - ((char *)&msg);
	iov[1].iov_base = samp;
	while (numsamp > 0) {
		cnt = min(numsamp, 64);
		iov[1].iov_len = 2*cnt;
		numsamp -= cnt;
		msg.d.diagsamp.seq = sm->demodulator[demnr].diagseq[nr]++;
		rxpacket(sm, iov, 2);
		(char *)iov[1].iov_base += iov[1].iov_len;
	}
}

void sm_rawbits(void **state, unsigned nr, u_int8_t *rawb, int numraw)
{
	struct soundmodem_state *sm = GETSTATEPTR(state);
	unsigned demnr = GETNR(state);
	struct sm_message msg;
	struct iovec iov[2];

	if (!(sm->demodulator[demnr].diagmode & (0x100 << nr)))
		return;
	memcpy(msg.hdr.signature, SOUNDMODEM_SIGNATURE, sizeof(msg.hdr.signature));
	msg.hdr.signature[0] |= (demnr << 4);
	msg.hdr.cmd = SOUNDMODEM_CMD_DIAGDATA + 8 + nr;
	msg.d.diagraw.flags = 0;
	iov[0].iov_base = &msg;
	iov[0].iov_len = ((char *)&msg.d.diagraw.data) - ((char *)&msg);
	iov[1].iov_base = rawb;
	while (numraw > 0) {
		iov[1].iov_len = min(numraw, 128);
		msg.d.diagraw.seq = sm->demodulator[demnr].diagseq[nr+8]++;
		rxpacket(sm, iov, 2);
		(char *)iov[1].iov_base += iov[1].iov_len;
		numraw -= iov[1].iov_len;
	}
}

void sm_setdcd(void **state, int dcd)
{
	struct soundmodem_state *sm = GETSTATEPTR(state);
	unsigned nr = GETNR(state);

	if (dcd)
		sm->dcd |= 1 << nr;
	else
		sm->dcd &= ~(1 << nr);
}

int sm_getptt(void **state)
{
	struct soundmodem_state *sm = GETSTATEPTR(state);

	return (sm->ptt != 0);
}

/* --------------------------------------------------------------------- */

static void sighandler(int sig)
{
	errprintf(LOG_INFO, "terminating on receiving signal %i\n", sig);
	exit(0);
}

/* --------------------------------------------------------------------- */

static void sig_usr1(int sig)
{
	int i;

	printf("Kissinptr: %u\n", state->kissin.ptr);

	printf("Statistics:\n  KISS out: %ld errors: %ld  in: %ld errors: %ld\n"
	       "  PTT keyed: %ld  DMA residue1: %d residue2: %d  IntRate: %d\n",
	       state->stat.kiss_out, state->stat.kiss_outerr,
	       state->stat.kiss_in, state->stat.kiss_inerr,
	       state->stat.ptt_keyed, state->stat.dma_residue1,
	       state->stat.dma_residue2, state->stat.int_rate);
	for (i = 0; i < state->nrmodulator; i++) {
		printf(" Modulator %i: Packets: %ld dropped: %ld errors: %ld cycles: %ld\n", i,
		       state->modulator[i].stat.tx_packets,
		       state->modulator[i].stat.tx_dropped,
		       state->modulator[i].stat.tx_errors,
		       state->modulator[i].stat.tx_cycles);
	}
	for (i = 0; i < state->nrdemodulator; i++) {
		printf(" Demodulator %i: Packets: %ld errors: %ld cycles: %ld\n", i,
		       state->demodulator[i].stat.rx_packets,
		       state->demodulator[i].stat.rx_errors,
		       state->demodulator[i].stat.rx_cycles);
	}
	printf("Channel access parameters:\n  TxDelay: %d Slottime: %d "
	       "P-Persistence: %d TxTail: %d Fulldup: %d\n",
	       state->chaccesspar.txdelay, state->chaccesspar.slottime,
	       state->chaccesspar.ppersist, state->chaccesspar.txtail,
	       state->chaccesspar.fulldup);
	printf("Modem state:\n  DCD: %d PTT: %d\n",
	       !!state->dcd, !!state->ptt);
	if (signal(SIGUSR1, sig_usr1) == SIG_ERR)
		errstr(SEV_ERROR, "signal");
}

/* --------------------------------------------------------------------- */

#ifdef WORDS_BIGENDIAN
#define AFMT_S16_NATIVE  AFMT_S16_BE
#define AFMT_S16_SWAPPED AFMT_S16_LE
#else
#define AFMT_S16_NATIVE  AFMT_S16_LE
#define AFMT_S16_SWAPPED AFMT_S16_BE
#endif

static struct soundmodem_state *init(const char *configfile)
{
	struct soundmodem_state *sm;
	unsigned pgsz = getpagesize();
	char aname[128] = "/dev/dsp";
	char ptyname[128] = "";
	char serptt[128] = "";
	char parptt[128] = "";
	char midiptt[128] = "";
	static const char *delim = " \t\n";
	struct modulator *mod;
	struct demodulator *demod;
	FILE *f;
	char buf[256];
	char *cp, *cp2, *cp3;
	unsigned sr;
	int apar;

	apar = (sizeof(struct soundmodem_state) + pgsz + 1) & (~(pgsz - 1));
	sm = (struct soundmodem_state *)mmap(NULL, apar, PROT_READ | PROT_WRITE, 
					     MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);
	if (sm == MAP_FAILED)
		errstr(SEV_FATAL, "Cannot allocate memory for state info");
	sm->chaccesspar = (struct chaccesspar){ 10, 10, 40, 1, 0 };
	if (!(f = fopen(configfile, "r")))
		errprintf(SEV_FATAL, "Cannot open config file \"%s\"\n", configfile);
	while (fgets(buf, sizeof(buf), f)) {
		cp = strtok(buf, delim);
		if (!cp || *cp == '#')
			continue;
		if (!strcmp(cp, "device")) {
			cp3 = strtok(NULL, delim);
			if (!cp3)
				errstr(SEV_FATAL, "invalid audio device path");
			strncpy(aname, cp3, sizeof(aname));
			continue;
		}
		if (!strcmp(cp, "modulator")) {
			if (sm->nrmodulator >= MAXMODEM)
				errstr(SEV_FATAL, "too many modulators");
			mod = sm->modulator+sm->nrmodulator;
			cp = strtok(NULL, delim);
			if (!cp)
				errstr(SEV_FATAL, "invalid modulator library");
			mod->dlhandle = dlopen(cp, RTLD_NOW);
			if ((cp2 = dlerror()))
				errprintf(SEV_FATAL, "cannot open modulator library \"%s\", error %s\n",
					  cp, cp2);
			mod->code = dlsym(mod->dlhandle, "modulator_0");
			if ((cp2 = dlerror()) || !mod->code)
				errprintf(SEV_FATAL, "invalid modulator library \"%s\", error %s\n", cp, cp2);
			cp = strtok(NULL, delim);
			sm->state[sm->nrmodulator].mod = mod->code->init(cp, &sr);
			if (sr > sm->samplerate)
				sm->samplerate = sr;
			sm->nrmodulator++;
			continue;
		}
		if (!strcmp(cp, "demodulator")) {
			if (sm->nrdemodulator >= MAXMODEM)
				errstr(SEV_FATAL, "too many demodulators");
			demod = sm->demodulator+sm->nrdemodulator;
			cp = strtok(NULL, delim);
			if (!cp)
				errstr(SEV_FATAL, "invalid demodulator library");
			demod->dlhandle = dlopen(cp, RTLD_NOW);
			if ((cp2 = dlerror()))
				errprintf(SEV_FATAL, "cannot open demodulator library \"%s\", error %s\n",
					  cp, cp2);
			demod->code = dlsym(demod->dlhandle, "demodulator_0");
			if ((cp2 = dlerror()) || !demod->code)
				errprintf(SEV_FATAL, "invalid demodulator library \"%s\", error %s\n", cp, cp2);
			cp = strtok(NULL, delim);
			sm->state[sm->nrdemodulator].demod = demod->code->init(cp, &sr);
			if (sr > sm->samplerate)
				sm->samplerate = sr;
			sm->nrdemodulator++;
			continue;
		}
		if (!strcmp(cp, "serptt")) {
			cp3 = strtok(NULL, delim);
			if (!cp3)
				errstr(SEV_FATAL, "invalid serial ptt device path");
			strncpy(serptt, cp3, sizeof(serptt));
			continue;
		}
		if (!strcmp(cp, "parptt")) {
			cp3 = strtok(NULL, delim);
			if (!cp3)
				errstr(SEV_FATAL, "invalid parallel ptt device path");
			strncpy(parptt, cp3, sizeof(parptt));
			continue;
		}
		if (!strcmp(cp, "midiptt")) {
			cp3 = strtok(NULL, delim);
			if (!cp3)
				errstr(SEV_FATAL, "invalid midi ptt device path");
			strncpy(midiptt, cp3, sizeof(midiptt));
			continue;
		}
		if (!strcmp(cp, "pty")) {
			cp3 = strtok(NULL, delim);
			if (!cp3)
				errstr(SEV_FATAL, "invalid pty path");
			strncpy(ptyname, cp3, sizeof(ptyname));
			continue;
		}
		errprintf(SEV_WARNING, "unknown config file entry \"%s\"\n", cp);
       	}
	fclose(f);
	/* open KISS device */
#ifdef HAVE_MKISS
	if (!ptyname[0]) {
		int ttyfds[2];
		char devname[IFNAMSIZ];

		find_pty(ttyfds);
		set_termios(ttyfds, devname);
		startif(devname);
		sm->kissfd = ttyfds[1];
	} else
#endif /* HAVE_MKISS */
	if ((sm->kissfd = open(ptyname, O_RDWR)) == -1)
		errprintf(SEV_FATAL, "cannot open pty device \"%s\"\n", ptyname);
	{
		struct termios tm;

		memset(&tm, 0, sizeof(tm));
		tm.c_cflag = CS8 | CREAD | CLOCAL;
		if (tcsetattr(sm->kissfd, TCSANOW, &tm))
			errstr(SEV_INFO, "master: tcsetattr");
	}
	/* open PTT devices */
	if (serptt[0]) {
		sm->serpttfd = open(serptt, O_RDWR | O_NONBLOCK);
		if (sm->serpttfd == -1)
			errprintf(SEV_ERROR, "cannot open serial ptt device \"%s\"\n", serptt);
	} else
		sm->serpttfd = -1;
	if (parptt[0]) {
		sm->parpttfd = open(parptt, O_RDWR | O_NONBLOCK);
		if (sm->parpttfd == -1)
			errprintf(SEV_ERROR, "cannot open parallel ptt device \"%s\"\n", parptt);
	} else
		sm->parpttfd = -1;
	if (midiptt[0]) {
		sm->midipttfd = open(serptt, O_RDWR | O_NONBLOCK);
		if (sm->midipttfd == -1)
			errprintf(SEV_ERROR, "cannot open midi ptt device \"%s\"\n", midiptt);
	} else
		sm->midipttfd = -1;
	/* audio device */
	if ((sm->audiofd = open(aname, O_RDWR)) == -1)
		errprintf(SEV_FATAL, "cannot open audio device \"%s\"\n");
#ifdef HAVE_SYS_SOUNDCARD_H
	if (ioctl(sm->audiofd, SNDCTL_DSP_GETCAPS, &apar) == -1)
		errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_GETCAPS");
	if (!(apar & DSP_CAP_TRIGGER))
		errprintf(SEV_FATAL, "Sound driver does not support trigger\n");
	if (!(apar & DSP_CAP_DUPLEX))
		sm->drvmode |= SNDDRVMODE_HALFDUPLEX;
	/* set audio format */
	apar = AFMT_S16_NATIVE;
	if (ioctl(sm->audiofd, SNDCTL_DSP_SETFMT, &apar) == -1)
		errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SETFMT");
	if (apar != AFMT_S16_NATIVE) {
		apar = AFMT_S16_SWAPPED;
		if (ioctl(sm->audiofd, SNDCTL_DSP_SETFMT, &apar) == -1)
			errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SETFMT");
		if (apar == AFMT_S16_SWAPPED) {
			sm->drvmode |= SNDDRVMODE_CONVBYTESWAP;
		} else {
			apar = AFMT_U8;
			if (ioctl(sm->audiofd, SNDCTL_DSP_SETFMT, &apar) == -1)
				errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SETFMT");
			if (apar != AFMT_U8)
				errstr(SEV_FATAL, "audio driver supports neither S16LE, S16BE nor U8 format\n");
			sm->drvmode |= SNDDRVMODE_CONVU8;
		}
	}
	/* set number of channels */
	apar = 0;
	if (ioctl(sm->audiofd, SNDCTL_DSP_STEREO, &apar) == -1)
		errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_STEREO");
	if (apar != 0)
		errstr(SEV_FATAL, "audio driver does not support mono");
	/* set sampling rate */
	apar = sm->samplerate;
	if (ioctl(sm->audiofd, SNDCTL_DSP_SPEED, &apar) == -1)
		errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SPEED");
	errprintf(SEV_INFO, "audio driver \"%s\", fmt %s, samplerate %u/%u, %sduplex\n",
		  aname, (sm->drvmode & SNDDRVMODE_CONVU8) ? "U8" : 
		  (sm->drvmode & SNDDRVMODE_CONVBYTESWAP) ? "S16 (swapped)" : "S16",
		  sm->samplerate, apar,
		  (sm->drvmode & SNDDRVMODE_HALFDUPLEX) ? "half" : "full");
	sm->samplerate  = apar;
	sm->samplerateby100 = (apar+50) / 100;
	/* we need to set the block size to something useful, i.e. approx 10ms */
	apar = log2(sm->samplerateby100 + (sm->samplerateby100 >> 1));
	if (!(sm->drvmode & SNDDRVMODE_CONVU8))
		apar++;
	apar |= 0x00030000; /* 0x7fff0000; */
	errprintf(SEV_INFO, "audio driver fragment size %08x\n", apar);
	if (ioctl(sm->audiofd, SNDCTL_DSP_SETFRAGMENT, &apar) == -1)
		errstr(SEV_ERROR, "ioctl: SNDCTL_DSP_SETFRAGMENT");
	if (ioctl(sm->audiofd, SNDCTL_DSP_NONBLOCK, 0) == -1)
		errstr(SEV_ERROR, "ioctl: SNDCTL_DSP_NONBLOCK");
#endif /* HAVE_SYS_SOUNDCARD_H */
	/* initialize all modems */
	for (apar = 0; apar < sm->nrmodulator; apar++)
		sm->modulator[apar].code->setup(&sm->state[apar].mod, sm->samplerate);
	for (apar = 0; apar < sm->nrdemodulator; apar++) {
		sm->demodulator[apar].code->setup(&sm->state[apar].demod, sm->samplerate, &sr);
		if (sr > sm->overlap)
			sm->overlap = sr;
	}
	if (sm->overlap > SAMPLEBUF_SIZE/4)
		errprintf(SEV_FATAL, "overlap requested too big\n");
	/*
	 * sigusr1 is used to display the current state
	 */
	if (signal(SIGUSR1, sig_usr1) == SIG_ERR)
		errstr(SEV_ERROR, "signal");
	/*
	 * set the process to realtime privs
	 */
#ifdef HAVE_SCHED_H
	{
		struct sched_param schp;
		memset(&schp, 0, sizeof(schp));
		schp.sched_priority = sched_get_priority_min(SCHED_RR)+1;
#if 0
		if (sched_setscheduler(0, SCHED_RR, &schp) != 0)
			errstr(SEV_ERROR, "sched_setscheduler");
#endif
	}
#endif /* HAVE_SCHED_H */
	/*
	 * lock memory down
	 */
	if (mlockall(MCL_CURRENT) == -1) 
		errstr(SEV_ERROR, "mlockall");
	return sm;
}

/* --------------------------------------------------------------------- */

static const char *drvname = "Usersoundmodem, HB9JNX";
static const char *modename = "Blah";

extern inline void process_pkt(struct soundmodem_state *s, u_int8_t *pkt, unsigned int len)
{
	int nr;
	char buf[512];
	struct sm_message *msg = (struct sm_message *)pkt;
	struct sm_message msg2;

	if (len < 2)
		return;
	nr = (pkt[0] >> 4) & 0xf;
	if (nr >= MAXMODEM)
		return;
	switch (pkt[0]) {
	case KISS_CMD_DATA:
		if (nr >= s->nrmodulator)
			return;
		if (s->modulator[nr].code->enqueue(&s->state[nr].mod, pkt+1, len-1))
			s->modulator[nr].stat.tx_dropped++;
		else
			s->modulator[nr].stat.tx_packets++;
		if (snprintpkt(buf, sizeof(buf), pkt+1, len-1) != -1)
			printf("tx: %s", buf);
		return;

	case KISS_CMD_TXDELAY:
		s->chaccesspar.txdelay = pkt[1];
		errprintf(SEV_INFO, "kiss: txdelay = %ums\n", s->chaccesspar.txdelay*10);
		return;
			
	case KISS_CMD_SLOTTIME:
		s->chaccesspar.slottime = pkt[1];
		errprintf(SEV_INFO, "kiss: slottime = %ums\n", s->chaccesspar.slottime*10);
		return;

	case KISS_CMD_PPERSIST:
		s->chaccesspar.ppersist = pkt[1];
		errprintf(SEV_INFO, "kiss: ppersist = %u/256\n", s->chaccesspar.ppersist);
		return;

	case KISS_CMD_TXTAIL:
		s->chaccesspar.txtail = pkt[1];
		errprintf(SEV_INFO, "kiss: txtail = %ums\n", s->chaccesspar.txtail*10);
		return;

	case KISS_CMD_FULLDUP:
		s->chaccesspar.fulldup = !!pkt[1];
		errprintf(SEV_INFO, "kiss: %sduplex\n", s->chaccesspar.fulldup ? "full" : "half");
		return;

	case KISS_CMD_HARDWARE:
		if (len < sizeof(msg->hdr) || memcmp(pkt, SOUNDMODEM_SIGNATURE, sizeof(msg->hdr.signature))) {
			errprintf(SEV_WARNING, "unknown hw kiss packet: 0x%02x 0x%02x\n", pkt[0], pkt[1]);
			return;
		}
		switch (msg->hdr.cmd) {
		case SOUNDMODEM_CMD_REQCHACCESS:
			memcpy(msg2.hdr.signature, SOUNDMODEM_SIGNATURE, sizeof(msg2.hdr.signature));
			msg2.hdr.signature[0] |= (nr << 4);
			msg2.hdr.cmd = SOUNDMODEM_CMD_ACKCHACCESS;
			msg2.d.cp.tx_delay = s->chaccesspar.txdelay;
			msg2.d.cp.tx_tail = s->chaccesspar.txtail; 
			msg2.d.cp.slottime = s->chaccesspar.slottime;
			msg2.d.cp.ppersist = s->chaccesspar.ppersist;
			msg2.d.cp.fulldup = s->chaccesspar.fulldup; 
			simple_rxpacket(s, &msg2, sizeof(msg2.hdr) + sizeof(msg2.d.cp));
			return;

		case SOUNDMODEM_CMD_REQSTAT:
			memcpy(msg2.hdr.signature, SOUNDMODEM_SIGNATURE, sizeof(msg2.hdr.signature));
			msg2.hdr.signature[0] |= (nr << 4);
			msg2.hdr.cmd = SOUNDMODEM_CMD_ACKSTAT;
			msg2.d.cs.ptt_keyed = s->stat.ptt_keyed;
			msg2.d.cs.tx_packets = s->modulator[nr].stat.tx_packets;
			msg2.d.cs.tx_errors = s->modulator[nr].stat.tx_errors;
			msg2.d.cs.rx_packets = s->demodulator[nr].stat.rx_packets;
			msg2.d.cs.rx_errors = s->demodulator[nr].stat.rx_errors;
                        msg2.d.cs.int_rate = s->stat.int_rate;
                        msg2.d.cs.mod_cycles = s->modulator[nr].stat.tx_cycles;
                        msg2.d.cs.demod_cycles = s->demodulator[nr].stat.rx_cycles;
                        msg2.d.cs.dma_residue1 = s->stat.dma_residue1;
                        msg2.d.cs.dma_residue2 = s->stat.dma_residue2;
			msg2.d.cs.ptt = (s->ptt != 0);
			msg2.d.cs.dcd = (s->dcd != 0);
			simple_rxpacket(s, &msg2, sizeof(msg2.hdr) + sizeof(msg2.d.cs));
			return;

		case SOUNDMODEM_CMD_DIAGPAR:
			if (len < sizeof(msg->hdr) + sizeof(msg->d.diagpar)) {
				errprintf(SEV_WARNING, "invalid diagpar packet\n");
				return;
			}
			if (nr >= s->nrdemodulator)
				return;
			s->demodulator[nr].diagmode = msg->d.diagpar.modes;
			return;

		case SOUNDMODEM_CMD_CALIB:
			if (nr >= s->nrmodulator)
				return;
			if (len < sizeof(msg->hdr) + sizeof(msg->d.calib)) {
				errprintf(SEV_WARNING, "invalid calib packet\n");
				return;
			}
			s->modulator[nr].calibrate = msg->d.calib * s->samplerateby100;
			errprintf(SEV_INFO, "kiss: calibrate %u\n", msg->d.calib);
			return;

		case SOUNDMODEM_CMD_DRIVERNAMEREQ:
			memcpy(msg2.hdr.signature, SOUNDMODEM_SIGNATURE, sizeof(msg2.hdr.signature));
			msg2.hdr.signature[0] |= (nr << 4);
			msg2.hdr.cmd = SOUNDMODEM_CMD_DRIVERNAMEACK;
			strcpy(msg2.d.str, drvname);
			simple_rxpacket(s, &msg2, sizeof(msg2.hdr) + strlen(drvname) + 1);
			return;

		case SOUNDMODEM_CMD_MODENAMEREQ:
			memcpy(msg2.hdr.signature, SOUNDMODEM_SIGNATURE, sizeof(msg2.hdr.signature));
			msg2.hdr.signature[0] |= (nr << 4);
			msg2.hdr.cmd = SOUNDMODEM_CMD_MODENAMEACK;
			strcpy(msg2.d.str, modename);
			simple_rxpacket(s, &msg2, sizeof(msg2.hdr) + strlen(modename) + 1);
			return;
		}
		errprintf(SEV_WARNING, "invalid hw command packet: 0x%04x\n", msg->hdr.cmd);
		return;

	default:
		errprintf(SEV_WARNING, "unknown kiss packet: 0x%02x 0x%02x\n", pkt[0], pkt[1]);
		return;
	}
}

/* --------------------------------------------------------------------- */

/* we decode on the fly in place */

extern inline void kiss_decode(struct soundmodem_state *s, u_int8_t *b, int len)
{
	int nlen = 0;
	u_int8_t *p1 = b, *p2 = b;

//  printf("KISS in:");
      	while (len > 0) {
//  printf(" %02x", *p1);
		if (*p1 != KISS_FESC) {
			*p2++ = *p1++;
			nlen++;
			len--;
			continue;
		}
		if (len < 2)
			goto err; /* invalid escape */
		if (p1[1] == KISS_TFEND)
			*p2++ = KISS_FEND;
		else if (p1[1] == KISS_TFESC)
			*p2++ = KISS_FESC;
		else
			goto err; /* invalid escape */
		nlen++;
//  printf(" %02x", p1[1]);
		p1 += 2;
		len -= 2;
	}
//  printf("  nlen=%d\n", nlen);
	process_pkt(s, b, nlen);
	s->stat.kiss_in++;
	return;
 err:
//  printf("  error\n");
	s->stat.kiss_inerr++;
}

/* --------------------------------------------------------------------- */

extern inline void kiss_read(struct soundmodem_state *s)
{
	char *cp1, *cp2, *endp;
	int i;

	if (s->kissin.ptr >= KISSINBUF_SIZE)  /* discard oversized packets */
		s->kissin.ptr = 0;
	i = read(s->kissfd, s->kissin.b + s->kissin.ptr, KISSINBUF_SIZE - s->kissin.ptr);
	if (i < 0) {
		if (errno == EAGAIN)
			return;
		errstr(SEV_FATAL, "read");
	}
	if (i == 0)
		return;
	// printf("KISS: read %d ptr %u\n", i, s->kissin.ptr);
	s->kissin.ptr += i;
	endp = s->kissin.b + s->kissin.ptr;
	cp1 = memchr(s->kissin.b, KISS_FEND, s->kissin.ptr);
	while (cp1) {
		cp2 = memchr(cp1+1, KISS_FEND, endp - cp1 - 1);
		if (!cp2) {
			s->kissin.ptr = endp-cp1;
			memmove(s->kissin.b, cp1, s->kissin.ptr);
			return;
		}
		kiss_decode(s, cp1+1, cp2-cp1-1);
		cp1 = cp2;
	}
}

/* --------------------------------------------------------------------- */

#ifdef __i386__
#define GETTICK(x)                                                \
({                                                                \
	__asm__ __volatile__("rdtsc" : "=a" (x) : : "dx");        \
})
#else /* __i386__ */
#define GETTICK(x)  ((x)=0)
#endif /* __i386__ */

/* --------------------------------------------------------------------- */

extern inline void processsamples(struct soundmodem_state *sm)
{
	int16_t *p1 = sm->snd.rxsbuf + sm->overlap;
	int8_t *p2 = (int8_t *)p1;
	int ret, i, tick1, tick2;

	i = SAMPLEBUF_SIZE-sm->overlap;
	if (!(sm->drvmode & SNDDRVMODE_CONVU8))
		i <<= 1;
	ret = read(sm->audiofd, p1, i);
	if (ret < 0) {
		if (errno == EINTR || errno == EAGAIN)
			return;
		errstr(SEV_FATAL, "read");
	}
	if (ret == 0)
		return;
	sm->stat.dma_residue2 = sm->stat.dma_residue1;
	sm->stat.dma_residue1 = ret / sizeof(int16_t);
	sm->stat.int_rate++;
	//errprintf(SEV_DEBUG, "audioread: ret %i\n", ret);
	if (sm->drvmode & SNDDRVMODE_CONVU8) {
		p1 += ret;
		p2 += ret;
		for (i = 0; i < ret; i++)
			*--p1 = ((*--p2) - 0x80) << 8;
	} else {
		if (ret & 1)
			errprintf(SEV_FATAL, "audioread: ret %i, nonintegral number of samples read\n", ret);
		ret >>= 1;
		if (sm->drvmode & SNDDRVMODE_CONVBYTESWAP) {
			for (i = 0; i < ret; i++, p1++) {
				u_int16_t x = *p1;
				*p1 = ((x >> 8) & 0x00ff) | ((x << 8) & 0xff00);
			}
		}
	}
	p1 = sm->snd.rxsbuf + sm->overlap;
	for (i = 0; i < sm->nrdemodulator; i++) {
		GETTICK(tick1);
		sm->demodulator[i].code->demodulate(&sm->state[i].demod, p1, ret);
		GETTICK(tick2);
		sm->demodulator[i].stat.rx_cycles = tick2 - tick1;
	}
	for (i = 0; i < sm->nrdemodulator; i++)
		sm_diag(&sm->state[i].demod, 0, 0, 0, p1, ret);
	memmove(sm->snd.rxsbuf, sm->snd.rxsbuf + ret, sm->overlap * sizeof(int16_t));
	sm->access.slotcnt -= ret;
}

/* --------------------------------------------------------------------- */

extern __inline__ void output_status(struct soundmodem_state *s, unsigned int ptt)
{
	int o[2] = { 0, 0 };
	int xptt = s->invert_ptt ^ (s->ptt = !!ptt);
	int xdcd = s->invert_dcd ^ !!s->dcd;
	char c = 0;

	if (s->serpttfd >= 0) {
		o[xptt] |= TIOCM_RTS;
		o[xdcd] |= TIOCM_DTR;
		if (o[0])
			if (ioctl(s->serpttfd, TIOCMBIC, o))
				errstr(SEV_FATAL, "ioctl: TIOCMBIC");
		if (o[1])
			if (ioctl(s->serpttfd, TIOCMBIS, o+1))
				errstr(SEV_FATAL, "ioctl: TIOCMBIS");
		if (xptt)
			write(s->serpttfd, &c, 1);
	}
	if (s->midipttfd >= 0 && xptt)
		write(s->midipttfd, &c, 1);
	if (s->parpttfd >= 0) {
		c = (xdcd << 1) | xptt;
		write(s->parpttfd, &c, 1);
	}
}

/* --------------------------------------------------------------------- */

static void prkissout(u_int8_t *cp, int len)
{
	printf("KISS out:");
	for (; len > 0; len--, cp++)
		printf(" %02x", *cp);
	printf("\n");
}

static short processinput(struct soundmodem_state *sm, short pttevmask)
{
	struct pollfd pfd[2];
	int ret;

	pfd[0].fd = sm->audiofd;
	pfd[0].events = pttevmask;
	if (!(sm->drvmode & SNDDRVMODE_HALFDUPLEX) || !pttevmask)
		pfd[0].events |= POLLIN;
	pfd[1].fd = sm->kissfd;
	pfd[1].events = POLLIN | ((sm->kissout.rd != sm->kissout.wr) ? POLLOUT : 0);
	for (;;) {
		ret = poll(pfd, 2, 1000);
		if (ret == -1 && errno != EINTR)
			errstr(SEV_FATAL, "poll");
		if (ret == 0) {
			errprintf(SEV_WARNING, "poll timeout, soundcard DMA problem?\n");
			continue;
		}
		if (pfd[1].revents & POLLIN)
			kiss_read(sm);
		if (pfd[1].revents & POLLOUT && sm->kissout.rd != sm->kissout.wr) {
			ret = (sm->kissout.rd >= sm->kissout.wr) ? KISSOUTBUF_SIZE : sm->kissout.wr;
			ret = write(sm->kissfd, sm->kissout.b + sm->kissout.rd, ret - sm->kissout.rd);
			if (ret < 0) {
				if (errno != EAGAIN && errno != EINTR)
					errstr(SEV_FATAL, "write");
			} else {
				sm->kissout.rd = (sm->kissout.rd + ret) % KISSOUTBUF_SIZE;
			}
		}
		if (pfd[0].revents & POLLIN)
			processsamples(sm);
		if (pfd[0].revents & (POLLIN|POLLOUT)) {
			output_status(sm, pttevmask);
			return pfd[0].revents;
		}
	}
}

/* ---------------------------------------------------------------------- */

static void transmit(struct soundmodem_state *sm, int16_t *buf, int samples)
{
	int16_t *p1;
	int8_t *p2;
	int i, bytes;

	if (sm->drvmode & SNDDRVMODE_CONVU8) {
		p1 = buf;
		p2 = (int8_t *)buf;
		for (i = 0; i < samples; i++)
			*p2++ = ((*p1++) >> 8) + 0x80;
		bytes = samples;
	} else {
		bytes = 2*samples;
		if (sm->drvmode & SNDDRVMODE_CONVBYTESWAP) {
			p1 = buf;
			for (i = 0; i < samples; i++, p1++) {
				u_int16_t x = *p1;
				*p1 = ((x >> 8) & 0x00ff) | ((x << 8) & 0xff00);
			}
		}
	}
	p2 = (int8_t *)buf;
	while (bytes > 0) {
		while (!(processinput(sm, POLLOUT) & POLLOUT));
		i = write(sm->audiofd, p2, bytes);
		if (i < 0) {
			if (errno == EINTR || errno == EAGAIN)
				return;
			errstr(SEV_FATAL, "write");
		} else {
			p2 += i;
			bytes -= i;
		}
	}
}

/* ---------------------------------------------------------------------- */

static u_int16_t random_seed;

extern inline u_int16_t random_num(void)
{
	random_seed = 28629 * random_seed + 157;
	return random_seed;
}

/* --------------------------------------------------------------------- */

extern inline int find_txready(struct soundmodem_state *sm)
{
	int i;

	for (i = 0; i < sm->nrmodulator; i++)
		if (sm->modulator[i].code->can_send(&sm->state[i].mod))
			return i;
	return -1;
}	

extern inline int find_calib(struct soundmodem_state *sm)
{
	int i;

	for (i = 0; i < sm->nrmodulator; i++)
		if (sm->modulator[i].calibrate > 0)
			return i;
	return -1;
}	

/* --------------------------------------------------------------------- */

#define TXBUFSIZE 1024

static void modem_run(struct soundmodem_state *sm)
{
	int16_t txbuf[TXBUFSIZE];
	int apar, txnr, i, j, tick1, tick2;

 errprintf(SEV_INFO, "modem_run\n");
#ifdef HAVE_SYS_SOUNDCARD_H
	apar = PCM_ENABLE_INPUT;
//	if (!(sm->drvmode & SNDDRVMODE_HALFDUPLEX))
//		apar |= PCM_ENABLE_OUTPUT;
	if (ioctl(sm->audiofd, SNDCTL_DSP_SETTRIGGER, &apar) == -1)
		errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SETTRIGGER");
#endif /* HAVE_SYS_SOUNDCARD_H */

  dorx:
#ifdef HAVE_SYS_SOUNDCARD_H
	if (sm->drvmode & SNDDRVMODE_HALFDUPLEX) {
		apar = PCM_ENABLE_INPUT;
		if (ioctl(sm->audiofd, SNDCTL_DSP_SETTRIGGER, &apar) == -1)
			errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SETTRIGGER");
	}
#endif /* HAVE_SYS_SOUNDCARD_H */
	sm->access.slotcnt = sm->samplerateby100 * sm->chaccesspar.slottime;
	for (;;) {
		processinput(sm, 0);
		if (sm->access.slotcnt > 0)
			continue;
		sm->access.slotcnt = sm->samplerateby100 * sm->chaccesspar.slottime;
		if (find_calib(sm) != -1)
			goto keyup;
		if (find_txready(sm) != -1)
			goto arbitrate1;
       		continue;
		
	  arbitrate1:
		if (sm->chaccesspar.fulldup)
			goto keyup;
		if (sm->dcd)
			continue;
		if ((random_num() % 256) > sm->chaccesspar.ppersist)
			continue;
		goto keyup;
	}
		
  keyup:
	sm->stat.ptt_keyed++;
	if ((txnr = find_txready(sm)) != -1)
		goto starttx;
  startcalib:
	while ((txnr = find_calib(sm)) != -1) {
		i = sm->modulator[txnr].calibrate;
		if (i > TXBUFSIZE)
			i = TXBUFSIZE;
		GETTICK(tick1);
		i = sm->modulator[txnr].code->modulate_calib(&sm->state[txnr].mod, txbuf, i);
		GETTICK(tick2);
		sm->modulator[txnr].stat.tx_cycles = tick2 - tick1;
		if (i <= 0)
			errprintf(SEV_FATAL, "modulator modulate_calib error\n");
		sm->modulator[txnr].calibrate -= i;
		transmit(sm, txbuf, i);
		if ((i = find_txready(sm)) != -1) {
			txnr = i;
			goto starttx;
		}
	}
	/* terminate transmission */
#ifdef HAVE_SYS_SOUNDCARD_H
	if (sm->drvmode & SNDDRVMODE_HALFDUPLEX) {
		if (ioctl(sm->audiofd, SNDCTL_DSP_SYNC, 0) == -1)
			errstr(SEV_FATAL, "ioctl: SNDCTL_DSP_SYNC");
	}
#endif /* HAVE_SYS_SOUNDCARD_H */
	goto dorx;

  starttx:
	j = sm->samplerateby100 * sm->chaccesspar.txdelay;
	while (j > 0) {
		i = j;
		if (i > TXBUFSIZE)
			i = TXBUFSIZE;
		GETTICK(tick1);
		i = sm->modulator[txnr].code->modulate_calib(&sm->state[txnr].mod, txbuf, i);
		GETTICK(tick2);
		sm->modulator[txnr].stat.tx_cycles = tick2 - tick1;
		if (i <= 0)
			errprintf(SEV_FATAL, "modulator modulate_calib error\n");
		transmit(sm, txbuf, i);
		j -= i;
	}
	for (;;) {
		for (;;) {
			GETTICK(tick1);
			i = sm->modulator[txnr].code->modulate(&sm->state[txnr].mod, txbuf, TXBUFSIZE);
			GETTICK(tick2);
			sm->modulator[txnr].stat.tx_cycles = tick2 - tick1;
			if (i <= 0)
				break;
			transmit(sm, txbuf, i);
		}
		if ((j = find_txready(sm)) == -1)
			break;
		txnr = j;
		j = sm->samplerateby100; /* inter mode txdelay 10 ms for now */
		while (j > 0) {
			i = j;
			if (i > TXBUFSIZE)
				i = TXBUFSIZE;
			GETTICK(tick1);
			i = sm->modulator[txnr].code->modulate_calib(&sm->state[txnr].mod, txbuf, i);
			GETTICK(tick2);
			sm->modulator[txnr].stat.tx_cycles = tick2 - tick1;
			if (i <= 0)
				errprintf(SEV_FATAL, "modulator modulate_calib error\n");
			transmit(sm, txbuf, i);
			j -= i;
		}
	}
	j = sm->samplerateby100 * sm->chaccesspar.txtail;
	while (j > 0) {
		i = j;
		if (i > TXBUFSIZE)
			i = TXBUFSIZE;
		GETTICK(tick1);
		i = sm->modulator[txnr].code->modulate_calib(&sm->state[txnr].mod, txbuf, i);
		GETTICK(tick2);
		sm->modulator[txnr].stat.tx_cycles = tick2 - tick1;
		if (i <= 0)
			errprintf(SEV_FATAL, "modulator modulate_calib error\n");
		transmit(sm, txbuf, i);
		j -= i;
	}
	goto startcalib;
}

/* --------------------------------------------------------------------- */

static void write_pid(pid_t pid)
{
	FILE *f = fopen(PIDFILE, "a");
	if (!f) {
		errprintf(SEV_INFO, "cannot open PID file %s\n", PIDFILE);
		return;
	}
	fprintf(f, "%i\n", pid);
	fclose(f);
}

/* --------------------------------------------------------------------- */

static void kill_soundmodemd(void)
{
	FILE *f = fopen(PIDFILE, "r");
	pid_t pid;

	if (!f) {
		errprintf(SEV_INFO, "cannot open PID file %s\n", PIDFILE);
		return;
	}
	while (fscanf(f, " %i \n", &pid) == 1) 
		if (pid <= 0 || pid == getpid())
			errprintf(SEV_INFO, "invalid pid %i\n", pid);
		else {
			kill(pid, SIGTERM);
			errprintf(SEV_INFO, "terminating process %i\n", pid);
		}
	fclose(f);
	unlink(PIDFILE);
}

/* --------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
	int c;
	int err = 0;
	pid_t pid;
	const char *cfgname = CONFIGDIR "/soundmodem.conf";
	int daemon = 1;

	while ((c = getopt(argc, argv, "c:lkn")) != -1) 
		switch (c) {
		case 'c':
			cfgname = optarg;
			break;

		case 'l':
			logging = 1;
			break;

		case 'k':
			kill_soundmodemd();
			exit(0);

		case 'n':
			daemon = 0;
			break;

		default:
			err++;
			break;
		}
	if (err) {
		fprintf(stderr, "usage: soundmodem [-c <cfgfilename>] [-l] [-k] [-n]\n");
		exit(1);
	}
	if (!daemon) {
		setsid(); /* become a process group leader and drop controlling terminal */
		fclose(stdin); /* no more standard in */
		if (logging)
			openlog("soundmodem", LOG_PID, LOG_DAEMON);
		state = init(cfgname);
		signal(SIGHUP, SIG_IGN);
		signal(SIGTERM, sighandler);
		modem_run(state);
		errprintf(SEV_FATAL, "dying\n");
		exit(1);
	}
	switch (pid = fork()) {
	case -1:
		errstr(SEV_FATAL, "fork\n");
		exit(1);

	case 0: /* child process */
		setsid(); /* become a process group leader and drop controlling terminal */
		fclose(stdin); /* no more standard in */
		if (logging)
			openlog("soundmodem", LOG_PID, LOG_DAEMON);
		state = init(cfgname);
		signal(SIGHUP, SIG_IGN);
		signal(SIGTERM, sighandler);
		modem_run(state);
		errprintf(SEV_FATAL, "dying\n");
		exit(1);

	default:
		write_pid(pid);
		fprintf(stderr, "soundmodem: child pid: %u\n", pid);
		break;
	}
	exit(0);
}
 
/* --------------------------------------------------------------------- */
