#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

#include "soundmodem.h"
#include "hdlc.h"
#include "complex.h"
#include "nqconf.h"
#include "fec.h"
#include "newqpskrx.h"
#include "tbl.h"
#include "misc.h"
#include "filter.h"

/* --------------------------------------------------------------------- */

static void rxidle(void **);
static void rxtune(void **);
static void rxdata(void **);

/* --------------------------------------------------------------------- */

void init_newqpskrx(void **state)
{
	struct rxstate *s = *state;
	int i;

	/* clear dcd */
	sm_setdcd(state, 0);
	/* switch to idle mode */
	s->rxroutine = rxidle;
	s->rxwindowfunc = ToneWindowInp;
	s->carrierfreq = 0.0;
	s->acceptance = 0;
	for (i = 0; i < TuneCarriers; i++) {
		s->tunepower[i] = 0.0;
		s->tunephase[i] = 0.0;
		s->tunecorr[i].re = 0.0;
		s->tunecorr[i].im = 0.0;
	}
}

/* --------------------------------------------------------------------- */

static void putbitbatch(void **state, unsigned data)
{
	struct rxstate *s = *state;
	unsigned i, bit;

	for (i = 0; i < s->fec.bitbatchlen; i++) {
		bit = (data & (1 << i)) ? 1 : 0;
		s->shreg |= bit << 9;
		if (s->shreg & 1) {
			hdlc_putbits(&s->hdlc, (s->shreg >> 1), state);
			s->shreg &= ~0xff;
			s->shreg |= 0x100;
		}
		s->shreg >>= 1;
	}
}

/* --------------------------------------------------------------------- */

/*
 * Anti-alias filter, Hilbert transform and mix with the internal NCO.
 */
static void rxfilter(void **state, float *in, complex * out)
{
	struct rxstate *s = *state;
	complex z;
	int i;

	/* make room for new samples at the end of RX window */
	for (i = 0; i < WindowLen - SymbolLen / 2; i++)
		out[i] = out[i + SymbolLen / 2];
	in -= DecimateRatio * SymbolLen / 2 + AliasFilterLen;
	for (i = WindowLen - SymbolLen / 2; i < WindowLen; i++) {
		in += DecimateRatio;
		out[i].re = mac(in, AliasFilterInpI, AliasFilterLen);
		out[i].im = mac(in, AliasFilterInpQ, AliasFilterLen);
		s->rxphase += s->carrierfreq;
		if (s->rxphase > M_PI)
			s->rxphase -= 2 * M_PI;
		if (s->rxphase < -M_PI)
			s->rxphase += 2 * M_PI;
		z.re = cos(s->rxphase);
		z.im = sin(s->rxphase);
		out[i] = cmul(out[i], z);
	}
}

static void fft(complex * in, complex * out)
{
	int i, j, k;
	int s, sep, width, top, bot;
	float tr, ti;

	/* order the samples in bit reverse order */
	for (i = 0; i < WindowLen; i++) {
		j = rbits8(i) >> (8 - WindowLenLog);
		out[j] = in[i];
	}
	/* in-place FFT */
	sep = 1;
	for (s = 1; s <= WindowLenLog; s++) {
		width = sep;    /* butterfly width =  2^(s-1) */
		sep <<= 1;      /* butterfly separation = 2^s */
		for (j = 0; j < width; j++) {
			k = WindowLen * j / sep;
			for (top = j; top < WindowLen; top += sep) {
				bot = top + width;
				tr = out[bot].re * CosTable[k] + out[bot].im * SinTable[k];
				ti = out[bot].im * CosTable[k] - out[bot].re * SinTable[k];
				out[bot].re = out[top].re - tr;
				out[bot].im = out[top].im - ti;
				out[top].re = out[top].re + tr;
				out[top].im = out[top].im + ti;
			}
		}
	}
}

/* --------------------------------------------------------------------- */

void newqpskrx(void **state)
{
	struct rxstate *s = *state;
	complex tmp[WindowLen];
	complex z;
	int i, j;

	rxfilter(state, s->rxbuf + s->bufptr, s->rxwin);

	/* apply the window function */
	for (i = 0; i < WindowLen; i++) {
		tmp[i].re = s->rxwin[i].re * s->rxwindowfunc[i];
		tmp[i].im = s->rxwin[i].im * s->rxwindowfunc[i];
	}

	/* fft */
	fft(tmp, s->fftbuf);

	/* select the wanted FFT bins and adjust the phases */
	s->rxphasecorr = (s->rxphasecorr + SymbolLen / 2) % WindowLen;
	j = FirstDataCarr;
	for (i = 0; i < DataCarriers; i++) {
		z.re = CosTable[(j * s->rxphasecorr) % WindowLen];
		z.im = -SinTable[(j * s->rxphasecorr) % WindowLen];
		s->rxpipe[s->rxptr][i] = cmul(s->fftbuf[j], z);
		j += DataCarrSepar;
	}

	/* process the data */
	s->rxroutine(state);

	s->rxptr = (s->rxptr + 1) % RxPipeLen;
}

/* --------------------------------------------------------------------- */

static void rxidle(void **state)
{
	struct rxstate *s = *state;
	float x;
	complex z;
	int i, j, cntr;
	unsigned int prev, curr;
	char buf[256];

	curr = s->rxptr;
	prev = (curr - 1) % RxPipeLen;

	j = (FirstTuneCarr - FirstDataCarr) / DataCarrSepar;

	cntr = 0;
	for (i = 0; i < TuneCarriers; i++) {
		x = cpwr(s->rxpipe[curr][j]);
		s->tunepower[i] = avg(s->tunepower[i], x, RxAverFollow);

		z = ccor(s->rxpipe[curr][j], s->rxpipe[prev][j]);
		s->tunecorr[i].re = avg(s->tunecorr[i].re, z.re, RxAverFollow);
		s->tunecorr[i].im = avg(s->tunecorr[i].im, z.im, RxAverFollow);

		if (2 * cmod(s->tunecorr[i]) > s->tunepower[i])
			cntr++;

		j += TuneCarrSepar / DataCarrSepar;
	}

	if (cntr >= TuneCarriers - 1)
		s->acceptance++;
	else if (s->acceptance > 0)
		s->acceptance--;

	if (s->acceptance < 2 * RxMinTune)
		return;

#ifdef RxAvoidPTT
	if (sm_getptt(state))
		return;
#endif

	/* ok, we have a carrier */
	sm_setdcd(state, 1);

	for (i = 0; i < TuneCarriers; i++)
		s->tunephase[i] = carg(s->tunecorr[i]);

	x = phaseavg(s->tunephase, TuneCarriers);
	s->carrierfreq += x * 4.0 / WindowLen;

	buf[0] = 0;
	for (i = 0; i < TuneCarriers; i++) {
		x = s->tunepower[i] / (s->tunepower[i] - cmod(s->tunecorr[i]));
		x = 10 * log10(x);
		j = strlen(buf);
		sprintf(buf + j, "%+.1fHz/%.1fdB  ",
			  s->tunephase[i] * 4 / WindowLen / Hertz, x);
	}
	errprintf(SEV_INFO, "Tune tones: %s\n", buf);

	/* switch to tune mode */
	s->rxroutine = rxtune;
	s->rxwindowfunc = DataWindowInp;
	s->atsymbol = 1;
	s->acceptance = 0;
	s->statecntr = 2 * RxTuneTimeout;
	for (i = 0; i < TuneCarriers; i++) {
		s->power_at[i] = s->tunepower[i];
		s->corr1_at[i].re = s->tunepower[i];
		s->corr1_at[i].im = 0.0;
		s->corr2_at[i].re = s->tunepower[i];
		s->corr2_at[i].im = 0.0;

		s->power_inter[i] = s->tunepower[i];
		s->corr1_inter[i].re = s->tunepower[i];
		s->corr1_inter[i].im = 0.0;
		s->corr2_inter[i].re = s->tunepower[i];
		s->corr2_inter[i].im = 0.0;
	}
}

/* --------------------------------------------------------------------- */

static void rxtune(void **state)
{
	struct rxstate *s = *state;
	int i, j, cntr;
	unsigned int prev2, prev1, curr;
	complex *cor1, *cor2, z;
	float *pwr, x;

	if (s->statecntr-- <= 0) {
		/* timeout waiting sync - go back to idling */
		init_newqpskrx(state);
		return;
	}
	curr = s->rxptr;
	prev1 = (curr - 1) % RxPipeLen;
	prev2 = (curr - 2) % RxPipeLen;

	s->atsymbol ^= 1;

	if (s->atsymbol) {
		pwr = s->power_at;
		cor1 = s->corr1_at;
		cor2 = s->corr2_at;
	} else {
		pwr = s->power_inter;
		cor1 = s->corr1_inter;
		cor2 = s->corr2_inter;
	}

	j = (FirstTuneCarr - FirstDataCarr) / DataCarrSepar;
	for (i = 0; i < TuneCarriers; i++) {
		x = cpwr(s->rxpipe[curr][j]);
		pwr[i] = avg(pwr[i], x, RxAverFollow - 1);

		z = ccor(s->rxpipe[curr][j], s->rxpipe[prev1][j]);
		cor1[i].re = avg(cor1[i].re, z.re, RxAverFollow - 1);
		cor1[i].im = avg(cor1[i].im, z.im, RxAverFollow - 1);

		z = ccor(s->rxpipe[curr][j], s->rxpipe[prev2][j]);
		cor2[i].re = avg(cor2[i].re, z.re, RxAverFollow - 1);
		cor2[i].im = avg(cor2[i].im, z.im, RxAverFollow - 1);

		j += TuneCarrSepar / DataCarrSepar;
	}

	if (!s->atsymbol)
		return;

	cntr = 0;
	for (i = 0; i < TuneCarriers; i++) {
		if (s->power_at[i] > s->power_inter[i]) {
			x = s->power_at[i];
			z = s->corr2_at[i];
		} else {
			x = s->power_inter[i];
			z = s->corr2_inter[i];
		}

		if (-z.re > x / 2)
			cntr++;
	}

	if (cntr >= TuneCarriers - 1)
		s->acceptance++;
	else if (s->acceptance > 0)
		s->acceptance--;

	if (s->acceptance < RxMinSync)
		return;

	cntr = 0;
	for (i = 0; i < TuneCarriers; i++) {
		if (s->corr2_at[i].re < s->corr2_inter[i].re)
			cntr++;
		else
			cntr--;
	}

	if (cntr < 0) {
		s->atsymbol = 0;
		pwr = s->power_inter;
		cor1 = s->corr1_inter;
		cor2 = s->corr2_inter;
	}

	for (i = 0; i < TuneCarriers; i++) {
		z.re = -cor2[i].re;
		z.im = cor1[i].re - (pwr[i] + cor2[i].re) / 2;

		s->syncdelay[i] = carg(z);

		z.re = -cor2[i].re;
		z.im = cor2[i].im;
		s->syncphase[i] = carg(z);
	}

	x = phaseavg(s->syncdelay, TuneCarriers) / M_PI;
	s->skip = (0.5 - x) * DecimateRatio * SymbolLen;

	errprintf(SEV_INFO, "Sync: %d (%s-symbol)\n", s->skip, s->atsymbol ? "at" : "inter");

	x = phaseavg(s->syncphase, TuneCarriers);
	s->carrierfreq -= x * 2.0 / WindowLen;
	errprintf(SEV_INFO, "Preamble at: %+.2fHz\n", s->carrierfreq / Hertz);

	/* switch to data mode */
	s->rxroutine = rxdata;
	s->atsymbol ^= 1;
	s->acceptance = DCDMaxDrop / 2;
	s->updhold = RxUpdateHold;
	s->bitbatches = 0;
	/* init deinterleaver */
	init_inlv(&s->fec);
	for (i = 0; i < DataCarriers; i++) {
		s->pherr[i] = 0.0;
		s->pheavg[i] = 0.0;
		s->dcdavg[i] = 0.0;
		s->power[i] = 0.0;
		s->correl[i] = 0.0;
		s->fecerrors[i] = 0;
	}
}

/* --------------------------------------------------------------------- */

static void rxdata(void **state)
{
	struct rxstate *s = *state;
	int rxword[SymbolBits], errword[SymbolBits];
	int i, j, bits, data, dcd, errs;
	unsigned int prev2, prev1, curr;
	complex z;
	float x;
	char buf[256];

	/* nothing to do if inter-symbol */
	if ((s->atsymbol ^= 1) == 0)
		return;

	curr = s->rxptr;
	prev1 = (curr - 1) % RxPipeLen;
	prev2 = (curr - 2) % RxPipeLen;

	for (i = 0; i < SymbolBits; i++) {
		rxword[i] = 0;
		errword[i] = 0;
	}

	/*
	 * Delay the dcd/pherror/power/sync updates for the first
	 * `RxUpdateHold symbols because tune phase hasn't
	 * necessarily ended yet.
	 */
	if (s->updhold)
		s->updhold--;

	for (i = 0; i < DataCarriers; i++) {
		/* get the angle and add bias */
		z = ccor(s->rxpipe[curr][i], s->rxpipe[prev2][i]);
		x = carg(z) + M_PI / PhaseLevels;

		if (x < 0)
			x += 2 * M_PI;

		/* work out the bits */
		bits = (int) (x * PhaseLevels / (2 * M_PI));
		bits &= (1 << SymbolBits) - 1;

		/* calculate phase error (`0.5 compensates the bias) */
		s->pherr[i] = x - (bits + 0.5) * 2 * M_PI / PhaseLevels;

		/* flip the top bit back */
		bits ^= 1 << (SymbolBits - 1);

		/* gray decode */
		data = 0;
		for (j = 0; j < SymbolBits; j++)
			data ^= bits >> j;

		/* put the bits to rxword (top carrier first) */
		for (j = 0; j < SymbolBits; j++)
			if (data & (1 << (SymbolBits - 1 - j)))
				rxword[j] |= 1 << i;

		/*
		 * Update DCD, phase error, power and
		 * sync correlation averages.
		 */
		if (!s->updhold) {
			s->dcdavg[i] = s->pherr[i] * s->pherr[i] * DCDTuneAverWeight + s->dcdavg[i] * (1 - DCDTuneAverWeight);

			s->pheavg[i] = s->pherr[i] * DCDTuneAverWeight + s->pheavg[i] * (1 - DCDTuneAverWeight);

			x = cpwr(s->rxpipe[curr][i]);
			s->power[i] = avg(s->power[i], x, RxDataSyncFollow);

			x = ccorI(s->rxpipe[prev1][i], s->rxpipe[curr][i]) - ccorI(s->rxpipe[prev1][i], s->rxpipe[prev2][i]);
			s->correl[i] = avg(s->correl[i], x, RxDataSyncFollow);
		}
	}

	/* count carriers that have small enough phase error */
	dcd = 0;
	for (i = 0; i < DataCarriers; i++)
		if (s->dcdavg[i] < DCDThreshold / PhaseLevels / PhaseLevels)
			dcd++;

	/* decide if this was a "good" symbol */
	if (dcd < DataCarriers / 2)
		dcd = 0;
	else
		dcd = 1;

	/* feed the data to the decoder */
	for (i = 0; i < SymbolBits; i++) {
		rxword[i] = deinlv(&s->fec, rxword[i]);
		rxword[i] = fecdecode(&s->fec, rxword[i], &errword[i]);
		putbitbatch(state, rxword[i]);
	}

	/* count FEC errors */
	if (!s->updhold && dcd) {
		for (i = 0; i < SymbolBits; i++) {
			s->bitbatches++;
			for (j = 0; j < DataCarriers; j++)
				if (errword[i] & (1 << j))
					s->fecerrors[j]++;
		}
	}

	/* if the symbol wasn't "good", drop acceptance */
	if (!dcd)
		s->acceptance--;
	else if (s->acceptance < DCDMaxDrop)
		s->acceptance++;

	if (s->acceptance)
		return;

	/* DCDMaxDrop subsequent "bad" symbols, it's gone... */
	errprintf(SEV_INFO, "Carrier lost at: %+.2fHz\n", s->carrierfreq / Hertz);
	errs = 0;
	buf[0] = 0;
	for (i = 0; i < DataCarriers; i++) {
		errs += s->fecerrors[i];
		j = strlen(buf);
		sprintf(buf + j, "%02d ", s->fecerrors[i]);
	}
	errprintf(SEV_INFO, "FEC errors: %s: %03d / %03d\n", buf, errs, s->bitbatches);

	/* go back to idling */
	init_newqpskrx(state);
	return;
}

/* --------------------------------------------------------------------- */

