/*****************************************************************************/

/*
 *      libmodnewqpsk.c  --  Linux Userland Soundmodem NEWQPSK modulator.
 *
 *      Copyright (C) 1999  Tomi Manninen, OH2BNS (tomi.manninen@hut.fi)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*****************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "soundmodem.h"
#include "hdlc.h"
#include "nqconf.h"
#include "fec.h"
#include "newqpsktx.h"
#include "complex.h"
#include "tbl.h"
#include "fec.h"

/* --------------------------------------------------------------------- */

#define SAMPLERATE(x) 	((x) * DecimateRatio*SymbolLen/DataCarriers/SymbolBits)

static void *init(const char *config, unsigned *samplerate)
{
	struct txstate *s = calloc(1, sizeof(struct txstate));
	unsigned fec = 3, inlv = 8, bps = 3000;
	char *cp;

	if (!s)
		errstr(SEV_FATAL, "modnewqpsk: cannot allocate memory");
	hdlc_inittx(&s->hdlc);
	cp = strtok(config, ",");
	while (cp) {
		if (!strncmp(cp, "fec=", 4))
			fec = strtoul(cp+4, NULL, 0);
		else if (!strncmp(cp, "inlv=", 5))
			inlv = strtoul(cp+5, NULL, 0);
		else if (!strncmp(cp, "bps=", 4))
			bps = strtoul(cp+4, NULL, 0);
		else
			errprintf(SEV_WARNING, "modneqpsk: invalid parameter \"%s\"\n", cp);
		cp = strtok(NULL, ",");
	}
	errprintf(SEV_INFO, "modqpsk: bps %u, fec %u, inlv %u\n", bps, fec, inlv);
	if (!(bps == 2500 || bps == 3000))
		errstr(SEV_FATAL, "modnewqpsk: bps must be 2500 or 3000\n");
	if (fec < 0 || fec > 4)
		errstr(SEV_FATAL, "modnewqpsk: fec must be 0...4\n");
	if (inlv < 0 || inlv > 16)
		errstr(SEV_FATAL, "modnewqpsk: inlv must be 0...16\n");
	*samplerate = SAMPLERATE(bps);
	s->bps = bps;
	s->fec.feclevel = fec;
	s->fec.inlv = inlv;
	return s;
}

static int setup(void **state, unsigned samplerate)
{
	struct txstate *s = *state;

	if (samplerate != SAMPLERATE(s->bps))
		errprintf(SEV_FATAL, "modnewqpsk: need %usps, got %usps\n",
			  SAMPLERATE(s->bps), samplerate);
	init_fec(&s->fec);
	init_newqpsktx(state);
	return 0;
}

static int modulate(void **state, int16_t *samples, int numsamp)
{
	struct txstate *s = *state;
	int n, cnt = 0;

	if (s->txdone) {
		s->txdone = 0;
		return 0;
	}
	while (!s->txdone && numsamp >= SymbolLen * DecimateRatio) {
		n = newqpsktx(state, samples);
		samples += n;
		numsamp -= n;
		cnt += n;
	}
	return cnt;
}

static int modulate_calib(void **state, int16_t *samples, int numsamp)
{
	struct txstate *s = *state;
	int n, cnt = 0;

	s->tuneonly = 1;
	while (numsamp >= SymbolLen * DecimateRatio) {
		n = newqpsktx(state, samples);
		samples += n;
		numsamp -= n;
		cnt += n;
	}
	s->tuneonly = 0;
	return cnt;
}

static int enqueue(void **state, u_int8_t *pkt, int pktlen)
{
	struct txstate *s = *state;

	return hdlc_enqueue(&s->hdlc, pkt, pktlen);
}

static int can_send(void **state)
{
	struct txstate *s = *state;

	return !hdlc_txempty(&s->hdlc);
}

/* --------------------------------------------------------------------- */

static const char copyright[] = "(C) 1999 by Tomi Manninen, OH2BNS";
static const char modename[] = "newqpsk";

struct soundmodem_modulator modulator_0 = {
	modename, copyright,
	init, setup, modulate, modulate_calib, enqueue, can_send
};

/* --------------------------------------------------------------------- */
