//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

// SYNTAX.CPP

// Performs syntactic analysis of the tokens returned by the LexicalAnalyzer.

#ifndef  __CONIO_H
#include <conio.h>
#endif

#ifndef  __CTYPE_H
#include <ctype.h>
#endif

#ifndef  __STDIO_H
#include <stdio.h>
#endif

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __TIME_H
#include <time.h>
#endif

#ifndef  __LEXICAL_H
#include "lexical.h"
#endif

#ifndef  __SLIST_H
#include "slist.h"
#endif

#ifndef  __SSTACK_H
#include "sstack.h"
#endif

#ifndef  __SYNTAX_H
#include "syntax.h"
#endif

// Possible error codes

#define E_DIVIDEBYZERO  0
#define E_INVALIDEXPR   2
#define E_RPEXPECTED    3
#define E_ENVARNDEF     4
#define E_OUTOFMEMORY   5
#define E_PRESSANYKEY   6
#define E_UNEXPTOKEN    7
#define E_NOPFHITEM     8
#define E_NOCOMPARE     9
#define E_CONSTANT     10
#define E_EMPTYEQN     11
#define E_PARENBAL     12
#define E_TOOLONG      13
#define E_REDEF        14


short SyntaxAnalyzer::AddOps[] = { LexicalAnalyzer::ADDOP,
                                   LexicalAnalyzer::SUBOP };

short SyntaxAnalyzer::RelOps[] = { LexicalAnalyzer::EQOP,
                                   LexicalAnalyzer::NEQOP,
                                   LexicalAnalyzer::LESOP,
                                   LexicalAnalyzer::LEQOP,
                                   LexicalAnalyzer::GEQOP,
                                   LexicalAnalyzer::GTOP };

short SyntaxAnalyzer::PfhItems[] = {
                                   LexicalAnalyzer::AX25_UPLOADER,
                                   LexicalAnalyzer::BBS_MESSAGE_TYPE,
                                   LexicalAnalyzer::BULLETIN_ID_NUMBER,
                                   LexicalAnalyzer::COMPRESSION_DESC,
                                   LexicalAnalyzer::COMPRESSION_TYPE,
                                   LexicalAnalyzer::CREATE_TIME,
                                   LexicalAnalyzer::DESTINATION,
                                   LexicalAnalyzer::DOWNLOAD_COUNT,
                                   LexicalAnalyzer::EXPIRE_TIME,
                                   LexicalAnalyzer::FILE_DESCRIPTION,
                                   LexicalAnalyzer::FILE_EXT,
                                   LexicalAnalyzer::FILE_NAME,
                                   LexicalAnalyzer::FILE_NUMBER,
                                   LexicalAnalyzer::FILE_SIZE,
                                   LexicalAnalyzer::FILE_TYPE,
                                   LexicalAnalyzer::KEYWORDS,
                                   LexicalAnalyzer::LAST_MODIFIED_TIME,
                                   LexicalAnalyzer::PRECEDENCE,
                                   LexicalAnalyzer::SEU_FLAG,
                                   LexicalAnalyzer::SOURCE,
                                   LexicalAnalyzer::TITLE,
                                   LexicalAnalyzer::UPLOAD_TIME,
                                   LexicalAnalyzer::USER_FILE_NAME };

char* SyntaxAnalyzer::errorMessages[ 15 ] = {
                          "Attempt to divide by zero.\n",
                          "",
                          "Invalid expression.\n",
                          "Right parenthesis ')' expected.\n",
                          "Environment variable not defines.\n",
                          "Out of memory.\n",
                          "Press any key to continue...\n",
                          "Enexpected token.\n",
                          "PFH item expected.\n",
                          "Comparison operator expected.\n",
                          "Constant '%s' type mismatch.\n",
                          "Empty equation definition.\n",
                          "Incomplete equation, parenthesis don't balance.",
                          "Equation exceeds %u bytes input.\n",
                          "Equation redefinition error.\n"
                          };

Equation::Equation()
         : stack( 0 ), keyTitle( 0 ), id( -1 )
// --------------------------------------------------------------------------
//  Contructor
// --------------------------------------------------------------------------
{}

Equation::~Equation()
// --------------------------------------------------------------------------
//  Destructor
// --------------------------------------------------------------------------
{
   delete keyTitle;
   delete stack;
}

SyntaxAnalyzer::SyntaxAnalyzer( char* mycall, char* alias )
                              : mycall( mycall ),
                                alias( alias ),
                                eqnCount( 0 ),
                                debug( 0 ),
                                maxEqnSize( 1000 ),
                                stackSize( 4000 / sizeof( Symbol ) )
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   stack = new SStack( stackSize );
}

SyntaxAnalyzer::~SyntaxAnalyzer()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   delete stack;
}

int SyntaxAnalyzer::equationFile( char* fileName )
// --------------------------------------------------------------------------
//  Read and parse the equation file
// --------------------------------------------------------------------------
{
   FILE* eqnFile;
   if ((eqnFile = fopen( fileName, "rt" )) == 0) // check for equation file
   {
      printf( "Equation file <%s> not found\n", fileName );
      return 1;
   }
   statementToParse = new char[ maxEqnSize ];
   if (!statementToParse)
   {
      fclose( eqnFile );
      return 1;
   }

   int      newLine = 1;
   int      skip = 0;
   unsigned len = 0;
   int      count = -1;
   int      rtnCode = 0;

   char* buf = new char[ 2048 ];
   while( 1 )
   {
      int cnt = fread( buf, 1, 2048, eqnFile);
      char* cp = buf;
      if (cnt == 0)
         break;
      char* end = cp + cnt;

      while (len < maxEqnSize && cp < end)
      {
         char ch = *cp++;
         if (newLine && ch == ';')
            skip = 1;
         if (ch == '\n')
         {
            newLine = 1;
            skip = 0;
            continue;
         }
         else
            newLine = 0;
         if (skip)
            continue;
         if (ch == ' ' && len == 0)
            continue;
         statementToParse[ len++ ] = ch;
         if (ch == '(')
            if (count == -1)
               count = 1;
            else
               count++;
         if (ch == ')')
            count--;
         if (count == 0)
         {
            statementToParse[ len ] = 0;
            if (debug)
               puts( statementToParse );
            Equation* eqn = selectEquation( statementToParse );
            int i;
            for (i = 0; i < eqnCount; i++)
               if (eqn->id == eqnList[ i ]->id)
               {
                  error( E_REDEF );
                  rtnCode = 1;
                  break;
               }
            if (eqn && i == eqnCount)
               eqnList[ eqnCount++ ] = eqn;
            len = 0;
            count = -1;
         }
      }
      if (len >= maxEqnSize)
      {
         error( E_TOOLONG );
         rtnCode = 1;
      }
   }
   if (count > 0)
   {
      error( E_PARENBAL );
      statementToParse[ len ] = 0;
      puts( statementToParse );
      rtnCode = 1;
   }
   delete statementToParse;
   fclose( eqnFile );
   return rtnCode;
}

void SyntaxAnalyzer::defaultEqns()
// --------------------------------------------------------------------------
//  Parse the default equations
// --------------------------------------------------------------------------
{
   char* equations[] = {
      "(priority (destination=(*+mycall+*))|(destination=(*+alias+*)))",
      "(auto (destination=*ALL*)&&(file_size<10000))",
      "(F1 \"All Mail\" keytitle((file_type!=2)&((file_type<200)|(file_type=255))))",
      "(F2 mycall keytitle(destination=(*+mycall+*))|(destination=(*+alias+*)))",
      "(F3 Bulletins keytitle(destination=*ALL*))",
      "(F4 Logs keytitle(file_type>200)&(file_type!=255))",
      "(F5 Done keytitle maskdone)",
      "(F6 Requested keytitle maskauto maskpriority)",
      "(F7 Keps keytitle(file_type=8)|(file_type=9))",
      "(F8 Images keytitle(file_type=211)|(keywords=*IMAGE*))",
      "(F9 NEWS keytitle(keywords=*NEWS*))",
//      "(F10 SatLink keytitle(keywords=*SatLink*))",
      "(F10 \"All Files\"keytitle(file_type >= 0))",
      "(F11 BBS keytitle(file_type=1)|(file_type=2))",
      "(F12 Urgent keytitle(precedence>0))" };

   Equation* eqn;
   for (int i = 0; i < 14; i++)
      if ((eqn = selectEquation( equations[ i ] )) != 0)
         eqnList[ eqnCount++ ] = eqn;
}

Equation* SyntaxAnalyzer::selectEquation( char* lineToParse )
// --------------------------------------------------------------------------
// Initializes the lexical analyzer and begins parsing 'line_to_parse'.
// --------------------------------------------------------------------------
{
#define NOID -1
#define DIRECTIVE  -2
   errorFlag = 0;
   this->stack->reset();
   lex.initLexicalyzer( lineToParse );
   eqn = new Equation;
   if (!eqn)
   {
      error( E_OUTOFMEMORY );
      return 0;
   }
   lex.getSymbol();
   switch (lex.token)
   {
      case lex.ENDOFLINE:
         break;
      case lex.LP:
         lex.getSymbol();
         switch (lex.token)
         {
            case lex.F1:
            case lex.F2:
            case lex.F3:
            case lex.F4:
            case lex.F5:
            case lex.F6:
            case lex.F7:
            case lex.F8:
            case lex.F9:
            case lex.F10:
            case lex.F11:
            case lex.F12:
            case lex.AUTO:
            case lex.NEVER:
            case lex.PRIORITY:
               eqn->id = lex.token;
               nameEquation();
               break;
            default:
               eqn->id = DIRECTIVE;
               directive();
         }
         if (lex.token != lex.RP)
            error( E_RPEXPECTED );
         lex.getSymbol();
   }
   if (lex.token != lex.ENDOFLINE)
      error( E_INVALIDEXPR );
   if (this->stack->errorFlag)
      this->errorFlag = 1;
   if (eqn->id != DIRECTIVE &&
       (eqn->id == NOID || (eqn->keyTitle == 0 && this->stack->sp == 0)))
   {
      error( E_EMPTYEQN );
      this->errorFlag = 1;
   }
   if (!debug && this->errorFlag)
      puts( this->statementToParse );
   if (eqn->id == DIRECTIVE || this->errorFlag)
   {
      delete eqn;
      eqn = 0;
   }
   else
   {
      this->stack->toLower();
      eqn->stack = new SStack( this->stack->sp );
      if (eqn->stack)
      {
         *eqn->stack = *this->stack;
         if (debug)
         {
            printf( "Current equation is type %i", eqn->id );
            if (eqn->keyTitle)
               printf( ", %s\n", eqn->keyTitle );
            else
               printf( "\n" );
            eqn->stack->print();
         }
      }
      else
         error( E_OUTOFMEMORY );
   }
   return eqn;
}

void SyntaxAnalyzer::nameEquation()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   lex.getSymbol();
   switch (lex.token)
   {
      case lex.LP:
         equation();
         break;
      case lex.MASKDONE:
      case lex.MASKPRIORITY:
      case lex.MASKNEVER:
      case lex.MASKAUTO:
         masks();
         if (lex.token == lex.LP)
         {
            equation();
            Symbol newS( lex.AND, (unsigned long) 0 );
            this->stack->push( newS );
         }
         break;
      default:
         keytitle();
         if (lex.token == lex.RP)                // no equation present
         {
            // Put a test for (file_type >= 0) on the stack so all
            // the test will always be true as the default

            Symbol newS( lex.FILE_TYPE, (unsigned long) 0 );
            this->stack->push( newS );
            newS.set( lex.NUMCONSTANT, (unsigned long) 0  );
            this->stack->push( newS );
            newS.set( lex.GEQOP, (unsigned long) 0 );
            this->stack->push( newS );
            break;
         }
         if (lex.token != lex.LP)
            masks();
         if (lex.token == lex.LP)
         {
            equation();
         }
   }
}

void SyntaxAnalyzer::masks()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   Symbol newS( lex.STATUS, (unsigned long) 0 );
   this->stack->push( newS );
   switch (lex.token)
   {
      case lex.MASKAUTO:
         newS.set( lex.NUMCONSTANT, (unsigned long) StatusList::AUTOMATIC );
         this->stack->push( newS );
         break;
      case lex.MASKPRIORITY:
         newS.set( lex.NUMCONSTANT, (unsigned long) StatusList::PRIORITY );
         this->stack->push( newS );
         break;
      case lex.MASKNEVER:
         newS.set( lex.NUMCONSTANT, (unsigned long) StatusList::NEVER );
         this->stack->push( newS );
         break;
      case lex.MASKDONE:
         newS.set( lex.NUMCONSTANT, (unsigned long) StatusList::DOWNLOADED );
         this->stack->push( newS );
         break;
      default:
         constant();
         if (lex.token != lex.EXTRAMASK)
            error( E_INVALIDEXPR );
   }
   newS.set( lex.EQOP, (unsigned long) 0 );
   this->stack->push( newS );
   lex.getSymbol();
   switch (lex.token)
   {
      case lex.ENDOFLINE:
      case lex.RP:
      case lex.LP:
         break;
      default:
         masks();
         newS.set( lex.OR, (unsigned long) 0 );
         this->stack->push( newS );
   }
}

void SyntaxAnalyzer::keytitle()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   constant();
   if (lex.token == lex.KEYTITLE)
   {
      Symbol* sp = this->stack->pop();
      eqn->keyTitle = new char[ strlen( sp->strVal ) + 1 ];
      strcpy( eqn->keyTitle, sp->strVal );
   }
   else
      error( E_INVALIDEXPR );
   lex.getSymbol();
}

void SyntaxAnalyzer::equation()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   test();
   while (lex.token == lex.AND || lex.token == lex.OR)
   {
      int savedToken = lex.token;
      lex.getSymbol();
      test();
      Symbol newS( savedToken, (unsigned long) 0 );
      this->stack->push( newS );
   }
}

void SyntaxAnalyzer::test()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   if (lex.token == lex.LP)
   {
      lex.getSymbol();
      equation();
      if (lex.token != lex.RP)
         error( E_RPEXPECTED );
      else
         lex.getSymbol();
   }
   else
   {
      if (inSet( PfhItems, lex.token, NUMPFHITEMS ))
      {
         Symbol newS( lex.token, (unsigned long) 0 );
         this->stack->push( newS );
      }
      else
         error( E_NOPFHITEM );
      lex.getSymbol();
      if (inSet( RelOps, lex.token, NUMRELOPS ))
      {
         int savedToken = lex.token;
         lex.getSymbol();
         constant();
         Symbol newS( savedToken, (unsigned long) 0 );
         this->stack->push( newS );
      }
      else
         error( E_NOCOMPARE );
      if (lex.token != lex.RP)
      {
         error( E_RPEXPECTED);
         while (lex.token != lex.RP && lex.token != lex.ENDOFLINE)
            lex.getSymbol();
      }
   }
}

void SyntaxAnalyzer::directive()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   switch (lex.token)
   {
      case lex.KEYPRESS:
         printf( "%s\n", errorMessages[ E_PRESSANYKEY ] );
         while (!kbhit());                       // pause the program
         getch();
         break;
      default:
         constant();
         switch (lex.token)
         {
            case lex.DEBUG:
               Symbol* sp;
               sp = this->stack->pop();
               switch ( int( sp->numVal ) )
               {
                  case 0:
                     this->stack->debug = lex.debug = debug = 0;
                     break;
                  case 2:
                     this->stack->debug = 0;
                     lex.debug = debug = 1;
                     break;
                  case 3:
                     this->stack->debug = lex.debug = debug = 1;
                     break;
                  case 1:
                  default:
                     this->stack->debug = lex.debug = 0;
                     debug = 1;
                     break;
               }
               break;
            case lex.ASCTIME:
               sp = this->stack->pop();
               printf( "%s", ctime( (time_t*) &sp->numVal ) );
               break;
            case lex.EQUATIONSIZE:
               sp = this->stack->pop();
               maxEqnSize = int( sp->numVal );
               stackSize = (4 * maxEqnSize) / sizeof( Symbol );
               delete stack;
               stack = new SStack( stackSize );
               break;
            default:
               error( E_INVALIDEXPR );
         }
   }
   lex.getSymbol();
   if (lex.token != lex.RP && lex.token != lex.ENDOFLINE)
      directive();
}

void SyntaxAnalyzer::constant()
// --------------------------------------------------------------------------
// Handles the addition operators
// --------------------------------------------------------------------------
{
   int savedToken;
   int UnaryOp = lex.OTHER;
   if (lex.token == lex.ADDOP)
   {
     UnaryOp = lex.ADDOP;
     lex.getSymbol();
   }
   else
      if (lex.token == lex.SUBOP)
      {
         UnaryOp = lex.SUBOP;
         lex.getSymbol();
      }
   value();
   if (UnaryOp == lex.SUBOP)
   {
      Symbol* sp = this->stack->pop();
      sp->numVal = - sp->numVal;
      this->stack->push( *sp );
   }

   // while token is an addition-type operator ...

   while (inSet( AddOps, lex.token, NUMADDOPS ) != 0)
   {
      savedToken = lex.token;
      lex.getSymbol();
      value();
      switch (savedToken)
      {
         case lex.ADDOP:
            Symbol* bp;
            bp = this->stack->pop();
            Symbol* ap;
            ap = this->stack->pop();
            if (*ap->strVal)
               ap->append( *bp );
            else
               ap->numVal += bp->numVal;
            this->stack->push( *ap );
            break;
         case lex.SUBOP:
            bp = this->stack->pop();
            ap = this->stack->pop();
            ap->numVal = - ap->numVal + bp->numVal;
            this->stack->push( *ap );
            break;
      }
   }
}

void SyntaxAnalyzer::value()
// --------------------------------------------------------------------------
// Implements the syntax of an expression's value.  This is the most finite
// component, such as a constant, a modifier, or another parenthethical
// expression.
// --------------------------------------------------------------------------
{
   char* entVarStr;
   Symbol newS;
   switch (lex.token)
   {
      case lex.MYCALL:
         strcpy( lex.symbol, mycall );
         lex.token = lex.STRCONSTANT;
         value();                                // note the recursion
         break;
      case lex.MYADDR:
      case lex.ALIAS:
         strcpy( lex.symbol, alias );
         lex.token = lex.STRCONSTANT;
         value();                                // note the recursion
         break;
      case lex.TODAY:
         lex.tokenValue = time( 0 );
         lex.token = lex.NUMCONSTANT;
         value();                                // note the recursion
         break;

      // Apply modifiers to the symbol values on the top of the stack

      case lex.MINUTE:
      case lex.MINUTES:
         Symbol* sp;
         sp = this->stack->pop();
         if (sp->type != lex.NUMCONSTANT )
            error( E_CONSTANT );
         sp->numVal *= 60;
         this->stack->push( *sp );
         lex.getSymbol();
         break;
      case lex.HOUR:
      case lex.HOURS:
         sp = this->stack->pop();
         if (sp->type != lex.NUMCONSTANT )
            error( E_CONSTANT );
         sp->numVal *= 3600;
         this->stack->push( *sp );
         lex.getSymbol();
         break;
      case lex.DAY:
      case lex.DAYS:
         sp = this->stack->pop();
         if (sp->type != lex.NUMCONSTANT )
            error( E_CONSTANT );
         sp->numVal *= 86400L;
         this->stack->push( *sp );
         lex.getSymbol();
         break;
      case lex.WEEK:
      case lex.WEEKS:
         sp = this->stack->pop();
         if (sp->type != lex.NUMCONSTANT )
            error( E_CONSTANT );
         sp->numVal *= 604800L;
         this->stack->push( *sp );
         lex.getSymbol();
         break;

      // Substitute environment variables

      case lex.GETENVAR:
         sp = this->stack->pop();                // get env variable
         entVarStr = getenv( sp->strVal );       // get env variable string
         if (entVarStr)                          // if we found it
         {
            // We create a new instance of the analyzer because we can't
            // use "lex" without affecting the initialization to the
            // current line we are parsing.

            LexicalAnalyzer* l = new LexicalAnalyzer;
            l->initLexicalyzer( entVarStr );          // process the string
            l->getSymbol();                           // get the symbol
            lex.token = l->token;                     // substitute for the
            lex.tokenValue = l->tokenValue;           // the current symbol
            strcpy( lex.symbol, l->symbol );
            delete l;
            value();                            // note the recursion
         }
         else
            error( E_ENVARNDEF );
         break;

      // Create symbols for numerical PFH items with a type corresponding
      // to the token and a numbrical value of 0

      case lex.BBS_MESSAGE_TYPE:
      case lex.COMPRESSION_TYPE:
      case lex.CREATE_TIME:
      case lex.DOWNLOAD_COUNT:
      case lex.EXPIRE_TIME:
      case lex.FILE_NUMBER:
      case lex.FILE_SIZE:
      case lex.FILE_TYPE:
      case lex.LAST_MODIFIED_TIME:
      case lex.PRECEDENCE:
      case lex.SEU_FLAG:
      case lex.UPLOAD_TIME:
      case lex.AX25_UPLOADER:
      case lex.BULLETIN_ID_NUMBER:
      case lex.COMPRESSION_DESC:
      case lex.DESTINATION:
      case lex.FILE_DESCRIPTION:
      case lex.FILE_EXT:
      case lex.FILE_NAME:
      case lex.KEYWORDS:
      case lex.SOURCE:
      case lex.TITLE:
      case lex.USER_FILE_NAME:
         newS.set( lex.token, (unsigned long) 0 );
         this->stack->push( newS );
         lex.getSymbol();
         break;

      case lex.NUMCONSTANT:
         newS.set( lex.token, lex.tokenValue );
         this->stack->push( newS );
         lex.getSymbol();
         break;
      case lex.STRCONSTANT:
         newS.set( lex.token, lex.symbol );
         this->stack->push( newS );
         lex.getSymbol();
         break;

      case lex.LP:
         lex.getSymbol();
         constant();
         if (lex.token != lex.RP)
            error( E_RPEXPECTED );
         else
            lex.getSymbol();
         break;
      default:
         error( E_INVALIDEXPR );
   }
}

void SyntaxAnalyzer::error( int errCode )
// --------------------------------------------------------------------------
// error displays an error message corresponding to errCode.
// --------------------------------------------------------------------------
{
   errorFlag = 1;
   switch (errCode)
   {
      case E_CONSTANT:
         printf( errorMessages[ E_CONSTANT ], lex.symbol );
         break;
      case E_TOOLONG:
         printf( errorMessages[ E_TOOLONG ], maxEqnSize );
         break;
      default:
         printf( errorMessages[ errCode ] );
   }
   if (errCode == E_OUTOFMEMORY)
      exit( 1 );
   printf( errorMessages[ E_PRESSANYKEY ] );
   while (!kbhit());
   getch();
}

int SyntaxAnalyzer::inSet( short* set, int toFind, int numEntries )
{
// --------------------------------------------------------------------------
// Searches for toFind in the array of short integers pointed to by set,
// returning 1 if toFind was found, or 0 if not found.
// --------------------------------------------------------------------------
   for (int i = 0; i < numEntries; i++)
      if (*(set + i) == toFind)
         return 1;
   return 0;
}

int SyntaxAnalyzer::largestStackSize()
// --------------------------------------------------------------------------
//  Return the size of the largest stack of any equation in the equation
//  list.
// --------------------------------------------------------------------------
{
   int maxStackSize = 0;
   for (int i = 0; i < eqnCount; i++)
      if (eqnList[ i ]->stack->size > maxStackSize)
         maxStackSize = eqnList[ i ]->stack->size;
   return maxStackSize;
}