//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

// -------------------------------------------------------------------------
// SLIP (Serial Line IP) encapsulation and control routines.
// -------------------------------------------------------------------------

#ifndef  __SLIP_H
#include "slip.h"
#endif

#ifndef  __DIR_H
#include <dir.h>
#endif

#ifndef  __MEM_H
#include <mem.h>
#endif

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __TIME_H
#include <time.h>
#endif

#ifndef  __CALLUP_H
#include "callup.h"
#endif

#ifndef  __HWDEVICE_H
#include "hwdevice.h"
#endif

#ifndef  __MSGBUF_H
#include "msgbuf.h"
#endif

Slip::Slip( unsigned maxNewBufSize,
            unsigned& logInFrames, unsigned& logOutFrames,
            unsigned& loggingStatus )
          :
            L1Protocol(),
            maxNewBufSize( maxNewBufSize ),
            maxFrameSize( 1024 ),
            dir( 0 ),
            slipFileIn( 0 ),
            slipFileOut( 0 )
// -------------------------------------------------------------------------
//  Constructor
// -------------------------------------------------------------------------
{
   // Assume the maximum size for a SLIP encoded AX.25 frame to be
   // ( source + destination + 8 digipeaters ) * (callsign length = 7) +
   // control + PID + (data == 256 bytes).  This gives (328 bytes * 2) + 2
   // for SLIP encoding, or 658 bytes.  A maximum frame size of 1024 provides
   // some safety margin.

   // The minimum size of the SLIP input buffer must be the maximum size of
   // an encoded, partial SLIP frame left the buffer from a previous call plus
   // the maximum size MBuf received in the current call to recvIn().

   inBufSize = maxNewBufSize + maxFrameSize;
   if ((inBuf = new char[ inBufSize ]) == 0)
      frWin->printFatal( "Out of memory" );
   this->loggingStatus = &loggingStatus;
   this->logInFrames = &logInFrames;
   this->logOutFrames = &logOutFrames;
}

Slip::~Slip()
// -------------------------------------------------------------------------
//  Destructor
// -------------------------------------------------------------------------
{
   if (inBuf)
      delete inBuf;
   if (slipFileIn)
      fclose( slipFileIn );
   if (slipFileOut)
      fclose( slipFileOut );
}

MBuf* Slip::decode( char* ibp, const unsigned cnt )
// -------------------------------------------------------------------------
//  Process incoming bytes in SLIP format.
//  The returned values are:
//           0 - Not complete frame in input.
//          !0 - pointer to the decoded MBuf.
// -------------------------------------------------------------------------
{
   int escaped = 0;
   if (*ibp != FR_END)                           // if no starting FR_END
      return 0;
   ibp++;                                        // discard first FR_END
   char* ibend = ibp + cnt;                      // input buffer end

   MBuf* obp = new MBuf( 2*cnt );                // allocate output buffer
   char* ocp = obp->data;                        // init output char ptr
   obp->cnt = 0;                                 // init output data count

   for (; ibp < ibend; ibp++)
   {
      if (obp->cnt == obp->size)                 // if frame will overflow
         obp->reAlloc( obp->size + 128 );        // reallocate it
      if (escaped)                               // translate the 2-byte
      {                                          // escape sequence back to
         switch (*ibp)                           // original character
         {
            case T_FR_ESC :
               *ocp++ = FR_ESC;
               obp->cnt++;
               break;
            case T_FR_END :
               *ocp++ = FR_END;
               obp->cnt++;
               break;
            default :
               recvErrCnt++;                     // count the errors
         }
         escaped = 0;                            // no longer escaped
      }
      else
         switch (*ibp)
         {
            case FR_END :                        // end of frame
               return obp;                       // return completed frame
            case FR_ESC :                        // frame escape
               escaped = 1;
               break;
            default :
               *ocp++ = *ibp;                    // append the char to buffer
               obp->cnt++;                       // increment data length
         }
   }
   recvErrCnt++;                                 // can only get here if
   delete obp;                                   // final FR_END is missing
   return 0;
}

MBuf* Slip::encode( const MBuf& bp ) const
// -------------------------------------------------------------------------
//  Encode a frame in SLIP format.
// -------------------------------------------------------------------------
{
   // Allocate output MBuf that's twice as long as the packet.
   // This is a worst-case guess (consider a packet full of FR_ENDs!)

   MBuf* lbp = new MBuf( 2*bp.cnt + 4 );         // line ready frame
   if (!lbp)
      return 0;
   char* cpo = lbp->data;                        // start of raw buffer
   *cpo++ = FR_END;                              // beginning frame flag
   char* cpi = bp.data;                          // start of incoming buffer

   // Copy input to output, escaping special characters

   for (int i = 0; i < bp.cnt; i++)
   {
      char c = *cpi++;
      switch (c)
      {
         case FR_ESC :
            *cpo++ = FR_ESC;
            *cpo++ = T_FR_ESC;
            break;
         case FR_END :
            *cpo++ = FR_ESC;
            *cpo++ = T_FR_END;
            break;
         default :
            *cpo++ = c;
      }
   }
   *cpo++ = FR_END;                              // ending frame flag
   lbp->cnt = ( int )(cpo - lbp->data);
   return lbp;
}

void Slip::link( HwDevice& loPro, L1Protocol& hiPro )
// -------------------------------------------------------------------------
// -------------------------------------------------------------------------
{
   hiProto = &hiPro;
   loProto = &loPro;
}

void Slip::logRecvIn()
// -------------------------------------------------------------------------
//  Log incoming a SLIP frame to a file.
// -------------------------------------------------------------------------
{
   log( inBuf, frLen );
}

void Slip::log( const char* bp, const unsigned cnt )
// -------------------------------------------------------------------------
// Log a SLIP frame to a file.
// -------------------------------------------------------------------------
{
   static char fileOpen = 0;                     // dump file status flag

   if (!fileOpen)                                // if dump file not open
   {
      time_t t = time( &t );                     // use the time to
      tm*    tm = gmtime( &t );                  // make a 00YYMMDD file name
      char name[9];
      sprintf( (char*) &name, "%04i%02i%02i\x0", tm->tm_year, tm->tm_mon + 1,
                                                 tm->tm_mday );
      char* path = new char[ MAXPATH ];          // allocate path
      fnmerge( path, 0, dir, name, ".SLP" );     // merge the path components
      if ((slipFileOut = fopen( path, "ab")) == 0)
      {
         sprintf( frWin->buf, "Error opening SLIP output file %s", path );
         frWin->printFatal( frWin->buf );
      }
      delete path;                               // free allocated memory
      fileOpen = 1;                              // raise file status flag
   }
   fwrite( bp, cnt, 1, slipFileOut);
}

void Slip::recvIn( MBuf& newBuf )
// -------------------------------------------------------------------------
//  Handle incoming SLIP data.
//
// If there is a complete SLIP frame already in the buffer, it should
// have been processed.  Delete it from the buffer, and reset the counts
// and pointers.  The SLIP frame must be left in the buffer so that
// Slip::dump() can be called based on PID from other parts of the
// program.
// -------------------------------------------------------------------------
{
   recvBytesIn += newBuf.cnt;                    // count bytes in
   recvFramesIn++;                               // count frames in

   static unsigned       len       = 0;          // data length in buffer
   static char*          bp        = inBuf;      // working pointer
   static unsigned long  lastCnt   = 0;          // last count of SLIP errors
   if (newBuf.cnt > maxNewBufSize)               // if too much new data
   {
      delete &newBuf;                            // kill new data
      len = 0;                                   // kill residual data
      bp = inBuf;
      recvErrCnt++;
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      return;
   }
   if (len + newBuf.cnt > inBufSize)             // if no room in accumulator;
   {                                             // this shouldn't happen
      len = 0;                                   // kill residual data
      bp = inBuf;
      recvErrCnt++;
   }
   memcpy( bp, newBuf.data, newBuf.cnt );        // copy new data into
   len += newBuf.cnt;                            // the accumulator
   bp += newBuf.cnt;
   delete &newBuf;

   static char* begin = 0;                       // beginning of SLIP frame
   static char* end = 0;                         // end of SLIP frame
   int incompleteFrame = 0;
   while (len > 0 && !incompleteFrame)           // while we have data
   {                                             // in the accumulator
      if (!begin)
         if ((begin = (char*) memchr(inBuf, FR_END, len)) == 0)
         {                                       // no starting FR_END
            len = 0;                             // kill data
            bp = inBuf;
            recvErrCnt++;
         }
         else
            if (begin != inBuf)                  // if bytes before starting
            {                                    // FR_END, kill these bytes
               len -= (unsigned)(begin - inBuf);
               memmove( inBuf, begin, len );
               begin = inBuf;
               bp = inBuf + len;
               recvErrCnt++;
            }
      if (begin)                                 // if we started a frame
         if ((end = (char*) memchr( inBuf + 1, FR_END, len - 1 )) != 0)
            if (end == inBuf + 1)                // if back-to-back FR_END
            {                                    // then kill one of them
               len--;
               memmove(inBuf, inBuf + 1, len);
               bp--;
               recvErrCnt++;
               end = 0;
            }
            else                                 // we have an complete frame
            {
               frLen = (unsigned)(end - inBuf + 1);
               if (*logInFrames)
                  log( inBuf, frLen );
               MBuf* fp;
               if ((fp = decode( inBuf, frLen )) != 0)
               {
                  recvBytesOut += fp->cnt;       // count bytes out
                  recvFramesOut++;               // count frames out
                  if (hiProto)                   // if destination
                     hiProto->recvIn( *fp );     // send it upstairs
                  else                           // otherwise
                     delete fp;                  // kill it

               }
               // The SLIP frame must be left in the buffer until after
               // it goes upstairs so that Slip::dump() can be
               // called based on PID from other parts of the
               // program.

               len -= frLen;                     // num data bytes remaining
               memmove( inBuf, end + 1, len );   // delete frame
               bp = inBuf + len;                 // next free space in buffer
               begin = end = 0;                  // reset pointers
            }
         else                                    // we have incomplete frame
            if (len > maxFrameSize)              // if excessive frame size
            {                                    // then kill the frame
               len = 0;
               bp = inBuf;
               recvErrCnt++;
            }
            else
               incompleteFrame = 1;              // incomplete frame

      if (eCallUp && recvErrCnt != lastCnt)      // if we need to report the
      {                                          // error count change
         eCallUp->func();
         lastCnt = recvErrCnt;
      }
   }
}

int Slip::send( MBuf& buf )
// -------------------------------------------------------------------------
//  Send an MBuf using SLIP; then delete the allocated memory.
//
//  The values returned are:
//           0 - Successfully sent the MBuf
//           1 - Error, MBuf not sent
// -------------------------------------------------------------------------
{
   if (!loProto)                                 // if link not set
      return 1;
// lastsent = msClock();                         // time sent
   MBuf* rp = encode( buf );                     // raw buffer ptr
   delete &buf;                                  // free incoming buffer
   if (!rp)                                      // encode as raw frame
      return 1;
   if (*logOutFrames)                            // dump raw-encoded frame
      log( rp->data, rp->cnt );

   sendFramesOut++;                              // count outgoing frames
   sendBytesOut += rp->cnt;                      // count outgoing bytes
   return loProto->send( *rp );                  // send to hw device
}
