//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __MSGBUF_H
#include "msgbuf.h"
#endif

#ifndef  __MEM_H
#include <mem.h>
#endif

#ifndef  __STDIO_H
#include <stdio.h>
#endif

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __WINDOW_H
#include "window.h"
#endif

MBuf*   MBuf::ZERO = 0;
Window* MBuf::win = 0;
char*   MBuf::errMsgMBuf1 = "MBuf : out of memory, allocation failed";
char*   MBuf::errMsgMBuf2 = "MBuf : data exceeded maximum MBuf size";

MBuf::MBuf() : size( 0 ), cnt( 0 ), data( 0 ), dupDataPtr( 0 )
// ------------------------------------------------------------------------
// Constructor function
//
// Summary
//    Constructor for an empty message buffer object
//
// Functional description
//    We assign zero to the MBuf's size and data count.  We also zero
//    the data pointers.
// ------------------------------------------------------------------------
{}


MBuf::MBuf( const unsigned size ) : size( size ), cnt( 0 )
// ------------------------------------------------------------------------
// Constructor function
//
// Summary
//    Constructor for a message buffer object
//
// Parameters
//    size - size of the data area of the MBuf object
//
// Functional description
//    We assign the parameter size to the MBuf's size and set the MBuf's
//    data count to zero.  Then we allocate memory for a data space.
//    We initialize dupDataPtr to also point to the start of the data 
//    space.
// ------------------------------------------------------------------------
{
   dupDataPtr = data = new char[ size ];
   if (!data)
   {
      if (win)
         win->printFatal( errMsgMBuf1 );
      else
      {
         puts( errMsgMBuf1 );
         exit( 1 );
      }
   }
}


MBuf::MBuf( const char* newData, const unsigned newCnt )
    :
    size( newCnt ), cnt( newCnt )
// -------------------------------------------------------------------------
// Constructor function
//
// Summary
//    Constructor for a message buffer object.  Loads the object with
//    data.
//
// Parameters
//    newData - pointer to data array
//    newCnt  - count of the number of bytes in the array
// -------------------------------------------------------------------------
{
   if (!newData)
   {
      cnt = 0;                                   // create an empty MBuf
      dupDataPtr = data = 0;
   }
   else
   {
      data = dupDataPtr = new char[ size ];      // allocate new data space
      if (!data)                                 // check for success
      {
         if (win)
            win->printFatal( errMsgMBuf1 );
         else
         {
            puts( errMsgMBuf1 );
            exit( 1 );
         }
      }
      memcpy( data, newData, cnt );      // copy the data
   }
}

MBuf::~MBuf()
// ------------------------------------------------------------------------
// Destructor function
//
// Summary
//    Destructor for a message buffer object.  Deletes the storage for the
//    data.
//
// ------------------------------------------------------------------------
{
   if (dupDataPtr)                               // deallocate data buffer
      delete dupDataPtr;
   data = dupDataPtr = 0;
   cnt = size = 0;
}


MBuf::MBuf( const MBuf& sourceMBuf ) : size( sourceMBuf.size ),
                                       cnt( sourceMBuf.cnt )
// -------------------------------------------------------------------------
// Member function
//
// Summary
//    Copy constructor.  The new MBuf retains the size and any data offset
//    of the source MBuf.
//
// Parameters
//    sourceMBuf - the source MBuf we are to copy from
//
// -------------------------------------------------------------------------
{
   if (!sourceMBuf.size)     
   {
      cnt = 0;                                   // create an empty MBuf
      dupDataPtr = data = 0;
   }
   else
   {
      size = sourceMBuf.size;                    // retain the size
      dupDataPtr = new char[ size ];             // allocate new data space        
      if (!dupDataPtr)                           // check for success
      {
         if (win)
            win->printFatal( errMsgMBuf1 );
         else
         {
            puts( errMsgMBuf1 );
            exit( 1 );
         }
      }
      unsigned offset = (unsigned)(sourceMBuf.data - sourceMBuf.dupDataPtr);
      data = dupDataPtr + offset;

      // Check to see if the data range is larger than the allocated space.
      // Set the count to zero if it is.

      if (offset + cnt > size)
         cnt = 0;
      memcpy( data, sourceMBuf.data, cnt );      // copy the data
   }
}


MBuf::MBuf( const MBuf& sourceMBuf, const unsigned skipCnt, 
                                    const unsigned copyCnt )
// -------------------------------------------------------------------------
// Member function
//
// Summary
//    Copy constructor for a subset of the source data.  Set the size to
//    the copyCnt.  If the copyCnt is greater than the available data, set
//    the cnt to the amount of available data copied.
//
// Parameters
//    sourceMBuf - source MBuf we are to copy from
//    skipCnt    - the number of data bytes to skip
//    copyCnt    - the number of data bytes to copy
//
// -------------------------------------------------------------------------
{
   if (!sourceMBuf.cnt || skipCnt >= sourceMBuf.cnt || copyCnt == 0)     
   {
      cnt = size = 0;                            // create an empty MBuf
      dupDataPtr = data = 0;
   }
   else
   {
      cnt = sourceMBuf.cnt - skipCnt;            // count data available
      if (cnt > copyCnt)
         cnt = copyCnt;
      size = copyCnt;                            // set size to copy count
      dupDataPtr = data = new char[ size ];
      if (!data)
      {
         if (win)
            win->printFatal( errMsgMBuf1 );
         else
         {
            puts( errMsgMBuf1 );
            exit( 1 );
         }
      }
      memcpy( data, sourceMBuf.data + skipCnt, cnt );
   }
}


void MBuf::append( MBuf & sourceMBuf )
// -------------------------------------------------------------------------
// Member function
//
// Summary
//    Attach the data in an MBuf to this one and delete the source MBuf
//
// Parameters
//    sourceMBuf - the source MBuf to append
//
// -------------------------------------------------------------------------
{
   if (data > dupDataPtr)                        // if any offset, remove it
   {
      memmove( dupDataPtr, data, cnt);           // shift data to eliminate it
      data = dupDataPtr;                         // adjust data pointer
   }
   if (cnt + sourceMBuf.cnt > size)              // if buffer too small
   {
      if (long( cnt ) + sourceMBuf.cnt > 0xffff) // check size of a new one
      {                                          // to make sure it is not
         if (win)                         // excessively large
            win->printFatal( errMsgMBuf2 );
         else
         {
            puts( errMsgMBuf2 );
            exit( 1 );
         }
      }
      unsigned offset = (unsigned)(data - dupDataPtr); // save any offset
      unsigned newSize = cnt + sourceMBuf.cnt;
      char* cp = new char[ newSize ];            // allocate new data space
      if (!cp)                                   // if allocation failed
      {
         if (win)
            win->printFatal( errMsgMBuf1 );
         else
         {
            puts( errMsgMBuf1 );
            exit( 1 );
         }
      }
      memcpy( cp, dupDataPtr, size );            // copy the data
      delete dupDataPtr;                         // free old data space
      dupDataPtr = cp;                           // update data space pointer
      data = cp + offset;                        // reset data pointer
      size = newSize;                            // reset the size
   }
   memcpy( data + cnt, sourceMBuf.data, sourceMBuf.cnt );  // append data
   cnt += sourceMBuf.cnt;                        // update data count
   delete &sourceMBuf;                           // delete source MBuf
}

int MBuf::pullChar()
// ------------------------------------------------------------------------
// Member function
//
// Summary
//    Pull single character from MBuf.  Return -1 if no data.
//
//    Returned value must be an int for this to work.  High bits
//    are zero for all characters.
// ------------------------------------------------------------------------
{
   if (cnt <= 0)                                 // if no data in buffer
      return -1;
   cnt--;                                        // decrement data count
   return *data++;                               // inc address on return
}

int MBuf::pullUp( char* destBuf, const unsigned pullCnt )
// --------------------------------------------------------------------------
// Member function
//
// Summary
//    Copy and delete pullCnt bytes from beginning of data storage.  Return
//    number of bytes actually pulled off.
//
// Parameters
//    destBuf - pointer to the buffer into which to copy the removed bytes.
//    pullCnt - number of bytes to remove
//
// --------------------------------------------------------------------------
{
   if (!destBuf)                                 // if null pointer
      return 0;                                  // return with nothing
                                                 // can't exceed what's there
   unsigned n = pullCnt <= cnt ? pullCnt : cnt;
   if (n == 1)                                   // common case optimization
      *destBuf = *data;
   else
      if (n > 1)
         memcpy( destBuf, data, n );
   data += n;
   cnt -= n;
	return n;
}

void MBuf::pushDown( const unsigned pushCnt )
// -------------------------------------------------------------------------
// Member function
//
// Summary
//    Insert specified amount of bytes of room in front of the existing data.
//
//    This operation is the logical inverse of pullUp(), hence the name.
//
// Parameters
//    pushCnt - the number of bytes of room to insert
//
// -------------------------------------------------------------------------
{
   if (dupDataPtr + pushCnt <= data)             // if enough free space
   {                                             // already at front, then
      data -= pushCnt;                           // adjust pointers and count
      cnt += pushCnt;
      return;
   }

   if (cnt + pushCnt > size)                     // if this MBuf too small
      reAlloc( cnt + pushCnt );                  // reallocate it      

   memmove( dupDataPtr + pushCnt, data, cnt );   // shift the data
   data = dupDataPtr;                            // set new data pointer
   cnt += pushCnt;                               // adjust cnt
}

void MBuf::reAlloc( const unsigned newSize )
// ------------------------------------------------------------------------
// Member function
//
// Summary
//    Reallocate a message buffer to a larger size
//
// Parameters
//    newSize - the new size for the data space
//
// ------------------------------------------------------------------------
{
   unsigned offset = (unsigned)(data - dupDataPtr);// save any offset
   char* cp = new char[ newSize ];               // allocate new data space
   if (!cp)
   {
      if (win)
         win->printFatal( errMsgMBuf1 );
      else
      {
         puts( errMsgMBuf1 );
         exit( 1 );
      }
   }
   if (size > newSize)                           // if reducing the size
   {
      size = newSize;                            // make adjustments
      if (cnt > size)
         cnt = size;
   }
   memcpy( cp, dupDataPtr, size );               // copy the data
   delete dupDataPtr;                            // free old data space
   dupDataPtr = cp;                              // update data space pointer
   data = cp + offset;                           // reset data pointer
   size = newSize;                               // reset the size
}


MBuf& MBuf::operator =( const MBuf& sourceMBuf )
// ------------------------------------------------------------------------
// Member function
//
// Assignment operator for MBufs
//
// Parameters
//    sourceMBuf - the MBuf we are to assign to this
//
// ------------------------------------------------------------------------
{
   if (this != &sourceMBuf )                     // ignore if it's us
   {
      delete dupDataPtr;
      if (!sourceMBuf.size)
      {
         cnt = size = 0;                         // create an empty MBuf
         dupDataPtr = data = 0;
      }
      else
      {
         size = sourceMBuf.size;                 // retain the size
         dupDataPtr = new char[ size ];          // allocate new data space
         if (!dupDataPtr)                        // check for success
         {
            if (win)
               win->printFatal( errMsgMBuf1 );
            else
            {
               puts( errMsgMBuf1 );
               exit( 1 );
            }
         }
         unsigned offset = (unsigned)(sourceMBuf.data - sourceMBuf.dupDataPtr);
         data = dupDataPtr + offset;

         // Check to see if the data range is larger than the allocated space.
         // Set the count to zero if it is.

         cnt = offset + sourceMBuf.cnt <= size ? sourceMBuf.cnt : 0;
         memcpy( data, sourceMBuf.data, cnt );   // copy the data
      }
   }
   return *this;                                 // point to us
}
