//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __MATCH_H
#include "match.h"
#endif

// character defines

#define MATCH_CHAR_SINGLE               '?'
#define MATCH_CHAR_KLEENE_CLOSURE       '*'
#define MATCH_CHAR_NULL                 '\0'

// match defines

#define MATCH_LITERAL  5        /* match failure on literal match */
#define MATCH_ABORT    3        /* premature end of text string */
#define MATCH_END      2        /* premature end of pattern string */
#define MATCH_VALID    1        /* valid match */

int matchAfterStar( register char* p, register char* t );
int matche( register char* p, register char* t );

unsigned match( register char* p, register char* t )
// --------------------------------------------------------------------------
// match() is a shell to matche() to return only logical values.
// --------------------------------------------------------------------------
{
    int errorType = matche( p, t );
    return errorType == MATCH_VALID ? 1 : 0;
}

int matche( register char* p, register char* t )
// --------------------------------------------------------------------------
// Match the pattern PATTERN against the string TEXT;
//
// returns MATCH_VALID if pattern matches, or an errorcode as follows
// otherwise:
//
//           MATCH_LITERAL  - match failure on literal mismatch
//           MATCH_ABORT    - premature end of text string
//           MATCH_END      - premature end of pattern string
//           MATCH_VALID    - valid match
//
//
// A match means the entire string TEXT is used up in matching.
//
// In the pattern string:
//      `*' matches any sequence of characters (zero or more)
//      `?' matches any character
// --------------------------------------------------------------------------
{
   for (; *p; p++, t++)
   {
      // If this is the end of the text then this is the end of the match

      if (!*t)
      {
         return *p == MATCH_CHAR_KLEENE_CLOSURE && *++p == MATCH_CHAR_NULL ?
                MATCH_VALID : MATCH_ABORT;
      }

      // Determine and react to pattern type

      switch (*p)
      {
         case MATCH_CHAR_SINGLE:                 // single any character match
            break;
         case MATCH_CHAR_KLEENE_CLOSURE:         // multiple any character match
            return matchAfterStar( p, t );
         default:                                // must match this character exactly
            if (*p != *t && *p - 32 != *t)
               return MATCH_LITERAL;
      }
   }

   // If end of text not reached then the pattern fails

   if (*t)
      return MATCH_END;
   else
      return MATCH_VALID;
}


int matchAfterStar( register char* p, register char* t )
// --------------------------------------------------------------------------
// Recursively call matche() with final segment of PATTERN and of TEXT.
// --------------------------------------------------------------------------
{
   register int match = 0;

   // Pass over existing ? and * in pattern

   while (*p == MATCH_CHAR_SINGLE || *p == MATCH_CHAR_KLEENE_CLOSURE)
   {
      if (*p == MATCH_CHAR_SINGLE)               // take one char for each ? and +
         if (!*t++)                              // if end of text then no match
            return MATCH_ABORT;
      p++;                                       // move to next char in pattern
   }

   // If end of pattern we have matched regardless of text left

   if (!*p)
      return MATCH_VALID;

   // Get the next character to match which must be a literal

   register nextp = *p;

   // Continue until we run out of text or definite result seen

   do
   {
      // Match the next character in the pattern match to the next character
      // in the text.  Increment text pointer as we go here.

      if (nextp == *t || nextp - 32 == *t)
         match = matche( p, t );
      if (!*t++)                                 // if the end of text is
           match = MATCH_ABORT;                  // reached then no match

   } while (match != MATCH_VALID && match != MATCH_ABORT);

   return match;                                 // return result
}

