//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __DIR_H
#include <dir.h>
#endif

#ifndef  __STDIO_H
#include <stdio.h>
#endif

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MISC_H
#include "misc.h"
#endif

#ifndef  __HOLFILE_H
#include "holfile.h"
#endif

#ifndef  _WINDOW_H
#include "window.h"
#endif

char HolFile::fileName[ MAXPATH ];
char HolFile::inRec[ HolFile::MAXBYTES ];

HolFile::HolFile(       char*         path,
                  const unsigned long fileNumber,
                        Window&       frWin )
                  :
                  path( path ),
                  fileNumber( fileNumber ),
                  frWin( frWin ),
                  fileSize( 0 ),
                  holCount( 0 ),
                  holTotal( 0 ),
                  list( 0 ),
                  listSize( 0 )
// ------------------------------------------------------------------------
//  Constructor
// ------------------------------------------------------------------------
{}


HolFile::~HolFile()
// -------------------------------------------------------------------------
//  Destructor
// -------------------------------------------------------------------------
{
   delete list;
}

void HolFile::addHol( const unsigned long loLimit,
                      const unsigned long upLimit )
// -------------------------------------------------------------------------
//  Add a hole to hole list
// -------------------------------------------------------------------------
{
   if (!listSize)
   {
      if ((list = new FHole[ LIST_SIZE_START ]) == 0)
         frWin.printFatal( "Out of memory" );
      listSize = LIST_SIZE_START;
   }
   if (holCount == listSize)
   {
      // The following test is made because of the array size and memcpy()
      // limitations of a size_t type (unsigned int).

      long newSize = listSize + LIST_SIZE_INC; // inc list size
      if (newSize * sizeof( FHole ) > 0xffff)
         frWin.printFatal( "FHole list exceeded 0xffff bytes" );
      FHole* newList = new FHole[ unsigned( newSize ) ];
      if (newList)
      {
         memcpy( newList, list, listSize * sizeof( FHole ) );
         delete list;
         list = newList;
         listSize = unsigned( newSize );
      }
      else
      {
         delete list;
         frWin.printFatal( "Out of memory" );
      }
   }
   int i;
   for (i = 0; i < holCount; i++)                // insert the new record
      if (loLimit < list[ i ].lowerLimit)
         break;
   memmove( &list[ i+1 ], &list[ i ], (holCount - i)*sizeof( FHole ) );
   list[ i ].lowerLimit = loLimit;
   list[ i ].upperLimit = upLimit;

   holCount++;                                   // inc hole count
   holTotal += upLimit - loLimit + 1;            // adjust hole total
}

void HolFile::deleteHol( const unsigned i )
// -------------------------------------------------------------------------
//  Delete record from file hole-list
// -------------------------------------------------------------------------
{
   if (i >= holCount)
      frWin.printFatal( "FHole index exceeds list size" );
   if (!holCount)
      frWin.printFatal( "Deleting hole in empty list" );

   // Decrement hole count and adjust hole total before hole is deleted

   holCount--;
   holTotal -= list[ i ].upperLimit - list[ i ].lowerLimit + 1;
   memmove( &list[ i ], &list[ i+1 ], (holCount - i) * sizeof( FHole ) );
}

int HolFile::needData( const unsigned long offset,
                       const unsigned dLen,
                       const char     flag )
// -------------------------------------------------------------------------
//  Determine if the hole list indicates that the data is needed.  If the
//  data is needed, then update the hole list to reflect the new data before
//  returning.
// -------------------------------------------------------------------------
{
   unsigned long firstByte = offset;
   unsigned long lastByte = firstByte + dLen - 1;

   // If bit 5 in the flag byte is a one, this indicates that the last byte
   // in the frame is the last byte in the file.  If the data length is
   // greater than zero, we can calculate the file size.

   // However, if the data length is zero, then the offset is apparently
   // the EOF point which "usually" is the same as the file size.  The offset
   // has been seen to be greater than the filesize, but in this case, the
   // hole will not be adjusted sufficiently smaller.  This does not cause a
   // problem, assuming that a good adjustment would be made later.  This use
   // of the offset and zero data frames does not seem consistent with
   // protocol specs.

   // We save the file size in either case and make the hole adjustment if
   // necessary.

   if (flag & 0x20)                              // if last frame for file
      if(dLen)                                   // if data
      {
         if (fileSize && fileSize < lastByte + 1)// dynamically changing file
         {
            adjustLastHolUp( lastByte + 1 );     // adjust last hole up
            fileSize = lastByte + 1;
         }
         else
         {
            fileSize = lastByte + 1;             // save file size
            adjustLastHolDn();                   // adjust last hole down
         }
      }
      else                                       // offset >= EOF
      {                                          
         if (fileSize && fileSize < offset )     // dynamically changing file
         {
            adjustLastHolUp( offset );           // adjust last hole up
            fileSize = offset;
         }
         else
         {
            fileSize = offset;
            adjustLastHolDn();                   // adjust last hole down
         }
      }

   if (!dLen)                                    // if no data to save
      return 0;                                  // return

   // An incomplete file will have at least one record in the hole list.
   // If no holes exist, then the last hole was removed by the preceding
   // adjustment and the file is complete.

   int i;
   for (i = 0; i < holCount; i++)
      if (firstByte <= list[ i ].upperLimit)
         break;
   if (i == holCount)
      return 0;                                  // first > upper limit of last hole rec

   if (firstByte <= list[ i ].lowerLimit)        // first <= lower limit
   {
      if (lastByte < list[ i ].lowerLimit)       // data < hole
         return 0;
      else
         if (lastByte >= list[ i ].upperLimit)   // data spans entire hole
         {
            deleteHol( i );
            return 1;
         }
         else
         // (lastByte < list[ i ].upperLimit)    // data includes lower limit
         {
            holTotal -= list[ i ].upperLimit - list[ i ].lowerLimit + 1;
            list[ i ].lowerLimit = lastByte + 1;
            holTotal += list[ i ].upperLimit - list[ i ].lowerLimit + 1;
            return 1;
         }
   }
   else
   // (firstByte > list[ i ].lowerLimit)         // first > lower limit
   {
      if (lastByte >= list[ i ].upperLimit)      // data spans includes upper limit
      {
         holTotal -= list[ i ].upperLimit - list[ i ].lowerLimit + 1;
         list[ i ].upperLimit = firstByte - 1;
         holTotal += list[ i ].upperLimit - list[ i ].lowerLimit + 1;
         return 1;
      }
      else // (lastByte < list[ i ].upperLimit)  // data doesn't include lower or upper limit
      {
         unsigned long loLimit = lastByte + 1;
         unsigned long upLimit = list[ i ].upperLimit;
                                                 // modify current hole rec for lower hole
         holTotal -= list[ i ].upperLimit - list[ i ].lowerLimit + 1;
         list[ i ].upperLimit = firstByte - 1;
         holTotal += list[ i ].upperLimit - list[ i ].lowerLimit + 1;
         addHol( loLimit, upLimit );             // add new rec for upper hole
         return 1;
      }
   }
}

void HolFile::adjustLastHolDn()
// -------------------------------------------------------------------------
//  If an intermediate frame is received as the first frame then a hole
//  record is created as "lowerLimit, 0xfffffff".  Once we know the file
//  size, we can correct the upperLimit of this hole.
// -------------------------------------------------------------------------
{
   if (!holCount)                                // don't search if there
      return;                                    // are no hole records

   int i = holCount - 1;
   if (list[ i ].upperLimit >= fileSize)         // if adjustment is required
      if (list[ i ].lowerLimit >= fileSize)      // if hole > fileSize
      {
         deleteHol( i );                         // kill the hole
         return;
      }
      else                                       // else kill part of the hole
      {
         holTotal -= list[ i ].upperLimit - list[ i ].lowerLimit + 1;
         list[ i ].upperLimit = fileSize - 1;    // set new upper Limit
         holTotal += fileSize - list[ i ].lowerLimit;
         return;
      }
   else                                          // else no adjustment needed
      return;
}

void HolFile::adjustLastHolUp( const unsigned long newSize )
// -------------------------------------------------------------------------
//  This function is called for files that have dynamically increasing
//  file sizes.  The sizes increase so holes need to be added.  DO NOT call
//  this function if the fileSize is ZERO!
// -------------------------------------------------------------------------
{
   if (!holCount)                                // don't search if there
      return;                                    // are no hole records

   if (holCount)
   {
      int i = holCount - 1;
      if (list[ i ].upperLimit == fileSize - 1)
      {
         holTotal -= list[ i ].upperLimit - list[ i ].lowerLimit + 1;
         list[ i ].upperLimit = newSize - 1;
         holTotal += newSize - list[ i ].lowerLimit;
         return;
      }
      else
      {
         addHol( fileSize - 1, newSize - 1 );
         return;
      }
   }
   else                                          // file already complete
      addHol( fileSize - 1, newSize - 1 );
}

int HolFile::holeStrFromMem( FPair* pr )
// -------------------------------------------------------------------------
//  Create hole string for file fill request.  Return the number of holes in
//  the string.  If the entire file is needed (e.g., hole size 0xffff)
//  return 0;
// -------------------------------------------------------------------------
{
   if (!holCount)
      return 0;
   int i = 0;
   int count = 0;
   do
   {
      // Limit length to max size for unsigned int

      if (list[ i ].upperLimit - list[ i ].lowerLimit + 1 < 0xFFFF)
            pr[ count ].len = ( unsigned )(list[ i ].upperLimit -
                                           list[ i ].lowerLimit + 1);
      else
            pr[ count ].len = 0xFFFF;
      memcpy( &pr[ count ].offset, &list[ i ].lowerLimit, 3 );
      i++;
      count++;
   } while (i < holCount && count < MAX_PAIRS);
   return count;
}

int HolFile::holeStrFromFile( const char*         path,
                              const unsigned long fileNumber,
                                    FPair*        pr         )
// -------------------------------------------------------------------------
//  Create hole string for file fill request.  Return the number of holes in
//  the string.  If the entire file is needed (e.g., hole size 0xffff)
//  return 0;
// -------------------------------------------------------------------------
{
   sprintf( fileName, "%s%-lx.HOL", path, fileNumber);

   FILE *file;
   int count = 0;
   if ((file = fopen( fileName, "rt" )) != 0)
   {
      fgets( inRec, MAXLINE, file );             // pfh headers received
      fgets( inRec, MAXLINE, file );             // pfh file length
      fgets( inRec, MAXLINE, file );             // number of holes
      while (fgets( inRec, MAXLINE, file) != 0 && count < MAX_PAIRS)
      {
         unsigned long lowerLimit, upperLimit;
         sscanf( inRec, "%lu,%lu", &lowerLimit, &upperLimit );
         if (upperLimit - lowerLimit + 1 < 0xFFFF)
            pr[ count ].len = ( unsigned )(upperLimit - lowerLimit + 1);
         else
            pr[ count ].len = 0xFFFF;
         memcpy( &pr[count].offset, &lowerLimit, 3 );
         count++;
      }
      fclose( file );
      if (count == 1 && pr[0].len == 0)          // hole size is 0xffff + 1
         count = 0;
   }
   return count;
}

void HolFile::open()
// -------------------------------------------------------------------------
// Read .HOL file for a file
// -------------------------------------------------------------------------
{
   sprintf( fileName, "%s%-lx.HOL", path, fileNumber);
   FILE* file;
   unsigned long lowerLimit, upperLimit;
   if ((file = fopen( fileName, "rt" )) != 0)
   {
      fgets( inRec, MAXBYTES, file );            // pfh headers received
      fgets( inRec, MAXBYTES, file );            // pfh file length
      if (*inRec != '0')
         fileSize = atol( inRec );

      fgets( inRec, MAXBYTES, file );            // number of holes
      while (fgets( inRec, MAXBYTES, file ) != 0)
      {
         sscanf( inRec, "%lu,%lu", &lowerLimit, &upperLimit );
         addHol( lowerLimit, upperLimit );
      }
      fclose( file );
   }
   else                                          // no .HOL file found
   {
      lowerLimit = 0;                            // create a hole
      upperLimit = 0xfffffffL;
      addHol( lowerLimit, upperLimit );
   }
}

void HolFile::close( const int closeAction )
// -------------------------------------------------------------------------
//  Update .HOL file from hole list for files.
// -------------------------------------------------------------------------
{
   char* oldFileName = 0;
   char* newFileName = 0;
   switch (closeAction)
   {
      case SAVE:
         oldFileName = new char[ MAXPATH ];
         newFileName = new char[ MAXPATH ];
         if (!oldFileName || !newFileName)
            frWin.printFatal( "Out of memory" );
         sprintf( oldFileName, "%s%-lx.HOL", path, fileNumber);
         sprintf( newFileName, "%s%-lx.OLH", path, fileNumber);
         remove( newFileName );
         rename( oldFileName, newFileName );
         FILE* file;
         if ((file = fopen(oldFileName, "wt")) == 0)
         {
            sprintf( frWin.buf, "Error creating %-lx.HOL, %s",
                     fileNumber, sys_errlist[errno] );
            frWin.printFatal( frWin.buf );
         }
         fputs( "0 pfh header received\n", file );
         fprintf( file, "%lu pfh file length\n", fileSize );
         fprintf( file, "%u holes\n", holCount );
         int i;
         for (i = 0; i < holCount; i++)
            fprintf( file, "%lu, %lu\n", list[ i ].lowerLimit,
                                         list[ i ].upperLimit );
         fclose( file );
         break;
      case PURGE:
         oldFileName = new char[ MAXPATH ];
         if (!oldFileName)
            frWin.printFatal( "Out of memory" );
         sprintf( oldFileName, "%s%-lx.HOL", path, fileNumber);
         remove( oldFileName );
         sprintf( oldFileName, "%s%-lx.OLH", path, fileNumber);
         remove( oldFileName );
         break;
   }
   delete oldFileName;
   delete newFileName;
}

unsigned long HolFile::firstHol()
// -------------------------------------------------------------------------
//  Returns the offset of the first hole.  If no holes are found,
//  returns 0xffffffff.
// -------------------------------------------------------------------------
{
   if (holCount)
      return list[ 0 ].lowerLimit;
   return 0xffffffffL;
}

