//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __ACTFILE_H
#include "actfile.h"
#endif

#ifndef  __DIR_H
#include "dir.h"
#endif

#ifndef  __DOS_H
#include <dos.h>
#endif

#ifndef  __ERRNO_H
#include <errno.h>
#endif

#ifndef  __FCNTL_H
#include <fcntl.h>
#endif

#ifndef  __IO_H
#include <io.h>
#endif

#ifndef  __MATH_H
#include <math.h>
#endif

#ifndef  __STDIO_H
#include <stdio.h>
#endif

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STAT_H
#include <sys/stat.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __DIRBCST_H
#include "dirbcst.h"
#endif

#ifndef  __DIRECTRY_H
#include "directry.h"
#endif

#ifndef  __HOLFILE_H
#include "holfile.h"
#endif

#ifndef  __MISC_H
#include "misc.h"
#endif

#ifndef  __MSGBUF_H
#include "msgbuf.h"
#endif

#ifndef  __PFHEADER_H
#include "pfheader.h"
#endif

#ifndef  _WINDOW_H
#include "window.h"
#endif

ActFile::ActFile( const char*         path,
                  const unsigned long fileNumber,
                  const unsigned      position,
                        Directory*    dir,
                        unsigned      silent,
                  const unsigned      maxPfhSize,
                        Window&       fiWin,
                        Window&       iWin,
                        Window&       frWin )
                  :
                  accessCount( 0 ),
                  dir( dir ),
                  fileHandle( -1 ),
                  fileNumber( fileNumber ),
                  fileType( -1 ),
                  holFile( 0 ),
                  maxPfhSize( maxPfhSize ),
                  position( position ),
                  silent( silent ),
                  fiWin( fiWin ),
                  iWin( iWin ),
                  frWin( frWin )
// ------------------------------------------------------------------------
//  Constructor
// ------------------------------------------------------------------------
{
   int size = strlen( path ) + 1;
   dlDir = new char[ size ];
   if (!dlDir)
      frWin.printFatal( "Out of memory" );
   strcpy( dlDir, path );
}

ActFile::~ActFile()
// ------------------------------------------------------------------------
//  Destructor
// ------------------------------------------------------------------------
{
   delete dlDir;
}

void ActFile::close( const int closeAction )
// ------------------------------------------------------------------------
//  Close the open file
// ------------------------------------------------------------------------
{
   char* oldFileName = 0;
   char* newFileName = 0;
   ::close( fileHandle );
   switch (closeAction)
   {
      case SAVE:
         holFile->close( HolFile::SAVE );
         break;
      case PURGE:
         oldFileName = new char[ MAXPATH ];
         if (!oldFileName)
            frWin.printFatal( "Out of memory" );
         sprintf( oldFileName, "%s%-lx.ACT", dlDir, fileNumber);
         remove( oldFileName );
         holFile->close( HolFile::PURGE );
         break;
      case COMPLETE:
         if (!silent)
            sound( 800 );
         oldFileName = new char[ MAXPATH ];
         newFileName = new char[ MAXPATH ];
         if (!oldFileName || !newFileName)
            frWin.printFatal( "Out of memory" );
         sprintf( oldFileName, "%s%-lx.ACT", dlDir, fileNumber);
         sprintf( newFileName, "%s%-lx.DL", dlDir, fileNumber);
         if (rename( oldFileName, newFileName ) != 0)
         {
            sprintf( iWin.buf, "File %lx.DL already exists", fileNumber);
            iWin.print( iWin.buf );
            remove( oldFileName );
         }
         holFile->close( HolFile::PURGE );
         if (!silent)
            nosound();
         break;
   }
   delete oldFileName;
   delete newFileName;
   delete holFile;
}

int  ActFile::complete()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   return holFile->holCount > 0 ? 0 : 1;
}

void ActFile::display()
// --------------------------------------------------------------------------
//  Display an activity file pfh in the info window.  Triggered by "w" key.
// --------------------------------------------------------------------------
{
   PFileHeader* pfh = dir->getPfh( fileNumber );
   if (pfh)
   {
      pfh->display( position );
      return;
   }
   if (fileType < 0 && (holFile->holCount == 0 || holFile->firstHol() != 0))
   {
      char* bp = new char[ maxPfhSize ];
      PFileHeader* pfh = new PFileHeader( iWin );
      unsigned len = readPfh( *pfh, bp );
      if (len)                                   // if we have good pfh
         pfh->display( position );
      delete bp;
      delete pfh;
   }
}

void ActFile::display( const unsigned long offset )
// --------------------------------------------------------------------------
//  Display an activity file in the file window.
// --------------------------------------------------------------------------
{
   char* cp = frWin.buf;
   sprintf( cp, "%-8lx%6u", fileNumber, holFile->holCount );
   cp += 14;
   float percent = 0;
   unsigned long size = holFile->fileSize;
   if (size)
   {
      percent = floor( 100.0 - 100.0 * holFile->holTotal / size );
      if (percent >= 0.0 && percent <= 100.0 )
         sprintf( cp, "%8lu%8lu%4.0f%", size, offset, percent );
      else
         sprintf( cp, "     N/A%8lu     ", offset );
   }
   else
      sprintf( cp, "     N/A%8lu     ", offset );
   fiWin.printLine( position + 2, fiWin.buf );
   if (percent < 0 || percent > 100.0)
   {
      sprintf( iWin.buf, "Invalid file size in %lx header", fileNumber );
      iWin.print( iWin.buf );
   }
}

int ActFile::needData( const unsigned long offset,
                       const unsigned dLen,
                       const char     flag )
// ------------------------------------------------------------------------
//  Is the data needed?
// ------------------------------------------------------------------------
{
   if (fileType < 0)
   {
      // If file is in the directory, record its information from the
      // directory record.  The file size and type are needed.

      PFileHeader* pfh = dir->getPfh( fileNumber );
      if (pfh)
         savePfhInfo( *pfh );
   }
   return holFile->needData( offset, dLen, flag );
}

void ActFile::open()
// ------------------------------------------------------------------------
// ------------------------------------------------------------------------
{
   // See if the .ACT file already exists; create one if it doesn't.

   char* fileName = new char[ MAXPATH ];
   if (!fileName)
      frWin.printFatal( "Out of memory" );
   sprintf( fileName, "%s%-lx.ACT", dlDir, fileNumber);
   fileHandle = ::open( fileName, O_CREAT|O_RDWR|O_BINARY, S_IWRITE );
   if (fileHandle == -1)
   {
      sprintf( frWin.buf, "Open error for file %s, %s",
                          fileName, sys_errlist[ errno ] );
      frWin.printFatal( frWin.buf );
   }
   delete fileName;

   holFile = new HolFile( dlDir, fileNumber, frWin ); // create a hole file
   if (!holFile)
      frWin.printFatal( "Out of memory" );
   holFile->open();                              // open it
}

unsigned ActFile::readPfh( PFileHeader& pfh, char* bp )
// ------------------------------------------------------------------------
//  Read the header from the .ACT file into the buffer, bp.  The buffer must
//  be at least as large as "maxPfhSize".  Decode the header into pfh.
//  Return the length of the header in the buffer.
// ------------------------------------------------------------------------
{
   if ((holFile->holCount != 0 && holFile->firstHol() == 0) // if hole in
       || !bp )                                             // header or
      return 0;                                             // null buffer
   if (lseek( fileHandle, 0, SEEK_SET ) == -1)
   {
      sprintf( frWin.buf, "Seek error for file %-lx.act, %s", fileNumber, sys_errlist[ errno ]);
      frWin.printFatal( frWin.buf );
   }
   int res = read( fileHandle, bp, maxPfhSize );
   if (res == -1)
   {
      sprintf( frWin.buf, "Read error for file %-lx.act, %s", fileNumber, sys_errlist[ errno ]);
      frWin.printFatal( frWin.buf );
   }
   return pfh.get( bp, res );
}

void ActFile::savePfhInfo( PFileHeader& pfh )
// ------------------------------------------------------------------------
// ------------------------------------------------------------------------
{
   fileType = pfh.file_type;
   if (holFile->fileSize == 0)
   {
      holFile->fileSize = pfh.file_size;

      //@@@@@@@@@@@@@@@@@@@@@
      // If the following code is temporarily commented out, it is to try to
      // fix the problem with dynamically changing file sizes.  The only
      // adjustment will be made by the HolFile objects.

      holFile->adjustLastHolDn();                // adjust last hole down
   }
   else
      if (holFile->fileSize < pfh.file_size)
      {                                          // adjust last hole up
         holFile->adjustLastHolUp( pfh.file_size );
         holFile->fileSize = pfh.file_size;
      }
}

void ActFile::update( const unsigned long offset,
                      const char*         data,
                      const unsigned      len )
// ------------------------------------------------------------------------
// Save broadcast frame data
// ------------------------------------------------------------------------
{
   // Get to the correct point in the file.  Expand the file to the offset
   // if necessary.

   unsigned long fSize = filelength( fileHandle );
   if (offset > fSize)
//chsize( fileHandle, offset);
   {
      if (lseek( fileHandle, fSize, SEEK_SET ) == -1)
      {
         sprintf( frWin.buf, "Seek error for file %-lx.act, %s",
                             fileNumber, sys_errlist[ errno ] );
         frWin.printFatal( frWin.buf );
      }
      unsigned long nBytes = offset - fSize;     // bytes yet to output

      // Can only write 64K at a time; write garbage

      unsigned res;
      while (nBytes > 65534L)
      {
         if ((res = write( fileHandle, 0, (unsigned) 65534L )) != 65534L)
         {
            sprintf( frWin.buf,
                     "Error filling file %-lx.act with 65534L bytes",
                     fileNumber );
            if ((int) res == -1)
               sprintf( frWin.buf + strlen( frWin.buf ), ", %s", sys_errlist[ errno ] );
            else
               sprintf( frWin.buf + strlen( frWin.buf ), ", wrote only %lu bytes", res );
            frWin.printFatal( frWin.buf );
         }
         nBytes -= 65534L;
      }
      if ((res = write( fileHandle, 0, (unsigned) nBytes )) != nBytes)
      {
         sprintf( frWin.buf,
                  "Error filling file %-lx.act with %lu bytes", fileNumber, nBytes );
         if ((int) res == -1)
            sprintf( frWin.buf + strlen( frWin.buf ), ", %s", sys_errlist[ errno ] );
         else
            sprintf( frWin.buf + strlen( frWin.buf ), ", wrote only %li bytes", res );
         frWin.printFatal( frWin.buf );
      }

   }
   else
      if (lseek( fileHandle, offset, SEEK_SET ) == -1)
      {
         sprintf( frWin.buf, "Seek error for file %-lx.act, %s", fileNumber, sys_errlist[ errno ]);
         frWin.printFatal( frWin.buf );
      }
   if (write( fileHandle, data, len ) != len)
   {
      sprintf( frWin.buf, "Write error for file %-lx.act", fileNumber );
      frWin.printFatal( frWin.buf );
   }

   // We check for a good pfh after we write the data.  The pfh may span
   // more than one file frame so we need to read from the file to ensure
   // that we have a complete pfh.  The frames can come in any order.
   // We use fileType to track whether or not we have already done this.

   // Get the file type from the pfh.  Pass the file type to the StatusList
   // so that it can record whether or not this file should be blocked based
   // on file type.

   // Some of the servers may not set the header flag bit that indicates
   // that the last byte of the frame is the last byte of the file.
   // Therefore, we can't tell what is the size of the file file without
   // reading and using the pfh.  The file size is kept by the hole file
   // object.

   // We may already know the file size.  When ActFile::needData() is called,
   // it looks up the file in the directories that have been loaded.  The file
   // may or may not be in the directory.

   if (fileType < 0 && (holFile->holCount == 0 || holFile->firstHol() != 0))
   {
      char* bp = new char[ maxPfhSize ];
      PFileHeader pfh( iWin );
      unsigned len = readPfh( pfh, bp );
      if (len)                                   // if we 't have good pfh
      {
         // Put the pfh in the directory so that we can see it.  We also save
         // the directory at this point.  This will create duplicates in the
         // directory files, although they are not displayed.  This can't be
         // avoided if we want to see files for which we have not received
         // a broadcast directory entry.

         dir->addPfh( pfh );                     // put in dir display
         // save_dir( bp, len );                 // put in dir file
         savePfhInfo( pfh );
      }
      delete bp;
   }
}

