//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//
// --------------------------------------------------------------------------
//
//  The files ASY.* and PC.* provide a set of callable serial port driver
//  functions.  They do not require any other source code to support the
//  functions.  I give special thanks to Phil Karn, KA9Q, who suggested
//  using the code from his NOS program.  I have used some of the logic
//  from some of the NOS functions, but there are more differences than
//  similarities.
//
//  These functions are not meant to be comprehensive by any means, but they
//  do work for my purposes.  They do not support software flow control, but
//  I have included comments where the logic might be added.
//
// --------------------------------------------------------------------------

#ifndef  __8250_H
#define  __8250_H

#ifndef  __PC_H
#include "pc.h"
#endif

// --------------------------------------------------------------------------
// Output pseudo-dma control structure
// --------------------------------------------------------------------------

class DMA
{
public:
   const    char*    data;                       // current output pointer
            unsigned cnt;                        // byte count remaining
   volatile char     busy;                       // transmitter active

                     DMA() : data( 0 ), cnt( 0 ), busy( 0 ) {}
};

// --------------------------------------------------------------------------
// Read fifo control structure
// --------------------------------------------------------------------------

class FIFO
{
public:
            char*        buf;                    // ring buffer
            unsigned     bufSize;                // size of ring buffer
            char*        wp;                     // write pointer
            char*        rp;                     // read pointer
   volatile unsigned     cnt;                    // count of characters in buffer
            unsigned     hiWat;                  // high water mark
            unsigned     overflow;               // count of sw fifo buffer overflows

                         FIFO() : buf( 0 ), bufSize( 0 ), wp( 0 ), rp( 0 ),
                                  cnt( 0 ), hiWat( 0 ), overflow( 0 ) {}
};

// --------------------------------------------------------------------------
//  Asychronous Serial Port Object Class
// --------------------------------------------------------------------------

class Asy
{
public:

   // If you increase this, you must add additional interrupt vector
   // hooks in isr.cpp

   static enum         { ASY_MAX     =  8 };

   static enum         { BAUDRATE    =  1,
                         PARITYOPT   =  2,
                         DATABITS    =  3,
                         STOPBITS    =  4,
                         DTR         =  5,
                         OUT1        =  6,
                         OUT2        =  7,
                         RTS         =  8,
                         CTSmode     =  9,
                         RTSmode     = 10,
                         SOFTFLOW    = 11,
                         STARTCHAR   = 12,
                         STOPCHAR    = 13,
                         DOWN,
                         UP,
                         RLSDmode,
                         FIFODEPTH           };

   static Asy*           asyList[ ASY_MAX ];

// NOTE:  There are no default values for port numbers greater than four;
//        You must provide portAddr and portIRQ.
//
//        Default addresses and IRQs:
//                port 1:       0x03f8,   4
//                port 2:       0x02f8,   3
//                port 3:       0x03e8,   4
//                port 4:       0x02e8,   3

                         Asy( const int portNum,
                              const int portAddr,
                              const int portIRQ );
                         ~Asy();

           int           open( char& buffer, unsigned size );
           int           close();

           void          iFlush();
           void          oFlush();

           unsigned      iQSize();

           int           setOpt( int option, unsigned long value );
           unsigned long getOpt( int option );

           int           read( char* bp, unsigned len, unsigned& iQSize );
           int           readCh( char* c, unsigned& iQSize );

           int           write( const char* buf, unsigned cnt );

           int           checkInt();

                         // Asy interrupt handlers

   static void interrupt asy0vec( ... );
   static void interrupt asy1vec( ... );
   static void interrupt asy2vec( ... );
   static void interrupt asy3vec( ... );
   static void interrupt asy4vec( ... );
   static void interrupt asy5vec( ... );
   static void interrupt asy6vec( ... );
   static void interrupt asy7vec( ... );

          unsigned       errBreak;
          unsigned       errFraming;
          unsigned       errOverrun;
          unsigned       errParity;
          FIFO           fifo;

private:
   static enum         { STOPBIT1    = 0x00,        // bit masks for parity, stopbits
                         STOPBIT2    = 0x04,
                         NOPARITY    = 0x00,
                         ODDPARITY   = 0x08,
                         EVENPARITY  = 0x18 };

   //  National 8250 asynch UART definitions
   //  Control/status register offsets from base address

   static enum         { THR = 0,                   // transmitter holding register
                         RBR = 0,                   // receiver buffer register
                         DLL = 0,                   // divisor latch LSB
                         DLM = 1,                   // divisor latch MSB
                         IER = 1,                   // interrupt enable register
                         IIR = 2,                   // interrupt ident register
                         FCR = 2,                   // FIFO control register (16550A only)
                         LCR = 3,                   // line control register
                         MCR = 4,                   // modem control register
                         LSR = 5,                   // line status register
                         MSR = 6 };                 // modem status register

   //  Interrupt Enable Register bits

   static enum         { IER_DAV = 0x01,            // rx data available interrupt
                         IER_TxE = 0x02,            // tx holding register empty interrupt
                         IER_RLS = 0x04,            // receive line status interrupt
                         IER_MS  = 0x08 };          // modem status interrupt

   //  Interrupt Identification Register

   static enum         { IIR_IP           = 0x01,   // 0 if interrupt pending
                         IIR_MSTAT        = 0x00,   // modem status interrupt
                         IIR_THRE         = 0x02,   // transmitter holding register empty int
                         IIR_RDA          = 0x04,   // receiver data available interrupt
                         IIR_RLS          = 0x06,   // receiver line status interrupt
                         IIR_ID_MASK      = 0x07,   // mask for interrupt ID
                         IIR_FIFO_TIMEOUT = 0x08,   // FIFO timeout interrupt pending - 16550A
                         IIR_FIFO_ENABLED = 0xc0 }; // FIFO enabled (FCR0,1 = 1) - 16550A only

   // Line Control Register

   static enum         { LCR_5BITS    = 0x00,       // 5 bit words
                         LCR_6BITS    = 0x01,       // 6 bit words
                         LCR_7BITS    = 0x02,       // 7 bit words
                         LCR_8BITS    = 0x03,       // 8 bit words
                         LCR_BIT_MASK = 0x03,       // mask of bit size
                         LCR_NSB      = 0x04,       // number of stop bits
                         LCR_PEN      = 0x08,       // parity enable
                         LCR_EPS      = 0x10,       // even parity select
                         LCR_SP       = 0x20,       // stick parity
                         LCR_SB       = 0x40,       // set break
                         LCR_DLAB     = 0x80 };     // divisor latch access bit

   // 8250 Line Status Register

   static enum         { LSR_DR       = 0x01,       // data ready
                         LSR_OE       = 0x02,       // overrun error
                         LSR_PE       = 0x04,       // parity error
                         LSR_FE       = 0x08,       // framing error
                         LSR_BI       = 0x10,       // break interrupt
                         LSR_THRE     = 0x20,       // transmitter line holding register empty
                         LSR_TSRE     = 0x40,       // transmitter shift register empty
                         LSR_ERROR    = 0x0e };     // LST error mask

   // Modem Control Register

   static enum         { MCR_DTR          = 0x01,   // data terminal ready
                         MCR_RTS          = 0x02,   // request to send
                         MCR_RFR          = 0x02,   // ready for receiving (same as RTS)
                         MCR_OUT1         = 0x04,   // out 1 (not used)
                         MCR_OUT2         = 0x08,   // master interrupt enable (actually OUT 2)
                         MCR_LOOP         = 0x10 }; // loopback test mode

   // 8250 Modem Status Register

   static enum         { MSR_DCTS         = 0x01,   // delta clear-to-send
                         MSR_DDSR         = 0x02,   // delta data set ready
                         MSR_TERI         = 0x04,   // trailing edge ring indicator
                         MSR_DRLSD        = 0x08,   // delta rx line signal detect
                         MSR_CTS          = 0x10,   // clear to send
                         MSR_DSR          = 0x20,   // data set ready
                         MSR_RI           = 0x40,   // ring indicator
                         MSR_RLSD         = 0x80,   // rx line signal detect
                         MSR_DELTAS       = MSR_DCTS | MSR_DDSR |
                                            MSR_TERI | MSR_DRLSD,
                         MSR_CLEAR        = 0xf0,
                         MSR_NO_CLEAR     = 0x0f };

   // 16550A FIFO control register values

   static enum         { FIFO_ENABLE     = 0x01,    // enable tx & rx fifo
                         FIFO_CLR_RX     = 0x02,    // clear rx fifo
                         FIFO_CLR_TX     = 0x04,    // clear tx fifo
                         FIFO_START_DMA  = 0x08,    // enable txrdy/rxrdy pin DMA handshake
                         FIFO_SIZE_1     = 0x00,    // rx fifo trigger levels
                         FIFO_SIZE_4     = 0x40,
                         FIFO_SIZE_8     = 0x80,
                         FIFO_SIZE_14    = 0xC0,
                         FIFO_SIZE_MASK  = 0xC0,
                         FIFO_TRIGGER_LEVEL = FIFO_SIZE_4,
                         FIFO_SETUP         = FIFO_ENABLE |
                                              FIFO_CLR_RX |
                                              FIFO_CLR_TX |
                                              FIFO_TRIGGER_LEVEL };

   // The portStatus indicates the current device status.

   static enum {         ACTIVE          = 0x0001,  // interrupts enabled
                         ERROR_FOUND     = 0x0002,  // error reported (see below)
                         CARRIER_DET     = 0x0004,  // carrier detected
                         TIMER_ON        = 0x0008,  // ETC timer has been started
                         REM_FLOW        = 0x0010,  // remote system uses flow ctl
                         REM_XOFF        = 0x0020,  // rem sys has sent us an xoff
                         LOC_FLOW        = 0x0040,  // rem sys understands flow ctl
                         XOFF_SENT       = 0x0080,  // we have sent rem sys an xoff
                         SEND_XOFF       = 0x0100,  // tell driver to send an xoff
                         TRIM_BIT_7      = 0x0200,  // set bit 7 input byte to 0
                         WRITE_BIT_7     = 0x0400,  // force bit 7 of output to 1
                         BUFFER_FULL     = 0x0800,  // read buffer is full
                         XON_SENT        = 0x2000,  // XON sent to remote system
                         SEND_XON        = 0x4000 };// tell driver to send an XON

   static enum         { OUTPUT_FIFO_SIZE = 16 };


          unsigned       base;
          unsigned long  baud;
   static long           BAUDCLK;
          char*          buf;
          unsigned       bufSize;
          unsigned       busy;
          int            ctsReq;                 // use CTS flow control
          char           dataBits;
          DMA            dma;
          unsigned       fifoDepth;
          unsigned       fifoTimeOuts;
          unsigned       irq;
          char           is16550a;
          char           lsr;
   static long           MAXBPS;
          char           msr;                    // modem status bits
          unsigned       overrun;
          char           parity;
          unsigned       port;
   static CastISR        portISRs[ ASY_MAX ];
          char           portOpen;
          unsigned       portStatus;
          int            rlsdReq;                // use Received Line Signal Detect (aka CD)
          unsigned       rxChar;
          unsigned       rxInts;
          unsigned char  saveDivH;
          unsigned char  saveDivL;
          unsigned char  saveIER;
          unsigned char  saveIIR;
          unsigned char  saveLCR;
          int            saveMask;
          unsigned char  saveMCR;
          unsigned char  saveMSR;
          CastISR        saveVec;
          char           stopBits;
          unsigned       stInts;
          unsigned       txChar;
          unsigned       txInts;
                                                 // dynamic dialup params

          long           answers;                // count of incoming calls
          long           remdrops;               // count of times link dropped remotely

          void           readLSR();
          void           readMSR( const char clearFlag );
          void           rxInt();
          int            speed( unsigned long bps );
          void           stInt();
          void           txInt();
};

#endif   __ASY_H
