#include "Yaesu.h"
#include "math.h"
#include "ctype.h"
#include "string.h"


/* This software commands the Computer Aided Transceiver (CAT) port on  */
/* The Yaesu FT-736R and FT-847 transceivers.  This library contains    */
/* functions needed by satellite tracking software.  Future upgrades    */
/* will have a complete command set for these transceivers.             */

/* Copyright 1998 by Kenneth J. Ernandes, N2WWD  email: n2wwd@amsat.org */
/* Web page: <http://www.mindpsring.com/~n2wwd                          */

/* This code may be freely incorporated into any software distributed   */
/* by AMSAT, including AMSAT-NA and all international affiliates.  This */
/* code may also be freely incorporated into any personal or public     */
/* domain software.  Commercial software not distributed by AMSAT may   */
/* also use this code for a $10 donation to AMSAT-NA, 850 Sligo Avenue, */
/* Suite 600, Silver Spring, MD 20910-4703, USA or any international    */
/* affiliate.                                                           */

int catYaesuCommand( unsigned catPortID, char *msgByte )
{
  char i;

  for( i = 0; i < 5; i++ )
  {
    outp( catPortID, msgByte[i] );
    delay( 50 );                   // 50 msec delay between command bytes
  }

  return 0;
}


int cat736CtcssEnc( unsigned catPortID )
{
  char catCtcssEnc[5];

/* Only the last byte is relevant for enabling CTCSS encoding. */

  catCtcssEnc[4] = 0x4A;

  catYaesuCommand( catPortID, catCtcssEnc );

  return 0;
}


int cat847CtcssEnc( unsigned catPortID, char vfoID )
{
  char catCtcssEnc[5];

/* First byte commands enabling CTCSS encoding. */

  catCtcssEnc[0] = 0x4A;

/* Last byte indicates which VFO in which the CTCSS should encode. */

  switch( vfoID )
  {
    case 0:                            // Main VFO
      catCtcssEnc[4] = 0x0A;
      break;
    case 1:                            // Sat RX VFO
      catCtcssEnc[4] = 0x1A;
      break;
    case 2:                            // Sat TX VFO
      catCtcssEnc[4] = 0x2A;
      break;
    default:                           // Default to Main VFO
      catCtcssEnc[4] = 0x0A;
  }

  catYaesuCommand( catPortID, catCtcssEnc );

  return 0;
}


int cat736CtcssEncDec( unsigned catPortID )
{
  char catCtcssEncDec[5];

/* Only the last byte is relevant for enabling CTCSS encoding and decoding. */

  catCtcssEncDec[4] = 0x0A;

  catYaesuCommand( catPortID, catCtcssEncDec );

  return 0;
}


int cat847CtcssEncDec( unsigned catPortID, char vfoID )
{
  char catCtcssEncDec[5];

/* First byte commands enabling CTCSS encoding and decoding. */

  catCtcssEncDec[0] = 0x2A; 

/* Last byte indicates which VFO in which the CTCSS should encode & decode. */

  switch( vfoID )
  {
    case 0:                            // Main VFO
      catCtcssEncDec[4] = 0x0A;
      break;
    case 1:                            // Sat RX VFO
      catCtcssEncDec[4] = 0x1A;
      break;
    case 2:                            // Sat TX VFO
      catCtcssEncDec[4] = 0x2A;
      break;
    default:                           // Default to Main VFO
      catCtcssEncDec[4] = 0x0A;
  }

  catYaesuCommand( catPortID, catCtcssEncDec );

  return 0;
}


int cat736CtcssFreqInput( unsigned catPortID, double ctcssFreq )
{
  char codeIndex;
  int status;

  if( ctcssFreq > 65.00  &&  ctcssFreq < 69.45 )
    codeIndex = 0;
  else if( ctcssFreq > 69.45  &&  ctcssFreq < 73.15 )
    codeIndex = 1;
  else if( ctcssFreq > 73.15  &&  ctcssFreq < 75.70 )
    codeIndex = 35;
  else if( ctcssFreq > 75.70  &&  ctcssFreq < 78.35 )
    codeIndex = 2;
  else if( ctcssFreq > 78.35  &&  ctcssFreq < 81.10 )
    codeIndex = 37;
  else if( ctcssFreq > 81.10  &&  ctcssFreq < 83.95 )
    codeIndex = 3;
  else if( ctcssFreq > 83.95  &&  ctcssFreq < 86.95 )
    codeIndex = 39;
  else if( ctcssFreq > 86.95  &&  ctcssFreq < 90.00 )
    codeIndex = 4;
  else if( ctcssFreq > 90.00  &&  ctcssFreq < 93.15 )
    codeIndex = 41;
  else if( ctcssFreq > 93.15  &&  ctcssFreq < 97.40 )
    codeIndex = 5;
  else if( ctcssFreq > 97.40  &&  ctcssFreq < 101.75 )
    codeIndex = 6;
  else if( ctcssFreq > 101.75  &&  ctcssFreq < 105.35 )
    codeIndex = 7;
  else if( ctcssFreq > 105.35  &&  ctcssFreq < 109.05 )
    codeIndex = 8;
  else if( ctcssFreq > 109.05  &&  ctcssFreq < 112.85 )
    codeIndex = 9;
  else if( ctcssFreq > 112.85  &&  ctcssFreq < 116.80 )
    codeIndex = 10;
  else if( ctcssFreq > 116.80  &&  ctcssFreq < 120.90 )
    codeIndex = 11;
  else if( ctcssFreq > 120.90  &&  ctcssFreq < 125.15 )
    codeIndex = 12;
  else if( ctcssFreq > 125.15  &&  ctcssFreq < 129.55 )
    codeIndex = 13;
  else if( ctcssFreq > 101.75  &&  ctcssFreq < 105.35 )
    codeIndex = 14;
  else if( ctcssFreq > 134.15  &&  ctcssFreq < 138.90 )
    codeIndex = 15;
  else if( ctcssFreq > 138.90  &&  ctcssFreq < 143.75 )
    codeIndex = 16;
  else if( ctcssFreq > 143.75  &&  ctcssFreq < 148.80 )
    codeIndex = 17;
  else if( ctcssFreq > 148.80  &&  ctcssFreq < 154.05 )
    codeIndex = 18;
  else if( ctcssFreq > 154.05  &&  ctcssFreq < 159.45 )
    codeIndex = 19;
  else if( ctcssFreq > 159.45  &&  ctcssFreq < 165.05 )
    codeIndex = 20;
  else if( ctcssFreq > 165.05  &&  ctcssFreq < 170.85 )
    codeIndex = 21;
  else if( ctcssFreq > 170.85  &&  ctcssFreq < 176.85 )
    codeIndex = 22;
  else if( ctcssFreq > 176.85  &&  ctcssFreq < 183.05 )
    codeIndex = 23;
  else if( ctcssFreq > 183.05  &&  ctcssFreq < 189.50 )
    codeIndex = 24;
  else if( ctcssFreq > 189.50  &&  ctcssFreq < 198.15 )
    codeIndex = 25;
  else if( ctcssFreq > 198.15  &&  ctcssFreq < 207.10 )
    codeIndex = 26;
  else if( ctcssFreq > 207.10  &&  ctcssFreq < 214.40 )
    codeIndex = 27;
  else if( ctcssFreq > 214.40  &&  ctcssFreq < 221.90 )
    codeIndex = 28;
  else if( ctcssFreq > 221.90  &&  ctcssFreq < 229.65 )
    codeIndex = 29;
  else if( ctcssFreq > 229.65  &&  ctcssFreq < 237.35 )
    codeIndex = 30;
  else if( ctcssFreq > 207.10  &&  ctcssFreq < 214.40 )
    codeIndex = 31;
  else if( ctcssFreq > 207.10  &&  ctcssFreq < 214.40 )
    codeIndex = 32;
  else
    codeIndex = -1;

  status = ( codeIndex == -1 );

  if( !status )
    cat736CtcssToneCode( catPortID, codeIndex );

  return status;
}


int cat847CtcssFreqInput( unsigned catPortID, double ctcssFreq, char vfoID )
{
  char codeIndex;
  int status;

  if( ctcssFreq > 65.00  &&  ctcssFreq < 68.15 )
    codeIndex = 38;
  else if( ctcssFreq > 68.15  &&  ctcssFreq < 70.60 )
    codeIndex = 32;
  else if( ctcssFreq > 70.60  &&  ctcssFreq < 73.15 )
    codeIndex = 31;
  else if( ctcssFreq > 73.15  &&  ctcssFreq < 75.70 )
    codeIndex = 37;
  else if( ctcssFreq > 75.70  &&  ctcssFreq < 78.35 )
    codeIndex = 15;
  else if( ctcssFreq > 78.35  &&  ctcssFreq < 81.10 )
    codeIndex = 36;
  else if( ctcssFreq > 81.10  &&  ctcssFreq < 83.95 )
    codeIndex = 30;
  else if( ctcssFreq > 83.95  &&  ctcssFreq < 86.95 )
    codeIndex = 35;
  else if( ctcssFreq > 86.95  &&  ctcssFreq < 90.00 )
    codeIndex = 14;
  else if( ctcssFreq > 90.00  &&  ctcssFreq < 93.15 )
    codeIndex = 34;
  else if( ctcssFreq > 93.15  &&  ctcssFreq < 96.10 )
    codeIndex = 29;
  else if( ctcssFreq > 96.10  &&  ctcssFreq < 98.70 )
    codeIndex = 33;
  else if( ctcssFreq > 98.70  &&  ctcssFreq < 101.75 )
    codeIndex = 13;
  else if( ctcssFreq > 101.75  &&  ctcssFreq < 105.35 )
    codeIndex = 28;
  else if( ctcssFreq > 105.35  &&  ctcssFreq < 109.05 )
    codeIndex = 12;
  else if( ctcssFreq > 109.05  &&  ctcssFreq < 112.85 )
    codeIndex = 27;
  else if( ctcssFreq > 112.85  &&  ctcssFreq < 116.80 )
    codeIndex = 11;
  else if( ctcssFreq > 116.80  &&  ctcssFreq < 120.90 )
    codeIndex = 26;
  else if( ctcssFreq > 120.90  &&  ctcssFreq < 125.15 )
    codeIndex = 10;
  else if( ctcssFreq > 125.15  &&  ctcssFreq < 129.55 )
    codeIndex = 25;
  else if( ctcssFreq > 129.55  &&  ctcssFreq < 134.15 )
    codeIndex = 9;
  else if( ctcssFreq > 134.15  &&  ctcssFreq < 138.90 )
    codeIndex = 24;
  else if( ctcssFreq > 138.90  &&  ctcssFreq < 143.75 )
    codeIndex = 8;
  else if( ctcssFreq > 143.75  &&  ctcssFreq < 148.80 )
    codeIndex = 23;
  else if( ctcssFreq > 148.80  &&  ctcssFreq < 154.05 )
    codeIndex = 7;
  else if( ctcssFreq > 154.05  &&  ctcssFreq < 159.45 )
    codeIndex = 22;
  else if( ctcssFreq > 159.45  &&  ctcssFreq < 165.05 )
    codeIndex = 6;
  else if( ctcssFreq > 165.05  &&  ctcssFreq < 170.85 )
    codeIndex = 21;
  else if( ctcssFreq > 170.85  &&  ctcssFreq < 176.85 )
    codeIndex = 5;
  else if( ctcssFreq > 176.85  &&  ctcssFreq < 183.05 )
    codeIndex = 20;
  else if( ctcssFreq > 183.05  &&  ctcssFreq < 189.50 )
    codeIndex = 4;
  else if( ctcssFreq > 189.50  &&  ctcssFreq < 198.15 )
    codeIndex = 19;
  else if( ctcssFreq > 198.15  &&  ctcssFreq < 207.10 )
    codeIndex = 3;
  else if( ctcssFreq > 207.10  &&  ctcssFreq < 214.40 )
    codeIndex = 18;
  else if( ctcssFreq > 214.40  &&  ctcssFreq < 221.90 )
    codeIndex = 2;
  else if( ctcssFreq > 221.90  &&  ctcssFreq < 229.65 )
    codeIndex = 17;
  else if( ctcssFreq > 229.65  &&  ctcssFreq < 237.70 )
    codeIndex = 1;
  else if (ctcssFreq > 237.70  &&  ctcssFreq < 246.05 )
    codeIndex = 16;
  else if (ctcssFreq > 246.05  &&  ctcssFreq < 252.30 )
    codeIndex = 0;
  else
    codeIndex = -1;

  status = ( codeIndex == -1 );

  if( !status )
    cat847CtcssToneCode( catPortID, codeIndex, vfoID );

  return status;
}


int cat736CtcssOff( unsigned catPortID )
{
  char catCtcssOff[5];

/* Only the last byte is relevant for disabling the CTCSS. */

  catCtcssOff[4] = 0x8A;

  catYaesuCommand( catPortID, catCtcssOff );

  return 0;
}


int cat847CtcssOff( unsigned catPortID, char vfoID  )
{
  char catCtcssOff[5];

/* First byte commands disabling the CTCSS. */

  catCtcssOff[0] = 0x8A;

/* Last byte indicates which VFO in which the CTCSS should be disabled. */

  switch( vfoID )
  {
    case 0:                            // Main VFO
      catCtcssOff[4] = 0x0A;
      break;
    case 1:                            // Sat RX VFO
      catCtcssOff[4] = 0x1A;
      break;
    case 2:
      catCtcssOff[4] = 0x2A;           // Sat TX VFO
      break;
    default:
      catCtcssOff[4] = 0x0A;           // Default to Main VFO
  }

  catYaesuCommand( catPortID, catCtcssOff );

  return 0;
}


int cat736CtcssPlInput( unsigned catPortID, char *plCode )
{
  char codeIndex, i, plCodeUC[4];
  int status;

/* Make upper case copy of PL code string. */

  for( i = 0; i < 2; i++ )
  {
    if( plCode[i] == NULL ) break;

    plCodeUC[i] = toupper( plCode[i] );
  }

/* Ensure null termination. */

  plCodeUC[3] = NULL;

/* Search for PL code. */

  if( !strcmp( plCode, "XZ" ) )
    codeIndex = 0;
  else if( !strcmp( plCodeUC, "XA" ) )
    codeIndex = 1;
  else if( !strcmp( plCodeUC, "WA" ) )
    codeIndex = 2;
  else if( !strcmp( plCodeUC, "YZ" ) )
    codeIndex = 3;
  else if( !strcmp( plCodeUC, "YB" ) )
    codeIndex = 4;
  else if( !strcmp( plCodeUC, "ZA" ) )
    codeIndex = 5;
  else if( !strcmp( plCodeUC, "1Z" ) )
    codeIndex = 6;
  else if( !strcmp( plCodeUC, "1A" ) )
    codeIndex = 7;
  else if( !strcmp( plCodeUC, "1B" ) )
    codeIndex = 8;
  else if( !strcmp( plCodeUC, "2E" ) )
    codeIndex = 9;
  else if( !strcmp( plCodeUC, "2A" ) )
    codeIndex = 10;
  else if( !strcmp( plCodeUC, "2B" ) )
    codeIndex = 11;
  else if( !strcmp( plCodeUC, "3Z" ) )
    codeIndex = 12;
  else if( !strcmp( plCodeUC, "3A" ) )
    codeIndex = 13;
  else if( !strcmp( plCodeUC, "3B" ) )
    codeIndex = 14;
  else if( !strcmp( plCodeUC, "4Z" ) )
    codeIndex = 15;
  else if( !strcmp( plCodeUC, "4A" ) )
    codeIndex = 16;
  else if( !strcmp( plCodeUC, "4B" ) )
    codeIndex = 17;
  else if( !strcmp( plCodeUC, "5Z" ) )
    codeIndex = 18;
  else if( !strcmp( plCodeUC, "5A" ) )
    codeIndex = 19;
  else if( !strcmp( plCodeUC, "5B" ) )
    codeIndex = 20;
  else if( !strcmp( plCodeUC, "6Z" ) )
    codeIndex = 21;
  else if( !strcmp( plCodeUC, "6A" ) )
    codeIndex = 22;
  else if( !strcmp( plCodeUC, "6B" ) )
    codeIndex = 23;
  else if( !strcmp( plCodeUC, "7Z" ) )
    codeIndex = 24;
  else if( !strcmp( plCodeUC, "7A" ) )
    codeIndex = 25;
  else if( !strcmp( plCodeUC, "M1" ) )
    codeIndex = 26;
  else if( !strcmp( plCodeUC, "M2" ) )
    codeIndex = 27;
  else if( !strcmp( plCodeUC, "M3" ) )
    codeIndex = 28;
  else if( !strcmp( plCodeUC, "M4" ) )
    codeIndex = 29;
  else if( !strcmp( plCodeUC, "M5" ) )
    codeIndex = 30;
  else if( !strcmp( plCodeUC, "M6" ) )
    codeIndex = 31;
  else if( !strcmp( plCodeUC, "M7" ) )
    codeIndex = 32;
  else if( !strcmp( plCodeUC, "XZQ" ) )
    codeIndex = 33;
  else if( !strcmp( plCodeUC, "XAQ" ) )
    codeIndex = 34;
  else if( !strcmp( plCodeUC, "WA" ) )
    codeIndex = 35;
  else if( !strcmp( plCodeUC, "XBQ" ) )
    codeIndex = 36;
  else if( !strcmp( plCodeUC, "WB" ) )
    codeIndex = 37;
  else if( !strcmp( plCodeUC, "YZQ" ) )
    codeIndex = 38;
  else if( !strcmp( plCodeUC, "YA" ) )
    codeIndex = 39;
  else if( !strcmp( plCodeUC, "YBQ" ) )
    codeIndex = 40;
  else if( !strcmp( plCodeUC, "ZZ" ) )
    codeIndex = 41;
  else
    codeIndex = -1;

  status = ( codeIndex == -1 );

  if( !status )
    cat736CtcssToneCode( catPortID, codeIndex );

  return status;
}


int cat847CtcssPlInput( unsigned catPortID, char *plCode, char vfoID )
{
  char codeIndex, i, plCodeUC[4];
  int status;

/* Make upper case copy of PL code string. */

  for( i = 0; i < 2; i++ )
  {
    if( plCode[i] == NULL ) break;

    plCodeUC[i] = toupper( plCode[i] );
  }

/* Ensure null termination. */

  plCodeUC[3] = NULL;

/* Search for PL code. */

  if( !strcmp( plCode, "M7" ) )
    codeIndex = 0;
  else if( !strcmp( plCodeUC, "M5" ) )
    codeIndex = 1;
  else if( !strcmp( plCodeUC, "M3" ) )
    codeIndex = 2;
  else if( !strcmp( plCodeUC, "M1" ) )
    codeIndex = 3;
  else if( !strcmp( plCodeUC, "7Z" ) )
    codeIndex = 4;
  else if( !strcmp( plCodeUC, "6A" ) )
    codeIndex = 5;
  else if( !strcmp( plCodeUC, "5B" ) )
    codeIndex = 6;
  else if( !strcmp( plCodeUC, "5Z" ) )
    codeIndex = 7;
  else if( !strcmp( plCodeUC, "4A" ) )
    codeIndex = 8;
  else if( !strcmp( plCodeUC, "3B" ) )
    codeIndex = 9;
  else if( !strcmp( plCodeUC, "3Z" ) )
    codeIndex = 10;
  else if( !strcmp( plCodeUC, "2A" ) )
    codeIndex = 11;
  else if( !strcmp( plCodeUC, "1B" ) )
    codeIndex = 12;
  else if( !strcmp( plCodeUC, "1Z" ) )
    codeIndex = 13;
  else if( !strcmp( plCodeUC, "YB" ) )
    codeIndex = 14;
  else if( !strcmp( plCodeUC, "XB" ) )
    codeIndex = 15;
  else if( !strcmp( plCodeUC, "M6" ) )
    codeIndex = 16;
  else if( !strcmp( plCodeUC, "M4" ) )
    codeIndex = 17;
  else if( !strcmp( plCodeUC, "M2" ) )
    codeIndex = 18;
  else if( !strcmp( plCodeUC, "7A" ) )
    codeIndex = 19;
  else if( !strcmp( plCodeUC, "6B" ) )
    codeIndex = 20;
  else if( !strcmp( plCodeUC, "6Z" ) )
    codeIndex = 21;
  else if( !strcmp( plCodeUC, "5A" ) )
    codeIndex = 22;
  else if( !strcmp( plCodeUC, "4B" ) )
    codeIndex = 23;
  else if( !strcmp( plCodeUC, "4Z" ) )
    codeIndex = 24;
  else if( !strcmp( plCodeUC, "3A" ) )
    codeIndex = 25;
  else if( !strcmp( plCodeUC, "2B" ) )
    codeIndex = 26;
  else if( !strcmp( plCodeUC, "2E" ) )
    codeIndex = 27;
  else if( !strcmp( plCodeUC, "1A" ) )
    codeIndex = 28;
  else if( !strcmp( plCodeUC, "ZA" ) )
    codeIndex = 29;
  else if( !strcmp( plCodeUC, "YZ" ) )
    codeIndex = 30;
  else if( !strcmp( plCodeUC, "XA" ) )
    codeIndex = 31;
  else if( !strcmp( plCodeUC, "WZ" ) )
    codeIndex = 32;
  else if( !strcmp( plCodeUC, "ZB" ) )
    codeIndex = 33;
  else if( !strcmp( plCodeUC, "ZZ" ) )
    codeIndex = 34;
  else if( !strcmp( plCodeUC, "YA" ) )
    codeIndex = 35;
  else if( !strcmp( plCodeUC, "WB" ) )
    codeIndex = 36;
  else if( !strcmp( plCodeUC, "WA" ) )
    codeIndex = 37;
  else if( !strcmp( plCodeUC, "XZ" ) )
    codeIndex = 38;
  else
    codeIndex = -1;

  status = ( codeIndex == -1 );

  if( !status )
    cat847CtcssToneCode( catPortID, codeIndex, vfoID );

  return status;
}


int cat736CtcssToneCode( unsigned catPortID, char codeIndex )
{
  char toneCode[5];
  int status;

  if( codeIndex < 0 )
  {
    status = -1;
  }

  else if( codeIndex > 41 )
  {
    status = 1;
  }

  else
  {
    status = 0;

/* Set specific CTCSS frequency code. */

    toneCode[0] = 0x3E - codeIndex;

/* Command to set CTCSS frequency. */

    toneCode[4] = 0xFA;

/* Send command to CAT port. */

    catYaesuCommand( catPortID, toneCode );
  }

  return status;
}


int cat847CtcssToneCode( unsigned catPortID, char codeIndex, char vfoID )
{
  char toneCode[5];
  int status;

  if( codeIndex < 0 )
  {
    status = -1;
  }

  else if( codeIndex > 38 )
  {
    status = 1;
  }

  else
  {
    status = 0;

/* Set specific CTCSS frequency code. */

    if( codeIndex < 32 )
      toneCode[0] = codeIndex;
    else
      toneCode[0] = 0x39 + codeIndex - 32;

/* Command to set CTCSS frequency. */

    switch( vfoID )
    {
      case 0:                          // Main VFO
        toneCode[4] = 0x0B;
        break;
      case 1:                          // Sat RX VFO
        toneCode[4] = 0x1B;
        break;
      case 2:                          // Sat TX VFO
        toneCode[4] = 0x2B;
    }

/* Send command to CAT port. */

    catYaesuCommand( catPortID, toneCode );
  }

  return status;
}


int catYaesuDisable( unsigned catPortID )
{
  char i;

  for( i = 0; i < 5; i++ )
  {
    outp( catPortID, 0x80 );
    delay( 50 );
  }



  return 0;
}


int catYaesuEnable( unsigned catPortID, char catPortNumber )
{
  unsigned char i, comm_byte, modemCtrlReg, modemStatReg;
  unsigned int comm_set;

/* Set register addresses. */

  modemCtrlReg = catPortID + 4;
  modemStatReg = catPortID + 6;

/* Init serial port at 4800 baud, 8 data bits, 2 stop bits, no parity. */

/* Note that the FT-847 can accept other baud rates -- 4800 was        */
/* selected for compatibility with the FT-847 -- i.e., simplicity      */

  _bios_serialcom( _COM_INIT, catPortNumber,
                   _COM_CHR8 | _COM_STOP2 | _COM_NOPARITY | _COM_4800 );

/* Set RTS low and DTR high. */

  comm_set = inp( modemCtrlReg );  // Read register
  comm_set &= ~2;                  // RTS bit low
  comm_set |= 1;                   // DTR bit high
  outp( modemCtrlReg, comm_set );  // Write register

/* Set CTS low. */

  comm_set = inp( modemStatReg );  // Read register
  comm_set &= ~16;                 // CTS bit low
  outp( modemStatReg, comm_set );  // Write register

  delay( 100 );                    // 100 msec delay

/* Write enable command to CAT port. */

  for( i = 0; i < 5; i++ )
  {
    outp( catPortID, 0x00 );
    delay( 50 );                   // 50 msec delay
  }

  return 0;
}


int catYaesuFreqEncode( double freqMHz, char *msgByte )
{
  char i;
  int digits[8];
  double freq10Hz, temp;

/* Compute frequency in 10 Hz units -- force roundoff. */

  freq10Hz = floor( freqMHz * 1.0E5 + 0.5 );

/* Separate frequencies into individual digits. */

  for( i = 0; i < 7; i++ )
  {
    temp = freq10Hz / 10.0;
    freq10Hz = floor( temp );

/* Most significant digit has index of 0; least significant digit has */
/* index of 7.                                                        */

    digits[7-i] = (int) ( 10.0 * (temp - freq10Hz) + 0.5 );
  }

/* Most significant digit keeps residual upper frequency to accommodate */
/* the 1.2 GHz band.                                                    */

  digits[0] = freq10Hz;

/* Frequency is eight packed BCD digits in first 4 characters of msgByte. */

  for( i = 0; i < 4; i++ )
  {
    msgByte[i] = (char) ( 16 * digits[2*i]  +  digits[2*i+1] );
  }

  return 0;
}



int catYaesuFreqSet( unsigned catPortID, double freq )
{
  char freqSetCmd[5];

/* Encode the input frequency in the first 4 bytes of command string. */

  catYaesuFreqEncode( freq, freqSetCmd );

/* Set Instruction code to set VFO frequency. */

  freqSetCmd[4] = 0x01;

/* Set the frequency. */

  catYaesuCommand( catPortID, freqSetCmd );

  return 0;
}


int catYaesuModeSet( unsigned catPortID, char mode )
{
  char catCommand[5];

/* Set first 4 characters. */

  catYaesuModeString( mode, catCommand );

/* Mode set instruction code character. */

  catCommand[4] = 0x07;

/* Send command character string to CAT. */

  catYaesuCommand( catPortID, catCommand );

  return 0;
}


int catYaesuModeString( char mode, char *catCommand )
{

/* Set command mode byte according to mode character. */

  switch( mode )
  {
    case 'L':
      catCommand[0] = 0x00;
      break;
    case 'U':
      catCommand[0] = 0x01;
      break;
    case 'C':
      catCommand[0] = 0x02;
      break;
    case 'c':
      catCommand[0] = 0x82;
      break;
    case 'R':
      catCommand[0] = 0x03;
      break;
    case 'r':
      catCommand[0] = 0x83;
      break;
    case 'A':
      catCommand[0] = 0x04;
      break;
    case 'a':
      catCommand[0] = 0x84;
      break;
    case 'F':
      catCommand[0] = 0x08;
      break;
    case 'f':
      catCommand[0] = 0x88;
      break;
    default:
      catCommand[0] = 0x08;
  }

/* Padding Characters. */

  catCommand[1] = 0;
  catCommand[2] = 0;
  catCommand[3] = 0;

  return 0;
}


int catYaesuSatModeEnable( unsigned catPortID, char satModeEnable,
                           char transceiverType )
{
  char i, catCommand[5];

  for( i = 0; i < 5; i++ )
  {
    catCommand[i] = 0;
  }

  if( satModeEnable )        // Turn on Sat Mode if satModeEnable != 0
  {
    if( transceiverType )
      catCommand[4] = 0x4E;            // FT-847
    else
      catCommand[4] = 0x0E;            // FT-736R
  }
  else
  {
    catCommand[4] = 0x8E;    // Turn off Sat Mode if satModeEnable = 0
  }

  catYaesuCommand( catPortID, catCommand );

  return 0;
}


int catYaesuSatRxFreq( unsigned catPortID, double freq, char transceiverType )
{
  char freqSetCmd[5];

/* Encode the input frequency in the first 4 bytes of command string. */

  catYaesuFreqEncode( freq, freqSetCmd );

/* Set Instruction code to set VFO frequency. */

  if( transceiverType )
    freqSetCmd[4] = 0x11;              // FT-847
  else
    freqSetCmd[4] = 0x1E;              // FT-736R

/* Set the frequency. */

  catYaesuCommand( catPortID, freqSetCmd );

  return 0;
}


int catYaesuSatRxMode( unsigned catPortID, char mode )
{
  char catCommand[5];

/* Set first 4 characters. */

  catYaesuModeString( mode, catCommand );

/* Mode set instruction code character. */

  catCommand[4] = 0x17;

/* Send command character string to CAT. */

  catYaesuCommand( catPortID, catCommand );

  return 0;
}


int catYaesuSatTxFreq( unsigned catPortID, double freq, char transceiverType )
{
  char freqSetCmd[5];

/* Encode the input frequency in the first 4 bytes of command string. */

  catYaesuFreqEncode( freq, freqSetCmd );

/* Set Instruction code to set VFO frequency. */

  if( transceiverType )
    freqSetCmd[4] = 0x21;              // FT-847
  else
    freqSetCmd[4] = 0x2E;              // FT-736R

/* Set the frequency. */

  catYaesuCommand( catPortID, freqSetCmd );

  return 0;
}


int catYaesuSatTxMode( unsigned catPortID, char mode )
{
  char catCommand[5];

/* Set first 4 characters. */

  catYaesuModeString( mode, catCommand );

/* Mode set instruction code character. */

  catCommand[4] = 0x27;

/* Send command character string to CAT. */

  catYaesuCommand( catPortID, catCommand );

  return 0;
}


int catYaesuSplit( unsigned catPortID, double rcvFreq, double xmtFreq,
                   char transceiverType )
{
  char catCommand[5], splitDir;
  double deltaFreq;

  deltaFreq = xmtFreq - rcvFreq;

  if( deltaFreq < 0.0 )
  {
    splitDir = '-';
  }

  else if( deltaFreq > 0.0 )
  {
    if( deltaFreq > 0.0 )
    {
      splitDir = '+';
    }

    else
    {
      splitDir = ' ';
    }
  }

  catYaesuFreqSet( catPortID, rcvFreq );
  catYaesuSplitOffset( catPortID, fabs(deltaFreq) );
  catYaesuSplitDir( catPortID, splitDir, transceiverType );

  return 0;
}


int catYaesuSplitDir( unsigned catPortID, char dir, char transceiverType )
{
  char i, simplexCode, catCommand[5];

/* Set simplex code */

  if( transceiverType )
    simplexCode = 0x88;                // FT-847
  else
    simplexCode = 0x89;                // FT-736R

/* Set padding characters. */

  for( i = 0; i < 4; i++ )
  {
    catCommand[i] = 0;
  }

/* Set Split Instruction Code. */
  switch( dir )
  {
    case '-':
      catCommand[4] = 0x09;
      break;
    case '+':
      catCommand[4] = 0x49;
      break;
    case ' ':
      catCommand[4] = simplexCode;
      break;
    default:
      catCommand[4] = simplexCode;
  }

  catYaesuCommand( catPortID, catCommand );

  return 0;
}


int catYaesuSplitOffset( unsigned catPortID, double freqOffset )
{
  char catCommand[5];

  catYaesuFreqEncode( freqOffset, catCommand );
  catCommand[4] = 0xF9;

  catYaesuCommand( catPortID, catCommand );

  return 0;
}

