/* Copyright (C) 1999 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

// tnc.cpp

#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <iostream.h>
#include <strstream.h>
#include "tnc.h"
#include "pipes.h"
#include "widgets.h"

// the following define set the amount of clear bytes in the Kam buffer that
// will be left when send a file
#define KEYBOARD_PRIORITY (WORDWRAP + 2)

#define CHANGEMODE_LATENCY 3
#define CHANGE_CONNECTED_STATUS_LATENCY 10
#define FILESEND_COUNT 3

Tnc::Tnc(Transmit_buffer& a, Pipe_fifo& b, Pipe_fifo& c,
	 int vhf_paclen, int hf_paclen, int no_rx_cw_flag, int rx_bell_flag, int slow_write_flag):
           Tnc_base(Tnc_base(a, b, c)), check_status_flag(0), info_requested_flag(0),
           rx_frame_complete(FALSE), unsent_frame_flag(FALSE), unsent_file_frame_flag(FALSE),
	   tx_frame_index(0), file_frame_index(0), CMDflag(FALSE), tx_data(Tnc::tx_frame + 4),
	   file_data(Tnc::file_frame + 4), CMDchar(0), de_flag(0), find_call_index(0),
           modechange_count(0), connected_status_count(0), can_send_packet_file_flag(FALSE),
           can_send_tor_file_flag(FALSE) {
    rx_frame.command = rx_frame.port = rx_frame.stream = rx_frame.data[0] =
      rx_frame.data_count = rx_frame.FEND_count = rx_frame.bytes_afterFEND = 0;
    rx_frame.got_FESC = FALSE;
    *callbuffer = 0;
    kam_freebytes.packet = KEYBOARD_PRIORITY;
    kam_freebytes.tor = KEYBOARD_PRIORITY;
    int port;
    int stream;
    for (port = 0; port < 2; port++) {
        for (stream = 0; stream < MAXUSERS; stream++) {
	    tnc_func.stream_status[stream][port] = Tnc_func::disconnected;
	    *(tnc_func.hisCall[stream][port]) = 0;
	}
    }
    *(tnc_func.selCall) = 0;
    tnc_func.hfmode = Tnc_func::packet;
    tnc_func.hisCall_lock = Tnc_func::off;
    tnc_func.speed_lock = Tnc_func::off;
    tnc_func.tx_status = Tnc_func::rx;
    tnc_func.have_sent_lock = Tnc_func::no;
    tnc_func.active_hf_stream = tnc_func.active_vhf_stream = tnc_func.active_port = 0;
    tnc_func.capturefile_flag = FALSE;
    tnc_func.capture_stream.port = tnc_func.capture_stream.stream = 0;
    tnc_func.got_disconnect_flag = FALSE;
    tnc_func.disconnect_stream.port = tnc_func.disconnect_stream.stream = 0;
    tnc_func.paclen.vhf_paclen = vhf_paclen;
    tnc_func.paclen.hf_paclen = hf_paclen;
    tnc_func.no_rx_cw_flag = no_rx_cw_flag;
    tnc_func.rx_bell_flag = rx_bell_flag;
    tnc_func.slow_write_flag = slow_write_flag;
    tnc_func.sending_cw_flag = FALSE;
    if ((tnc_func.download_list_ptr = new DownloadList) == 0) {
        cerr << "Memory allocation error in Tnc::Tnc()" << endl;
	exit(MEM_ERROR);
    }
    if ((tnc_func.print_list_ptr = new PrintList) == 0) {
        cerr << "Memory allocation error in Tnc::Tnc()" << endl;
	exit(MEM_ERROR);
    }
}

Tnc::~Tnc(void) {
    delete tnc_func.download_list_ptr;
    delete tnc_func.print_list_ptr;
}

void Tnc::process_received_bytes(void) {

    char modemchar;
    int index;
    int result;
    while ((result = receive_pipe.read(combuffer, PIPE_BUF)) > 0) {
        for (index = 0; index < result; index++) {
	    modemchar = combuffer[index];
	    if ((uchar)modemchar == FEND) { // we are either at the beginning or end of a frame
	        if (!rx_frame.FEND_count || !rx_frame.bytes_afterFEND) { // we are at the beginning
		    rx_frame.FEND_count = 1;
		    rx_frame.bytes_afterFEND = 0;
		    rx_frame.data_count = 0;
		}
		else {                      // we are at the end
		    rx_frame_complete = TRUE;
		    rx_frame.FEND_count = 0;
		}
	    }

	    else if (!rx_frame.bytes_afterFEND && rx_frame.FEND_count) { // command byte
	        rx_frame.command = modemchar;
		// if query reply frame, pretend we have received port and stream bytes
		if (modemchar == '?') {
		    rx_frame.bytes_afterFEND = 3; 
		    info_requested_flag = 0;
		}
		else rx_frame.bytes_afterFEND++;
	    }

	    else if (rx_frame.bytes_afterFEND == 1) {   // port byte
	        rx_frame.port = modemchar;
		rx_frame.bytes_afterFEND++;
	    }
    
	    else if (rx_frame.bytes_afterFEND == 2) {   // stream byte
	        rx_frame.stream = modemchar;
		rx_frame.bytes_afterFEND++;
	    }

	    else if (rx_frame.FEND_count) {             // data byte
	        if ((uchar)modemchar == FESC) rx_frame.got_FESC = TRUE;
		else if (rx_frame.got_FESC && (uchar)modemchar == TFEND) {
		    rx_frame.data[rx_frame.data_count++] = FEND;
		    rx_frame.got_FESC = FALSE;
		}
		else if (rx_frame.got_FESC && (uchar)modemchar == TFESC) {
		    rx_frame.data[rx_frame.data_count++] = FESC;
		    rx_frame.got_FESC = FALSE;
		}
		else {
		    rx_frame.data[rx_frame.data_count++] = modemchar;
		    rx_frame.got_FESC = FALSE;
		}
	    }
	    
	    if (rx_frame_complete ) {
	        rx_frame_complete = FALSE;
		if (rx_frame.data_count) {
		    process_rx_frame();
		}
	    }
	}
    }
}

void Tnc::send_to_tnc(int guard_flush) {

// if guard_flush is TRUE, then the requirement to buffer a number of characters
// equal to GUARD_QUEUE will be ignored

// send anything in the buffer or unsent frame
// test with 4 and not 3 in the following formula to allow for an added '\n' in HF
// Tor modes when sending text
    if (get_active_freebytes() > 4
	&& (tr_buffer.view_letter() != -1 || unsent_frame_flag)) {
        int letter = 0;
        do {
	    if (!unsent_frame_flag) {  // make a new frame if none waiting to be sent
	        tx_frame[0] = FEND;
		tx_frame[1] = 'D';
		tx_frame[2] = tnc_func.active_port ? '2' : '1';
		
		char temp;
		if ((tnc_func.active_port && tnc_func.hfmode != Tnc_func::packet)
		      || tnc_func.stream_status[tnc_func.active_stream()]
		      [tnc_func.active_port] == Tnc_func::disconnected) {
		    temp = '0';
		}
		else temp = ((tnc_func.active_stream() | 0x40) + 1);
		tx_frame[3] = temp;   // now add the stream byte

		int freebytes = get_active_freebytes();

		for (tx_frame_index = 0; CMDflag == FALSE
		       && tx_frame_index < tnc_func.active_paclen()
		       && tx_frame_index < freebytes - 4
		       && (guard_flush
			   || !tnc_func.active_port
			   || tnc_func.hfmode == Tnc_func::packet
			   || prog_func.send_mode != Prog_func::guard
			   || tr_buffer.letters_used() > GUARD_QUEUE)
		       && (letter = tr_buffer.extract_letter()) != -1;) { // this test must come last to prevent
		                                                          // incorrect abstraction from tr_buffer
		    if ((uchar)letter == FEND) {
		        tx_data[tx_frame_index++] = FESC;
			tx_data[tx_frame_index++] = TFEND;
		    }
		    else if ((uchar)letter == FESC) {
		        tx_data[tx_frame_index++] = FESC;
			tx_data[tx_frame_index++] = TFESC;
		    }

		    else if (letter == '\n') {
		        tx_data[tx_frame_index++] = '\r';
			guard_flush = FALSE;
			if (tnc_func.active_port
			    && tnc_func.hfmode != Tnc_func::packet
			    && tnc_func.hfmode != Tnc_func::cw) {
			    tx_data[tx_frame_index++] = '\n';
			}
		    }
    // check to see if there is a special hf command in tr_buffer (the only command
    // normally placed in the buffer is the receive command 'E': others are placed
    // directly in send_pipe)
		    else if (letter == CMDinbuffer) {
		        CMDchar = tr_buffer.extract_letter();    // now get the command char itself
			tr_buffer.extract_letter();              // extract the second CMDinbuffer token
			CMDflag = TRUE;
		    }

#if CHAR_SET==LATIN_1
		    else if ((letter = Latin1_to_cp437(letter)) != 0) tx_data[tx_frame_index++] = letter;
#else
		    else tx_data[tx_frame_index++] = letter;
#endif
		}
		tx_frame[tx_frame_index + 4] = FEND;
	    }
	    else letter = 0;  // put dummy value in letter if we have an unsent frame

	    if (tx_frame_index) {
	        int result = send_pipe.write(tx_frame, tx_frame_index + 5);
		if (result < 1) unsent_frame_flag = TRUE;
		else {
		    unsent_frame_flag = FALSE;
		    if (tnc_func.active_port && tnc_func.hfmode != Tnc_func::packet) {
		        kam_freebytes.tor -= tx_frame_index + 3;
			if (kam_freebytes.tor < 0) kam_freebytes.tor = 0;
		    }
		    else {
		        kam_freebytes.packet -= tx_frame_index + 3;
			if (kam_freebytes.packet < 0) kam_freebytes.packet = 0;
		    }
		}
	    }
	} while (unsent_frame_flag == FALSE
		   && CMDflag == FALSE
		   && get_active_freebytes() > 4
		   && (guard_flush
		       || !tnc_func.active_port
		       || tnc_func.hfmode == Tnc_func::packet
		       || prog_func.send_mode != Prog_func::guard
		       || tr_buffer.letters_used() > GUARD_QUEUE)
		   && letter != -1);
    }
    if (CMDflag && !unsent_frame_flag) {          // now send any command
        CMDflag = FALSE;
	send_specialcommand(CMDchar);
	usleep(100000);
    }
}

void Tnc::send_file(FileBuffer* file_buffer_ptr, const int max_buffer_count, const int this_buffer) {

// max_buffer_count and this_buffer are intended to enable a number of files
// to be sent without one hogging the Kam memory

// 'max_buffer_count' should provide the total number of files being sent at any
// one time and 'this_buffer' should send the file number in question being sent
// in this iteration throuh Tnc::send_file(), beginning at 0 and ending at
// (max_buffer_count - 1)

// we ensure in the 'while' test in the do-while loop below that only one packet is sent if
// the free bytes in the Kam buffer are less than (max_paclen * buffer_count), to stop
// Kam memory hogging

// set up some necessary constants
    const int max_paclen = tnc_func.paclen.vhf_paclen > tnc_func.paclen.hf_paclen ?
                                tnc_func.paclen.vhf_paclen : tnc_func.paclen.hf_paclen;
    const int required_packet_bytes = 3 + KEYBOARD_PRIORITY + max_paclen + 
	                                           (max_paclen * max_buffer_count);
    const int required_tor_bytes = 3 + KEYBOARD_PRIORITY + (2 * tnc_func.paclen.hf_paclen);

// if this is the first file to be sent in this iteration, make any write 
// of packet files atomic for them all and check if we can send a file
// in Tor mode

    if (!this_buffer) { 
        if (get_packet_freebytes() > required_packet_bytes) {
	    can_send_packet_file_flag = TRUE;
	}
	else can_send_packet_file_flag = FALSE;
	if (tnc_func.hfmode != Tnc_func::packet
	    && get_tor_freebytes() > required_tor_bytes) {
	    can_send_tor_file_flag = TRUE;
	}
	else can_send_tor_file_flag = FALSE;
    }

    int count = 0;  // we want to limit the number of times we loop through the while/do loop
                    // so we get back to the keyboard without too long a wait when loading
                    // a big file

    if (((can_send_packet_file_flag && (!file_buffer_ptr->get_port()
				       || tnc_func.hfmode == Tnc_func::packet)) 
	|| (can_send_tor_file_flag && file_buffer_ptr->get_port()
				       && tnc_func.hfmode != Tnc_func::packet))
	 && (file_buffer_ptr->is_used() || unsent_file_frame_flag)) {
        int letter = 0;
        do {
	    if (!unsent_file_frame_flag) {  // make a new frame if none waiting to be sent
	        file_frame[0] = FEND;
		file_frame[1] = 'D';
		file_frame[2] = file_buffer_ptr->get_port() ? '2' : '1';

		char temp;
		if ((file_buffer_ptr->get_port() && tnc_func.hfmode != Tnc_func::packet)
		      || tnc_func.stream_status[file_buffer_ptr->get_stream()]
		      [file_buffer_ptr->get_port()] == Tnc_func::disconnected) {
		    temp = '0';
		}
		else temp = ((file_buffer_ptr->get_stream() | 0x40) + 1);
		file_frame[3] = temp;   // now add the stream byte

		int send_paclen;
		if (!file_buffer_ptr->get_port()) send_paclen = tnc_func.paclen.vhf_paclen;
		else if (tnc_func.hfmode == Tnc_func::packet) send_paclen = tnc_func.paclen.hf_paclen;
		else send_paclen = tnc_func.paclen.hf_paclen - 1; // leave room for final '\n' after '\r' in frame

		for (file_frame_index = 0; CMDflag == FALSE
		       && file_frame_index < send_paclen
		       && file_buffer_ptr->is_used();) {

		    letter = file_buffer_ptr->front(Qqueue_enum::remove);   
		    if ((uchar)letter == FEND) {
		        file_data[file_frame_index++] = FESC;
			file_data[file_frame_index++] = TFEND;
		    }
		    else if ((uchar)letter == FESC) {
		        file_data[file_frame_index++] = FESC;
			file_data[file_frame_index++] = TFESC;
		    }
    // if we are in binary or 7-plus mode we want no further translations
    // other than LF -> CR translation if in 7-plus mode
		    else if (file_buffer_ptr->get_buffer_mode() != FileBuffer::text) {
		        if (letter == '\n' && file_buffer_ptr->get_buffer_mode() == FileBuffer::s_plus) {
			    file_data[file_frame_index++] = '\r';
			}
		        else file_data[file_frame_index++] = letter;
		    }

		    else if (letter == '\n') {
		        file_data[file_frame_index++] = '\r';
			if (file_buffer_ptr->get_port()
			    && tnc_func.hfmode != Tnc_func::packet
			    && tnc_func.hfmode != Tnc_func::cw) {
			    file_data[file_frame_index++] = '\n';
			}
		    }


    // check to see if there is a special hf command in cq_buffer (the only command
    // normally placed in the buffer is the receive command 'E': others are placed
    // directly in send_pipe)
		    else if (letter == CMDinbuffer) {
		        CMDchar = file_buffer_ptr->front(Qqueue_enum::remove);    // now get the command char itself
			file_buffer_ptr->front(Qqueue_enum::remove);              // extract the second CMDinbuffer token
			CMDflag = TRUE;
		    }

#if CHAR_SET==LATIN_1
		    else if ((letter = Latin1_to_cp437(letter)) != 0) file_data[file_frame_index++] = letter;
#else
		    else file_data[file_frame_index++] = letter;
#endif
		}
		file_frame[file_frame_index + 4] = FEND;
	    }
	    
	    if (file_frame_index) {
	        int result = send_pipe.write(file_frame, file_frame_index + 5);
		if (result < 1) unsent_file_frame_flag = TRUE;
		else {
		    unsent_file_frame_flag = FALSE;
		    if (file_buffer_ptr->get_port() && tnc_func.hfmode != Tnc_func::packet) {
		        kam_freebytes.tor -= file_frame_index + 3;
			if (kam_freebytes.tor < 0) kam_freebytes.tor = 0;
		    }
		    else {
		        kam_freebytes.packet -= file_frame_index + 3;
			if (kam_freebytes.packet < 0) kam_freebytes.packet = 0;
		    }
		}
	    }
	    count++;
	} while (unsent_file_frame_flag == FALSE
		   && count < FILESEND_COUNT
		   && CMDflag == FALSE
		   && file_buffer_ptr->is_used()
		   && (file_buffer_ptr->get_port() && tnc_func.hfmode != Tnc_func::packet ?
		        get_tor_freebytes() > required_tor_bytes : 
	                get_packet_freebytes() > required_packet_bytes));
    }
    file_buffer_ptr->show_bytes();
    if (count >= FILESEND_COUNT) usleep(200000); // the Kam needs some recovery time to work out how
                                                 // much buffer it has got left
    if (CMDflag && !unsent_file_frame_flag) {    // now send any command
        CMDflag = FALSE;
	send_specialcommand(CMDchar);
	usleep(100000);
    }
}

void Tnc::process_rx_frame(void) {

    if (rx_frame.command == '?') {
        char mode = rx_frame.data[1];                   // "M" byte
        if (mode == 'H' || mode == 'J') {
	    tr_buffer.set_can_transmit_delete(TRUE);
	    tnc_func.tor_connected_mode = Tnc_func::pactor_gtor;
	}
	else {
	    tr_buffer.set_can_transmit_delete(FALSE);
	    tnc_func.tor_connected_mode = Tnc_func::not_pactor_gtor;
	}

	switch (mode) {
	case 'A':
	    if (tnc_func.hfmode != Tnc_func::packet) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    tnc_func.hfmode = Tnc_func::packet;
		    mainscreen_p->display_mode();
		    mainscreen_p->show_packetmenu_streams();
		    modechange_count = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'B':
	    if (tnc_func.hfmode != Tnc_func::rtty) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::rtty;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'C':
	    if (tnc_func.hfmode != Tnc_func::ascii) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::ascii;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'D':
	    if (tnc_func.hfmode != Tnc_func::amtor && tnc_func.hfmode != Tnc_func::tor) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::amtor;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'E':
	    if (tnc_func.hfmode != Tnc_func::fec) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::fec;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'G':
	    if (tnc_func.hfmode != Tnc_func::lamtor) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::lamtor;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'H':
	    if (tnc_func.hfmode != Tnc_func::pactor && tnc_func.hfmode != Tnc_func::tor) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::pactor;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'J':
	    if (tnc_func.hfmode != Tnc_func::gtor && tnc_func.hfmode != Tnc_func::tor) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::gtor;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'L':
	    if (tnc_func.hfmode != Tnc_func::cw) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::cw;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'M':
	    if (tnc_func.hfmode != Tnc_func::tor) {
	        modechange_count++;
		// we have a larger latency when receiving the information that we are in Tor mode,
		// to ensure we do not have problems changing from Tor mode to Pactor, Amtor or G-tor
		// (if we change to, say, Pactor, we will get a "<PACTOR STANDBY>" message: latency in
		// the status information sent by the Kam can cause this block to reset hfmode to Tor
		// mode, and the "<PACTOR STANDBY>" message will then cause Tnc::read_state_message()
		// to set us back to Tor mode again, where we will be stuck)
		if (modechange_count > CHANGEMODE_LATENCY + 15) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::tor;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	case 'N':
	    if (tnc_func.hfmode != Tnc_func::gmon) {
	        modechange_count++;
		if (modechange_count > CHANGEMODE_LATENCY) {
		    if (tnc_func.hfmode == Tnc_func::packet) mainscreen_p->show_tormenu_streams();
		    tnc_func.hfmode = Tnc_func::gmon;
		    mainscreen_p->display_mode();
		    modechange_count = 0;
		    tnc_func.active_hf_stream = 0;
		    tnc_func.speed_lock = Tnc_func::off;
		    tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_func.sending_cw_flag = FALSE;
		    if (tnc_func.active_port) {
		        mainscreen_p->display_current_stream();
			mainscreen_p->display_callsign();
			mainscreen_p->display_connected_status();
			mainscreen_p->make_torline();
			mainscreen_p->display_freebytes();
			mainscreen_p->set_call_lock_button();
			mainscreen_p->set_connect_button();
			mainscreen_p->set_disconnect_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_auto_cq_button();
			mainscreen_p->set_speed_lock_button();
			mainscreen_p->set_ident_button();
			mainscreen_p->set_sync_button();
			mainscreen_p->set_abort_button();
			mainscreen_p->set_rx_button();
			mainscreen_p->set_tx_button();
			mainscreen_p->set_cw_speed_button();
			receivewin_p->buffer_flush();
			receivewin_p->raiseWidget(MAXUSERS);
		    }
		}
	    }
	    else modechange_count = 0;
	    break;
	}

	mode = rx_frame.data[2];                        // "S" byte
	if (tnc_func.hfmode != Tnc_func::packet
	    && ((mode != '2' &&  tnc_func.stream_status[0][1] == Tnc_func::connected)
		|| (mode == 2 && tnc_func.stream_status[0][1] == Tnc_func::disconnected))) {
	    connected_status_count++;
	}
	else connected_status_count = 0;
	if (connected_status_count > CHANGE_CONNECTED_STATUS_LATENCY) {
	    if (tnc_func.stream_status[0][1] == Tnc_func::connected) {
	        tnc_func.stream_status[0][1] = Tnc_func::disconnected;
		mainscreen_p->display_connected_status();
		mainscreen_p->set_connect_button();
		mainscreen_p->set_sync_button();
		mainscreen_p->set_auto_cq_button();
		if (tnc_func.hfmode == Tnc_func::tor && !prog_func.sending_autocq) {
		  // go back to TOR mode (the Kam will have reset to the mode of the connecting
		  // station - we don't want this after that link finishes)
	
		    usleep(100000);
		    send_specialcommand(packetCMD);
		    usleep(300000);
		    send_kamcommand(torCMD, '2', '0');
		    usleep(100000);
		}
	    }
	    else {
	        tnc_func.stream_status[0][1] = Tnc_func::connected;
		mainscreen_p->display_connected_status();
		mainscreen_p->set_connect_button();
		mainscreen_p->set_sync_button();
		mainscreen_p->set_auto_cq_button();
	    }
	    connected_status_count = 0;
	}

	mainscreen_p->update_torinfo(rx_frame.data[3]); // "X" byte
	mainscreen_p->update_txinfo(rx_frame.data[4]);  // "Y" byte
	if (rx_frame.data[3] & 32) {                    // we are an ISS
	    tnc_func.have_sent_lock = Tnc_func::no;
	}
	else if (tnc_func.stream_status[0][1] == Tnc_func::connected
	      && tnc_func.tor_connected_mode == Tnc_func::pactor_gtor
	      && tnc_func.speed_lock == Tnc_func::on
	      && tnc_func.have_sent_lock == Tnc_func::no) {
	    tnc_func.have_sent_lock = Tnc_func::yes;
	    send_specialcommand('1');
	}
	if (rx_frame.data[4] & 2) tnc_func.tx_status = Tnc_func::tx;
	else tnc_func.tx_status = Tnc_func::rx;
    }
    else {
        int port;
	int stream;
	int index;
        port = rx_frame.port & 0xf;  // remove upper nibble to convert ASCII to number
	if (port) port--;            // put in format for active_port
    
	if (rx_frame.stream == '0') stream = 0;
	else {
	    stream = rx_frame.stream & 0x1f; // remove top 3 bits to convert to number
	    if (stream) stream--;            // put in format for active_stream()
	}

	char* message;
	if (!(message = new char[rx_frame.data_count + 1])) {
	    cerr << "Memory allocation error in Tnc::process_rx_frame()" << endl;
	    exit(MEM_ERROR);
	}

	if ((unsigned int)stream >= MAXUSERS || (unsigned int)port > 1) {    // check for sanity
	    *rx_frame.data = 0;
	}
    
	else if (rx_frame.command == 'C') {  // buffer contains response to a command
	    int received_status_flag = FALSE;
	    if (check_status_flag) {
		memcpy(message, rx_frame.data, rx_frame.data_count);
		message[rx_frame.data_count] = 0;  // null terminate the string
		if (check_status_flag && strstr(message, "FREE BYTES")) {  // we are checking free bytes in the Kam buffers
		    check_status_flag = 0;
		    parse_status(message);
		    mainscreen_p->display_freebytes();
		    received_status_flag = TRUE;
		}
	    } 

	    if (received_status_flag == FALSE) {
	        char* char_p = message;
	        for (index = 0; index < rx_frame.data_count; index++) {
		    if (rx_frame.data[index] < 127  && rx_frame.data[index] > 31) {
		        *char_p = rx_frame.data[index];
			char_p++;
		    }
		    else if (rx_frame.data[index] == '\r') {
		        *char_p = 0;  // null terminate the string
			receivewin_p->write(message);
			receivewin_p->newline();
		        char_p = message;
		    }
		}
		if (char_p > message) {  // anything left to display?
		    *char_p = 0;  // null terminate the string
		    receivewin_p->write(message);
		}
	    }
	}

	else if (rx_frame.command == 'D') {   // buffer contains data
	    if (!port || !tnc_func.no_rx_cw_flag || tnc_func.hfmode != Tnc_func::cw) {
	        int stream_capture_flag = FALSE;
		if (tnc_func.capturefile_flag &&
		    tnc_func.capture_stream.stream == stream &&
		    tnc_func.capture_stream.port == port) {
		    stream_capture_flag = TRUE;
		}

		DownloadFile* dload_file_ptr = 0;
// first check whether file(s) being download
		if (!tnc_func.download_list_ptr->is_empty()
		    && tnc_func.download_list_ptr->get_download_status(stream, port)
		       == DownloadList::on) {
// we now need to find the download file object in the list object
// which is receiving the file on the active stream and port
		    tnc_func.download_list_ptr->reset(DList_enum::bottom_end);
		    while ((dload_file_ptr = (DownloadFile*)tnc_func.download_list_ptr->inspect(DList_enum::up)) != 0
			   && !(dload_file_ptr->get_stream() == stream
				&& dload_file_ptr->get_port() == port)) {}
		    if (!dload_file_ptr) {
		        receivewin_p->write("\nOops - can't find the download file object to write to\n");
			QApplication::beep();
		    }
		}

		PrintFile* print_file_ptr = 0;
// first check whether there is a print mark set
		if (!tnc_func.print_list_ptr->is_empty()
		    && tnc_func.print_list_ptr->get_print_status(stream, port)
		       == PrintList::on) {
// we now need to find the print file object in the list object
// which is receiving the file on the active stream and port
		    tnc_func.print_list_ptr->reset(DList_enum::bottom_end);
		    while ((print_file_ptr = (PrintFile*)tnc_func.print_list_ptr->inspect(DList_enum::up)) != 0
			   && !(print_file_ptr->get_stream() == stream
				&& print_file_ptr->get_port() == port)) {}
		    if (!print_file_ptr) {
		        receivewin_p->write("\nOops - can't find the print file object to write to\n");
			QApplication::beep();
		    }
		}

	        char* char_p = message;
	        for (index = 0; index < rx_frame.data_count; index++) {
		    if (dload_file_ptr) {
		        if (rx_frame.data[index] == '\r' 
			    && dload_file_ptr->get_download_mode() == DownloadFile::s_plus) {
			    dload_file_ptr->add_char('\n');
			}
			else dload_file_ptr->add_char(rx_frame.data[index]);
		    }

#if CHAR_SET==LATIN_1
		    if ((uchar)rx_frame.data[index] < 255  && (uchar)rx_frame.data[index] > 32 &&
			  (rx_frame.data[index] = cp437_to_Latin1(rx_frame.data[index])) != 0) {
#elif CHAR_SET==CP437
		    if ((uchar)rx_frame.data[index] < 255  && (uchar)rx_frame.data[index] > 32) {
#else
		    if (rx_frame.data[index] < 127  && rx_frame.data[index] > 32) {
#endif
		        *char_p = rx_frame.data[index];
			char_p++;
			if (stream_capture_flag) tnc_func.capturefile.put(rx_frame.data[index]);
			if (print_file_ptr) print_file_ptr->add_char(rx_frame.data[index]);
		    }
		    else if (rx_frame.data[index] == '\r') {
		        *char_p = 0;  // null terminate the string
			if (port) receivewin_p->buffer_flush();
			if (char_p > message) {
			    receivewin_p->write(message, stream, port);
			    char_p = message;
			}
			receivewin_p->newline(stream, port);
			if (stream_capture_flag) tnc_func.capturefile.put('\n');
			if (print_file_ptr) print_file_ptr->add_char('\n');
		    }
		    else if (rx_frame.data[index] == ' ') {
		        if (port) receivewin_p->buffer_flush();
			*char_p = ' ';
			char_p++;
			if (stream_capture_flag) tnc_func.capturefile.put(' ');
			if (print_file_ptr) print_file_ptr->add_char(' ');
		    }
		    else if (rx_frame.data[index] == 8) {
		        if (char_p > message) char_p--;
			else if (tnc_func.slow_write_flag
			    && port
			    && receivewin_p->buffer_letters()) {
			    receivewin_p->buffer_del();
			}
			else receivewin_p->del_letter(stream, port);
			if (stream_capture_flag) {
			    tnc_func.capturefile.put(8);
			    tnc_func.capturefile.put(' ');
			    tnc_func.capturefile.put(8);
			}
			if (print_file_ptr) {
			    print_file_ptr->add_char(8);
			    print_file_ptr->add_char(' ');
			    print_file_ptr->add_char(8);
			}
		    }
		    else if (rx_frame.data[index] == 7 && tnc_func.rx_bell_flag) {
		        QApplication::beep();
		    }
		    if (tnc_func.hisCall_lock == Tnc_func::off && port
			&& tnc_func.hfmode != Tnc_func::packet) {
		        find_call(rx_frame.data[index]);
		    }
		}
		if (char_p > message) {  // anything left to display?
		    *char_p = 0;  // null terminate the string
		    if (tnc_func.slow_write_flag
			  && port
			  && tnc_func.active_port
			  && (tnc_func.hfmode == Tnc_func::rtty
			      || tnc_func.hfmode == Tnc_func::ascii
			      || tnc_func.hfmode == Tnc_func::cw
			      || tnc_func.hfmode == Tnc_func::amtor
			      || tnc_func.hfmode == Tnc_func::lamtor
			      || tnc_func.hfmode == Tnc_func::fec
			      || (tnc_func.hfmode == Tnc_func::gtor
				  && tnc_func.stream_status[0][1] == Tnc_func::disconnected)
				|| (tnc_func.hfmode == Tnc_func::tor
				    && (tnc_func.stream_status[0][1] == Tnc_func::disconnected
					|| tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor)))) {
		        char* temp_p;
		        for (temp_p = message; *temp_p; temp_p++) receivewin_p->buffer_add(*temp_p);
		    }
		    else receivewin_p->write(message, stream, port);
		}
		if (dload_file_ptr) dload_file_ptr->show_bytes();
	    }
	}
    
	else if (rx_frame.command == 'M') {// buffer contains unconnected packet data
	    int temp;
	    if (port) temp = tnc_func.active_hf_stream;
	    else temp = tnc_func.active_vhf_stream;
	    if (tnc_func.stream_status[temp][port] == Tnc_func::disconnected) {
	        char* char_p = message;
	        for (index = 0; index < rx_frame.data_count; index++) {
#if CHAR_SET==LATIN_1
		    if ((uchar)rx_frame.data[index] < 255  && (uchar)rx_frame.data[index] > 31 &&
			  (rx_frame.data[index] = cp437_to_Latin1(rx_frame.data[index])) != 0) {
#elif CHAR_SET==CP437
		    if ((uchar)rx_frame.data[index] < 255  && (uchar)rx_frame.data[index] > 31) {
#else
		    if (rx_frame.data[index] < 127  && rx_frame.data[index] > 31) {
#endif
		        *char_p = rx_frame.data[index];
			char_p++;
		    }
		    else if (rx_frame.data[index] == '\r') {
		        *char_p = 0;  // null terminate the string
			receivewin_p->write(message, temp, port);
			receivewin_p->newline(temp, port);
		        char_p = message;
		    }
		    else if (rx_frame.data[index] == 8) {
		        if (char_p > message) char_p--;
			else receivewin_p->del_letter(stream, port);
		    }
		    else if (rx_frame.data[index] == 7 && tnc_func.rx_bell_flag) {
		        QApplication::beep();
		    }
		}
		if (char_p > message) {  // anything left to display?
		    *char_p = 0;  // null terminate the string
		    receivewin_p->write(message, temp, port);
		}
		// try to beautify the display of unconnected packets
		if (index && rx_frame.data[index - 1] == ':') {
		    receivewin_p->newline(temp, port);
		}
	    }
	}

	else if (rx_frame.command == 'E') {  // my transmitted data in HF mode
	    char* char_p = message;
	    for (index = 0; index < rx_frame.data_count; index++) {
#if CHAR_SET==LATIN_1
	        if ((uchar)rx_frame.data[index] < 255  && (uchar)rx_frame.data[index] > 32 &&
		    (rx_frame.data[index] = cp437_to_Latin1(rx_frame.data[index])) != 0) {
#elif CHAR_SET==CP437
	        if ((uchar)rx_frame.data[index] < 255  && (uchar)rx_frame.data[index] > 32) {
#else
	        if (rx_frame.data[index] < 127  && rx_frame.data[index] > 32) {
#endif
		    *char_p = rx_frame.data[index];
		    char_p++;
		}
		else if (rx_frame.data[index] == '\r') {
		    *char_p = 0;  // null terminate the string
		    receivewin_p->buffer_flush();
		    if (char_p > message) {
		        receivewin_p->write(message, 0, 1);
			char_p = message;
		    }
		    receivewin_p->newline(0, 1);
		}
		else if (rx_frame.data[index] == ' ') {
		    receivewin_p->buffer_flush();
		    *char_p = ' ';
		    char_p++;
		}
		else if (rx_frame.data[index] == 8) {
		    if (char_p > message) char_p--;
		    else if (tnc_func.slow_write_flag
			     && receivewin_p->buffer_letters()) {
		        receivewin_p->buffer_del();
		    }
		    else receivewin_p->del_letter(0, 1);
		}
	    }
	    if (char_p > message) {  // anything left to display?
	        *char_p = 0;  // null terminate the string
		if (tnc_func.slow_write_flag
		    && tnc_func.active_port
		    && (tnc_func.hfmode == Tnc_func::rtty
			|| tnc_func.hfmode == Tnc_func::ascii
			|| tnc_func.hfmode == Tnc_func::cw
			|| tnc_func.hfmode == Tnc_func::amtor
			|| tnc_func.hfmode == Tnc_func::lamtor
			|| tnc_func.hfmode == Tnc_func::fec
			|| (tnc_func.hfmode == Tnc_func::gtor
			    && tnc_func.stream_status[0][1] == Tnc_func::disconnected)
			|| (tnc_func.hfmode == Tnc_func::tor
			    && (tnc_func.stream_status[0][1] == Tnc_func::disconnected
				|| tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor)))) {
		    char* temp_p;
		    for (temp_p = message; *temp_p; temp_p++) receivewin_p->buffer_add(*temp_p);
		}
		else receivewin_p->write(message, 0, 1);
	    }
	}
	else if (rx_frame.command == 'S') {   // buffer contains link state information
	    char* char_p = message;
	    for (index = 0; index < rx_frame.data_count; index++) {
	        if (rx_frame.data[index] < 127  && rx_frame.data[index] > 31) {
		    *char_p = rx_frame.data[index];
		    char_p++;
		}
		else if (rx_frame.data[index] == '\r') {
		    *char_p = '\n';
		    char_p++;
		}
	    }
	    *char_p = 0;  // null terminate the string
	    if (port) receivewin_p->buffer_flush(); // clear out any data before showing state message
	    receivewin_p->write(message, stream, port);
	    read_state_message(message, stream, port);
	}
	delete[] message;
    }
}

void Tnc::read_state_message(char* message, int stream, int port) {
    char* message_ptr = 0;
    char* temp_ptr = 0;
    if (strstr(message, "*** DISCONNECTED")) {
        if (tnc_func.download_list_ptr->get_download_status(stream, port) == DownloadList::on) {
// we now need to find the download file object receiving the
// file on the stream and port receiving the disconnect
	    tnc_func.download_list_ptr->reset(DList_enum::bottom_end);
	    DownloadFile* dload_file_ptr;
	    while ((dload_file_ptr = (DownloadFile*)tnc_func.download_list_ptr->inspect(DList_enum::up)) != 0
		   && !(dload_file_ptr->get_stream() == stream
			&& dload_file_ptr->get_port() == port)) {}
	    if (dload_file_ptr) {
	        tnc_func.download_list_ptr->extract();
		delete dload_file_ptr; // this will also delete the DownloadDialog object
	    }
	    else {
	        receivewin_p->write("\nOops - can't find the download file object to delete\n");
		QApplication::beep();
	    }
	}
        tnc_func.stream_status[stream][port] = Tnc_func::disconnected;
	mainscreen_p->display_connected_status();
	mainscreen_p->set_connect_button();
	mainscreen_p->set_sync_button();
	mainscreen_p->set_auto_cq_button();
	tnc_func.got_disconnect_flag = TRUE;
	tnc_func.disconnect_stream.port = port;
	tnc_func.disconnect_stream.stream = stream;
	mainscreen_p->beep(3);
    }

    else if ((message_ptr = strstr(message, "*** CONNECTED to ")) != 0) {
        message_ptr += 17;                  // update the callsign buffer
	// remove any trailing spaces (eg if CSTAMP is ON)
	for (temp_ptr = message_ptr; *temp_ptr && *temp_ptr != ' '; temp_ptr++) {}
	if (*temp_ptr == ' ') *temp_ptr = 0;
	
	strncpy(tnc_func.hisCall[stream][port], message_ptr, hisCall_SIZE);
	tnc_func.hisCall[stream][port][hisCall_SIZE] = 0;  // make sure the string is null terminated
	if (tnc_func.active_port && tnc_func.hfmode != Tnc_func::packet) {
	    tnc_func.hisCall_lock = Tnc_func::on;
	}
	mainscreen_p->display_callsign();
        tnc_func.stream_status[stream][port] = Tnc_func::connected;
	mainscreen_p->display_connected_status();
	mainscreen_p->set_connect_button();
	mainscreen_p->set_sync_button();
	mainscreen_p->set_auto_cq_button();
	mainscreen_p->beep(5);
    }

    else if ((message_ptr = strstr(message, "<LINKED TO ")) != 0
	       && (temp_ptr = strchr(message, '>')) != 0) {
 // put the call-sign in the call buffer (for a non-Amtor link) in case we didn't initiate it
        if (tnc_func.hfmode != Tnc_func::amtor) {
	    *temp_ptr = 0;    // terminate the string at the ">"
	    message_ptr += 11;
	    strncpy(tnc_func.hisCall[0][1], message_ptr, hisCall_SIZE);// Tor mode is always stream 0
	    tnc_func.hisCall[0][1][hisCall_SIZE] = 0;  // make sure the string is null terminated
	    tnc_func.hisCall_lock = Tnc_func::on;
	    mainscreen_p->display_callsign();
	    make_selcall(message_ptr, tnc_func.selCall);
	}

        tnc_func.stream_status[0][1] = Tnc_func::connected;  
	mainscreen_p->display_connected_status();
	mainscreen_p->set_connect_button();
	mainscreen_p->set_sync_button();
	mainscreen_p->set_auto_cq_button();
	mainscreen_p->beep(5);
    }
    
    else if (strstr(message, "<LINKED>")) { // Amtor link initiated by other station
        *(tnc_func.selCall) = 0; // we don't know the selcall (signal this)
	*(tnc_func.hisCall[0][1]) = 0; // or the callsign of the other station yet
	tnc_func.hisCall_lock = Tnc_func::off; // we can now pick the call-sign up
	mainscreen_p->display_callsign(); // blank the call-sign displayed

        tnc_func.stream_status[0][1] = Tnc_func::connected;  // Tor mode is always stream 0
	mainscreen_p->display_connected_status();
	mainscreen_p->set_connect_button();
	mainscreen_p->set_sync_button();
	mainscreen_p->set_auto_cq_button();
	mainscreen_p->beep(5);
    }

    else if (strstr(message, "TOR STANDBY>") || strstr(message, "<LINK FAILED")) {      // in Pactor, Amtor or Gtor
// Tor modes are always stream 0, port 1
        if (tnc_func.download_list_ptr->get_download_status(0, 1) == DownloadList::on) {
// we now need to find the download file object receiving the
// file on stream 0 and port 1
	    tnc_func.download_list_ptr->reset(DList_enum::bottom_end);
	    DownloadFile* dload_file_ptr;
	    while ((dload_file_ptr = (DownloadFile*)tnc_func.download_list_ptr->inspect(DList_enum::up)) != 0
		   && !dload_file_ptr->get_port()) {} // Tor modes are always the only stream on port 1
	    if (dload_file_ptr) {
	        tnc_func.download_list_ptr->extract();
		delete dload_file_ptr; // this will also delete the DownloadDialog object
	    }
	    else {
	        receivewin_p->write("\nOops - can't find the download file object to delete\n");
		QApplication::beep();
	    }
	}
        tnc_func.stream_status[0][1] = Tnc_func::disconnected;  // Tor modes are always stream 0
	mainscreen_p->display_connected_status();
	mainscreen_p->set_connect_button();
	mainscreen_p->set_sync_button();
	mainscreen_p->set_auto_cq_button();
	if (tnc_func.hfmode == Tnc_func::tor && !prog_func.sending_autocq) {
	    // go back to TOR mode (the Kam will have reset to the mode of the connecting
	    // station - we don't want this after that link finishes)
	
	    usleep(100000);
	    send_specialcommand(packetCMD);
	    usleep(300000);
	    send_kamcommand(torCMD, '2', '0');
	    usleep(100000);
	}
    }
    else if (strstr(message, "TOR_STANDBY>")             // in Kam TOR mode
	       && tnc_func.stream_status[0][1] == Tnc_func::connected) {
        tnc_func.stream_status[0][1] = Tnc_func::disconnected;  // Tor modes are always stream 0
	mainscreen_p->display_connected_status();
	mainscreen_p->set_connect_button();
	mainscreen_p->set_sync_button();
	mainscreen_p->set_auto_cq_button();
    }
}

void Tnc::parse_status(char* message) {
    char* temp_ptr = strchr(message, '\r'); // terminate the first line with a null
    if (temp_ptr) *temp_ptr = 0;
    else temp_ptr = message + strlen(message);
// if hfmode is Tor, we need to find the beginning of the second free byte number string
    if (tnc_func.hfmode != Tnc_func::packet) {
        temp_ptr = strchr(message, '/');
	if (temp_ptr) {
	    *temp_ptr = 0; // null terminate the first free byte number string
	    temp_ptr++;    // go to start of second free byte number string
	    kam_freebytes.tor = atoi(temp_ptr);
	}
    }
// now find the beginning of the first free byte number string
    if (temp_ptr) {
        for(; *temp_ptr != ' ' && temp_ptr > message; temp_ptr--);
	if (temp_ptr > message) { // check for sanity
	    temp_ptr++;    // go to start of first free byte number string
	    kam_freebytes.packet = atoi(temp_ptr);
	}
    }
}

void Tnc::find_freebytes(void) {
  // check_status_flag will prevent two concurrent requests
  // for a status update, but will only block 3 times
  // to prevent locking
    if (check_status_flag < 1 || !kam_freebytes.packet || !kam_freebytes.tor) {
        check_status_flag = 3;
	send_kamcommand(statusCMD, '2', '0'); // if we send the STATUS command on Port 2, Stream 0
                                              // (HF Tor stream) we minimise the amount of
                                              // extraneous information received
    }
    else check_status_flag--;
}

void Tnc::get_info(void) {
  // info_requested_flag will prevent two concurrent requests
  // for a tor info update (that may be what has brought us here),
  // but will only block 20 times to prevent locking
    if (info_requested_flag < 1) {
        info_requested_flag = 20;
	send_specialcommand('?');
    }
    else info_requested_flag--;
}

void Tnc::find_call(char modemchar) {
    char* message_ptr = 0;

    if (de_flag == 4) {
        if (find_call_index < CALLBUFFER_SIZE && modemchar != '\r' && modemchar != ' ') callbuffer[find_call_index++] = modemchar;
	else {
	    callbuffer[find_call_index] = 0;    // null terminate the string in callbuffer[]
	    find_call_index = 0;
	    de_flag = 0;
	    if ((message_ptr = strchr(callbuffer, ' ')) != 0) { // terminate string at first word space
	        *message_ptr = 0;
	    }
	    for(message_ptr = callbuffer; *message_ptr != 0; message_ptr++) {  // convert to upper case
	        *message_ptr = toupper(*message_ptr);
	    }
	    if (is_validcall(callbuffer)) {
	        strncpy(tnc_func.hisCall[0][1], callbuffer, hisCall_SIZE);
 // if an Amtor link is in progress which we did not initiate, lock the callsign
		if (tnc_func.stream_status[0][1] == Tnc_func::connected
		      && !*(tnc_func.selCall)) {
		    tnc_func.hisCall_lock = Tnc_func::on;
		}
		mainscreen_p->display_callsign();
 // now make up a selcall if we are not connected or we did not iniatate an Amtor link in progress
		if (tnc_func.stream_status[0][1] == Tnc_func::disconnected
		      || !*(tnc_func.selCall)) {
		    make_selcall(callbuffer, tnc_func.selCall);
		}
	    }
	}
    }

// look for " de " or " DE "

    if (modemchar == ' ' && de_flag < 3) de_flag = 1;
    else if (de_flag == 1  && (modemchar == 'd' || modemchar == 'D')) de_flag++;
    else if (de_flag == 2  && (modemchar == 'e' || modemchar == 'E')) de_flag++;
    else if (de_flag == 3  && modemchar == ' ') de_flag++;
    else if (de_flag < 4) {
        if (modemchar == '\r') de_flag = 1;
	else  de_flag = 0;
    }
}

int Tnc::is_validcall(const char* callsign) {  // returns TRUE if callsign valid, else returns FALSE
                                               // the contents of callsign must be in upper case
    char* char_ptr;
    if (!(char_ptr = new char[strlen(callsign) + 1])) {
        cerr << "Memory allocation error in int Tnc::is_validcall(char*)" << endl;
	exit(MEM_ERROR);
    }
    strcpy(char_ptr, callsign);
    char* temp_ptr;
    char* temp_ptr2 = 0;
    if ((temp_ptr = strchr(char_ptr, '/')) != 0) {  // split up the string if there is a '/'
        *temp_ptr = 0;
	temp_ptr++;   
	// temp_ptr now points at the right hand half, and char_ptr to the left hand half
	if (strlen(char_ptr) > strlen(temp_ptr)) temp_ptr = char_ptr; 
	// temp_ptr points to longer half now
	  // now end the string at any second '/'
	if ((temp_ptr2 = strchr(temp_ptr, '/')) != 0) *temp_ptr2 = 0;
    }
    else temp_ptr = char_ptr; 
        //  if no '/' in the callsign, temp_ptr points to a string holding callsign

    int invalid_call_flag = FALSE;
    temp_ptr++;  // move to second letter of main part of callsign
       // check whether the part of the call after any second '/' is too long
    if (temp_ptr2 && strlen(temp_ptr2) > 4) invalid_call_flag = TRUE;
    else if (*temp_ptr < '0' || *temp_ptr > '9') {  // second letter not number
        temp_ptr++;
	if (*temp_ptr < '0' || *temp_ptr > '9') {
	  // if second and third letters not numbers then invalid call
	    invalid_call_flag = TRUE;
	}
    }
    else if (*(temp_ptr - 1)  >= '0' && *(temp_ptr - 1) <= '9') { 
         // first and second letter is number - invalid call
        invalid_call_flag = TRUE;
    }
    else if (*(temp_ptr + 1)  >= '0' && *(temp_ptr + 1) <= '9') temp_ptr++;
         // second and third letter number - go to last number

    if (!invalid_call_flag) {
        temp_ptr++; // temp_ptr now points to the letter after the last or only number
        int num_letters = strlen(temp_ptr);
	if (num_letters  > 3 || !num_letters) {
         // if more than 3 letters or no letters after number, invalid call
	    invalid_call_flag = TRUE;
	}
	for (; *temp_ptr != 0; temp_ptr++) {
	    if (*temp_ptr < 'A' || *temp_ptr > 'Z') { // if any further numbers, invalid call
	        invalid_call_flag = TRUE;
	    }
	}
    }
    delete[] char_ptr;
    return (!invalid_call_flag);
}

void Tnc::make_selcall(const char* callsign,  char* selcall) {
  // the callsign should be in upper case
  // the selcall is copied into char* selcall
  
    char* char_ptr = new char[strlen(callsign) + 1];
    if (!char_ptr) {
        cerr << "Memory allocation error in Tnc::make_selcall(const char*, char*)" << endl;
	exit(MEM_ERROR);
    }
    strcpy(char_ptr, callsign);
    char* temp_ptr;
    char* temp_ptr2;
    if ((temp_ptr = strchr(char_ptr, '/')) != 0) {  // split up the string if there is a '/'
        *temp_ptr = 0;
	temp_ptr++;  
	  // temp_ptr now points at the right hand half, and char_ptr to the left hand half
	if (strlen(char_ptr) > strlen(temp_ptr)) temp_ptr = char_ptr;
	  // temp_ptr points to longer half now
	  // now end the string at any second '/'
	if ((temp_ptr2 = strchr(temp_ptr, '/')) != 0) *temp_ptr2 = 0;
    }
    else temp_ptr = char_ptr;
          //  if no '/' in the callsign, temp_ptr points to a string holding callsign
    
    for (temp_ptr2 = temp_ptr; *temp_ptr2 != 0; temp_ptr2++) {}; //find end of string
    temp_ptr2--;
    selcall[4] = 0;
    int lettercount = 3;
    while (lettercount > 0 && temp_ptr2 >= temp_ptr) {
        if (*temp_ptr2 >= 'A' && *temp_ptr2 <= 'Z') selcall[lettercount--] = *temp_ptr2;
	temp_ptr2--;
    }
          // find the first letter of the callsign (but stop at end of string)
    while (!(*temp_ptr == 0 || (*temp_ptr >= 'A' && *temp_ptr <= 'Z'))) temp_ptr++;
          // now fill in rest of selcall
    for(; lettercount >= 0; lettercount--) selcall[lettercount] = *temp_ptr;
    delete[] char_ptr;
}

uchar Tnc::cp437_to_Latin1(uchar letter) {
    if (letter > 127) {
        switch(letter) {
	    case 0x80: letter = 0xc7; break;
	    case 0x81: letter = 0xfc; break;
	    case 0x82: letter = 0xe9; break;
	    case 0x83: letter = 0xe2; break;
	    case 0x84: letter = 0xe4; break;
	    case 0x85: letter = 0xe0; break;
	    case 0x86: letter = 0xe5; break;
	    case 0x87: letter = 0xe7; break;
	    case 0x88: letter = 0xea; break;
	    case 0x89: letter = 0xeb; break;
	    case 0x8a: letter = 0xe8; break;
	    case 0x8b: letter = 0xef; break;
	    case 0x8c: letter = 0xee; break;
	    case 0x8d: letter = 0xec; break;
	    case 0x8e: letter = 0xc4; break;
	    case 0x8f: letter = 0xc5; break;

	    case 0x90: letter = 0xc9; break;
	    case 0x91: letter = 0xe6; break;
	    case 0x92: letter = 0xc6; break;
	    case 0x93: letter = 0xf4; break;
	    case 0x94: letter = 0xf6; break;
	    case 0x95: letter = 0xf2; break;
	    case 0x96: letter = 0xfb; break;
	    case 0x97: letter = 0xf9; break;
	    case 0x98: letter = 0xff; break;
	    case 0x99: letter = 0xd6; break;
	    case 0x9a: letter = 0xdc; break;
	    case 0x9b: letter = 0xa2; break;
	    case 0x9c: letter = 0xa3; break;
	    case 0x9d: letter = 0xa5; break;

	    case 0xa0: letter = 0xe1; break;
	    case 0xa1: letter = 0xed; break;
	    case 0xa2: letter = 0xf3; break;
	    case 0xa3: letter = 0xfa; break;
	    case 0xa4: letter = 0xf1; break;
	    case 0xa5: letter = 0xd1; break;
	    case 0xa8: letter = 0xbf; break;
	    case 0xab: letter = 0xbd; break;
	    case 0xac: letter = 0xbc; break;
	    case 0xad: letter = 0xa1; break;
	    case 0xae: letter = 0xab; break;
	    case 0xaf: letter = 0xbb; break;

	    case 0xe1: letter = 0xdf; break;
	    case 0xe6: letter = 0xb5; break;

	    case 0xf1: letter = 0xb1; break;
	    case 0xf6: letter = 0xf7; break;
	    case 0xf8: letter = 0xb0; break;
	    case 0xf9: case 0xfa: letter = 0xb7; break;
	    case 0xfd: letter = 0xb2; break;

	    default: letter = 0;
	}
    }
    return letter;
}

uchar Tnc::Latin1_to_cp437(uchar letter) {
    if (letter > 127) {
        switch(letter) {
	    case 0xa1: letter = 0xad; break;
	    case 0xa2: letter = 0x9b; break;
	    case 0xa3: letter = 0x9c; break;
	    case 0xa5: letter = 0x9d; break;
	    case 0xab: letter = 0xae; break;

	    case 0xb0: letter = 0xf8; break;
	    case 0xb1: letter = 0xf1; break;
	    case 0xb2: letter = 0xfd; break;
	    case 0xb5: letter = 0xe6; break;
	    case 0xb7: letter = 0xf9; break;
	    case 0xbb: letter = 0xaf; break;
	    case 0xbc: letter = 0xac; break;
	    case 0xbd: letter = 0xab; break;
	    case 0xbf: letter = 0xa8; break;

	    case 0xc4: letter = 0x8e; break;
	    case 0xc5: letter = 0x8f; break;
	    case 0xc6: letter = 0x92; break;
	    case 0xc7: letter = 0x80; break;
	    case 0xc9: letter = 0x90; break;

	    case 0xd1: letter = 0xa5; break;
	    case 0xd6: letter = 0x99; break;
	    case 0xdc: letter = 0x9a; break;
	    case 0xdf: letter = 0xe1; break;

	    case 0xe0: letter = 0x85; break;
	    case 0xe1: letter = 0xa0; break;
	    case 0xe2: letter = 0x83; break;
	    case 0xe4: letter = 0x84; break;
	    case 0xe5: letter = 0x86; break;
	    case 0xe6: letter = 0x91; break;
	    case 0xe7: letter = 0x87; break;
	    case 0xe8: letter = 0x8a; break;
	    case 0xe9: letter = 0x82; break;
	    case 0xea: letter = 0x88; break;
	    case 0xeb: letter = 0x89; break;
	    case 0xec: letter = 0x8d; break;
	    case 0xed: letter = 0xa1; break;
	    case 0xee: letter = 0x8c; break;
	    case 0xef: letter = 0x8b; break;

	    case 0xf1: letter = 0xa4; break;
	    case 0xf2: letter = 0x95; break;
	    case 0xf3: letter = 0xa2; break;
	    case 0xf4: letter = 0x93; break;
	    case 0xf6: letter = 0x94; break;
	    case 0xf7: letter = 0xf6; break;
	    case 0xf9: letter = 0x97; break;
	    case 0xfa: letter = 0xa3; break;
	    case 0xfb: letter = 0x96; break;
	    case 0xfc: letter = 0x81; break;
	    case 0xff: letter = 0x98; break;

	    default: letter = 0;
	}
    }
    return letter;
}

void Tnc::send_kamcommand(const char* text, char ascii_port, char ascii_stream) {
    int result;
    if (ascii_port == -1) ascii_port = tnc_func.active_port ? '2' : '1';
    if (ascii_stream == -1) {
        if (tnc_func.active_port && tnc_func.hfmode != Tnc_func::packet) {
	    ascii_stream = '0';
	}
	else ascii_stream = ((tnc_func.active_stream() | 0x40) + 1);
    }

    char command_string[5];
    command_string[0] = FEND;
    command_string[1] = 'C';
    command_string[2] = ascii_port;
    command_string[3] = ascii_stream;
    command_string[4] = 0;

    do {
        result = send_pipe.write(command_string);
    } while (!result);


    if (*text) {  // text may be null if just the Kam LEDs are to be redirected
        do {
	    result = send_pipe.write(text);
	} while (!result);
    }

    do {
        result = send_pipe.write(FEND);
    } while (!result);
    usleep(20000);
}

void Tnc::send_specialcommand(char letter) {
    int result;
    do {
        result = send_pipe.write(FEND);
    } while (!result);

    do {
        result = send_pipe.write(letter);
    } while (!result);

    do {
        result = send_pipe.write(FEND);
    } while (!result);
    usleep(20000);
}

void Tnc_setup::process_received_bytes(void) {
    int index;
    int result;
    while ((result = receive_pipe.read(combuffer, PIPE_BUF - 1)) > 0) {
        for (index = 0; index < result; index++) {
	    if (combuffer[index] == '\r') combuffer[index] = '\n';
	}
	combuffer[index] = 0;
	receivewin_p->write(combuffer, 0, 0);
    }
}


void Tnc_setup::send_to_tnc(int) {

  // we don't need to use the value of the parameter passed to the function
  // because it is a dummy in Tnc_setup

    int letter;
    if ((letter = tr_buffer.extract_letter()) != -1) {
        if (letter == '\n') letter = '\r';
	send_pipe.write(letter);
    }
}

