
#ifndef NETWORK_H
#define NETWORK_H

#include <time.h>

#include "ctypes.h"
#include "csock.h"
#include "cstring.h"

struct call_q {
	call_t call;
	struct call_q *next;
};

struct node_t {
	call_t call;		/* Callsign */
	struct link_t *via;	/* Via which link */
	time_t since;		/* Linked since */
	int version;		/* PC version number */
	int hops;		/* How many hops away (not accurate) */
	int hops_ok;		/* Is the hops value OK */
	int here;		/* Here status for the node */
	int users_known;	/* User list known */
	int users_ok;		/* Is the user list valid */
	int users;		/* How many users on the node (from PC50) */
	long rtt;		/* Round-trip time (sec) */
	int rtt_ok;		/* Round-trip time measured? */
	int pinging;		/* Is this node being pinged */
	int locked;		/* Local lock */
	/* Semi-doubly linked list: */
	struct node_t **prevp;	/* Pointer to this entry in the previous one */
	struct node_t *next;	/* Next entry */
};

enum linkmode_t {
	lm_normal,
	lm_listen,
	lm_incoming
};

enum linkstate_t {
	ls_disc,
	ls_init,
	ls_linked, 
	ls_backup,
	ls_held
};

enum dreason_t {
	dr_unknown,		/* disc received */
	dr_notcon,		/* Never connected */
	dr_sysop,		/* Sysop requested */
	dr_remote,		/* Remote node requested */
	dr_loop,		/* Loop detected */
	dr_pingto,		/* Ping timeout */
	dr_held			/* Link held */
};

enum linkproto_t {
	lp_pc,			/* PacketCluster */
	lp_clu
};

struct link_t {
	call_t name;		/* Destination callsign/name */
	call_t call;		/* Destination callsign */
	call_t mycall;		/* Local callsign used */
	call_t link_call;	/* Local AX.25 callsign used */
	
	enum linkproto_t proto; /* Protocol used */
	enum linkmode_t mode;	/* Link mode */
	enum linkstate_t state;	/* Link state */
	time_t since;		/* When the state last changed */
	enum dreason_t dreason; /* Why disconnected */
	int version;		/* Version number */
	
	struct csock_t *sock;	/* Socket */
	struct node_t *node;	/* Neighbour node record */
	
	struct link_t *backuped; /* Which link this is a backup to */
	struct link_t *backup;	 /* Which link is a backup for this one */
	
	/* These ones only for PC links */
	struct charset_t *charset;	/* character conversion table used */
	int pc_beacon_interval;	/* PC50 beacon interval */
	int pc_beacon_timer;	/* Timer used for the above */
	int merge_dx;		/* How many DX spots are merged on linkup */
	int merge_wwv;		/* The same for WWV */
	
	int ping_interval;	/* How often to ping (minutes) */
	int ping_timer;		/* Timer for the above */
	int ping_maxrtt;	/* Maximum RTT allowed (seconds) */
	
	struct stringlist_t *scripts;	/* Connection script filenames */
	
	/* If a protocol message is not
	   understood, and one of these
	   strings is found in the received
	   message, the link is disconnected. */
	char **discstrs;
	
	int retry_time;		/* Relink after... (minutes) */
	int retry_timer;	/* Minutes since last try */
	
	int tries;		/* How many tries after last disconnect */
	int failures;		/* How many link failures */
	
	int routes;		/* How many routes on the link */
	int linker_timeout;	/* Linker timeout */
	int traced;		/* Traced? */

	struct link_t *next;
	struct link_t **prevp;
};

struct nuser_t {
	time_t since;		/* Logged on since */
	call_t call;		/* Callsign */
	name_t name;		/* Name */
	struct node_t *node;	/* Node */
	int hops;		/* How many hops away */
	int here;		/* The here bit */
	char *away_str;		/* Away string */
	time_t away_time;	/* Away since */
	int sysop;		/* Onko sysop */
	int priviledged;	/* Muuten vaan hyv jtk */
	/* Semi-doubly linked list: */
	struct nuser_t **prevp;
	struct nuser_t *next;
};

struct dx_t {				/* dx.clu */
	unsigned long num;		/* Index number */
	time_t time;			/* Time */
	dxcall_t call;			/* DX callsign */
	unsigned long freq;		/* Frequency, 1 decimal */
	call_t fromcall;		/* Who submitted the spot */
	call_t fromnodec;		/* at which node */
	struct link_t *via;		/* What link port did it come from */
	char info[81];			/* Misc text */
	int hops;			/* How many hops */
};

struct ann_t {				/* announce.clu */
	unsigned long num;		/* Index number */
	time_t time;			/* Packed datetime */
	call_t fromcall;		/* From callsign */
	call_t fromnodec;			/* From node */
	call_t tonodec;			/* Distributed to node (* = full) */
	struct link_t *via;		/* What link port did it come from */
	int sysop;			/* Only to sysops */
	int wx;				/* Is it a WX spot */
	char message[256];		/* The message itself */
	int hops;
};

struct talk_t {
	unsigned long num;		/* Index number */
	time_t time;			/* time */
	call_t fromcall;		/* From callsign  */
	call_t fromnodec;		/* From node */
	call_t tocall;			/* Recipient's callsign */
	struct node_t *tonode;		/* Recipient's node */
	struct link_t *via;		/* What link port did it come from */
	char message[256];		/* The message */
};

struct wwv_t {				/* wwv.clu */
	unsigned long num;		/* Index number */
	time_t time;			/* When we got it in */
	int hour;			/* Hour of the WWV message */
	int sfi, a, k;
	char forecast[81];
	call_t fromcall;		/* Who submitted this one */
	call_t fromnodec;		/* ...on what node... */
	struct link_t *via;		/* What link port did it come from */
	int hops;
};

struct index_t {
	/* Index counters for various message types */
	unsigned long dx_first, dx, ann_first, ann, wwv_first, wwv, talk, mail;
};

enum cutlink_t {
	cl_nice, cl_rude
};

int listened;

/* Statistiikat: */
extern long pc_received, pc_unknown, pc_invalid;
extern int node_count;			/* How many nodes */
extern int node_max;			/* Nodes... */
extern int link_count;			/* How many links */
extern int link_max;			/* Nodes... */

/* Listat: */
extern struct node_t *localnode;	/* Pointer to the local Node record */
extern struct node_t *nodes;		/* Nodes */
extern struct link_t *links;		/* Links */
extern struct index_t indexes;		/* Index */

extern void free_nodel(struct node_t *n);	/* Free a list of nodes */

extern struct node_t *get_node(call_t *call);	/* Find node */
extern struct node_t *guess_node(call_t *call);	/* Guess node */
extern struct link_t *get_link(call_t *nodec);	/* Find link */

extern int count_links(void);			/* Count links */
extern int count_linksh(void);			/* Count held links */

extern void clear_nlocks(void);			/* Clear node locks */

extern int init_links(void);			/* Try to initiate all links */
extern void link_finished(struct link_t *l);	/* Linking finished */
extern int cut_links(enum cutlink_t method, char *reason); /* Cut all links */

extern int link_login(struct csock_t *sock, struct link_t *l);
extern void link_logout(struct csock_t *sock);

extern int caught_loop(struct link_t *l, char *reason, ...);

/* Cluster events */
extern int net_dx(struct dx_t *dx);			/* DX */
extern int net_announce(struct ann_t *ann);		/* Announce */
extern int net_talk(struct talk_t *talk);		/* Talk */
extern int net_wwv(struct wwv_t *wwv);			/* WWV */

/* Node list handling */
extern void net_nodeadd(struct node_t *n);			/* Add cluster node */
extern void net_nodedel(struct node_t *n, char *reason);	/* Del cluster node */

extern void net_minute(void);			/* Called every minute */
extern void net_init(void);			/* Initialize */

/* Initialize the unit */
extern void net_readconfig(void);	/* Read the configuration file, links.ini */

#endif
