
/*
 *	network.c
 *
 *	the main workhorse of the cluster networking code
 */
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "hmalloc.h"
#include "ctypes.h"
#include "config.h"
#include "network.h"
#include "net_ping.h"
#include "net_link.h"
#include "net_pc.h"
#include "net_user.h"
#include "net_rdb.h"
#include "luser.h"
#include "ctime.h"
#include "timer.h"
#include "log.h"
#include "mid.h"
#include "crc.h"
#include "linker.h"
#include "version.h"
#include "f_last.h"
#include "f_nuser.h"
#include "f_dx.h"
#include "f_ann.h"
#include "f_wwv.h"
#include "cluster.h"
#include "ui.h"

int listened;

int node_count;			/* How many nodes */
int node_max;			/* Nodes... */
int link_count;			/* How many links */
int link_max;			/* Nodes... */

struct node_t *localnode;	/* Pointer to the local Node record */
struct node_t *nodes;		/* Nodes */
struct link_t *links;		/* Links */
struct index_t indexes;		/* Index */


/* ====================================================================== */
/*  N O D E   T A B L E   H A N D L I N G                                 */
/* ====================================================================== */

/*
 *	Insert a node entry to the node table
 */

struct node_t *insert_node(struct node_t *n)
{
	struct node_t **prevp = &nodes;
	struct node_t *p = nodes;
	int i;

	while ((p) && ((i = strcmp(p->call, n->call))) < 0) {
		if (!i)
			return NULL;	/* We already have this one ! */
		prevp = &p->next;
		p = p->next;
	}
	
	*prevp = n;
	n->next = p;
	n->prevp = prevp;
	if (n->next)
		n->next->prevp = &n->next;
	node_count++;
	return n;
}

/*
 * 	Delete a node from the list
 */

void free_node(struct node_t *n)
{
	free_nusersn(n);
	if (n->pinging)
		ping_abort(n, "Node vanished");
	rdb_nodedel(n);
	*n->prevp = n->next;
	if (n->next)
		n->next->prevp = n->prevp;
	node_count--;
	n->via->routes--;
	free(n);
}

/*
 *	Find a record from the node list
 */

struct node_t *get_node(call_t *call)
{
	struct node_t *p = nodes;

	while ((p) && (strcmp(p->call, (char *)call)))
		p = p->next;

	return p;
}

/*
 *	Get a link for the destination call
 */

struct link_t *get_link(call_t *call)
{
	struct link_t *l = links;
	
	while ((l) && (strcmp(l->call, (char *)call) != 0)) {
		l = l->next;
	}
	
	return l;
}

/*
 *	Count active links
 */
 
int count_links(void)
{
	struct link_t *l = links;
	int i = 0;
	
	while (l) {
		if (l->state == ls_linked)
			i++;
		l = l->next;
	}
	
	return i;
}

/*
 *	Count held links
 */
 
int count_linksh(void)
{
	struct link_t *l = links;
	int i = 0;

	while (l) {
		if (l->state == ls_held)
			i++;
		l = l->next;
	}

	return i;
}

/*
 *	Clear node table locks
 */

void clear_nlocks(void)
{
	struct node_t *n = nodes;

	while (n) {
		n->locked = 0;
		n = n->next;
	}
}

/*
 *	Disconnection reason string
 */

char *dr_str(enum dreason_t reason)
{
	static char dr_strs[] = {
		"hard disconnect", 
		"never connected", 
		"by sysop", 
		"remote request",
		"loop detected",
		 "ping timeout", 
		 "link held",
		 "\0"
	};
	
	static char s[30];
	
	if ((reason >= (int)dr_unknown) && (reason <= (int)dr_held))
		return &dr_strs[(int) reason - (int) dr_unknown];
	else {
		sprintf(s, "weird %d", (int) reason);
		return s;
	}
}

/*
 *	Drops all users and nodes behind a link
 */

void drop_routes(struct link_t *l)
{
	struct node_t *h, *n;
	char s[256];
	
	send_us(M_NODE, "%d nodes vanished due to link failure$t.", l->routes);
	
	h = nodes;
	while (h) {
		n = h->next;
		if (h->via == l) {
			sprintf(s, "%s <> %s link failed", (char *)&clucall, (char *)&l->call);
			h->hops = 0;
			link_nodedel(h, s);
			free_node(h);
		}
		h = n;
	}
	l->routes = 0;

	count_users();		/* Calculate total users */
}

/* ====================================================================== */
/*  L I N K   H A N D L I N G                                             */
/* ====================================================================== */

/*
 *	Link connection
 */
 
void link_login(struct csock_t *sock, struct link_t *l)
{
	if (l->state == ls_disc || l->state == ls_held) {
		sock->charset = l->charset;
		if (l->traced) {
			sock->trace_file = (char *)hmalloc(strlen(l->call) + 5);
			sprintf(sock->trace_file, "%s.trc", l->call);
		}
		sock->link = l;
		l->sock = sock;
		
		if (l->state == ls_held) {	/* Held */
			l->dreason = dr_held;
			sock_disconnect(sock);
			return;
		}
		l->state = ls_init;
		
		l->sock->eolmode = eol_text;
		l->sock->in_handler = pc_handler;
		l->sock->disc_handler = pc_disc_handler;
		
		pc_conn_handler(l);
		
		return;
	}
	/* Whoa, duplikaatti! Katkaistaan molemmat... */
	sock_disconnect(sock);
	sock_disconnect(l->sock);
}

/*
 *	End of connection script processing
 */

void init_link_script_end(struct link_t *l, int res)
{
//	log(L_DEBUG, "init_link_script_end(): %s: %d", l->call, res);
	
	if (res == LJ_SUCCESS) {
		l->sock->eolmode = eol_text;
		l->sock->in_handler = pc_handler;
		l->sock->disc_handler = pc_disc_handler;
	} else
		link_logout(l->sock);
	
	return;
}

/*
 *	Initiate a link
 */

void init_link(struct link_t *l)
{
	l->scripts = l->scripts->next;
	
	log(L_LINK, "Linker: Trying to link to %s (%s)", l->call, l->scripts->s);
	l->sock = start_connect(l->scripts, (void (*)(void *ptr, int res))&init_link_script_end, (void *)l, l->linker_timeout);
	
	if (l->sock) {
		l->state = ls_init;
		l->retry_timer = 0;
		l->tries++;
		l->sock->link = l;
	}
}

/*
 *	Initiate all links that are disconnected at the moment
 */

int init_links(void)
{
	struct link_t *l = links;
	int i;
	
	while (l) {
		if (l->state == ls_disc && l->mode != lm_incoming) {
			init_link(l);
			i++;
		}
		l = l->next;
	}
	return i;
}

/*
 *	A link has been finished
 */

void link_finished(struct link_t *l)
{
	write_last("Linker", "%s %s - Linked to %s", 
		datestr_s_pad(now), timestr_s(now), l->call);
	link_count++;
	l->state = ls_linked;
	l->dreason = dr_unknown;
	l->since = now;
	l->tries = 0;
	l->pc_beacon_timer = 0;
	l->ping_timer = 0;
	if (l->mode != lm_incoming)
		done_connect(l->sock);
	user_linkadd(l);
}

/*
 *	Drop all links immediately
 */

int cut_links(enum cutlink_t method, char *reason)
{
	struct link_t *l = links;
	int i;
	
	while (l) {
		if (l->state == ls_init || l->state == ls_linked) {
			l->dreason = dr_sysop;
			if (method == cl_rude || l->state == ls_init || l->mode == lm_listen)
				sock_disconnect(l->sock);
			else {
				link_reqclose(l, reason);
			}
			if (l->tries == 0)
				l->tries++;
			i++;
		}
		l = l->next;
	}
	return i;
}

/*
 *	Link has been disconnected
 */

void link_logout(struct csock_t *sock)
{
	struct link_t *l;
	
	l = sock->link;
	
	if (l->state != ls_held) {
		if (l->state != ls_init) {
			log(L_LINK, "Link to %s failed (%s).",
				l->call, dr_str(l->dreason));
			write_last("Linker", "%s %s  - Link to %s failed.",
			 	datestr_s_pad(now), timestr_s(now), l->call);
			user_linkdel(l, "unknown");
			link_count--;
		}
		l->state = ls_disc;
	}
	
	drop_routes(l);
	l->failures++;
	l->since = now;
	l->sock = 0;
	if (l->tries == 0) {
		l->retry_timer = l->retry_time - 1;
		return;
	}
	l->retry_timer = 0;
	if (l->backup) {
		l->state = ls_backup;
		l->backup->state = ls_disc;
	}
}

/* ====================================================================== */
/*  C L U S T E R   E V E N T S                                           */
/* ====================================================================== */

/**************** Talk handling ***************/

/*
 *	Bounce a talk
 */

void talk_back(int from, struct talk_t *t, struct talk_t *nt)
{
	/* Send a message to the sender */
	nt->time = now;
	switch (from) {
	case 0:
		strcpy(nt->fromcall, (char *)clucall);
		break;
	case 1:
		strcpy(nt->fromcall, (char *)&t->tocall);
		break;
	}
	strcpy(nt->fromnodec, (char *)clucall);
	strcpy(nt->tocall, t->fromcall);
	nt->tonode = get_node(&t->fromnodec);
	nt->via = NULL;
	net_talk(nt);
}

/*
 *	Process talk messages to localnode
 */

void talk_tonode(struct talk_t *t)
{
	struct talk_t nt;
	
	/* Detect 'away' talkback loops, and internode talks */
	if ((strstr(t->message, "away:")) || (get_node(&t->fromcall)))
		return;
	
	switch (lowcase_ch[(int)t->message[0]]) {	/* Remote commands... */
		case 'e':	/* echo */
			strcpy(nt.message, t->message);
			break;

		case 's':	/* sysop */
			sprintf(nt.message, "The sysop of this node is %s",
				(char *)sysop_call);
			break;
			
		case 't':	/* time */
			sprintf(nt.message, "%s, uptime %s",
			     timestr(now), secs2str_l(uptime()));
			break;
		
		case 'q':
			sprintf(nt.message, "QTH: %s (%s)", qth, qth_loc);
			break;
			
		case '%':
		case 'm':
/*			sprintf(nt.message, "Mem fr %ld us %ld b, %ld allocs.",
				maxavail(), heap_used(),
				heap_allocs());
			break;
*/		
		case 'h':
		case '?':	/* Help */
			strcpy(nt.message, "Commands: Echo Memory Qth Sysop Time Version");
			break;

		default:
			sprintf(nt.message, SOFT_STRING);
			break;
	}
	talk_back(0, t, &nt);
}

/*
 *	Process talk message to local user
 */

void talk_touser(struct luser_t *p, struct talk_t *t)
{
	char s[256];
	struct talk_t nt;
	
	/* Send him a beep, if he wants beeps and he is not in talk mode
	   with the sender of this message */
	
	if ( (p->f) && ((p->f->flags & F_BEEPS) == F_BEEPS) 
		&& ((p->f->beeps & M_TALK) == M_TALK) 
		&& (!((p->m2 == 1) && (!strcmp(t->fromcall, p->str)) )) )
			csputc(p->sock, '\007');
	
	sprintf(s, "%s talks", t->fromcall);
	if ((p->f) && ((p->f->flags & F_TIMESTAMP) == F_TIMESTAMP) ) {
		strcat(s, " ");
		strcat(s, timestr_s(t->time));
		strcat(s, "Z");
	}
	strcat(s, ":");
	
	csputs(p->sock, format(s, t->message, 80, 1));
	sprintf(p->last_talk_from, "%s@%s", t->fromcall, t->fromnodec);
	
	if (!( (p->nu->away_str) && (strcmp(t->fromcall, t->tocall) != 0) 
			   && (!strstr(hstrlwr(t->message), "away:")) ))
		return;
	
	sprintf(nt.message, "I'm away: %s", p->nu->away_str);
	talk_back(1, t, &nt);
	return;
}

/*
 *	Process a talk message
 */

void net_talk(struct talk_t *t)
{
	struct luser_t *p;
	struct nuser_t *u;
	struct talk_t nt;
	
	if (strcmp(t->fromcall, (char *)&clucall)) {
		log(L_TALK, "Talk: %s > %s: %s", t->fromcall, t->tocall, t->message);
		set_homenode(NULL, &t->fromcall, &t->fromnodec, 2);
	}
	
	if (t->tonode != localnode) {	/* Route the message */
		link_talk(t);
		return;
	}
	
	if (!strcmp(t->tocall, (char *)clucall)) { /* Is this a message to the local node itself */
		talk_tonode(t);
		return;
	}
	
	if ((p = get_luser(&t->tocall))) {	/* The user is logged on */
		if ((!p->f || ((p->f->messages & M_TALK) == M_TALK)) && !p->locked) {	/* Does he accept talk messages */
			talk_touser(p, t);
			return;
		}
		if (p->locked)
			sprintf(nt.message, "User %s is not able to talk right now.", t->tocall);
		else
			sprintf(nt.message, "User %s has disabled talk messages.", t->tocall);
		talk_back(0, t, &nt);
		return;
	}
	
	if (!strcmp(t->tocall, (char *)&clucall) && !strcmp(t->fromcall, (char *)&clucall))
		/* Catch a local loop (would cause a stack overflow) */
		return;
	
	u = get_nuser(&t->tocall);
	if ((u) && (u->node->via != t->via)) { /* Never route back, might loop */
		/* Forward the message to another cluster node (should check for loops!) */
		t->tonode = u->node;
		link_talk(t);
		return;
	}
	if (get_node(&t->fromcall) == NULL) {
		/* User is not on the node table */
		sprintf(nt.message, "User or node %s is not known here.", t->tocall);
		talk_back(0, t, &nt);
	}
}


/* ====================================================================== */

/*
 *	DX
 */

void net_dx(struct dx_t *dx)
{
	mid_t mid;
	unsigned short w;
	
	/* Dupe checking */
	memset(&mid, 0, MID_LEN);
	mid[0] = (char)MIDT_DX;
	
	w = CRC16_SEED;
	ccrc16l(dx->call, strlen(dx->call), &w);
	memmove(&mid[1], &w, sizeof(w));
	
	memmove(&mid[3], &dx->time, sizeof(dx->time));
	memmove(&mid[3 + sizeof(dx->time)], &dx->freq, sizeof(dx->freq));
	
	if (dx->freq > 0 && valid_call(&dx->fromcall) &&
	  valid_call(&dx->fromnodec) && new_mid(&mid)) {
		indexes.dx++;
		if (indexes.dx >= 10000)	/* Wraparound */
			indexes.dx = 1;
		dx->num = indexes.dx;
		
		log(L_DX, "DX de %s: %s %s \"%s\"",
			dx->fromcall, freq2str(dx->freq), dx->call,
			dx->info);
		
		ui_dx(dx);	/* Broadcast */
		user_dx(dx);	/* Notify the local users */
		if (!listened)	/* Route */
			link_dx(dx);
		
		write_dx(dx);	/* Write to file */
		set_homenode(NULL, &dx->fromcall, &dx->fromnodec, 2);
		free(dx);
	}
}

/* ====================================================================== */

void net_announce(struct ann_t *ann)
{

	if (valid_call(&ann->fromcall) && valid_call(&ann->fromnodec)) {
		indexes.ann++;
		if (indexes.ann >= 10000)
			indexes.ann = 1;
		ann->num = indexes.ann;
		
		log(L_ANN, "%s shouts @%s: %s", ann->fromcall, ann->tonodec, 
			ann->message);
		
		if (!strcmp(ann->tonodec, "*") || !strcmp(ann->tonodec, (char *)&clucall)) {
			user_announce(ann);	/* Notify local users */
			if (!(ann->wx || ann->sysop)) {
				ui_ann(ann);	/* Broadcast */
				write_ann(ann);	/* Write to file */
			}
		}
		if (!listened)	/* Route it to other links */
			link_announce(ann);
		set_homenode(NULL, &ann->fromcall, &ann->fromnodec, 2);
	}
	free(ann);
}

/* ====================================================================== */

void net_wwv(struct wwv_t *wwv)
{
	mid_t mid;
	struct tm *tm;

	/* Dupe checking */
	memset(&mid, 0, MID_LEN);
	mid[0] = (char)MIDT_WWV;
	
	tm = gmtime(&wwv->time);
	
	mid[1] = tm->tm_mday;
	mid[2] = tm->tm_mon;
	mid[3] = tm->tm_year - 1900;
	mid[4] = wwv->hour;
	
	if (!new_mid(&mid)) {
		free(wwv);
		return;
	}
		
	indexes.wwv++;
	if (indexes.wwv >= 10000)
		indexes.wwv = 1;
	wwv->num = indexes.wwv;
	
	log(L_WWV, "WWW for hour %d de %s: SFI %d A %d K %d - \"%s\"",
		wwv->hour, wwv->fromcall, wwv->sfi, wwv->a, wwv->k, wwv->forecast);
	
	ui_wwv(wwv);		/* Broadcast */
	user_wwv(wwv);		/* Local users */
	if (!listened)		/* Links */
		link_wwv(wwv);
	write_wwv(wwv);	/* File */
	set_homenode(NULL, &wwv->fromcall, &wwv->fromnodec, 2);
	
	free(wwv);
}


/*
 *	Add a list of nodes
 */

void net_nodeadd(struct node_t *n)
{
	struct node_t *nl, *p, *next;
	
	nl = p = n;
	p->prevp = &nl;
	
	while (p) {
		next = p->next;
		if (get_node(&p->call)) {
			if (p->next)
				p->next->prevp = p->prevp;
			*p->prevp = p->next;
			hfree(p);
		} else {
			log(L_NODE, "Node: %s added", p->call);
			user_nodeadd(p);
		}
		p = next;
	}
	
	link_nodeadd(nl);	/* feed links */
	
	/* insert in node list */
	p = nl;
	while (p) {
		next = p->next;
		
		p->via->routes++;
		
		if (!strcmp(p->call, p->via->call)) {
			p->via->node = p;	/* Its our neighbour! */
			p->via->version = p->version;
		}
		
		insert_node(p);
		
		if (node_count > node_max)	/* high water mark */
			node_max = node_count;
		
		p = next;
	}
}

void net_nodedel(struct node_t *n, char *reason)
{
	log(L_NODE, "Node: %s vanished", n->call);
	user_nodedel(n, reason);	/* Notify local users */
	link_nodedel(n, reason);	/* Tell others */
	free_node(n);			/* Remove from the list */
	count_users();			/* Calculate total users */
}

/* ====================================================================== */
/* This one is called once every minute */

void net_minute(void)
{
	struct link_t *l = links;
	
	while (l) {
		if (l->state == ls_disc && l->mode != lm_incoming) {
			l->retry_timer++;
			if (l->retry_timer >= l->retry_time)
				init_link(l);
		}
		
		/* PC50 user count beacon */
		if (l->state == ls_linked &&
		    (l->mode == lm_normal || l->mode == lm_incoming) &&
		    l->pc_beacon_interval != 0) {
			l->pc_beacon_timer++;
			if (l->pc_beacon_timer == l->pc_beacon_interval) {
				l->pc_beacon_timer = 0;
				link_beacon(l);
			}
		}
		
		/* Ping */
		if (l->state == ls_linked && l->ping_interval != 0) {
			l->ping_timer++;
			if (l->ping_timer == l->ping_interval) {
				if (l->mode == lm_listen) /* Just "ping" */
					link_ping(l, &l->call, &l->mycall, 1);
				else
					ping_add(l->node, &l->mycall, l->ping_maxrtt);

				l->ping_timer = 0;
			}
		}
		l = l->next;
	}

}

/* ====================================================================== */
/* Initialize the protocol unit */

void net_init(void)
{
	int i;
	
	for (i = 0; i <= 49; i++)
		pc_msgstats[i] = 0;
	
	/* Local node */
	localnode = nodes = hmalloc(sizeof(struct node_t));
	node_count = 1;
	node_max = 1;
	
	strcpy(localnode->call, (char *)clucall);
	localnode->via = NULL;
	localnode->since = now;
	localnode->version = pc_version;
	localnode->hops = 0;
	localnode->hops_ok = 1;
	localnode->here = 0;
	localnode->users_known = 1;
	localnode->users_ok = 1;
	localnode->users = 0;
	localnode->rtt = 0;
	localnode->rtt_ok = 1;
	localnode->pinging = 0;
	localnode->locked = 0;
	localnode->prevp = &nodes;
	localnode->next = NULL;
	
	set_timer(now + 5, 60, (void (*)(void *ptr))&init_links, NULL);
}

