
/*
 *	net_ping.c
 *
 *	pinging another network node
 */
 
#include <stdlib.h>
#include <string.h>

#include "net_ping.h"
#include "csock.h"
#include "network.h"
#include "luser.h"
#include "cstring.h"
#include "ctime.h"
#include "log.h"

int ping_jobs;		/* Amount of pinger jobs */
struct ping_t *pings;	/* Pinger list */

/* ====================================================================== */
/*  P I N G E R                                                           */
/* ====================================================================== */

void ping_add(struct node_t *node, call_t *fromcall, int rtt_max)
{
	struct ping_t *p = pings;
	struct ping_t **prev = &pings;
	
	log(L_PING, "Pinger: %s pings %s", fromcall, node->call);
	node->pinging = 1;
	
	while (p) {
		prev = &p->next;
		p = p->next;
	}
	
	p = malloc(sizeof(struct ping_t));
	ping_jobs++;
	*prev = p;
	
	strcpy(p->fromcall, (char *)&fromcall);
	time(&p->time);
	p->rtt_max = rtt_max;
	p->node = node;
	p->next = NULL;
	
	csprintf(node->via->sock, "PC51^%s^%s^1^\n", node->call, node->via->mycall);
}

/* ====================================================================== */

int ping_del(struct node_t *node)
{
	struct ping_t *p = pings;
	struct ping_t **prevp = &pings;
	
	while ((p) && (p->node != node)) {
		prevp = &p->next;
		p = p->next;
	}
	
	if (!p)
		return 1;
	
	p->node->pinging = 0;
	*prevp = p->next;
	free(p);
	ping_jobs--;
	return 0;
}

/* ====================================================================== */

void ping_abort(struct node_t *node, char *reason)
{
	struct ping_t *p = pings;
	struct luser_t *u;
	
	while ((p) && (p->node != node))
		p = p->next;
	
	if (!p)
		return;
	
	if ((u = get_luser(&p->fromcall)) && !u->locked)
		csprintf(u->sock, "Ping %s failed - %s\n", p->node->call, reason);

	ping_del(node);
}

/* ====================================================================== */
/* This is called every second */

void pinger_second(void)
{
	struct ping_t *p, *next = pings;
	
	while ((p = next)) {
		next = p->next;
		if (time(NULL) - p->time >= p->rtt_max) {
			log(L_PING, "Pinger: %s timed out", p->node->call);
			
			if (p->node == p->node->via->node) {	/* Naapuri? */
				p->node->via->dreason = dr_pingto;
				sock_disconnect(p->node->via->sock);
			}
			ping_abort(p->node, "Timeout");
		}
	}
}

/* ====================================================================== */

int cmd_ping(struct csock_t *s, int argc, char **argv)
{
	struct node_t *n;
	call_t *call;
	
	if (argc <= 1) {
		csputs(s, "Usage: Ping <node>\n");
		return 0;
	}
	
	call = (call_t *)hstrupr(argv[1]);
	
	n = get_node(call);
	
	if (!n) {
		csprintf(s, "Node %s not found.\n", call);
		return 0;
	}
	
	if (n->pinging) {
		csprintf(s, "Node %s is already being pinged.\n", call);
		return 0;
	}
	
	if (n == localnode) {
		csputs(s, "Cannot ping local node. The RTT is 0 anyway...\n");
	} else {
		csprintf(s, "Pinging %s...\n", call);
		ping_add(n, &s->lu->f->call, PING_TIMEOUT);
	}
	return 0;
}

/* ====================================================================== */

int cmd_pstatus(struct csock_t *s, int argc, char **argv)
{
	struct ping_t *p = pings;
	
	csputs(s, "Pinging:\n");
	if (!p) {
		csputs(s, "None.\n");
		return 0;
	}
	csputs(s, "Node      From      Time");
	while (p) {
		csprintf(s, "%s%s%s\n",
			padleft((char *)&p->node->call, 10),
			padleft((char *)&p->fromcall, 10),
			secs2str(time(NULL) - p->time));
		p = p->next;
	}
	return 0;
}

