
//////////////////////////////////////////////////////////////
//
// FBB driver for AGW packet engine
//
// File : fbb_agw.dll
//
// (C) F6FBB 1999
//
//////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include <ddeml.h>
#include "../fbb_drv.h"

#define AGW_MAJOR		1
#define AGW_MINOR		2

#define DISCONNECTED	0
#define CONNECTED		1
#define CONNECT_PENDING 2
#define WAITING_CALL	3
#define CONNECT_CALL	4

#define MAXITEMS 6

//------------

typedef struct DATAIN_t {
	char	*pData;
	int		nChannel;
	int		nType;
	int		nLen;
	DRVUI	*pUi;
	struct	DATAIN_t *pNext;
} DATAIN;

typedef struct CHANNEL_t {
	int		nChan;
	int		nState;
	int		nLastState;
	int		nLastNbFrames;
	int		nFrameNb;
	BOOL	bBinary;
	char	szMyCall[10];
	char	szCall[100];
} CHANNEL;

typedef struct FBBDRV_t
{
	HWND	hWnd;
	int		nPort;
	int		nNbChan;
	int		nLastChan;
	int		nAgwPort;
	BOOL	bNext;
	DRVUI	*pUi;
	DATAIN	*pHostHead;	// Head of data_in list
	DATAIN	*pHostTail;	// Tail of data_in list
	CRITICAL_SECTION	hCritical;
	char	szMyCall[10];
	CHANNEL	*pChan;
	struct	FBBDRV_t *pNext;
} FBBDRV;

struct PORTS
{
	int	nNbPorts;
	char **szName;
} Port;


//////////////////////////////////////////////////////////////
// Local prototypes
//////////////////////////////////////////////////////////////

static DWORD WINAPI AgwLoop( LPSTR lpData );
static FBBDRV *SearchPort(int nPort);
static FBBDRV *SearchAgwPort(int nPort);
static DATAIN *GetHost(FBBDRV *pFbbDrv);
static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi);
static LPSTR SockerrToString(DWORD serr );
static int SockMsgBox(UINT  fuType, LPSTR pszFormat,	... );
static BOOL AgwInit(FBBDRV *pFbbDrv, char *pPtr);
static BOOL Connected(FBBDRV *pFbbDrv, char *szCall);
static BOOL Disconnect(FBBDRV *pFbbDrv, int nChan);
static BOOL Disconnected(FBBDRV *pFbbDrv, int nChan);
static BOOL ConnectTo(FBBDRV *pFbbDrv, int nChan, char *szRemoteCall);
static void ToHost(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi);
static void InitChannel(FBBDRV *pFbbDrv, int nChan);

static void SendDataUN(FBBDRV *pFbbDrv, DRVUI *pUi, char *szData, int nDataLen);
static void SendDataTo(FBBDRV *pFbbDrv, int nChan, char *szData, int nDataLen);
static void AskStats(FBBDRV *pFbbDrv, int nChan);
static void ReceivedData(HDDEDATA hData);
static void GetPorts(void);
static void TXDATA(FBBDRV *pFbbDrv, char *str,int count);

static HDDEDATA CALLBACK CallBack(WORD wType, WORD t1, HCONV hConv1, HSZ hsz1, HSZ hsz2, HDDEDATA hData, DWORD t2, DWORD t3 );

//////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////

FBBDRV	*pDrvHead = NULL;	// Head of port structures
DWORD	nIdInst = 0;
HCONV	hConv[2];
HSZ		hszService, hszTopic[2], hszItem[8];

static char szTemp[4000];

//////////////////////////////////////////////////////////////
//
// Exported functions (5).
// They are :
// BOOL OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo);
// BOOL CloseFbbDriver(int nPort);
// BOOL ReadFbbDriver(int nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi);
// BOOL WriteFbbDriver(int nPort, int *nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi);
// BOOL StatFbbDriver(int nPort, int nChan, int nCmd, void *pPtr, int nLen);
//
//////////////////////////////////////////////////////////////


BOOL APIENTRY DllMain( HANDLE hModule, 
					  DWORD  ul_reason_for_call, 
					  LPVOID lpReserved
					  )
{
    switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
	case DLL_THREAD_ATTACH:
	case DLL_THREAD_DETACH:
		break;
	case DLL_PROCESS_DETACH:
		// Clean all linked ports
		while (pDrvHead)
			CloseFbbDriver(pDrvHead->nPort);
		break;
    }
    return TRUE;
}


BOOL WINAPI OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo)
{
	FBBDRV	*pFbbDrv;
//	HANDLE	hFbbThread;
//	DWORD	dwThreadID;
	DRVINFO *pInfo;
	
	// Already used ?
	pFbbDrv = SearchPort(nPort);
	if (pFbbDrv)
		return FALSE;
	
	pInfo = (DRVINFO *)pDrvInfo;

	// Delete the ssid 0 in mycall, otherwise AGW does not understand
	char *ptr = strchr(pInfo->szMyCall, '-');
	if (ptr && *(ptr+1) == '0')
		*ptr = '\0';
	
	// Connect to the AGW PE if not already done
	if (pDrvHead == NULL)
	{
		
		//start ddeml
		DWORD dwTempResult;
		HDDEDATA Result;

		if (DdeInitialize(&nIdInst, (PFNCALLBACK)(FARPROC)CallBack, APPCMD_CLIENTONLY, 0) != DMLERR_NO_ERROR)
			return FALSE;
		
		hszService  = DdeCreateStringHandle( nIdInst, "SV2AGW Packet Engine Server", CP_WINANSI );
		hszTopic[0] = DdeCreateStringHandle( nIdInst, "SV2AGW HUMAN", CP_WINANSI );
		hszTopic[1] = DdeCreateStringHandle( nIdInst, "AGWFWD DATA", CP_WINANSI );
		hszItem[0]  = DdeCreateStringHandle( nIdInst, "MONDATA", CP_WINANSI );
		hszItem[1]  = DdeCreateStringHandle( nIdInst, "INFODATA", CP_WINANSI );
		hszItem[2]  = DdeCreateStringHandle( nIdInst, "PORTS", CP_WINANSI );
		hszItem[3]  = DdeCreateStringHandle( nIdInst, "HEARD", CP_WINANSI );
		hszItem[4]  = DdeCreateStringHandle( nIdInst, "USERS", CP_WINANSI );
		hszItem[5]  = DdeCreateStringHandle( nIdInst, "STATUSDATA", CP_WINANSI );
		
		if( (hszService == NULL) || (hszTopic[0] == NULL)||(hszTopic[1] == NULL) || (hszItem[0] == NULL)|| (hszItem[1] == NULL)
			|| (hszItem[2] == NULL)|| (hszItem[3] == NULL)|| (hszItem[4] == NULL)|| (hszItem[5] == NULL))
			return FALSE;
		
		//CONNECT first conversation for monitor and data send
		hConv[0] = DdeConnect( nIdInst, hszService, hszTopic[0], NULL );
		if( hConv[0] == 0 )
			return FALSE;
		
		//START MONITOR ADVISE LOOP
		Result=DdeClientTransaction(NULL, 0, hConv[0], hszItem[0],
							CF_TEXT,XTYP_ADVSTART, TIMEOUT_ASYNC, &dwTempResult);
		if (!Result) return FALSE;
		
		//REGISTER with PACKETENGINE THIS APPLICATION STRING FOR DATA ADVISE LOOP
		char* szName = "AGWFWD DATA";
		char szStr[100];
		short nCount = strlen(szName);
		szStr[1] = 0;
		szStr[0] ='X';
		memcpy(szStr+22, &nCount, 2);
		memcpy(szStr+24, szName, nCount+1);

		if (!DdeClientTransaction( (LPBYTE)szStr, nCount+1+24, hConv[0], hszItem[0],
							CF_TEXT,XTYP_POKE, TIMEOUT_ASYNC, NULL ))
			return FALSE;
		
		//CONNECT Second Conversation For connected DATA
		hConv[1] = DdeConnect( nIdInst, hszService, hszTopic[1], NULL );
		if( hConv[1] == 0 )
			return FALSE;
		
		//START Connected Data ADVISE LOOP
		Result = DdeClientTransaction( NULL, 0, hConv[1], hszItem[1],
							CF_TEXT,XTYP_ADVSTART, TIMEOUT_ASYNC, &dwTempResult);
		if (!Result)
			return FALSE;
		
		//Register MYCALL
		if (!DdeClientTransaction( (LPBYTE)pInfo->szMyCall, strlen(pInfo->szMyCall)+1,
							hConv[1], hszItem[1], CF_TEXT,XTYP_POKE, TIMEOUT_ASYNC, NULL))
			return FALSE;
		
		//Start ADVISE LOOP for Status msgs (Connect,Disconnect,Retryout etc)
		Result = DdeClientTransaction( NULL, 0, hConv[1], hszItem[5],
							CF_TEXT,XTYP_ADVSTART, TIMEOUT_ASYNC, &dwTempResult);
		if (!Result)
			return FALSE;
		
		//Start MHEARD ADVISE LOOP
		Result = DdeClientTransaction( NULL, 0, hConv[1], hszItem[3],
							CF_TEXT,XTYP_ADVSTART, TIMEOUT_ASYNC, &dwTempResult);
		if (!Result)
			return FALSE;
		
		if (hWnd)
			WM_NOTIFY_MSG = RegisterWindowMessage(FBBDRV_NOTIFY);
		
		//Get Available RadioPorts with Descriptions
		GetPorts();
	}
	
	pFbbDrv = (FBBDRV *)LocalAlloc(LPTR, sizeof(FBBDRV));
	if (pFbbDrv == NULL)
		return FALSE;
	
	strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
	pFbbDrv->pUi		= NULL;
	pFbbDrv->bNext		= FALSE;
	pFbbDrv->hWnd		= hWnd;
	pFbbDrv->nPort		= nPort;
	pFbbDrv->nAgwPort	= pInfo->nMultCh;
	pFbbDrv->nNbChan	= pInfo->nNbChan;
	
	strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
	
	// Includes the unused channel 0
	pFbbDrv->pChan      = (CHANNEL *)LocalAlloc(LPTR, (pFbbDrv->nNbChan+1) * sizeof(CHANNEL));
	if (pFbbDrv->pChan == NULL)
	{
		LocalFree(pFbbDrv);
		return FALSE;
	}

	for (int n = 0 ; n <= pFbbDrv->nNbChan ; n++)
		InitChannel(pFbbDrv, n);
		
	// Insert the structure in the list
	pFbbDrv->pNext = pDrvHead;
	pDrvHead = pFbbDrv;
	
	InitializeCriticalSection(&pFbbDrv->hCritical);
	
	return TRUE;
	
} // end of OpenFbbDriver()

BOOL WINAPI CloseFbbDriver(int nPort)
{
	int nNb;
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	// Disconnect all connected channels
	for (nNb = 0 ; nNb <= pFbbDrv->nNbChan ; nNb++)
	{
		if (pFbbDrv->pChan[nNb].nState != DISCONNECTED)
			Disconnect(pFbbDrv, nNb);
	}
	
	DeleteCriticalSection(&pFbbDrv->hCritical);
	
	// Remove structure from list
	if (pDrvHead == pFbbDrv)
		pDrvHead = pDrvHead->pNext;
	else
	{
		FBBDRV *pTmp = pDrvHead;
		while (pTmp->pNext)
		{
			if (pTmp->pNext == pFbbDrv)
			{
				pTmp->pNext = pTmp->pNext->pNext;
				break;
			}
			pTmp = pTmp->pNext;
		}
	}
	
	// Close the AGW-PE connection if last
	if (pDrvHead == NULL)
	{
		int x;
		
		if( hConv[0] != 0 )
		{
			DdeDisconnect( hConv[0] );     // Let the Packet Engine know we are leaving
			hConv[0] = 0;
		}

		if( hConv[1] != 0 )
		{
			DdeDisconnect( hConv[1] );     // Let the Packet Engine know we are leaving
			hConv[1] = 0;
		}
		
		if( nIdInst != 0 )
		{
			DdeFreeStringHandle( nIdInst, hszService );
			if( hszTopic[0] != NULL )
			{
				DdeFreeStringHandle( nIdInst, hszTopic[0] );
				hszTopic[0] = NULL;
			}
			
			if( hszTopic[1] != NULL )
			{
				DdeFreeStringHandle( nIdInst, hszTopic[1] );
				hszTopic[1] = NULL;
			}
			
			for ( x = 0 ; x < MAXITEMS ; x++)
			{
				if( hszItem[x] != NULL )
				{
					DdeFreeStringHandle( nIdInst, hszItem[x] );
					hszItem[x] = NULL;
				}
				DdeUninitialize(nIdInst); //End DDEML manager
			}

			nIdInst = 0;

			// free the Port structure
			if (Port.szName)
			{
				for (x = 0 ; x < Port.nNbPorts ; x++)
				{
					if (Port.szName[x])
						LocalFree(Port.szName[x]);
				}
				LocalFree(Port.szName);
			}

		}
	}
	
	LocalFree(pFbbDrv);
	
	return ( TRUE ) ;
	
} // end of CloseFbbDriver()

BOOL WINAPI ReadFbbDriver(int *nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	FBBDRV *pFbbDrv = SearchPort(*nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	pPtr = GetHost(pFbbDrv);
	if (pPtr == NULL)
		return FALSE;
	
	if (pPtr->nLen)
	{
		memcpy(pszBuf, pPtr->pData, pPtr->nLen);
		LocalFree(pPtr->pData);
	}
	*nLen = pPtr->nLen;
	*nChan = pPtr->nChannel;
	*nType = pPtr->nType;
	
	if (pPtr->pUi)
	{
		*pUi = *pPtr->pUi;
		LocalFree(pPtr->pUi);
	}
	LocalFree(pPtr);
	
	return TRUE;
} // end of ReadFbbDriver()

BOOL WINAPI WriteFbbDriver(int nPort, int nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	return ToTnc(pFbbDrv, nChan, nType, pszBuf, nLen, pUi);
} // end of WriteFbbDriver()

BOOL WINAPI StatFbbDriver(int nPort, int nCmd, int nChan, void *pPtr, int nLen)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return 0;
	
	switch (nCmd)
	{
	case DRV_INIT:
		return AgwInit(pFbbDrv, (char *)pPtr);
	case DRV_SNDCMD:
	case DRV_ECHOCMD:
		return (ToTnc(pFbbDrv, nChan, DRV_COMMAND, (char *)pPtr, strlen((char *)pPtr), NULL));
	case DRV_PORTCMD:
		return (ToTnc(pFbbDrv, 0, DRV_COMMAND, (char *)pPtr, strlen((char *)pPtr), NULL));
	case DRV_PACLEN:
		*((int *) pPtr) = 250;
		return TRUE;
	case DRV_VERSION:
		wsprintf((char *)pPtr, 
			"v%d.%02d FBB driver for AGW-Engine (F6FBB-%s)", 
			AGW_MAJOR, AGW_MINOR, __DATE__);
		return TRUE;
	}
	
	return FALSE;
} // end of StatFbbDriver()


//////////////////////////////////////////////////////////////
//
// Local functions
//
//////////////////////////////////////////////////////////////

static void InitChannel(FBBDRV *pFbbDrv, int nChan)
{
	CHANNEL *pChan = &pFbbDrv->pChan[nChan];
	
	memset(pChan, 0, sizeof(CHANNEL));
	strcpy(pChan->szMyCall, pFbbDrv->szMyCall);
	pChan->nState = DISCONNECTED;
}

static BOOL AgwInit(FBBDRV *pFbbDrv, char *pPtr)
{
	char cCmd = *pPtr++;

	while (*pPtr && isspace(*pPtr))
		++pPtr;

	/*
	switch (toupper(cCmd))
	{
	}
	*/

	return FALSE;
}

static BOOL AgwCommand(FBBDRV *pFbbDrv, int nChan, char *pPtr)
{
	char cCmd = *pPtr++;

	while (*pPtr && isspace(*pPtr))
		++pPtr;

	switch (toupper(cCmd))
	{
	case 'C':
		// Connect
		if (nChan > 0)
			return ConnectTo(pFbbDrv, nChan, pPtr);
		break;
	case 'D':
		// Disconnect
		if (nChan > 0)
			return Disconnect(pFbbDrv, nChan);
		break;
	}

	return FALSE;
}

static BOOL Disconnect(FBBDRV *pFbbDrv, int nChan)
{
	char szStr[2];
	short int nCount;

	if (nChan < 1 || nChan > pFbbDrv->nNbChan)
		return FALSE;

	if (pFbbDrv->pChan[nChan].nState == DISCONNECTED)
		return FALSE;

	// Command
	szStr[0] = 'D';
	szStr[1] = '\0';
	nCount = 1;
	memmove(szTemp+22, &nCount, 2);
	memmove(szTemp+24, szStr, nCount+1);

	// Data
	szTemp[0]='S';
	szTemp[1] = pFbbDrv->nAgwPort;
	memcpy(szTemp+2, pFbbDrv->pChan[nChan].szMyCall, strlen(pFbbDrv->pChan[nChan].szMyCall)+1);
	memcpy(szTemp+12, pFbbDrv->pChan[nChan].szCall, strlen(pFbbDrv->pChan[nChan].szCall)+1);
	TXDATA(pFbbDrv, szTemp, nCount+24);

	return TRUE;
}

static BOOL Disconnected(FBBDRV *pFbbDrv, int nChan)
{
	char szBuf[256];

	if (nChan < 1 || nChan > pFbbDrv->nNbChan)
		return FALSE;

	if (pFbbDrv->pChan[nChan].nState == DISCONNECTED)
		return FALSE;

	// Send disconnect message
	wsprintf(szBuf, "(%d) DISCONNECTED fm %s", nChan, pFbbDrv->pChan[nChan].szCall);
	ToHost(pFbbDrv, nChan, DRV_COMMAND, szBuf, -1, NULL);

	// Clean the channel structure
	InitChannel(pFbbDrv, nChan);

	return TRUE;
}

static BOOL Connected(FBBDRV *pFbbDrv, char *szCall)
{
	char szBuf[256];
	int nChan;

	// Look for a free channel
	for (nChan = 1 ; nChan <= pFbbDrv->nNbChan ; nChan++)
	{
		if (pFbbDrv->pChan[nChan].nState == DISCONNECTED)
		{
	
			strcpy(pFbbDrv->pChan[nChan].szCall, szCall);
			pFbbDrv->pChan[nChan].nState = CONNECTED;
			
			// Send connect message
			wsprintf(szBuf, "(%d) CONNECTED to %s", nChan, szCall);
			ToHost(pFbbDrv, nChan, DRV_COMMAND, szBuf, -1, NULL);
		}
		return TRUE;
	}
	
	// No channel available ... Disconnect !
	char szStr[2];
	short int nCount;

	// Command
	szStr[0] = 'D';
	szStr[1] = '\0';
	nCount = 1;
	memmove(szTemp+22, &nCount, 2);
	memmove(szTemp+24, szStr, nCount+1);

	// Data
	szTemp[0]='S';
	szTemp[1] = pFbbDrv->nAgwPort;
	memcpy(szTemp+2, pFbbDrv->szMyCall, strlen(pFbbDrv->szMyCall)+1);
	memcpy(szTemp+12, szCall, strlen(szCall)+1);
	TXDATA(pFbbDrv, szTemp, nCount+24);

	return FALSE;
}

static FBBDRV *SearchPort(int nPort)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nPort == nPort)
			break;
		pTmp = pTmp->pNext;
	}
	
	return pTmp;
}

static FBBDRV *SearchAgwPort(int nAgwPort)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nAgwPort == nAgwPort)
			return pTmp;
		pTmp = pTmp->pNext;
	}
	
	return NULL;
}

static int Call2Chan(FBBDRV *pFbbDrv, char *szCall)
{
	int nChan;
	
	for (nChan = 1 ; nChan <= pFbbDrv->nNbChan ; nChan++)
	{
		if (strcmp(pFbbDrv->pChan[nChan].szCall, szCall) == 0)
			return (nChan);
	}
	return 0;
}

static BOOL SendStats(FBBDRV *pFbbDrv, int nChannel, int nNbFrames)
{
	DRVSTATS DrvStats;
	BOOL bOk = FALSE;
				
	DrvStats.nFrameNb = pFbbDrv->pChan[nChannel].nFrameNb;

	bOk |= (pFbbDrv->pChan[nChannel].nLastNbFrames != nNbFrames);
	pFbbDrv->pChan[nChannel].nLastNbFrames = nNbFrames;
	DrvStats.nNbFrames = nNbFrames;
	
	DrvStats.nNbRetry = 0;

	switch (pFbbDrv->pChan[nChannel].nState)
	{
	case CONNECTED :
		DrvStats.nState = DRV_CONNECTED;
		break;
	case DISCONNECTED :
		DrvStats.nState = DRV_DISCONNECTED;
		break;
	case CONNECT_PENDING :
	case WAITING_CALL :
	case CONNECT_CALL :
		DrvStats.nState = DRV_CONNPENDING;
		break;
	}

	bOk |= (pFbbDrv->pChan[nChannel].nLastState != DrvStats.nState);
	pFbbDrv->pChan[nChannel].nLastState = DrvStats.nState;

	DrvStats.nBuffers = 500;	// Why not ?
				
	if (bOk)	// Notify if any change
		ToHost(pFbbDrv, nChannel, DRV_STATS, (char *)&DrvStats, sizeof(DRVSTATS), NULL);

	if (nNbFrames)
		AskStats(pFbbDrv, nChannel);
		
	return TRUE;
}

static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	if (nLen <= 0 || nLen > 256)
		return FALSE;
	
	switch (nType)
	{
	case DRV_UNPROTO:
		if (pUi)
			SendDataUN(pFbbDrv, pUi, pData, nLen);
		break;
	case DRV_DATA:
		if (pFbbDrv->pChan[nChannel].nState == CONNECTED)
		{
			// Frame number.
			pFbbDrv->pChan[nChannel].nFrameNb = *((int *)pUi);

			SendDataTo(pFbbDrv, nChannel, pData, nLen);
			/* ++pFbbDrv->pChan[nChannel].nLastNbFrames;
			SendStats(pFbbDrv, nChannel, 0); */
			AskStats(pFbbDrv, nChannel);
		}
		break;
	case DRV_COMMAND:
		return AgwCommand(pFbbDrv, nChannel, pData);
	}
	
	return TRUE;
}

static void ToHost(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	
	if (nLen == -1)
		nLen = strlen(pData);
	
	if (nLen < 0 || nLen > 256)
		return;
	
	pPtr           = (DATAIN *)LocalAlloc( LMEM_FIXED, sizeof(DATAIN));
	pPtr->nChannel = nChannel;
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pUi      = pUi;
	pPtr->pNext    = NULL;
	
	if (nLen > 0)
	{
		pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
		memcpy(pPtr->pData, pData, nLen);
	}
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	if (pFbbDrv->pHostHead == NULL)
		pFbbDrv->pHostHead = pPtr;
	else
		pFbbDrv->pHostTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pHostTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	// Send Notification message
	if (pFbbDrv->hWnd)
		PostMessage(pFbbDrv->hWnd, WM_NOTIFY_MSG, nType, MAKELONG(pFbbDrv->nPort, nChannel));
}

static DATAIN *GetHost(FBBDRV *pFbbDrv)
{
	DATAIN *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pHostHead;
	if (pPtr)
		pFbbDrv->pHostHead = pFbbDrv->pHostHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static void HostInfo(FBBDRV *pFbbDrv, int nChannel, char *sFormat, ...)
{
	char	szStr[512];
	va_list pArgPtr;
	int nCnt;
	
	va_start(pArgPtr, sFormat);
	nCnt = wvsprintf(szStr, sFormat, pArgPtr);
	va_end(pArgPtr);
	
	if (nCnt > 0)
		ToHost(pFbbDrv, nChannel, DRV_INFO, szStr, nCnt, NULL);
}

// Decode the monitoring header and return its length
static int AgwGetUi(FBBDRV *pFbbDrv, char *pPtr, DRVUI *pUi)
{
	int nLen = 0;
	char *pScan;

	memset(pUi, 0, sizeof(DRVUI));

	// port is not used. Could be remapped in future evolutions
	while (*pPtr != ':')
	{
		++pPtr;
		++nLen;
	}
	++pPtr;
	++nLen;
	pUi->nPort = pFbbDrv->nPort;
	
	// Fm
	while (*pPtr != ' ')
	{
		++pPtr;
		++nLen;
	}
	++pPtr;
	++nLen;
	
	// fm Callsign
	pScan = pUi->szFrom;
	while (*pPtr != ' ')
	{
		*pScan++ = *pPtr++;
		++nLen;
	}
	*pScan = '\0';
	++pPtr;
	++nLen;

	// To
	while (*pPtr != ' ')
	{
		++pPtr;
		++nLen;
	}
	++pPtr;
	++nLen;
	
	// to callsign
	pScan = pUi->szTo;
	while (*pPtr != ' ')
	{
		*pScan++ = *pPtr++;
		++nLen;
	}
	*pScan = '\0';
	++pPtr;
	++nLen;
	
	if (*pPtr == 'V')
	{
		// Via
		while (*pPtr != ' ')
		{
			++pPtr;
			++nLen;
		}
		++pPtr;
		++nLen;
	
		pScan = pUi->szVia;
		while (*pPtr != '<')
		{
			if (*pPtr == ',')
			{
				*pScan++ = ' ';
				pPtr++;
			}
			else
				*pScan++ = *pPtr++;
			++nLen;
		}
		*pScan = '\0';
	}

	// ctrl
	++pPtr;	// skip <
	++nLen;
	
	pScan = pUi->szCtl;
	while (*pPtr != ' ')
	{
		*pScan++ = *pPtr++;
		++nLen;
	}
	*pScan = '\0';
	pUi->bUi = (strncmp("UI", pUi->szCtl, 2) == 0);
	++pPtr;
	++nLen;
	
	// pid
	while (*pPtr != '\r')
	{
		if (*pPtr == '=' && pUi->nPid == 0)
			sscanf(pPtr+1, "%x", &pUi->nPid);
		++pPtr;
		++nLen;
	}

	++pPtr;
	++nLen;

	return nLen;
}

///////////////////////////////////////////////////////////////////////////////
static int PrepareViaList(char *InCall, char *OutTo, char *OutVia)
{
	char *szToken;
	char szTmp[15];
	char HowVia=0;

	szToken = strtok(InCall,", ");
	if (OutTo)
	{
		if (szToken)
		{
			// To Call
			strcpy(OutTo, szToken);
		}
		szToken = strtok(NULL,", ");
	}
	
	for (;;)
	{
		// Digis
		if (szToken)
		{
			strcpy(szTmp, szToken);
			memmove(OutVia+1+(HowVia*10), szTmp, 10);
			HowVia++;
		} 
		else 
			break;
		szToken = strtok(NULL,", ");
	}//end for

	OutVia[0] = HowVia;
	return(HowVia);
}

///////////////////////////////////////////////////////////////////////////////
static BOOL ConnectTo(FBBDRV *pFbbDrv, int nChan, char *szRemoteCall)
{
	char szToCall[10]; //Destination Call
	char szViaCalls[1+10*8]; //Via Calls formatted
	char szStr[10];
	short int nCount;
	int nNbVia;

	if (pFbbDrv->pChan[nChan].nState != DISCONNECTED)
	{
		ToHost(pFbbDrv, nChan, DRV_COMMAND, "CHANNEL ALREADY CONNECTED", -1, NULL);
		return TRUE;
	}

	pFbbDrv->pChan[nChan].nState = CONNECT_PENDING;

	nNbVia = PrepareViaList(szRemoteCall, szToCall, szViaCalls);
	strcpy(pFbbDrv->pChan[nChan].szMyCall, pFbbDrv->szMyCall);
	strcpy(pFbbDrv->pChan[nChan].szCall, strupr(szToCall));

	if (nNbVia)
	{
		// Command with digis
		nCount = nNbVia*10 + 1;
		szStr[0] = 'V';
		memmove(szStr+1, szViaCalls, nCount);
		nCount++;
		memmove(szTemp+22, &nCount, 2);
		memmove(szTemp+24, szStr, nCount);
	}
	else
	{
		// Command without digi
		szStr[0] = 'C';
		szStr[1] = '\0';
		nCount = 1;
		memcpy(szTemp+22, &nCount, 2);
		memcpy(szTemp+24, szStr, nCount+1);
	}
	
	// Data
	szTemp[0] = 'S';
	szTemp[1] = pFbbDrv->nAgwPort;
	memcpy(szTemp+2, pFbbDrv->szMyCall, strlen(pFbbDrv->szMyCall)+1);
	memcpy(szTemp+12, szToCall, strlen(szToCall)+1);
	
	TXDATA(pFbbDrv, szTemp, nCount+24);

	return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
static void AskStats(FBBDRV *pFbbDrv, int nChan)
{
	char szStr[2];
	short int nCount;
	
	// Command
	szStr[0] = 'S';
	szStr[1] = '\0';
	nCount = 2;
	memmove(szTemp+22, &nCount, 2);
	memmove(szTemp+24, szStr, nCount);

	// Data
	szTemp[0] = 'S';
	szTemp[1] = pFbbDrv->nAgwPort;
	memcpy(szTemp+2, pFbbDrv->pChan[nChan].szMyCall, strlen(pFbbDrv->pChan[nChan].szMyCall)+1);
	memcpy(szTemp+12, pFbbDrv->pChan[nChan].szCall, strlen(pFbbDrv->pChan[nChan].szCall)+1);

	TXDATA(pFbbDrv, szTemp, nCount+24);

	pFbbDrv->nLastChan = nChan;
}

//////////////////////////////////////////////////////////////////////////////
static void SendDataUN(FBBDRV *pFbbDrv, DRVUI *pUi, char *szData, int nDataLen)
{
	short int nLen;
	short int nCount;
	int nNbVia;
	char szViaCalls[1+10*8]; //Via Calls formatted
	
	nNbVia = PrepareViaList(pUi->szVia, NULL, szViaCalls);

	if (nDataLen > 0)
		nCount = nDataLen;
	else
		nCount = strlen(szData);
		
	if (nNbVia)
	{
		// Command with digis
		szTemp[0] = 'V';
		nLen = nNbVia*10 + 1;
		memmove(szTemp+24, szViaCalls, nLen);
	}
	else
	{
		// Command without digi
		nLen = 0;
		szTemp[0] = 'M';
	}
	
	// Data
	szTemp[1] = pFbbDrv->nAgwPort;
	memcpy(szTemp+2, pFbbDrv->szMyCall, strlen(pFbbDrv->szMyCall)+1);
	memcpy(szTemp+12, pUi->szTo, strlen(pUi->szTo)+1);
	memmove(szTemp+22, &nCount, 2);
	memmove(szTemp+24+nLen, szData, nCount+1);

	TXDATA(pFbbDrv, szTemp, nCount+nLen+24);
}

//////////////////////////////////////////////////////////////////////////////
static void SendDataTo(FBBDRV *pFbbDrv, int nChan, char *szData, int nDataLen)
{
	short int nCount = nDataLen;

	szTemp[0] = 'D';
	szTemp[1] = pFbbDrv->nAgwPort;
	memcpy(szTemp+2, pFbbDrv->pChan[nChan].szMyCall, strlen(pFbbDrv->pChan[nChan].szMyCall)+1);
	memcpy(szTemp+12, pFbbDrv->pChan[nChan].szCall, strlen(pFbbDrv->pChan[nChan].szCall)+1);

	if (nCount == 0)
		nCount = strlen(szData);
	memcpy(szTemp+22, &nCount, 2);
	memcpy(szTemp+24, szData, nCount);

	TXDATA(pFbbDrv, szTemp, nCount+24);
}

///////////////////////////////////////////////////////////////////////////////
//
// This Function Get the Available RadioPorts from AGW Packet Engine
//
static void GetPorts(void)
{
	if (hConv[0])
	{
		HDDEDATA hData;
		static char szData[3010];
		char *szToken;

		hData = DdeClientTransaction(NULL, 0, hConv[0], hszItem[2],
						CF_TEXT, XTYP_REQUEST, 100000L, NULL);
		if (hData == NULL)
		{
			UINT z;
			z = DdeGetLastError(nIdInst);
			wsprintf(szData, "Error %d", z);
			MessageBox(NULL, szData, "AGWDLL32", MB_OK);
			return;
		}

		DdeGetData(hData,(LPBYTE)szData, 1999, 0);
		szData[1999] = '\0';
		szToken = strtok(szData, ";");
		Port.nNbPorts = atoi(szToken);
		Port.szName = (char **)LocalAlloc(LPTR, Port.nNbPorts * sizeof(char *));
		if (Port.szName)
		{
			for (int x = 0 ; x < Port.nNbPorts ; x++)
			{
				szToken = strtok(NULL, ";");
				if (szToken == NULL)
					break;
				Port.szName[x] = (char *)LocalAlloc(LMEM_FIXED, strlen(szToken)+1);
				if (Port.szName[x])
					memcpy(Port.szName[x], szToken, strlen(szToken)+1);
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
//
// This Function sends data to Packet Engine
//  The Format of the string has been done by the calling function
//
static void TXDATA(FBBDRV *pFbbDrv, char *str,int nCount)
{
	static int in = 0;

	++in;
	if (hConv[0])
	{
		if (DdeClientTransaction( (LPBYTE)str,(DWORD) nCount, hConv[0], hszItem[1],
							(UINT)CF_TEXT, XTYP_POKE, TIMEOUT_ASYNC ,NULL ) == NULL)
		{
			char szStr[80];
			UINT z = DdeGetLastError(nIdInst);
			wsprintf(szStr, "TxData error %d (%d) : \r\ncould not send %d bytes", z, in, nCount);
			MessageBox(NULL, szStr, "AGW driver", MB_ICONINFORMATION);
		}
	}
	--in;
}

//////////////////////////////////////////////////////////////////////////////
//
//  This Function Receives the Data from AGW Packet Engine
//   Is called from the callback function
//
//
static void ReceivedData( HDDEDATA hData )
{
/*
This function is called when the callback function is notified of
available data.
	*/
	short int nSize;
	int Result;
	int RadioPort;
	int nLen;
	char *szToken;
	char *szCmd;
	FBBDRV *pFbbDrv;
	static unsigned char szData[4000];
	DRVUI *pUi;

    if( hData != NULL )
    {
		Result = DdeGetData(hData, szData, sizeof(szData), 0 );
		if (Result<1)
			return;
		szData[3999]=NULL;
		//
		//we get here only 800bytes of data
		// AGW Packet Engine can send up to 3000bytes
		//this is possible if tcpip monitoring is done
		//but is rather unusual
		//
		if (Result>3999)
			Result=3999;
		
		memmove(&nSize, szData+2, 2);
		
		RadioPort = szData[0];
		pFbbDrv = SearchAgwPort(RadioPort);
		if (pFbbDrv == NULL)
			return;

		switch(szData[1]) // What kind of data is this??
		{
		case 'D'://DATA APO CONNECT
			ToHost(pFbbDrv, Call2Chan(pFbbDrv, (char *)szData+4), DRV_DATA, (char *)szData+15, nSize, NULL);
            break;
			
		case 'U'://MONITOR DATA UNPROTO
		case 'T'://TXDATA MONITOR DIKA MAS
		case 'S'://MONITOR HEADER
		case 'I'://MONITOR  HEADER+DATA CONNECT OTHER STATIONS
			pUi = (DRVUI *)LocalAlloc(LMEM_FIXED, sizeof(DRVUI));
			if (pUi)
			{
				nLen = AgwGetUi(pFbbDrv, (char *)szData+4, pUi);
				ToHost(pFbbDrv, 0, DRV_UNPROTO, (char *)szData+4+nLen, nSize-nLen, pUi);
			}
            break;

		case 'C'://CONNECT DISCONNECT
		
			//"*** DISCONNECTED RETRYOUT With %s\r",callfrom;
			//"*** CONNECTED With Station %s\r",CallFrom);
			//"*** DISCONNECTED From Station %s\r",CallFrom);

			//get the call
			szToken=strtok((char *)szData+4," ");	//***
			szCmd  =strtok(NULL," ");		//Connecetd
			szToken=strtok(NULL," ");		//to
			szToken=strtok(NULL," \r");	//station
			szToken=strtok(NULL," \r");	//SV2BBO
			if (szToken)
			{
				if (strstr(szCmd, "DISCONNECTED") != NULL)
					Disconnected(pFbbDrv, Call2Chan(pFbbDrv, szToken));
				else
					Connected(pFbbDrv, szToken);
			}
			break;
			
		case 'M'://MAXFRAME
			SendStats(pFbbDrv, pFbbDrv->nLastChan, nSize);
			break;
			
		case 'H'://MHeardList
			break;
			
		default: 
			return;
        }//END SWITCH
		//if callback call the callback now
    }
}

///////////////////////////////////////////////////////////////////////////////
static HDDEDATA CALLBACK CallBack( WORD wType, WORD t1, HCONV hConv1, HSZ hsz1, HSZ hsz2, HDDEDATA hData, DWORD t2, DWORD t3 )
{
    switch( wType )
    {
	case XTYP_ADVDATA :
		ReceivedData(hData);
		return (HDDEDATA)DDE_FACK;
	case XTYP_DISCONNECT :
		MessageBox( 0, "Disconnected From Packet Engine.\n Program Enters Idle Mode !", "AGW32.DLL", MB_ICONINFORMATION );
		if (hConv1 == hConv[0])
			hConv[0] = 0;
		else if (hConv1 == hConv[1])
			hConv[1] = 0;
		break;
    case XTYP_ERROR :
		MessageBox(0, "A critical DDE error has occured.","AGW32.DLL" , MB_ICONINFORMATION );
		break;
    default :
		break;
    }
    return NULL;
}

