
/******************************************************
 *
 * KCTT-T project.
 *
 * Jean-Paul ROUBELAT - F6FBB - jpr@f6fbb.org
 *
 * Version 0.1 14/03/99 : Initial release
 * Version 0.2 17/12/99 : Added fonctions in degrees
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************/

/******************************************************
 *                                                    *
 * tracker.c                                          *
 *                                                    *
 * Interface with the kct driver                      *
 *                                                    *
 ******************************************************/
 
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <linux/kctt.h>
#include <ctype.h>
#include <string.h>
#include "kctlib.h"

struct kctinfo {
	int OrigSite;
	int OrigAzim;
	int MinSite;
	int MaxSite;
	int MinAzim;
	int MaxAzim;
	int Radio1Type;
	int Radio1Addr;
	int Radio1Baud;
	int Radio1Bits;
	int Radio1Par;
	int Radio2Type;
	int Radio2Addr;
	int Radio2Baud;
	int Radio2Bits;
	int Radio2Par;
};

struct devlist {
	kctdev dev;
	struct kctinfo info;
	struct devlist * next;
};

struct devlist *devhead = NULL;


static struct kctinfo *KctInfo(kctdev dev)
{
	struct devlist *tmp;
	
	for (tmp = devhead ; tmp ; tmp = tmp->next)
	{
		if (tmp->dev == dev)
			return &(tmp->info);
	}
	
	return NULL;
}

static int KctBaud(int baud)
{
	switch(baud)
	{
	case 300:
		return KCTT_B300;
	case 600:
		return KCTT_B600;
	case 1200:
		return KCTT_B1200;
	case 2400:
		return KCTT_B2400;
	case 4800:
		return KCTT_B4800;
	case 9600:
		return KCTT_B9600;
	case 19200:
		return KCTT_B19200;
	}
	
	return 0;
}

static int KctBits(int bits)
{
	if (bits == 7)
		return KCTT_7BITS;
		
	return 0;
}

static int KctPar(int par)
{
	switch(par)
	{
	case 1:
		return KCTT_PARODD;
	case 2:
		return KCTT_PAREVEN;
	}
	
	return 0;
}

static int KctSetLine(kctdev dev, int rig)
{
	struct kctinfo *info = KctInfo(dev);
	unsigned char buf[4];
	int param = KCTT_2STOPS;

	if (info == NULL)
		return -1;
		
	switch (rig)
	{
	case KCTT_TRX1:
		param |= KctBaud(info->Radio1Baud);
		param |= KctBits(info->Radio1Bits);
		param |= KctPar(info->Radio1Par);
		break;
	case KCTT_TRX2:
		param |= KctBaud(info->Radio2Baud);
		param |= KctBits(info->Radio2Bits);
		param |= KctPar(info->Radio2Par);
		break;
	default:
		return -1;
	}
	 
	buf[0] = rig;
	buf[1] = KCTT_SETBAUD;
	buf[2] = 1;
	buf[3] = param;	
	return write(dev, buf, 4);
}

/* Frequency given in KHz */
int KctSetFrequency(kctdev dev, int rig, double freq)
{
	struct kctinfo *info = KctInfo(dev);
	int len = -1;
	char buf[80];
	int type;
	int addr;
	
	if (info == NULL)
		return -1;
		
	switch (rig)
	{
	case KCTT_TRX1:
		type = info->Radio1Type;
		addr = info->Radio1Addr;
		break;
	case KCTT_TRX2:
		type = info->Radio2Type;
		addr = info->Radio2Addr;
		break;
	default:
		return -1;
	}
	 
	switch (type)
	{
	case KCTT_ICOM :
		len = KctSetFrequencyIcom(buf+3, sizeof(buf) - 3, addr, freq);
		break;
	default :
		return -1;
	}
	
	if (len > 0)
	{
		buf[0] = rig;
		buf[1] = KCTT_SENDTRX;
		buf[2] = len;
#define DEBUG 0
#if DEBUG
		{
			int i;

			for (i = 0 ; i < len+3 ; i++)
				printf("%02x ", buf[i] & 0xff);
			printf("\n");
		}
#endif
		return write(dev, buf, len+3);
	}
	return -1;
}


/* returns the rig type from a string */
static int KctRadioType(char *RadioName)
{
	if (strcasecmp(RadioName, "icom") == 0)
		return KCTT_ICOM;
	if (strcasecmp(RadioName, "yaesu") == 0)
		return KCTT_YAESU;
	if (strcasecmp(RadioName, "kenwood") == 0)
		return KCTT_KENWOOD;
	return KCTT_UNKNOWN;
}

/* Cleans the beginning and the end of a string */
static char *strclean(char *str)
{
	int len;

	while (isspace(*str))
		++str;

	len = strlen(str);

	while ((len > 0) && (!isgraph(str[len-1])))
	{
		str[len-1] = '\0';
		--len;
	}

	return(str);
}

#define OTHER 0
#define CALIBRATION 1
#define RADIO 2

/* Read the calibration values */
static int KctReadDefaults(struct kctinfo *info)
{
	char str[256];
	char key[80];
	FILE *fptr;
	int block = OTHER;
		
	fptr = fopen(KCTT_CONF, "r");
	if (fptr == NULL)
		return 0;
		
	while (fgets(str, sizeof(str), fptr))
	{
		char *ptr;

		ptr = strclean(str);
		while (*ptr && !isgraph(*ptr))
			++ptr;
		
		if (*ptr == '\0' || *ptr == '#')
			continue;

		if (*ptr == '[')
		{
			if (sscanf(ptr+1, "%s", key) == 1)
			{
				int len;
				
				len = strlen(key);
				if (len > 1 && key[len-1] == ']')
					key[len-1] = '\0';

				if (strcasecmp(key, "calibration") == 0)
					block = CALIBRATION;
				else if (strcasecmp(key, "radio") == 0)
					block = RADIO;
				else
					block = OTHER;
			}
		}
		else
		{
			char *key, *value;
			int val = 0;
			
			switch(block)
			{
			case CALIBRATION :
				value = strchr(ptr, '=');
				if (value)
				{
					*value++ = '\0';
					val = atoi(strclean(value));
				}
			
				key = strclean(ptr);

				if (strcasecmp(key, "AZIMUTH_ORIGIN") == 0)
					info->OrigAzim = val;
				else if (strcasecmp(key, "AZIMUTH_MIN") == 0)
					info->MinAzim = val;
				else if (strcasecmp(key, "AZIMUTH_MAX") == 0)
					info->MaxAzim = val;
				else if (strcasecmp(key, "ELEVATION_ORIGIN") == 0)
					info->OrigSite = val;
				else if (strcasecmp(key, "ELEVATION_MIN") == 0)
					info->MinSite = val;
				else if (strcasecmp(key, "ELEVATION_MAX") == 0)
					info->MaxSite = val;
				break;

			case RADIO :
				value = strchr(ptr, '=');
				if (value)
				{
					*value++ = '\0';
					val = atoi(strclean(value));
				}
			
				key = strclean(ptr);

				if (strcasecmp(key, "RADIO1_TYPE") == 0)
					info->Radio1Type = KctRadioType(value);
				else if (strcasecmp(key, "RADIO1_ADDR") == 0)
					info->Radio1Addr = val;
				else if (strcasecmp(key, "RADIO1_BAUD") == 0)
					info->Radio1Baud = val;
				else if (strcasecmp(key, "RADIO1_BITS") == 0)
					info->Radio1Bits = val;
				else if (strcasecmp(key, "RADIO1_PAR") == 0)
					info->Radio1Par = val;
				else if (strcasecmp(key, "RADIO2_TYPE") == 0)
					info->Radio2Type = KctRadioType(value);
				else if (strcasecmp(key, "RADIO2_ADDR") == 0)
					info->Radio2Addr = val;
				else if (strcasecmp(key, "RADIO2_BAUD") == 0)
					info->Radio2Baud = val;
				else if (strcasecmp(key, "RADIO2_BITS") == 0)
					info->Radio2Bits = val;
				else if (strcasecmp(key, "RADIO2_PAR") == 0)
					info->Radio2Par = val;
				break;
			}
		}
	}

	return 1;
}

double KctGetSite(kctdev dev)
{
	struct kctinfo *info = KctInfo(dev);
	kctstat kstat;
	double site;
	int pos;
	
	if (info == NULL)
		return -1;

	KctGetStatus(dev, &kstat);
	
	pos = kstat.site_pos - info->MinSite;
	site = (double)pos * 180.0 / (double)(info->MaxSite - info->MinSite);

	site -= (double)info->OrigSite;
	if (site < 0.0)
		site += 180.0;

	return site;
}

int KctSetSite(kctdev dev, double site)
{
	struct kctinfo *info = KctInfo(dev);
	double val;
	int pos;
	
	if (info == NULL)
		return -1;

	if (site < 0.0 || site > 180.0)
		return -1;
		
	/* Convert angle to hex value */
	site += (double)info->OrigSite;
	if (site >= 180.0)
		site -= 180.0;
	
	val = (double)(info->MaxSite - info->MinSite) * site / 180.0;
	
	pos = (int)(val + .5) + info->MinSite;
	
	return KctSetPosition(dev, KCTT_SITE, pos);
}

double KctGetAzimuth(kctdev dev)
{
	struct kctinfo *info = KctInfo(dev);
	kctstat kstat;
	double azim;
	int pos;
	
	if (info == NULL)
		return -1;

	KctGetStatus(dev, &kstat);
	
	pos = kstat.azim_pos - info->MinAzim;
	azim = (double)pos * 360.0 / (double)(info->MaxAzim - info->MinAzim);
	azim -= (double)info->OrigAzim;
	if (azim < 0.0)
		azim += 360.0;

	return azim;
}

int KctSetAzimuth(kctdev dev, double azim)
{
	struct kctinfo *info = KctInfo(dev);
	double val;
	int pos;
	
	if (info == NULL)
		return -1;

	if (azim < 0.0 || azim > 360.0)
		return -1;
		
	/* Convert angle to hex value */
	azim += (double)info->OrigAzim;
	if (azim >= 360.0)
		azim -= 360.0;
	
	val = (double)(info->MaxAzim - info->MinAzim) * azim / 360.0;
	
	pos = (int)(val + .5) + info->MinAzim;
		
	return KctSetPosition(dev, KCTT_AZIMUTH, pos);
}

int KctSetSiteOrigin(kctdev dev, int orig, int min, int max)
{
	struct kctinfo *info = KctInfo(dev);
	
	if (info == NULL)
		return -1;
		
	info->OrigSite = orig;
	info->MinSite = min;
	info->MaxSite = max;
	
	return 0;
}

int KctSetAzimuthOrigin(kctdev dev, int orig, int min, int max)
{
	struct kctinfo *info = KctInfo(dev);
	
	if (info == NULL)
		return -1;
		
	info->OrigAzim = orig;
	info->MinAzim = min;
	info->MaxAzim = max;

	return 0;
}

kctdev KctOpen(int board)
{
	kctdev dev;
	char device[10];
	
	sprintf(device, "/dev/kctt%d", board);
	dev = open(device, O_RDWR);
	if (dev > 0)
	{
		/* add a new devlist structure */
		struct devlist *tmp;
		
		tmp = malloc(sizeof(struct devlist));
		if (tmp == NULL)
		{
			KctClose(dev);
			return -1;
		}
		
		tmp->dev = dev;

		tmp->info.OrigAzim = 180;
		tmp->info.MinAzim = 2;
		tmp->info.MaxAzim = 253;
		tmp->info.OrigSite = 0;
		tmp->info.MinSite = 2;
		tmp->info.MaxSite = 253;
		KctReadDefaults(&tmp->info);
		
		tmp->next = devhead;
		devhead = tmp;

		KctSetLine(dev, KCTT_TRX1);
		KctSetLine(dev, KCTT_TRX2);
	}
	
	return dev;
}

void KctClose(kctdev dev)
{
	struct devlist *tmp, *prev = NULL;
	
	if (dev != -1)
		close(dev);
		
	/* un-allocate the devlist structure */
	for (tmp = devhead ; tmp ; tmp = tmp->next)
	{
		if (tmp->dev == dev)
		{
			if (prev == NULL)
				devhead = tmp->next;
			else
				prev->next = tmp->next;
			free(tmp);
			return;
		}
		prev = tmp;
	}
}

int KctSetBrake(kctdev dev, int rotor, int tempo_rel, int tempo_set)
{
	int nb;
	unsigned char buf[4];

	tempo_rel = (tempo_rel + 99) / 100;
	buf[0] = rotor;
	buf[1] = KCTT_RELBRAKE;
	buf[2] = 1;
	buf[3] = tempo_rel;	
	nb = write(dev, buf, 4);
	if (nb != 4)
		return nb;
		
	tempo_set = (tempo_set + 99) / 100;
	buf[0] = rotor;
	buf[1] = KCTT_SETBRAKE;
	buf[2] = 1;
	buf[3] = tempo_set;	
	return write(dev, buf, 4);
}

int KctSetWindow(kctdev dev, int port, int val)
{
	unsigned char buf[4];

	if (val < 0 || val > 25)
		return -1;
		
	buf[0] = port;
	buf[1] = KCTT_SETSLEW;
	buf[2] = 1;
	buf[3] = val;	
	return write(dev, buf, 4);
}

int KctSetPosition(kctdev dev, int rotor, int pos)
{
	unsigned char buf[4];

	if (pos < 0 || pos > 255)
		return -1;
		
	buf[0] = rotor;
	buf[1] = 0;
	buf[2] = 1;
	buf[3] = (unsigned char)pos;

	return write(dev, buf, 4);
}

int KctGetStatus(kctdev dev, kctstat *stat)
{
	int ret;
	unsigned char buf[12];

	memset(stat, 0, sizeof(kctstat));
	
	ret = read(dev, buf, 12);
	if (ret != 12)
	{
		/* printf("read=%d control=%02x\n", ret, buf[3]); */
		return -1;
	}
	
	stat->control     = (int)buf[3] & 0xff;
	stat->site_pos    = (int)buf[4] & 0xff;
	stat->site_win    = (int)buf[5] & 0xff;
	stat->azim_pos    = (int)buf[8] & 0xff;
	stat->azim_win    = (int)buf[9] & 0xff;
	stat->site_brkset = ((int)buf[6] & 0xff) * 100;
	stat->site_brkrel = ((int)buf[7] & 0xff) * 100;
	stat->azim_brkset = ((int)buf[10] & 0xff) * 100;
	stat->azim_brkrel = ((int)buf[11] & 0xff) * 100;
	
	return 0xff & (int)buf[3];
}
