
/******************************************************
 *
 * KCTT-T project.
 * 
 * Jean-Paul ROUBELAT - F6FBB - jpr@f6fbb.org
 *
 * Version 0.1 14/03/99 : Initial release
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************/

/******************************************************
 *                                                    *
 * kctest.c                                           *
 *                                                    *
 * kct library test program                           *
 *                                                    *
 ******************************************************/
 
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include <syslog.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#ifdef __GLIBC__
#include <net/ethernet.h>
#else
#include <linux/if_ether.h>
#endif

#include "kctlib.h"

int verbose = 0;

#define CTRL_SOCK "/var/ax25/rotor/control"
#define ROTOR_CONF "/etc/ax25/rotor.conf"

typedef struct param {
	char *key;
	char *value;
	struct param *next;
} Param;

typedef struct satel {
	char *name;
	Param *param;
	struct satel *next;
} Satel;

Satel *sathead;

int reload = 0;
int close_request = 0;

void reload_signal(int dummy)
{
	reload = 1;
	signal(SIGHUP, reload_signal);
}

/* Cleans the beginning and the end of a string */
static char *strclean(char *str)
{
	int len;

	while (isspace(*str))
		++str;

	len = strlen(str);

	while ((len > 0) && (!isgraph(str[len-1])))
	{
		str[len-1] = '\0';
		--len;
	}

	return(str);
}

void display_conf(void)
{
	Satel *sat;
	Param *param;
	
	sat = sathead;
	while (sat)
	{
		syslog(LOG_INFO, "Satellite %s\n", sat->name);
		
		param = sat->param;
		while (param)
		{
			syslog(LOG_INFO, "  %s=%s\n", param->key, param->value);
			param = param->next;
		}
		sat = sat->next;		
	}
}

int read_conf(char *filename)
{
	FILE *fptr;
	char buf[256];
	char satellite[80];
	Satel *sat;
	Param *param;
	
	syslog(LOG_INFO, "open %s\n", filename);
	fptr = fopen(filename, "r");
	if (fptr == NULL)
	{
		syslog(LOG_ERR, "rotord : cannot open %s\n", filename);
		return 0;
	}
		
	/* free already allocated list */
	while (sathead)
	{
		sat = sathead;
		sathead = sat->next;
		
		while (sat->param)
		{
			param = sat->param;
			sat->param = param->next;
			free(param->key);
			free(param->value);
			free(param);
		}
		
		free(sat->name);
		free(sat);
	}
	
	sat = NULL;
	
	while (fgets(buf, sizeof(buf), fptr))
	{
		char *ptr;

		ptr = strclean(buf);
		while (*ptr && !isgraph(*ptr))
			++ptr;
		
		if (*ptr == '\0' || *ptr == '#')
			continue;
			
		if (*ptr == '[')
		{
			if (sscanf(ptr+1, "%s", satellite) == 1)
			{
				int len;
				
				len = strlen(satellite);
				if (len > 1 && satellite[len-1] == ']')
					satellite[len-1] = '\0';

				sat = calloc(sizeof(Satel), 1);
				sat->name = strdup(strclean(satellite));
				sat->next = sathead;
				sat->param = NULL;
			
				sathead = sat;
			}
		}
		else if (sat)
		{
			char *equal;
			
			equal = strchr(ptr, '=');
			if (equal)
			{
				*equal++ = '\0';
				
				param = calloc(sizeof(Param), 1);
				param->next = sat->param;

				param->key = strdup(strclean(ptr));
				param->value = strdup(strclean(equal));

				sat->param = param;
			}
		}
	}
		
	if (verbose)
		display_conf();
		
	return 1;
}

char *find_val(Satel *sat, char *key)
{
	Param *param = sat->param;
	while (param)
	{
		if (strcasecmp(param->key, key) == 0)
			return param->value;
		param = param->next;
	}
	
	return NULL;
}

Satel *find_sat(char *satname)
{
	Satel *sat = sathead;
	while (sat)
	{
		if (strcasecmp(satname, sat->name) == 0)
			return sat;
		sat = sat->next;		
	}
	
	return NULL;
}

void process(kctdev kct, char *buf)
{
	int dev, uplink;
	char *ptr;
	Satel *sat;
	double freq1, freq2, doppler;
	
	double azimuth, elevation, dop146;
	
	ptr = strtok(buf, "|");
	if (ptr == NULL)
		return;
	dev = atoi(ptr);
	
	ptr = strtok(NULL, "|");
	if (ptr == NULL)
		return;
	sat = find_sat(ptr);
	if (sat == NULL)
		return;
	
	ptr = strtok(NULL, "|");
	if (ptr == NULL)
		return;
	sscanf(ptr, "%lf", &azimuth);		
	
	ptr = strtok(NULL, "|");
	if (ptr == NULL)
		return;
	sscanf(ptr, "%lf", &elevation);		
			
	ptr = strtok(NULL, "|");
	if (ptr == NULL)
		return;
	sscanf(ptr, "%lf", &dop146);		
	
	/* freq1 direction up/dw */
	ptr = find_val(sat, "freq1_dir");
	if (ptr == NULL)
		return;
	uplink = (strcasecmp(ptr, "up") == 0);
	
	ptr = find_val(sat, "freq1");
	if (ptr == NULL)
		return;
	freq1 = atof(ptr) * 1000.;

	/* Compute doppler for freq1 */
	doppler = (freq1 / 146000) * (dop146 / 1000.);
	
	if (uplink)
		freq1 -= doppler;
	else
		freq1 += doppler;

	/* freq2 direction up/dw */
	ptr = find_val(sat, "freq2_dir");
	if (ptr == NULL)
		return;
	uplink = (strcasecmp(ptr, "up") == 0);
	
	ptr = find_val(sat, "freq2");
	if (ptr == NULL)
		return;
	freq2 = atof(ptr) * 1000.;
	
	/* Compute doppler for freq2 */
	doppler = (freq2 / 146000) * (dop146 / 1000.);
	
	if (uplink)
		freq2 -= doppler;
	else
		freq2 += doppler;
	
	if (verbose > 1)
		syslog(LOG_INFO, "name=%s az=%3.1f el=%3.1f 146=%1.2f 435=%1.2f\n", 
						sat->name, azimuth, elevation, freq2, freq1);
	
	if (kct != -1)
	{
		KctSetAzimuth(kct, azimuth);
		KctSetSite(kct, elevation);
		KctSetFrequency(kct, KCTT_TRX1, freq1);
		KctSetFrequency(kct, KCTT_TRX2, freq2); 
	}
}

void park(kctdev kct)
{
	Satel *sat;
	char *ptr;
	double azim, elev;
	
	sat = find_sat("park");
	if (sat == NULL)
	{
		KctClose(kct);
		kct = -1;
		if (verbose)
			syslog(LOG_INFO, "Kct closed\n");
		return;
	}
		
	ptr = find_val(sat, "azimuth");
	if (ptr == NULL)
		return;
	azim = atof(ptr);
		
	ptr = find_val(sat, "elevation");
	if (ptr == NULL)
		return;
	elev = atof(ptr);
	
	if (verbose)
		syslog(LOG_INFO, "Park  : az=%3.1f el=%3.1f\n", azim, elev);

	KctSetAzimuth(kct, azim);
	KctSetSite(kct, elev);

	/* Defer the driver close */
	close_request = 1;
}

int main(int argc, char **argv)
{
	int control_sock;
	int client_sock;
	int ctrl_len;
	int cnt;
	int parkok = 1;
	struct sockaddr_un ctrl_addr;
	kctdev kct;

	while ((cnt = getopt(argc, argv, "nv")) != EOF) {
		switch (cnt) {
		case 'n':
			parkok = 0;
			break;

		case 'v':
			++verbose;
			break;
         
		default:
			fprintf(stderr, "Usage: rotord [-n] [-v]\n");
			return 1;
		}
	}
		
	if (verbose)
	{
		openlog("rotord", LOG_PID, LOG_DAEMON);
		syslog(LOG_INFO, "starting");
	}
	
	read_conf(ROTOR_CONF);

	/* Create the rotor directory */
	{
		char dir[512];
		char *ptr;
		
		strcpy(dir, CTRL_SOCK);
		ptr = strrchr(dir, '/');
		if (ptr)
		{
			*ptr = '\0';
			mkdir(dir, 0755);
		}
	}
	
	if ((control_sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
		perror("rotord : socket");
		return 1;
	}
	
	unlink(CTRL_SOCK);

	ctrl_addr.sun_family = AF_UNIX;
	strcpy(ctrl_addr.sun_path, CTRL_SOCK);
	ctrl_len = sizeof(ctrl_addr.sun_family) + strlen(CTRL_SOCK);

	if (bind(control_sock, (struct sockaddr *)&ctrl_addr, ctrl_len) < 0)
	{
		perror("rotord : bind");
		return 2;
	}

	chmod(CTRL_SOCK, 0600);
	listen(control_sock, 1);
	
	signal(SIGUSR1, SIG_IGN);
	signal(SIGALRM, SIG_IGN);
	signal(SIGHUP,  reload_signal);
	
	client_sock = -1;
	kct = KctOpen(0);
	if (kct == -1)
	{
		perror("rotord : cannot open kctt");
		return 3;
	}

	switch (fork())
	{
	case 0 :
		if (verbose)
			syslog(LOG_INFO, "rotord becomes a daemon");
		break;
	case -1 :
		syslog(LOG_ERR, "rotord: cannot become a daemon");
		return 4;
	default :
		return 0;
	}

	/* Window of the rotor */
	KctSetWindow(kct, KCTT_AZIMUTH, 3);	 
	KctSetWindow(kct, KCTT_SITE, 3);	 
	if (parkok)
		park(kct);

	for (;;) 
	{
		int max_sock;
		fd_set fd_set;
		struct timeval to, *tv;
		
		FD_ZERO(&fd_set);
		FD_SET(control_sock, &fd_set);
		max_sock = control_sock;
		
		if (client_sock != -1)
		{
			if (client_sock > max_sock)
				max_sock = client_sock;
			FD_SET(client_sock, &fd_set);
		}

		if (close_request)
		{
			to.tv_sec = 60;
			to.tv_usec = 0;
			tv = &to;
		}
		else
		{
			tv = NULL;
		}
		
		select(max_sock+1, &fd_set, NULL, NULL, tv);
		
		if (reload)
		{
			read_conf(ROTOR_CONF);
			reload = 0;
			continue;
		}

		if (FD_ISSET(control_sock, &fd_set))
		{
			/* New connection */
			
			int new_sock;
			
		 	new_sock = accept(control_sock, (struct sockaddr *) &ctrl_addr, &ctrl_len);
			if (new_sock < 0)
			{
				perror("rotord : accept");
				exit(1);
			}
			
			if (client_sock == -1)
			{
				client_sock = new_sock;

				if (close_request)
					close_request = 0;
				else
					kct = KctOpen(0);
				if (kct != -1)
				{
					/* Window of the rotor */
					KctSetWindow(kct, KCTT_AZIMUTH, 3);	 
					KctSetWindow(kct, KCTT_SITE, 3);	 

					if (verbose)
						syslog(LOG_INFO, "Kct initialized");
				}
				else
					syslog(LOG_ERR, "Cannot initialize kct");
			}
			else
			{
				/* Only one connection allowed */
				close (new_sock);
			}
		}

		else if (client_sock != -1 && FD_ISSET(client_sock, &fd_set))
		{
			int nb;
			char buf[256];
			
			/* Read the client request */
			nb = read(client_sock, buf, sizeof(buf));
			if (nb > 0)
			{
				buf[nb] = '\0';
				process(kct, buf);
			}
			else 
			{
				/* Client disconnection */
				close(client_sock);
				client_sock = -1;
				if (parkok)
					park(kct);
			}
		}
		
		else if (close_request)
		{
			if (verbose)
				syslog(LOG_INFO, "Kct closed\n");
			KctClose(kct);
			close_request = 0;
			kct = -1;
		}
				
	}
	
	return 0;	/* Should never occur ! */
}
