/*
 * Kansas city tracker - tuner driver
 *
 * Driver for antenna rotors interface and transceiver frequency tuning.
 *
 * Copyright (C) 1999 by Jean-Paul ROUBELAT - F6FBB - jpr@f6fbb.org
 */

#include <linux/module.h>

#include <linux/config.h>
#include <linux/types.h>
#include <linux/major.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/fcntl.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/delay.h>

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>

#include <linux/kct.h>

/* No debugging trace */
/* #define KCT_DEBUG 1 */

#define KCT_BANNER "F6FBB Kansas-City-Tracker/Tuner Driver version 0.1 (Linux 2.0)\n"

/* Max number of boards */
#define KCT_NO		4

/* Max number of motors */
#define KCT_ROTOR	2

/* Address window */
#define KCT_SIZE	4

/* Period polling to read rotors = 50ms */
#define KCT_POLL	50

/* Max time rotor on = 2 minutes */
#define KCT_MAX		120000

/* flag bits */
#define KCT_EXIST	0x0001
#define KCT_BUSY	0x0002

/* registers */
#define KCT_STA		0
#define KCT_CTL		0
#define KCT_AD		1
#define KCT_UARTSEL	1
#define KCT_UARTA	2
#define KCT_UARTB	3

/* Bits of the status register */
#define KCT_STA_ADRDY	0x01
#define KCT_STA_ADSEL	0x02

/* Bits of the uartsel register */
#define KCT_UART_A1		0x01
#define KCT_UART_A0		0x02

/* Bits of the control register */
#define KCT_CTL_BRAKE	0x80
#define KCT_CTL_ADSEL	0x40
#define KCT_CTL_ADLATCH	0x20
#define KCT_CTL_ADSTART	0x10
#define KCT_CTL_LEFT	0x08
#define KCT_CTL_RIGHT	0x04
#define KCT_CTL_DOWN	0x02
#define KCT_CTL_UP		0x01

/* Bits of the uart register */
#define KCT_TXRDY		0x01
#define KCT_RXRDY		0x02

/* Macros */
#define SETBIT(val, bit) (val) |= (bit)
#define CLRBIT(val, bit) (val) &= (~(bit))

/* State machine */
#define KCT_STATE_IDLE	1
#define KCT_STATE_BRAKEREL	2
#define KCT_STATE_INCREMENT	3
#define KCT_STATE_DECREMENT	4
#define KCT_STATE_BRAKESET	5

typedef struct {
	int request;
	int delay;
	int watchdog;
	int brakerel;	/* multiples of 100ms */
	int brakeset;	/* multiples of 100ms */
	int position;	/* 0..255 */
	int slew;		/* 0..25 */
	int req_pos;	/* 0..255 */
	int state;
} Motor;

typedef struct {
	int base;
	int flags;
	unsigned char baud[2];
	unsigned char control;		/* control register */
	unsigned char clics;		/* select register */
	Motor motor[KCT_ROTOR];
	struct timer_list pos_t;
} Kct;

Kct kct[KCT_NO] = 
{
	{ 0x3e0,0,{0x07,0x07}},
	{ 0x3e4,0,{0x07,0x07}},
	{ 0x3e8,0,{0x07,0x07}},
	{ 0x3ec,0,{0x07,0x07}},
};

#define SETBIT(val, bit) (val) |= (bit)
#define CLRBIT(val, bit) (val) &= (~(bit))

/* rotor functions */
/*=================*/

static void pos_timer(unsigned long);

static __inline__ void set_pos_timer(Kct *kct)
{
	del_timer(&kct->pos_t);
	kct->pos_t.expires = jiffies + HZ * KCT_POLL / 1000;	/* 50 ms */
	kct->pos_t.function = pos_timer;
	kct->pos_t.data = (unsigned long) kct;
	add_timer(&kct->pos_t);
}

void set_brake(Kct *kct, int rotor, int val)
{
	static int brake = -1;
	
	if (rotor != KCT_AZIMUTH)
		return;
	
	if (val == brake)
		return;
		
	brake = val;
		
	if (val)
		SETBIT(kct->control, KCT_CTL_BRAKE);
	else
		CLRBIT(kct->control, KCT_CTL_BRAKE);
	outb(kct->control, kct->base+KCT_CTL);
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : Set brake=%u\n", val);
#endif
}

void stop_rotor(Kct *kct, int rotor)
{
	switch (rotor)
	{
	case KCT_SITE :
		CLRBIT(kct->control, KCT_CTL_UP|KCT_CTL_DOWN);
		break;
	case KCT_AZIMUTH :
		CLRBIT(kct->control, KCT_CTL_LEFT|KCT_CTL_RIGHT);
		break;
	default:
		return;
	}
	outb(kct->control, kct->base+KCT_CTL);
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : Stop rotor=%d position=%u\n", rotor, kct->motor[rotor].position);
#endif
}

void inc_rotor(Kct *kct, int rotor)
{
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : inc rotor=%d\n", rotor);
#endif
	switch (rotor)
	{
	case KCT_SITE :
		CLRBIT(kct->control, KCT_CTL_DOWN);
		SETBIT(kct->control, KCT_CTL_UP);
		break;
	case KCT_AZIMUTH :
		CLRBIT(kct->control, KCT_CTL_LEFT);
		SETBIT(kct->control, KCT_CTL_RIGHT);
		break;
	default:
		return;
	}
	outb(kct->control, kct->base+KCT_CTL);
}

void dec_rotor(Kct *kct, int rotor)
{
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : dec rotor=%d\n", rotor);
#endif
	switch (rotor)
	{
	case KCT_SITE :
		CLRBIT(kct->control, KCT_CTL_UP);
		SETBIT(kct->control, KCT_CTL_DOWN);
		break;
	case KCT_AZIMUTH :
		CLRBIT(kct->control, KCT_CTL_RIGHT);
		SETBIT(kct->control, KCT_CTL_LEFT);
		break;
	default:
		return;
	}
	outb(kct->control, kct->base+KCT_CTL);
}

static void start_conversion(Kct *kct, int rotor)
{
	/* Start conversion of the specified rotor */
	
	switch (rotor)
	{
	case KCT_SITE:
		CLRBIT(kct->control, KCT_CTL_ADSEL);
		break;
	case KCT_AZIMUTH:
		SETBIT(kct->control, KCT_CTL_ADSEL);
		break;
	}		
	outb(kct->control, kct->base+KCT_CTL);
	
	/* Latch select bit */
	SETBIT(kct->control, KCT_CTL_ADLATCH);
	outb(kct->control, kct->base+KCT_CTL);

	/* Start conversion */
	SETBIT(kct->control, KCT_CTL_ADSTART);
	outb(kct->control, kct->base+KCT_CTL);

	/* End request */
	CLRBIT(kct->control, KCT_CTL_ADLATCH|KCT_CTL_ADSTART);
	outb(kct->control, kct->base+KCT_CTL);

	/* Restart timer */
	if (kct->flags & KCT_BUSY)
		set_pos_timer(kct);
}

/* timer function : read the selected AD converter manage motor */
static void pos_timer(unsigned long kctptr)
{
	int nb;
	int status;
	int rotor;
	int position;
	Kct *kct = (Kct *)kctptr;
	Motor *motor;
	
	/* Read if the conversion results is available */
	status = inb(kct->base+KCT_STA);
	
	/* If conversion finished */
	if (status & KCT_STA_ADRDY)
		position = inb(kct->base+KCT_AD) & 0xff;
	else
	{
		/* Should never occur... Ask one more timer delay */
		set_pos_timer(kct);
		return;
	}
	
	/* Get the selected AD */
	if (status & KCT_STA_ADSEL)
		rotor = KCT_AZIMUTH;
	else
		rotor = KCT_SITE;

	motor = &kct->motor[rotor];
	motor->position = position;

	switch (motor->state)
	{
	
	case KCT_STATE_IDLE:
		if ((motor->request && (motor->req_pos > (motor->position + motor->slew))) ||
			(motor->request && (motor->req_pos < (motor->position - motor->slew))))
		{
			/* First release the brake */
			set_brake(kct, rotor, 1);
			
			motor->delay = 0;
			motor->state = KCT_STATE_BRAKEREL;
#ifdef KCT_DEBUG
			printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
		}
		else
		{
			/* Set brake on */
			set_brake(kct, rotor, 0);	
			motor->request = 0;
			motor->watchdog = 0;
		}
		break;
		
	case KCT_STATE_BRAKEREL:
		if (motor->delay >= motor->brakerel)
		{
			/* end of delay, start the motor in the good direction */
			if (motor->request && (motor->req_pos > (motor->position + motor->slew)))
			{
				motor->state = KCT_STATE_INCREMENT;
#ifdef KCT_DEBUG
				printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
				inc_rotor(kct, rotor);
			}
			else if (motor->request && (motor->req_pos < (motor->position - motor->slew)))
			{
				motor->state = KCT_STATE_DECREMENT;
#ifdef KCT_DEBUG
				printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
				dec_rotor(kct, rotor);
			}
			else
			{
				/* Motor does not have to change the position */
				motor->state = KCT_STATE_BRAKESET;
#ifdef KCT_DEBUG
				printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
			}
		}			
		break;
		
	case KCT_STATE_INCREMENT:
		/* motor is running and incrementing the AD */
		if (motor->position >= motor->req_pos)
		{
			/* Position is reached */
			motor->delay = 0;
			motor->request = 0;
			stop_rotor(kct, rotor);
			motor->state = KCT_STATE_BRAKESET;
#ifdef KCT_DEBUG
			printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
		}
		else
		{
			if (motor->watchdog > KCT_MAX)
			{
				/* More than 2 minutes on ! Stop the rotor */
				printk(KERN_INFO "kct : watchdog : stopping motor %d\n", rotor);
				motor->delay = 0;
				motor->request = 0;
				stop_rotor(kct, rotor);
				motor->state = KCT_STATE_BRAKESET;
#ifdef KCT_DEBUG
				printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
			}
		}
		break;
		
	case KCT_STATE_DECREMENT:
		/* motor is running and decrementing the AD */
		if (motor->position <= motor->req_pos)
		{
			/* Position is reached */
			motor->delay = 0;
			motor->request = 0;
			stop_rotor(kct, rotor);
			motor->state = KCT_STATE_BRAKESET;
#ifdef KCT_DEBUG
			printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
		}
		else
		{
			if (motor->watchdog > KCT_MAX)
			{
				/* More than 2 minutes on ! Stop the rotor */
				printk(KERN_INFO "kct : watchdog : stopping motor %d\n", rotor);
				motor->delay = 0;
				motor->request = 0;
				stop_rotor(kct, rotor);
				motor->state = KCT_STATE_BRAKESET;
#ifdef KCT_DEBUG
				printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
			}
		}
		break;
		
	case KCT_STATE_BRAKESET:
		if (motor->request)
		{
			/* Request during brake set. Abort brake set... */
			motor->state = KCT_STATE_BRAKEREL;
		}
		else
		{
			/* Set the brake after delay */
			if (motor->delay >= motor->brakeset)
			{
				/* out of delay, return to idle state */
				motor->state = KCT_STATE_IDLE;
#ifdef KCT_DEBUG
				printk(KERN_INFO "kct : state %d\n", motor->state);
#endif
			}
		}
		break;
		
	default:
		printk(KERN_ERR "Unknow state %u in kct : pos_timer\n", motor->state);
		break;
		
	}

	for (nb = 0 ; nb < KCT_ROTOR ; nb++)
	{	
		motor->delay += KCT_POLL;
		motor->watchdog += KCT_POLL;
	}
	
	/* Start conversion of next AD */
	if (++rotor == KCT_ROTOR)
		rotor = 0;
	start_conversion(kct, rotor);
}

static int rotor_request(Kct *kct, int rotor, int command, const char *buf, int len)
{
	if (len != 1)
		return -EINVAL;
		
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : rotor_request rotor=%d position=%u\n", rotor, (unsigned char)buf[0]);
#endif
	switch (command)
	{
	case KCT_SETPOS :
		kct->motor[rotor].req_pos = buf[0] & 0xff;
		kct->motor[rotor].request = 1;
		break;
	case KCT_SETSLEW :
		kct->motor[rotor].slew = buf[0] & 0xff;
		break;
	case KCT_SETBRAKE :
		kct->motor[rotor].brakeset = 100 * (buf[0] & 0xff);	/* n x 100ms */
		break;
	case KCT_RELBRAKE :
		kct->motor[rotor].brakerel = 100 * (buf[0] & 0xff);	/* n x 100ms */
		break;
	default:
		return -EINVAL;
	}
	return len + 3;
}

/* serial ports functions */
/*========================*/
static void uart_init(Kct *kct, int port)
{
	unsigned char val;
	int uart;
	int uart_nb;

	switch (port)
	{
	case KCT_TRX1:
		uart = kct->base + KCT_UARTA;
		uart_nb = 0;
		break;
	case KCT_TRX2:
		uart = kct->base + KCT_UARTB;
		uart_nb = 1;
		break;
	default:
		return;
	}
	
	/* Select command register */
	SETBIT(kct->clics, KCT_UART_A0);
	SETBIT(kct->clics, KCT_UART_A1);
	outb(kct->clics, kct->base+KCT_UARTSEL);
	
	/* Read command register to reset the mode register pointer */
	inb(uart);

	/* Select mode register */
	CLRBIT(kct->clics, KCT_UART_A0);
	SETBIT(kct->clics, KCT_UART_A1);
	outb(kct->clics, kct->base+KCT_UARTSEL);
	
	/* Set the line parameters */
	val = 0x02;
	if (kct->baud[uart_nb] & KCT_2STOPS)
		val |= 0xc0;
	else
		val |= 0x40;
	if (kct->baud[uart_nb] & KCT_PARODD)
		val |= 0x10;
	if (kct->baud[uart_nb] & KCT_PAREVEN)
		val |= 0x30;
	if (kct->baud[uart_nb] & KCT_7BITS)
		val |= 0x08;
	else
		val |= 0x0c;
	outb(val, uart);

#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : port=%d addr=%02x baud=%02x\n", port, (unsigned char)kct->clics, val);
#endif

	/* Set BaudRate */
	val = (kct->baud[uart_nb] & 0xf) | 0x30;
	outb(val, uart);

#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : port=%d addr=%02x baud=%02x\n", port, (unsigned char)kct->clics, val);
#endif

	/* Select command register */
	SETBIT(kct->clics, KCT_UART_A0);
	SETBIT(kct->clics, KCT_UART_A1);
	outb(kct->clics, kct->base+KCT_UARTSEL);
	
	/* Set Normal, Async, Tx and Rx enabled */
	outb(0x05, uart);
}

static int trx_request(Kct *kct, int port, int command, const char *buf, int len)
{
	int count;
	int uart;
	int uart_nb;
	
	switch (port)
	{
	case KCT_TRX1:
		uart = kct->base + KCT_UARTA;
		uart_nb = 0;
		break;
	case KCT_TRX2:
		uart = kct->base + KCT_UARTB;
		uart_nb = 1;
		break;
	default:
		return -EINVAL;
	}
	
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : trx_request : write %d bytes to port %d\n", len, port);
#endif
	
	switch(command)
	{
	case KCT_SENDTRX:
		for (count = 0 ; count < len ; count++) 
		{
			/* Select status register */
			SETBIT(kct->clics, KCT_UART_A0);
			CLRBIT(kct->clics, KCT_UART_A1);
			outb(kct->clics, kct->base+KCT_UARTSEL);
	
			/* wait for register empty */
			while ((inb(uart) & KCT_TXRDY) == 0)
				schedule();

			/* Select write holding register */
			CLRBIT(kct->clics, KCT_UART_A0);
			CLRBIT(kct->clics, KCT_UART_A1);
			outb(kct->clics, kct->base+KCT_UARTSEL);

			/* write the byte to the UART */
			outb(buf[count], uart);
		}
	
#ifdef KCT_DEBUG
		printk(KERN_INFO "kct : trx_request : done\n");
#endif
		break;
		
	case KCT_SETBAUD:
		if (len != 1)
			return -EINVAL;
		kct->baud[uart_nb] = buf[0];
		uart_init(kct, port);
		break;

	default:
		return -EINVAL;
	}
	
	return len + 3;
}

/* clics ports functions */
/*========================*/
static int clic_request(Kct *kct, const char *buf, int len)
{
	if (len != 1)
		return -EINVAL;
	kct->clics &= 0xc3;
	kct->clics |= ((buf[0] & 0xf) << 2);
	outb(kct->clics, kct->base+KCT_UARTSEL);
	return len + 3;
}

/* kernel functions */
/*==================*/

static int kct_read(struct inode * inode, struct file * file, char * buf, int count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	unsigned char control;
	int retval;
	char lbuf[14];

	/* Needs at least 12 characters in the buffer */
	if (count < 12)
		return -EINVAL;
	
	retval = verify_area(VERIFY_WRITE, buf, count);
	if (retval)
		return retval;

	/* Return the status */
	if ((kct[minor].motor[KCT_SITE].position != -1) &&
		(kct[minor].motor[KCT_SITE].position != -1))
		retval = 12;
	else
		retval = 4;

	control = kct[minor].control & 0x8f;
	if (kct[minor].motor[KCT_SITE].request)
		control |= 0x10;
	if (kct[minor].motor[KCT_AZIMUTH].request)
		control |= 0x20;

	lbuf[0]  = 0;
	lbuf[1]  = 0;
	lbuf[2]  = retval - 3;
	lbuf[3]  = control;
	lbuf[4]  = kct[minor].motor[KCT_SITE].position;
	lbuf[5]  = kct[minor].motor[KCT_SITE].slew;
	lbuf[6]  = kct[minor].motor[KCT_SITE].brakerel / 100;
	lbuf[7]  = kct[minor].motor[KCT_SITE].brakeset / 100;
	lbuf[8]  = kct[minor].motor[KCT_AZIMUTH].position;
	lbuf[9]  = kct[minor].motor[KCT_AZIMUTH].slew;
	lbuf[10] = kct[minor].motor[KCT_AZIMUTH].brakerel / 100;
	lbuf[11] = kct[minor].motor[KCT_AZIMUTH].brakeset / 100;
	
	memcpy_tofs(buf, lbuf, 12);

	/* Return the status */
	return retval;
}

static int kct_write(struct inode * inode, struct file * file, const char * buf, int count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	unsigned int len;
	int retval;
	int port;
	int command;
	char lbuf[20];

	if (count < 3 || count > 20)
		return -EINVAL;

	retval = verify_area(VERIFY_READ, buf, count);
	if (retval)
		return retval;

	memcpy_fromfs(lbuf, buf, count);

	len = 0xff & (unsigned int)lbuf[2];
	if (count != (len + 3))
		return -EINVAL;
		
	port = lbuf[0];
	command = lbuf[1];
	
#ifdef KCT_DEBUG
	printk(KERN_INFO "kct : write minor=%d port=%d command=%d len=%d\n", minor, port, command, len);
#endif
	switch (port)
	{
	case KCT_SITE :
	case KCT_AZIMUTH :
		return rotor_request(&kct[minor], port, command, lbuf+3, len);
	case KCT_TRX1 :
	case KCT_TRX2 :
		return trx_request(&kct[minor], port, command, lbuf+3, len);
	case KCT_CLICS :
		return clic_request(&kct[minor], lbuf+3, len);
	}

	return -EINVAL;
}

static int kct_open(struct inode * inode, struct file * file)
{
	unsigned int minor = MINOR(inode->i_rdev);

	if (minor >= KCT_NO)
		return -ENXIO;
	if ((kct[minor].flags & KCT_EXIST) == 0)
		return -ENXIO;
	if (kct[minor].flags & KCT_BUSY)
		return -EBUSY;

	MOD_INC_USE_COUNT;

	kct[minor].flags |= KCT_BUSY;
	
	/* Init UARTs */
	uart_init(&kct[minor], KCT_TRX1);
	uart_init(&kct[minor], KCT_TRX2);
	
	memset(&kct[minor].motor[KCT_SITE], 0 , sizeof(Motor));
	memset(&kct[minor].motor[KCT_AZIMUTH], 0 , sizeof(Motor));

	kct[minor].motor[KCT_SITE].position = -1;
	kct[minor].motor[KCT_AZIMUTH].position = -1;

	kct[minor].motor[KCT_SITE].state = KCT_STATE_IDLE;	
	kct[minor].motor[KCT_AZIMUTH].state = KCT_STATE_IDLE;	

	/* Start rotor polling timer */
	start_conversion(&kct[minor], KCT_SITE);
	
	return 0;
}

static void kct_release(struct inode * inode, struct file * file)
{
	unsigned int minor = MINOR(inode->i_rdev);

	kct[minor].flags &= ~KCT_BUSY;
	stop_rotor(&kct[minor], KCT_SITE);
	stop_rotor(&kct[minor], KCT_AZIMUTH);
	del_timer(&kct->pos_t);
	if (kct->motor[KCT_AZIMUTH].state != KCT_STATE_IDLE)
		set_brake(kct, KCT_AZIMUTH, 0);

	MOD_DEC_USE_COUNT;
}


static struct file_operations kct_fops = {
	NULL,		/* seek */
	kct_read,	/* read */
	kct_write,	/* write */
	NULL,		/* readdir */
	NULL,		/* select */
	NULL,		/* ioctl */
	NULL,		/* mmap */
	kct_open,	/* open */
	kct_release	/* release */
};

static int kct_probe(int minor)
{
	int base, size;
	unsigned int testvalue;

	size = KCT_SIZE;
	base = kct[minor].base;
	
	kct[minor].flags = 0;
	kct[minor].clics = 0;
	
	if (check_region(base, size) < 0)
		return -1;
		
	/* write a AD select and read back to check */
	outb_p(KCT_CTL_ADSEL, base);
	testvalue = inb_p(base);
	if ((testvalue & KCT_STA_ADSEL) == KCT_STA_ADSEL) 
	{
		outb_p(0, base);
		testvalue = inb_p(base);
		if ((testvalue & KCT_STA_ADSEL) == 0) 
		{
			kct[minor].flags |= KCT_EXIST;
			printk(KERN_INFO "kct%d at 0x%04x\n", minor, base);
			request_region(base, size, "kct");
			return 1;
		}
	}
	
	return 0;
}

#ifdef MODULE
#define kct_init init_module
#endif

int kct_init(void)
{
	int minor;
	int count = 0;

	printk(KERN_INFO KCT_BANNER);

	if (register_chrdev(KCT_MAJOR, "kct", &kct_fops)) 
	{
		printk("kct: unable to get major %d\n", KCT_MAJOR);
		return -EIO;
	}

	for (minor = 0 ; minor < KCT_NO ; minor++) 
	{
		if (kct_probe(minor) <= 0) 
			continue;
		++count;
	}

	if (count == 0) 
	{
		unregister_chrdev(KCT_MAJOR,"kct");
		printk(KERN_INFO "kct : no device found\n");
		return -EIO;
	}
	
	return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
	int minor;

	del_timer(&kct->pos_t);
	unregister_chrdev(KCT_MAJOR, "kct");
	for (minor = 0; minor < KCT_NO; minor++) 
		if (kct[minor].flags & KCT_EXIST)
			release_region(kct[minor].base, KCT_SIZE);
}
#endif
