
/*
 *	config.c
 *
 *	command line parsing, configuration file reading and command lists
 */
 
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>

#include "config.h"
#include "ctypes.h"
#include "cfgfile.h"
#include "cfg_iface.h"
#include "cfg_links.h"
#include "log.h"
#include "csock.h"
#include "cstring.h"
#include "hmalloc.h"

#define HELP_STRING "Usage: clussed [-b <base directory>]"

/*
 *	Configuration file commands
 */

#define _CFUNC_ (int (*)(void *dest, int argc, char **argv))

static struct cfgcmd cfg_cmds[] = {
	{ "hostname",		_CFUNC_ do_string,		&hostname		},
	{ "callsign",		_CFUNC_ do_callsign,		&clucall		},
	{ "qth",		_CFUNC_ do_string,		&qth			},
	{ "qth_loc",		_CFUNC_ do_string,		&qth_loc		},
	{ "sysopcall",		_CFUNC_ do_callsign,		&sysop_call		},
	{ "sysopname",		_CFUNC_ do_string,		&sysop_name		},
	{ "loglevel",		_CFUNC_ do_int,			&log_level		},
	{ "logdest",		_CFUNC_ do_int,			&log_dest		},
#ifdef HAVE_INET
	{ "tcplisten",		_CFUNC_ do_tcplisten,		&listenq		},
#endif
#ifdef HAVE_AX25
	{ "ax25listen",		_CFUNC_ do_ax25listen,		&listenq		},
#endif
#ifdef HAVE_NETROM
	{ "netromlisten",	_CFUNC_ do_netromlisten,	&listenq		},
#endif
	{ NULL,			NULL,				NULL			}
};

static struct cfgcmd links_cmds[] = {
	{ "link",		_CFUNC_ do_link,		&thislink		},
	{ "destcall",		_CFUNC_ do_destcall,		&thislink		},
	{ "mycall",		_CFUNC_ do_mycall,		&thislink		},
	{ "protocol",		_CFUNC_ do_protocol,		&thislink		},
	{ "script",		_CFUNC_ do_script,		&thislink		},
	{ "mode",		_CFUNC_ do_mode,		&thislink		},
	{ NULL,			NULL,				NULL			}
};

char *base_d = NULL;

char *etc_d = NULL;
char *config_f = NULL;
char *intconfig_f = NULL;
char *links_f = NULL;

char *data_d = NULL;
char *luser_f = NULL;
char *nuser_f = NULL;

call_t *clucall = NULL;
char *hostname = NULL;

call_t *sysop_call = NULL;
char *sysop_name = NULL;
char *qth = NULL;
char *qth_loc = NULL;

/*
 *	Parse command line arguments and derive directory and file names
 */

void parse_cmdl(int argc, char **argv)
{
	char s[PATHLEN+1];
	char c;
	
	s[PATHLEN] = '\0';
	
	while ((c = getopt(argc, argv, "b:")) != -1) {
	switch (c) {
		case 'b':
			base_d = hstrdup(optarg);
			break;;
		default:
			fprintf(stderr, "%s <%s@%s>\n%s\n", SOFT_STRING, COMPILE_BY, COMPILE_HOST, HELP_STRING);
			exit(1);
	}
	}
	
	if (base_d) {
		if (base_d[strlen(base_d)-1] != '/') {
			base_d = hrealloc(base_d, strlen(base_d) + 1);
			strcat(base_d, "/");
		}
	} else {
		snprintf(s, PATHLEN, "./");
		base_d = hstrdup(s);
	}
	
	if (!etc_d) {
		snprintf(s, PATHLEN, "%s%s", base_d, ETC_D_N);
		etc_d = hstrdup(s);
	}
	
	if (!data_d) {
		snprintf(s, PATHLEN, "%s%s", base_d, DATA_D_N);
		data_d = hstrdup(s);
	}
	
	snprintf(s, PATHLEN, "%s%s", etc_d, CONFIG_F_N);
	config_f = hstrdup(s);
	snprintf(s, PATHLEN, "%s%s", etc_d, INTCONFIG_F_N);
	intconfig_f = hstrdup(s);
	snprintf(s, PATHLEN, "%s%s", etc_d, LINKS_F_N);
	links_f = hstrdup(s);
	
	snprintf(s, PATHLEN, "%s%s", data_d, LUSER_F_N);
	luser_f = hstrdup(s);
	snprintf(s, PATHLEN, "%s%s", data_d, NUSER_F_N);
	nuser_f = hstrdup(s);
}

/*
 *	Read configuration files
 */

void read_config(void)
{
	int i;
	
	if (read_cfgfile(config_f, cfg_cmds))
		exit(1);
	if ((i = read_cfgfile(intconfig_f, cfg_cmds))) {
		if (i == -2)
			exit(0);
		log(L_ERR, "Continuing with no interfaces configured.");
	}
	
	if (!( (hostname) && (clucall) && (sysop_call) && (sysop_name)
	   && (qth) )) {
	   	log(L_CRIT, "Missing entries in configuration file!");
	   	exit(1);
	}
	hstrupr((char *)clucall);
	
	if ((i = read_cfgfile(links_f, links_cmds))) {
		if (i == -2)
			exit(0);
		log(L_ERR, "Continuing with no links configured.");
	}
}

