
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "cstring.h"
#include "hmalloc.h"
#include "ctypes.h"

/* character sets */
struct charset_t *charset_default = NULL;	/* Default charset */
struct charset_t *charsets = NULL;

/* Look-up tables */
chartable_t upcase_ch;
chartable_t lowcase_ch;
chartable_t clean_ch;

/*
 *	Initialize stuff
 */
 
void cstring_init(void)
{
	int i;

	for (i = 0; i < 255; i++)
		upcase_ch[i] = lowcase_ch[i] = clean_ch[i] = (char)i;
	
	for (i = 65; i <= 90; i++)
		lowcase_ch[i] = (char) i + 32;
	for (i = 97; i <= 122; i++)
		upcase_ch[i] = (char) i - 32;
	
	lowcase_ch[(int) '\x80'] = '\x87'; upcase_ch[(int) '\x87'] = '\x80';
	lowcase_ch[(int) '\x8e'] = '\x84'; upcase_ch[(int) '\x84'] = '\x8e';
	lowcase_ch[(int) '\x8f'] = '\x86'; upcase_ch[(int) '\x86'] = '\x8f';
	lowcase_ch[(int) '\x90'] = '\x82'; upcase_ch[(int) '\x82'] = '\x90';
	lowcase_ch[(int) '\x92'] = '\x91'; upcase_ch[(int) '\x91'] = '\x92';
	lowcase_ch[(int) '\x99'] = '\x94'; upcase_ch[(int) '\x94'] = '\x99';
	lowcase_ch[(int) '\x9a'] = '\x81'; upcase_ch[(int) '\x81'] = '\x9a';
	lowcase_ch[(int) '\xa5'] = '\xa4'; upcase_ch[(int) '\xa4'] = '\xa5';
	
	clean_ch[7] = (char) 14;
	clean_ch[8] = '~';
	clean_ch[10] = '~';
}

/*
 *	Is this an /ex or ctrl-z
 */

int isabortstr(char *s)
{
	if (!strcasecmp(s, "/ex\n"))
		return 1;
	if (!strcasecmp(s, "\032\n"))
		return 1;
	return 0;
}

/*
 *	Do some validity checking for callsign pointed to by `s'.
 */
 
int valid_call(const call_t *call)
{
	int len = 0;
	int nums = 0;
	int ssid = 0;
	char *s = (char *)call;
	char *p[1];
	
	if (s == NULL)
		return -1;
	while (*s && *s != '-') {
		if (!isalnum(*s))
			return -1;
		if (isdigit(*s))
			nums++;
		len++;
		s++;
	}
	if (*s == '-') {
		if (!isdigit(*++s))
			return -1;
		ssid = strtol(s, p, 10);
		if (**p)
			return -1;
	}
	if (len < 4 || len > 6 || !nums || nums > 2 || ssid < 0 || ssid > 15)
		return -1;
		
	return 0;
}

/*
 *	CR => LF
 */

extern char *cr2lf(char *s)
{
	char *c;
	for (c = s; (*c); c++) {
		if (*c == '\r')
			*c = '\n';
	}
	return s;
}

/*
 *	String to upper case
 */

extern char *hstrupr(char *s)
{
	char *c;
	for (c = s; (*c); c++) {
		*c = upcase_ch[(int)*c];
	}
	return s;
}

/*
 *	String to lower case
 */

extern char *hstrlwr(char *s)
{
	char *c;
	for (c = s; (*c); c++) {
		*c = lowcase_ch[(int)*c];
	}
	return s;
}

/*
 *	First character of each word to upper case, the rest to lower case
 */

char *capitalize(char *s)
{
	char *c, last;
	last = ' ';
	for (c = s; (*c); c++) {
		if (last == '$' || last == '&' || last == '%' || last == '#' 
		 || last == '"' || last == '+' || last == '/' || last == ','
		 || last == '~' || last == '^' || last == '_' || last == '-' 
		 || last == '.' || last == ' ')
			*c = upcase_ch[(int)*c];
		else
			*c = lowcase_ch[(int)*c];
		
		last = *c;
	}
	return s;
}

/*
 *	Translate against character set
 */

char *strtranslate(char *s, struct charset_t *set)
{
	char *c;
	
	if ((set) && (s)) {
		for (c = s; (*c); c++) {
			*c = set->table[(int)*c];
		}
	}
	return  s;
}

/*
 *	Safe strncpy
 */
 
char *hstrncpy(char *dest, char *src, int n)
{
	strncpy(dest, src, n);
	*(dest + n - 1) = '\0';
	
	return dest;
}

/*
 *	Return spaces
 */

char *spaces(int n)
{
	char *s = (char *)hmalloc(n+1);
	memset(s, (int) ' ', n);
	s[n] = '\0';
	return s;
}

char *backspaces(int n)
{
	char *s = (char *)hmalloc(n+1);
	memset(s, (int) '\007', n);
	s[n] = '\0';
	return s;
}

/*
 *	Pad to left
 */

char *padleft(char *s, int n)
{
	char *str = hmalloc(n+1);
	memset(s, ' ', n);
	strncpy(str, s, (strlen(s) > n) ? n : strlen(s));
	s[n] = '\0';
	return s;
}

/*
 *	Pad to right
 */

char *padright(char *s, int n)
{
	int nulls, chars;
	char *str;

	if (strlen(s) >= n) {
		nulls = 0;
		chars = n;
	}
	str = (char *)hmalloc(n+1);
	memset(s, ' ', n);
	strncpy((char *) str + nulls, s, chars);
	s[n] = '\0';
	return s;
}

/*
 *	Remove excess whitespace
 */

char *cleanstr(char *s)
{
	char *a, prev = 'a';
	char *t = hmalloc(strlen(s));
	char *b = t;

	for (a = s; (*a); a++) {
		if (*a == '\t')
			*a = ' ';
		if (*a == ' ' && prev == ' ')
			continue;
		*b = *a;
		prev = *a;
		b++;
	}
	*b = '\0';
	
	return s;
}

/*
 *	Word wrapping
 */

char *format(char *prefix, char *text, int linelen, int cr_end)
{
	char *f, *t, *x;
	int prefixlen, l, lw;
	static char buf[2048];
	linelen--;
	prefixlen = strlen(prefix);
	for (f = prefix, t = buf; *f; *t++ = *f++) ;
	l = t - buf;
	f = text;

	for (; ; ) {
		while (isspace(*f)) f++;
		if (!*f) {
			*t++ = '\n';
			*t = '\0';
			return buf;
		}

		for (x = f; *x && !isspace(*x); x++) ;

		lw = x - f;
		if (l > prefixlen && l + 1 + lw > linelen) {
			*t++ = '\n';
			l = 0;
		}

		do {
			*t++ = ' ';
			l++;
		} while (l < prefixlen);
		
		while (lw--) {
			*t++ = *f++;
			l++;
		}
	}
}

/*
 *	Frequency to string
 */

char *freq2str(long i)
{
	static char s[30];
	snprintf(s, 30, "%.1f", i / 10);
	s[29] = '\0';
	return s;
}

/*
 *	String to frequency
 */

long str2freq(char *s)
{
	double r;
	r = atof(s);
	return (long) floor(r * 10 + 0.5);
}

/*
 *	Integer to 4-char string with byte-style shortenings (kilo, mega...)
 */

char *bytes2str(int i)
{
	static char s[10];
	
	if (i < 1000)
		sprintf(s, "%-4d", i);
	else if (i < 1022976) {
		sprintf(s, "%-3.3f", (double)i / 1024);
		*strchr(s, '.') = 'k';
	} else if (i < 1047527424) {
		sprintf(s, "%-3.3f", (double)i / 1024 / 1024);
		*strchr(s, '.') = 'M';
	} else {
		sprintf(s, "%-3.3f", (double)i / 1024 / 1024 / 1024);
		*strchr(s, '.') = 'G';
	}
	s[4] = '\0';
	
	return s;
}

/*
 *	String to boolean
 */

int str2bool(char *s)
{
	/*  1 : True
	    0 : False
	   -1 : Unidentified */

	if (s) {
		switch (lowcase_ch[(int)s[0]]) {
		
		case 'y':	/* Yes */
		case '+':
		case '1':
		case 't':	/* True */
			return 1;

		case 'n':	/* No */
		case '-':
		case '0':
		case 'f':	/* False */
			return 0;

		}
	}

	if (!strcasecmp(s, "on"))
		return 1;
	if (!strcasecmp(s, "off"))
		return 0;

	return -1;
}

/*
 * **************** String list handling
 */

/*
 *	Add a string
 */
  
struct stringlist_t *add_string(struct stringlist_t *list, char *s)
{
	struct stringlist_t *new;
	
	new = hmalloc(sizeof(struct stringlist_t));
	
	if (list) {
		new->next = list->next;
		new->prevp = &list->next;
		list->next->prevp = &new->next;
		list->next = list;
	} else {
		new->next = new;
		new->prevp = &new->next;
	}
	
	new->s = s;
	
	return new;
}

