
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include <linux/ax25.h>
#include <linux/netrom.h>
#include <linux/rose.h>

#include "hmalloc.h"

#include "axutils.h"
#include "nrconfig.h"

#include "csock.h"
#include "af_netrom.h"
#include "log.h"

/*
 *	Bind() a netrom interface and listen() on it
 */

void netrom_listen(struct listenq_t *lq)
{
	log(L_SINFO, "Binding NET/ROM interface %s:%s%s",
		lq->port, lq->call, lq->compress ? " (compressed)" : "");
	
	if ((lq->fd = socket(lq->af_type, SOCK_SEQPACKET, 0)) < 0) {
		log(L_CRIT, "netrom_listen(): socket(): %s", strerror(errno));
		exit(1);
	}
	
	if (lq->fd > maxfd)
		maxfd = lq->fd;
	
	if (bind(lq->fd, (struct sockaddr *)&lq->sockaddr, lq->addrlen) < 0) {
		log(L_CRIT, "Could not bind() NET/ROM interface %s:%s: %s", 
			lq->port, lq->call, strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	if (listen(lq->fd, SOMAXCONN) < 0) {
		log(L_CRIT, "Could not listen() on NET/ROM interface %s:%s: %s",
			lq->port, lq->call, strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	FD_SET(lq->fd, &readfds);
}

/*
 *	Start listening on netrom interfaces
 */

void netrom_listens(void)
{
	struct listenq_t *lq;
	
	for (lq = listenq; (lq); lq = lq->next)
		if (lq->af_type == AF_NETROM)
			netrom_listen(lq);
}

/*
 *	Accept a NETROM connection
 */

void netrom_accept(struct listenq_t *lq)
{
	int i;
	int addrlen, new_fd;
	struct csock_t *s;
	union {
		struct full_sockaddr_ax25 ax25;
		struct sockaddr_rose rose;
	} sockaddr;
	
	i = 1;
	ioctl(lq->fd, FIONBIO, &i);	/* Set up a non-blocking accept() */
	
	addrlen = sizeof(struct full_sockaddr_ax25);
	new_fd = accept(lq->fd, (struct sockaddr *)&sockaddr, &addrlen);
	
	i = 0;
	ioctl(lq->fd, FIONBIO, &i);
	
	if (new_fd < 0) {
		if ((errno == EWOULDBLOCK) || (errno == EINTR))
			return;
		
		log(L_ERR, "NET/ROM accept() error: %s", strerror(errno));
		log(L_ERR, "Closing NET/ROM interface %s:%s (fd %d)", 
			lq->port, lq->call, lq->fd);
		
		close(lq->fd);
		lq->fd = -1;
		return;
	}
	
	i = 1;
	ioctl(new_fd, FIONBIO, &i);	/* Make the socket non-blocking */
	
	s = sock_alloc(new_fd, AF_NETROM, DEF_BUFLEN_NR, DEF_BUFLEN_NR, lq->compress);
	s->eoltype = ax25_eol;
	s->eol = AX25_EOL;
	
	s->call = hstrdup(ax2asc(&sockaddr.ax25.fsa_ax25.sax25_call));
	s->node = hstrdup(ax2asc(&sockaddr.ax25.fsa_digipeater[0]));
	
	addrlen = sizeof(struct full_sockaddr_ax25);
	s->port = hstrdup(lq->port);
	
	log(L_SINFO, "New NET/ROM connection on fd %d from %s@%s on port %s", 
		new_fd, s->call, s->node, s->port);
	
	sock_login(s);
}
