//------------------------------------------------------------------------------
// Copyright(c),  Ralph Holland, 1997, 2002 
//
// History:
//
// 004: 27 Apr 2002, Ralph B. Holland - Now uses a configuration file from 
//      The file's Format is first line contains somnec2d and then subsequent 
//      lines contain: nec2dx.exe,number_of_segments. These lines must be 
//      ordered in the configuration file with assending number of segments 
//      as this data is not sorted. If you don't supply a configuration
//      file a default one will be written for you. You can update this file
//      and add your executables and segment break points. You can use -s
//      to override the segment calculations in case you use a NGF or I calculate
//		incorrectly. 
//      If you type nec the command-line parameter usage will be displayed.
//
//	    NOTE: if you compile this with Visual C++ version 6.0 in debug mode 
//      lots of warnings are produced that state that the map, vector and
//      associated _Tree type debug information is truncated to 255 characters.
//		This has been fixed with the .NET Visual Studio C++ version 7.0.
//
// 003: 15 Apr 2002, Ralph B. Holland - Now processes GA, GH, GR, GX, SP, MP cards
//      in addition to GW and GN to determine which variant of NEC to execute.
//		Code modified to compile with Visual C++ version 6.0 (oh well).
//	    I also added machdep.h and machdep.cpp for machine dependent code, which 
//		you can modify when migrating to other operating systems, such as Unix or
//      Linux.
// 002: 29 Dec 1998, Ralph B. Holland - Fixed bug introduced on 14 Dec 98.
// 001: 14 Dec 1998, Ralph B. Holland - Refined so it may be adapted to Unix.
// 000: 29 Apr 1997, Ralph B. Holland - Initial writing
//
// Note:
//
// This version of nec is slower than its predecessor because it read a configuration file
// and processes more cards from the input stream
//
// Description:
//
// A program that can be used to execute NEC2D variants via the system() function.
//
// This program was designed for IBM compatibles and permits the selection of the variant 
// of nec2d which will support the number of segments and patches found in the input card
// specifications. The code was originally compiled with Borland C++ V5.01 and 
// targetted to the Windows 32 bit console model and has now been adapted to Visual
// C++ version 6.0 (which is a little different and doesn't provide some of the methods
// provided by Borland.) However the Borland version required.RTM.EXE and CW3220.DLL 
// which had to be placed in the executable path whereas the Visual C++ executable does 
// not require any ancillary support files or executables.
//
// The program opens and reads the specified NEC input file and determines
// the requirements such as frequency, ground parameters and number of segments
// and the number of patches.
//
// If necessary the program will execute somnec2d to formulate the Sommerfeld
// ground interpolation grid and save the data for re-use in the path
// specified by the SOMDATA envrionment variable, or the -p switch, or the current 
// directory by default.
//
// Then nec will execute the appropriate version of nec2d according to the number
// of segments. The user can define the segment break-points and the executable to
// be invoked below or equal to the break-point.(These executables are available 
// from the Unofficial NEC archives and I suggest you upgrade to the nec2dx variants
// as this corresponds to the default configuration file thatg is written if you
// do not supply one.)
//
// Because earlier nec2d programs were compiled with LAYHEY FORTRAN running in 
// protected mode they have been invoked via a batch file through the system() call 
// and the executable programs are passed the required parameters via the batch file 
// and the redirection of data to the program's standard input. This has not been
// changed for the nec2dx executables that run under Windows XP and this technique
// would work in the FORTRAN source if your compiler/library supported the system()
// command shell invocation.
//
// nec.cpp should be adaptable to Unix and Linux if required, although this has not 
// been done by the author. If you do this and make changes, please forward the
// source code to the author at mailto:ralph@arising.com.au for consideration and
// I will make adaptions to place in the NEC archives. (I may even download gcc to
// test or re-install Linux.) If you would like to contribute the make file that 
// would be appreciated too. Please help me out and also inform me of any #defines 
// that we can used for identification and conditional compilation on your platform.
//
// There is no waranty implied, explicit or otherwise for this program. You may freely 
// use, modify and republish this program for research purposes or personal use
// provided the source is acknowledge, the copyright is retained and you accept 
// no compensation for its use. (I think that is fair.)
//
// Report Bugs:
//
// If you have comments, suggestions or require assistance, you may contact the 
// author at mailto::ralph@arising.com.au. 
//
// The author also maintains a website http://www.arising.com.
//
// How to recompile for other platforms:
//
// Modify the declarations/definitions in machdep.h and the implementation in 
// machdep.cpp. If you find other dependencies they may best be worked into machdep
// as well. Please send updates to the author.
//
//------------------------------------------------------------------------------


#include <iostream>
#include <iomanip.h>
#include <fstream.h>
#include <io.h>
#include <strstrea.h>
#include "nec.h"


	// determine any line kind we are interested in
Nec::KLine Nec::lineKind(const char *line) {
	switch (line[0]) {
        case 'F':
   	        if (line[1]=='R') {
                return FR;
            }
            break;
        case 'G':
            switch (line[1]) {
				case 'A':
					return GA;
				case 'H':
					return GH;
                case 'N':
                    return GN;
				case 'R':
					return GR;
                case 'W':
                    return GW;
				case 'X':
					return GX;
            }
		case 'S':
			switch (line[1]) {
				case 'M':
					return SM;
				case 'P':
					return SP;
			}
   }
   return Non;
}


	// Count the number of segments in the input data
	// patches count as 2 segments (got this from doco not from source)
	// Warning: NGF file not processed nor accounted for, use -s to override.
void Nec::totalSegment(size_t &segments,size_t &patches) {
	segments = 0;
	patches = 0;
	for (TagMap::iterator iter = tagMap.begin();iter != tagMap.end(); iter++) {
		int tag = (*iter).first;
		int inc = (*iter).second;
		if (tag >= 0) {
			segments += inc;  // positive tag numbers mean segments
		}
		else {
			patches += inc;   // neagtive tag numbers mean patches (my convention)
		}
	}
}

	// this will be displayed on error with parameters.
void Nec::usage(const char *name) {
	string exec(name);
	string::size_type pos = exec.find_last_of('\\');
	if (pos != string::npos) {
		exec.replace(0,pos+1,"");
	}
	else {
		pos = exec.find_last_of('/');
		if (pos != string::npos) {
			exec.replace(0,pos+1,"");
		}
	}
	// Usage data, later on I will implement a formatter so this data will fit to the console size better.
	cout << endl << exec.c_str() << " is a shell program to execute nec2d variants." << endl << endl;
    cout << "Usage " << exec.c_str() << " [-sn] [-p[\"]SOMNEC_PATH[\"]] [-c[\"]CONFIG_PATH[\"]] [\"]inputfile[\"]" << endl;
	cout << "[[\"]outputfile[\"]] [-e]"  << endl << endl;
	cout << "where n is the maximum number of segments, -p specifies the SOMNEC_PATH, which" << endl;
	cout << "overrides the SOMDATA environment variable and -c specifies the config" << endl; 
	cout << "file override and -e says just execute for effect i.e. invokes somnec2d" << endl;
	cout << "only and not the nec2d variant." << endl << endl;
	cout << exec.c_str() << " reads the NEC input deck and processes GA, GH, GR, GW, GX, SP, FR" << endl;
	cout << "and SM cards to determine which nec2d variant to execute." << endl << endl;
	cout << "If a Somnec input file is required for a real ground (specified in the GN card)" << endl;
	cout << exec.c_str() << " will use the SOMDATA environment variable or -pPath to read or" << endl;
	cout << "write the appropriate Somnec input file." << endl << endl;
	cout << "You may also optionally supply a -cPath where Path is the pathname for the" << endl;
	cout << "configuration file which contains the maximum segments, a comma, and the" << endl;
	cout << "associated nec2d executable on each line. The first line of this configuration" << endl;
	cout << "file contains the name of the Somnec executable file." << endl << endl;
	cout << "If the configuration file does not exist the program will write a default" << endl;
	cout << "configuration file for you. This file will contain the somnec2d.exe on the" << endl;
	cout << "first line and the maximum segments followed by a comma and the corresponding" << endl;
	cout << "nec2dx executable which supports that many segments. You may edit this file" << endl;
	cout << "with your favourite text editor." << endl << endl;
	cout << "These executables, including " << exec.c_str() << " will run on all Microsoft Windows" << endl;
	cout << "platforms including Windows XP." << endl;
}


	// Restore the Sommerfeld data file for this K, S, and F
    // Sommerfeld data file is stored in SOMDATA directory specified
    // by the environment variable or in the ./SOMDATA if not specified
void Nec::sommerfeld(const char *sommerfeldExec, Numeric frequency) {
	char fileName[1024];
	ostrstream buffer(fileName,sizeof(fileName));
    long s = (long)((sigma * 1000.0)+0.5);
    long f = (long)(frequency * 100000L);

	// Ok we use this path
   	buffer << somnecPath.c_str() << '\0';

    // Now the dielectric constant is the first part of the pathname
    buffer. seekp(-1,iostream::cur);
    buffer << TREE_DELIMITER << epsilon << '\0';     // Note /

    // Next part of the path is conductivity
    buffer. seekp(-1,iostream::cur);
    buffer << TREE_DELIMITER << s << '\0';              // Note /
    if (!Exists(fileName)) {
   	    MakeDir(fileName);
    }

    // final part of the path (filename) is based on frequency
    buffer. seekp(-1,iostream::cur);
    buffer << TREE_DELIMITER << f << '\0';              // Note /

    cout << "SOMNEC data will be stored in " << fileName << endl;

    // create the NEC2 Sommerfeld data input file
    if (Exists(fileName)){
   	    CopyFile(fileName,SOMNEC_OUT,false);// 002 use SOMNEC_OUT not SOMNEC_IN!
    }
    else {
        ofstream batch(SOMNEC_SHELL);
        batch << sommerfeldExec << " <" << SOMNEC_IN << endl;
        batch. close();
		ofstream input(SOMNEC_IN);
        input << epsilon << ',' << sigma << ',' << frequency << ",0" << endl;
        input. close();
        chmod(SOMNEC_SHELL,777);  /// allow it to execute
   	    system(SOMNEC_SHELL);
        CopyFile(SOMNEC_OUT,fileName,false);
        DeleteFile(SOMNEC_IN);
        DeleteFile(SOMNEC_SHELL);
    }

}


	// The constructor
Nec::Nec(int argc, char *argv[]) {
	if (argc<1) {
		usage(argv[0]);
	}
	else {
		cSegmentOverride=0;
		forEffect = false;
		cout << endl << argv[0] << " version 2.0" << endl;

		// process the commandline arguments
		for( int i=1; i < argc; i++ ) {
			// cout << argv[i] << endl;
			string text(argv[i]);

			//	cout << "argv[i]=" << argv[i] << " text=" << text.c_str() << endl;
			size_t position = text.find("-");
			if (position!=string::npos) {
				text.replace(position,1,"");
				switch(text.c_str()[0]) {

					case 's':
						cSegmentOverride = atoi(&text.c_str()[1]);	// number of segments overridden by user				
						break;

					case 'p':
						text.replace(0,1,"");
						somnecPath = text;							// somnec database path supplied by user
						break;

					case 'c':
						text.replace(0,1,"");				
						configPath = text;								// configuration file pathname supplied by user
						break;

					case 'e':
						forEffect = true;
						break;

				}
			}
			else {
				if (!inputPath.length()) {
					inputPath = text;							// input pathname
				}
				else if (!outputPath.length()) {
					outputPath = text;							// output pathname
				}
			}
			
		}

		// fix the config file
		if (!configPath.length()) {
			char *p = getenv("NEC_CONFIG");
			if (p) {
				configPath = p; // use env variable
			}
			else {
				configPath = argv[0]; // use the exec path
			}
			string::size_type pos = configPath.find_last_of(".");
			if (pos == string::npos) {
				pos = configPath.length();
			}
			configPath.replace(pos,configPath.length()-pos,".cfg");
		}
		
		// let's determine the somnecPath
		if (!somnecPath.length()) {
			char *p = getenv("SOMDATA");
			if (p) {
				somnecPath = getenv("SOMDATA");
			}
		}
		if (!somnecPath.length()) {
			somnecPath = "SOMDATA";
		}

		// check if we have an input file path
		if (!inputPath.length()) {
			usage(argv[0]);
			return;
		}
		if (!Exists(inputPath.c_str())) {
			string::size_type pos = inputPath.find_last_of(".");
			if (pos == string::npos) {
				pos = inputPath.length();
			}
			inputPath.replace(pos,inputPath.length(),".nec");
			cout << inputPath.c_str() << endl;
			if (!Exists(inputPath.c_str())) {
				usage(argv[0]);
				return;
			}
		}
		
		// Now ensure that we have the output filename determined
		if (!outputPath.length()) {
			outputPath = inputPath;
			string::size_type pos = outputPath.find_last_of(".");
			if (pos == string::npos) {
				pos = outputPath.length();
			}
			outputPath.replace(pos,outputPath.length(),".out");
		}
		
		// OK let's determine the appropriate nec2d
		shell(argv[0]);
		
	}
}

void Nec::insertTag(int tag,int count) {
	map <int, int>::iterator iter = tagMap.find(tag);
	if (iter != tagMap.end() ) {
		iter->second += count;
	}
	else {
		tagMap.insert( std::pair<int,int>(tag, count) );
	}
}

	// construct a NEC2 object
void Nec::shell(const char *execName) {
    
	// add the .cfg to the file path
	if (!Exists(configPath.c_str())) {
		cout <<"Config File " << configPath.c_str() << " not found - writing default one"
			<< endl;
		ofstream stream(configPath.c_str());
		stream << "somnec2d.exe" << endl;
		stream << "nec2dx500.exe,500" << endl;
		stream << "nec2dx1500.exe,1500" << endl;
		stream << "nec2dx3000.exe,3000" << endl;
		stream << "nec2dx5000.exe,5000" << endl;
		stream.close();
	}
	patch = -1;
    epsilon = 0;
    sigma = 0;
    ifstream input(inputPath.c_str(),ios::in || ios::nocreate);
    if (!input. good()) {
   	    cout << inputPath.c_str() << " not found" << endl;
		usage(execName);
        return;
    }
    string line;
    uint16 count = 0;
	const size_t len = 1025;
	char * buffer = new char [len+1]; // accept 1024 chars
	input.seekg(0,ios::beg);
	size_t cFrequency = 0;

	// OK read every card and determine the number of segments for solution
	// Also acquire frequency in case Somnec data file required
    while (input. good() && !input. eof()) {
        input.getline(buffer,len);             // read a line upto \n
		buffer[len]='\0';
		line.assign(buffer,strlen(buffer));
        if (line. length()) {
			string type = line.substr(0,2);
			line = line.substr(2);
            switch (lineKind(type.c_str())) {
				
				case FR: {  // frequency specification
					int iFreq = atoi(stripToken(line,3).c_str());
					int nFreq = atoi(stripToken(line,5).c_str());
					if (nFreq==0) {
						nFreq=1;
					}
					string blank1 = stripToken(line,5);
					string blank2 = stripToken(line,5);
					Numeric frequency = (Numeric)atof(stripToken(line,10).c_str());
					Numeric delFrequency = (Numeric)atof(stripToken(line,10).c_str());
				
					// Now analyse the frequency card and insert frequency info
					for ( ; nFreq>0; nFreq-- ) {
						aFrequency.resize(cFrequency+1);
						aFrequency[cFrequency++] = frequency;
						if (iFreq==1) {
							frequency *= delFrequency;
						}
						else {
							frequency += delFrequency;
						}
					}
					break;
				}
				
				case GN: {  // ground specification
					int iPerf = atoi(stripToken(line,3).c_str());
					int nRadials = atoi(stripToken(line,5).c_str());
					string blank1 = stripToken(line,5);
					string blank2 = stripToken(line,5);
					epsilon = (Numeric)atof(stripToken(line,10).c_str());
					sigma = (Numeric)atof(stripToken(line,10).c_str()); 
					kGround = (KGround)(iPerf);
					break;
				}
				
				case GW:	// wire
				case GH:	// helix
				case GA:	// arc 
				{
					int tag = atoi(stripToken(line,3).c_str());
					int cSeg = atoi(stripToken(line,4).c_str());
					insertTag(tag,cSeg);
                    break;
				}
				
				case SP:  { // Surface patch - negative numbers are patches
					string blank = stripToken(line, 3);
					insertTag(-1,1);
					// tagMap.insert( std::pair<int,int>(patch--,1));
					break;
				}
				
				case SM:  {	// Multiple patches
					// we treat almost like segments - negative number are patches
					int nX = atoi(stripToken(line,3).c_str());
					int nY = atoi(stripToken(line,3).c_str());
					insertTag(-1,nX*nY);
					break;
				}
				
				case GR:  {	
					// replicate the whole stucture for non-zero tags
					// this may yield an over-estimate of the number of segments
					// (I haven't tested yet.)
					int inc = atoi(stripToken(line,3).c_str());
					TagMap copyMap( tagMap );
					for( int replication = atoi(stripToken(line,3).c_str()); replication; replication-- ) {
						for (TagMap::iterator iter = copyMap.begin();iter != copyMap.end(); iter++) {
							int tag = (*iter).first;
							if (tag != 0) {
								// non-zero tags add more
								int newTag = tag + inc;
								insertTag(newTag,(*iter).second );
							}
							else {
								// zero tags, just replicate
								(*iter).second++;
							}
						}
					}
					break;
				}
				
				case GX: { // reflect axes
					// this will yield an over-estimate of cSegs
					// (I haven't tested yet.)
					int inc = atoi(stripToken(line,3).c_str());
					int x = atoi(stripToken(line,1).c_str());
					int y = atoi(stripToken(line,1).c_str());
					int z = atoi(stripToken(line,1).c_str());
					int replication=1;
					if (x) {
						replication*=2;
					}
					if (y) {
						replication*=2;
					}
					if (z) {
						replication*=2;
					}
					for ( ; replication; replication-- ) {
						for (TagMap::iterator iter = tagMap.begin();iter != tagMap.end(); iter++) {
							(*iter).second *= replication;
						}
					}
					break;
				}
				
            }
        }
    }
    input. close();
	delete [] buffer;
	
	size_t cSegment;
	size_t cPatch;
	totalSegment(cSegment,cPatch); // patches are included*2 NGF is not currently included
	cout << "Number of segments = " << cSegment << " and number of patches = " << cPatch << endl;
	size_t total = cSegment + cPatch * 2;
	if (cSegmentOverride) {
		cout << "Segment override = " << cSegmentOverride << endl;
		total = cSegmentOverride;
	}
	cout << "Input file " << inputPath.c_str() << endl;
	cout << "Output file " << outputPath.c_str()<< endl;
	cout << "Config file " << configPath.c_str() << endl;
	if (epsilon) {
		cout <<"Ground dielectric constant and conductivity [" << epsilon << ','
			<< sigma << " S/m]"  << endl;
	}
	if (!aFrequency.empty()) {
		cout << "Frequency = " << aFrequency[0] << " MHz" << endl;
	}
	
	// Now write the shell files for NEC2x.EXE
	// and select the appropriate NEC2 files for speed
	ofstream batch(NEC_SHELL);
	
	// Generate the local batch files by reading the config file
	ifstream configFile(configPath.c_str());
	bool found = false;
	char necBuffer[256];
	char sommerfeldExec[256];
	sommerfeldExec[0]='\0';
	size_t size;
	while (!found && !configFile.eof()) {
		if (!sommerfeldExec[0]) {
			configFile >> sommerfeldExec;
		}
		else {
			configFile >> necBuffer;
			char *pSep = strchr(necBuffer,',');
			if (pSep) {
				size = atoi(pSep+1);
				*pSep = '\0';
			}
			if (total <= size) {
				batch << necBuffer;
				found = true;
			}
		}
	}
	configFile.close();
	if (!found) {
		cout << "Too many segments - using " << necBuffer << '.' << endl;
		batch << necBuffer;
	}
	batch << " <" << NEC2_IN << endl;
	batch.close();
	
	// NOTE: can only supply 1 frequency for Sommerfeld ground
	// we calculate them all but only one is used
	// nec2d will not work as you expect when doing frequency
	// stepping with SommerfeldGround
	if (kGround==SommerfeldGround) {
		cout << "Sommerfeld Executable " << sommerfeldExec << endl;
		for (size_t i = 0; i < aFrequency.size(); i++ ) {
			sommerfeld( sommerfeldExec, aFrequency[i]);
		}
	}

	if (!forEffect) {
	
		// Generate the commandline redirected input file for NEC2D
		ofstream out(NEC2_IN);
		out << inputPath.c_str() << endl;
		out << outputPath.c_str() << endl;
		out. close();
		chmod(NEC_SHELL,777); // for Unix
		cout << "Executing " << necBuffer << endl;
		system(NEC_SHELL);
	}

	// remove these files
	DeleteFile(NEC_SHELL);
	DeleteFile(NEC2_IN);
	if (kGround==SommerfeldGround) {
		DeleteFile(SOMNEC_OUT);
	}

}
	
	// Strip a token from the line and replace the line with the remainder
	// this mimics FORTRAN input fields of either size or separated by a delimiter
	// NOTE: If the input contains commas (,) then the width is ignored
string Nec::stripToken(string &line,size_t width) {
	string result;
	if (line. length()) {
		string::size_type posComma = line. find(',');
		if (posComma!=string::npos) {
			result = line.substr(0,posComma);
			line = line.substr(posComma+1);
		}
		else {
			if (line.length()>width) {
				result = line.substr(0,width);
				line = line.substr(width+1);
			}
			else {
				result = line;
				line = "";
			}
		}
	}
	else {
		result = "";
	}
    return result;
}



// The main program
int main(int argc,char **argv) {

#ifdef DEBUG
#	if PLATFORM==PLATFORM_UNIX
	cout << "Compiled for UNIX" << endl;
#	elif PLATFORM==PLATFORM_WIN
	cout << "Compiled for Windows" << endl;
#	elif PLATFORM==PLATFORM_DOS
	cout << "Compiled for DOS" << endl;
#	endif
#endif // DEBUG

	Nec nec(argc, argv); // construct a Nec object and do it
	return 0;
}


