//------------------------------------------------------------------------------
// Copyright(c),  Ralph Holland, 1997
//
// History:
// 002: 29 Dec 1998, Ralph B. Holland - Fixed bug introduced on 14 Dec 98.
// 001: 14 Dec 1998, Ralph B. Holland - Refined so it may be adapted to Unix.
// 000: 29 Apr 1997, Ralph B. Holland - Initial writing
//
// Description:
//
// This program is for IBM-compatibles running Windows 95. The code has
// been compiled with Borland C++ V5.01 and targetted to the Windows 32 bit
// console model. (Borland support / in pathnames - just like Unix!)
// It requires the Borland supplied files: RTM.EXE and CW3220.DLL (these must
// be placed in either the current directory or in the path).
//
// The program opens and reads the specified NEC input file and determines
// the requirements such as frequency, ground parameters and number of segments.
// If necessary the program will execute SOMNEC2D to formulate the Sommerfeld
// ground interpolation grid and save the data for re-use in the path
// specified by SOMDATA envrionment variable - or the current directory.
// Then NEC will execute the appropriate version of NEC2d
// (which is available from the NEC archives). As NEC2D programs are compiled
// with LAYHEY FORTRAN running in protected mode they have been invoked via a
// batch file via the system() call. The programs are passed the required
// parameters via the batch file and the redirection of data to the program's
// standard input.
//
// This program should be adaptable to Unix and Linux if required, although this
// has not been done.
//
// There is no waranty implied or otherwise for this program.
// You may freely use, modify and republish this program for research purposes
// provided the source is acknowledge. If you have comments, suggestions
// or require assistance, you may contact the author at
// mailto::vk1brh@dynamite.com.au. The author also maintains a website
// http:://www2.dynamite.com.au/vk1brh where programs may be posted from time
// to time.
//------------------------------------------------------------------------------

#include "File.h"           // file processing
#include "String.h"         // string processing
#include "Stream.h"	        // stream processing

#ifdef _Windows
    // Windows - leave these alone - they work for IBM
#  define NEC_SHELL       "nec2d.bat"
#  define NEC_IN          "nec2.in"
#  define SOMNEC_EXEC     "somnec2d.exe"
#  define SOMNEC_SHELL    "som2d.bat"
#  define SOMNEC_IN       "som2d.in"
#  define SOMNEC_OUT      "som2d.nec"
#else
    // Unix - you may adjust these if required
#  define NEC_SHELL       "nec2d.sh"
#  define NEC_IN          "nec2.in"
#  define SOMNEC_EXEC     "somnec2d"
#  define SOMNEC_SHELL    "som2d.sh"
#  define SOMNEC_IN       "som2d.in"
#  define SOMNEC_OUT      "som2d.nec"
#endif

	// Invoke Nec2d
class Nec {
public:
	Nec(const char *inputFileName, const char *outputFileName);
    Nec(const char *baseFileName);
private:

		// local model data
    enum KGround {FreeSpace=-1,Perfect=0,Finite=1,SommerfeldGround=2};
	Numeric frequency;    	    // frequency in MHz
    Numeric dielectric;         // relative dielectric constant of first ground
    Numeric conductivity;	    // relative dielectric constant of second ground
    int     cSegment;			// number of segments in geometetry
    KGround kGround;			// type of ground in the model

   	// Utilities
    void Sommerfeld();		    // Execute SOMMERFELD

    // Used to Shell to NEC2D
	void Shell(const char *inputFileName, const char *outputFileName);

		// Token Parsing
	enum KLine { Non, FR, GN, GW };
	static KLine LineKind(const char *line);

};

	// Process the inputFile and shell to NEC2D
Nec::Nec(const char *baseName) {
		// Establish the input and output name from the baseName
	string inputFileName = baseName;
    string outputFileName = baseName;
    inputFileName. set_case_sensitive(0);
    size_t position = inputFileName. find(".nec");
	if (position==NPOS) {
   	    inputFileName += ".nec";
	    outputFileName += ".out";
    }
    else {
   	    outputFileName. replace(position, 4, ".out");
    }
    Shell(inputFileName. c_str(),outputFileName. c_str());
}

	// process the inputFile and shell to NEC2D
Nec::Nec(const char *inputFileName,const char *outputFileName) {
	Shell(inputFileName,outputFileName);
}

Nec::KLine Nec::LineKind(const char *line) {
	switch (line[0]) {
        case 'F':
   	        if (line[1]=='R') {
                return FR;
            }
            break;
        case 'G':
            switch (line[1]) {
                case 'N':
                    return GN;
                case 'W':
                    return GW;
            }
   }
   return Non;
}

	// Restore the Sommerfeld data file for this K, S, and F
    // Sommerfeld data file is stored in SOMDATA directory specified
    // by the environment variable or in the ./SOMDATA if not specified
void Nec::Sommerfeld() {
	char fileName[128];
	ostrstream buffer(fileName,sizeof(fileName));
    long s = (conductivity * 1000.0)+0.5;
    long f = frequency * 100000L;

		// Locate the SOMDATA directory
	char *p = getenv("SOMDATA");
    if (p) {
   	    buffer << p << '\0';
    }
    else {
        buffer << "SOMDATA" << '\0';
    }
    cout << "SOMNEC data will be stored in " << fileName << endl;
    if (!Exists(fileName)){
        mkdir(fileName);
    }

    // Now the dielectric constant is the first part of the pathname
    buffer. seekp(-1,iostream::cur);
    buffer << '/' << dielectric << '\0';     // Note /
    if (!Exists(fileName)) {
		mkdir(fileName);
    }

    // Next part of the path is conductivity
    buffer. seekp(-1,iostream::cur);
    buffer << '/' << s << '\0';              // Note /
    if (!Exists(fileName)) {
   	    mkdir(fileName);
    }

    // final part of the path (filename) is based on frequency
    buffer. seekp(-1,iostream::cur);
    buffer << '/' << f << '\0';              // Note /


    // create the NEC2 Sommerfeld data input file
    if (Exists(fileName)){
   	    CopyFile(fileName,SOMNEC_OUT,false);// 002 use SOMNEC_OUT not SOMNEC_IN!
    }
    else {
        if (!Exists(SOMNEC_SHELL)) {
            ofstream batch(SOMNEC_SHELL);
            batch << SOMNEC_EXEC << " <" << SOMNEC_IN << endl;
            batch. close();
        }
		ofstream input(SOMNEC_IN);
        input << dielectric << ',' << conductivity << ',' << frequency << ",0" << endl;
        input. close();
#ifndef _Windows
        chmod(SOMNEC_SHELL,777);  /// allow it to execute
#endif
   	    system(SOMNEC_SHELL);
        CopyFile(SOMNEC_OUT,fileName,false);
        DeleteFile(SOMNEC_IN);
        DeleteFile(SOMNEC_SHELL);
    }
}

	// construct a NEC2 object
void Nec::Shell(const char *inputFileName,const char *outputFileName) {
    cSegment = 0;
    frequency = 0;
    dielectric = 0;
    conductivity = 0;
    ifstream input(inputFileName);
    if (!input. good()) {
   	    cout << inputFileName << " not found" << endl;
        cout << "Convention is either inputFile[.nec] or inputFile outputFile" << endl;
        exit(1);
    }
    string line;
    uint16 count = 0;
    while (input. good() && !input. eof()) {
        line. read_line(input);
        if (line. length()) {
            switch (LineKind(line.c_str())) {
                case FR:
                    frequency = atof(TokenIndex(line,',',5). c_str());
                    break;
                case GN:
 				    line = line(2,line.length()-2);
                    kGround = (KGround)atoi(line. c_str());
                    dielectric = atof(TokenIndex(line,',',5). c_str());
                    conductivity = atof(Token(line,','). c_str());
                    break;
                case GW:
         	        cSegment += atoi(TokenIndex(line,',',2). c_str());
                    break;
            }
        }
    }
    input. close();
	cout << "Input file " << inputFileName
        << " output file " << outputFileName << " for Frequency "
        << frequency << " MHz [" << dielectric << ','
        << conductivity << ']'  << " total segments = " << cSegment<< endl;
    if (kGround==SommerfeldGround) {
        Sommerfeld();
    }
   	// Now write the shell files for NEC2x.EXE
    // and select the appropriate NEC2 files for speed
    ofstream batch(NEC_SHELL);
   	// Generate the local batch files
    if (cSegment <= 255) {
        batch << "NEC2D.EXE";
    }
    else if (cSegment <= 512) {
   	    batch << "NEC2D512.EXE";
    }
    else if (cSegment <= 960) {
   	    batch << "NEC2D960.EXE";
    }
    else {
       	batch << "NEC2D1K4.EXE";  // may be too big for your machine
    }
    batch << " <" << NEC_IN << endl;
    batch.close();

	   // Generate the commandline redirected input file for NEC2D
	ofstream out(NEC_IN);
    out << inputFileName << endl;
    out << outputFileName << endl;
    out. close();

#ifndef _Windows
    chmod(NEC_SHELL,777); // for Unix
#endif

    system(NEC_SHELL);
    DeleteFile(NEC_SHELL);
    if (kGround==SommerfeldGround) {
   	    DeleteFile(SOMNEC_IN);
    }
}

	// A shell program that can be used to execute NEC2 data
int main(int argc,char **argv) {
		// Ensure that the shell BATCH file exists
    if (argc==2) {
   	    Nec nec(argv[1]);
    }
    else if (argc==3) {
   	    Nec nec(argv[1],argv[2]);
    }
    else {
		cout << "Shell program to execute NEC2 variants" << endl;
        cout << "Usage NEC inputfile" << endl;
        cout << "Usage NEC inputfile outputfile" << endl;
        char *p = getenv("SOMDATA");
        if (!p) {
            cout << "SOMDATA environment variable not set - will store data in ./SOMDATA" << endl;
        }
        else {
            cout << "Sommerfeld interpolation grid data stored in " << p << endl;
        }
    }
}
