//////////////////////////////////////////////////////////////////////////
//
// pgScript - PostgreSQL Tools
// RCS-ID:      $Id: pgsApplication.cpp,v 1.3 2008/08/10 17:45:38 pgunittest Exp $
// Copyright (C) 2002 - 2008, The pgAdmin Development Team
// This software is released under the Artistic Licence
//
//////////////////////////////////////////////////////////////////////////


#include "pgAdmin3.h"
#include "pgscript/pgsApplication.h"

#include "db/pgConn.h"
#include "pgscript/expressions/pgsAssign.h"
#include "pgscript/expressions/pgsIdent.h"
#include "pgscript/objects/pgsString.h"
#include "pgscript/utilities/pgsThread.h"

pgsApplication::pgsApplication(const wxString & host, const wxString & database,
		const wxString & user, const wxString & password, int port) :
	m_critical(1, 1), m_mutex(1,1), m_connection(pnew pgConn(host, database, user, password, port)),
			m_defined_conn(true), m_running(0), m_caller(0)
{
	if (m_connection->GetStatus() != PGCONN_OK)
	{
		wxLogError(wxT("PGSCRIPT: Cannot connect to database %s:%d/%s with ")
				wxT("credentials '%s'/'%s'"), host.c_str(), port, database.c_str(),
				user.c_str(), password.c_str());
	}
	
	wxLogScript(wxT("Application created"));
}

pgsApplication::pgsApplication(pgConn * connection) :
	m_critical(1, 1), m_mutex(1,1), m_connection(connection), m_defined_conn(false),
			m_running(0), m_caller(0)
{
	wxLogScript(wxT("Application created"));
}

pgsApplication::~pgsApplication()
{
	if (m_defined_conn)
	{
		pdelete(m_connection);
	}
	
	wxLogScript(wxT("Application destroyed"));
}

bool pgsApplication::parse_file(const wxString & file, pgsOutputStream & out,
		wxMBConv * conv)
{
	m_thread = new pgsThread(m_vars, m_critical, m_connection, file, out, *this, conv);
	return run_thread();
}

bool pgsApplication::parse_string(const wxString & string,
		pgsOutputStream & out)
{
	m_thread = new pgsThread(m_vars, m_critical, m_connection, string, out, *this);
	return run_thread();
}

bool pgsApplication::run_thread()
{
	m_mutex.Wait();
	++m_running;
	m_mutex.Post();
	
	if (m_thread != 0 && m_thread->Create() == wxTHREAD_NO_ERROR
			&& m_thread->Run() == wxTHREAD_NO_ERROR)
	{
		wxLogScript(wxT("Running..."));
		return true;
	}
	else
	{
		wxLogError(wxT("PGSCRIPT: Thread error"));
		delete m_thread;
		return false;
	}
}

bool pgsApplication::is_running()
{
	m_mutex.Wait();
	bool is_running = m_running > 0;
	m_mutex.Post();
	return is_running;
}

void pgsApplication::wait()
{
	if (is_running())
	{
		wxLogScript(wxT("Waiting for pgScript"));
		m_critical.Wait();
		m_critical.Post();
	}
}

void pgsApplication::terminate()
{
	if (is_running())
	{
		wxLogScript(wxT("Deleting pgScript"));
		m_thread->Delete();
	}
}

void pgsApplication::complete_execution()
{
	m_mutex.Wait();
	--m_running;
	m_mutex.Post();
	
#if !defined(NPGADMIN)
	if (m_caller != 0)
	{
		wxCommandEvent resultEvent(wxEVT_COMMAND_MENU_SELECTED, m_event_id);
		m_caller->AddPendingEvent(resultEvent);
    }
#endif
	
	wxLogScript(wxT("Execution completed"));
}

void pgsApplication::set_connection(pgConn * conn)
{
	m_critical.Wait();
	
	if (m_defined_conn)
	{
		pdelete(m_connection);
		m_defined_conn = false;
	}
	
	m_connection = conn;
	m_critical.Post();
}

void pgsApplication::clear_symbols()
{
	m_critical.Wait();
	m_vars.clear();
	m_critical.Post();
}

#if !defined(NPGADMIN)
void pgsApplication::set_caller(wxWindow * caller, long event_id)
{
	m_caller = caller;
	m_event_id = event_id;
}
#endif

pgsOperand pgsApplication::variable(const wxString & name)
{
	m_critical.Wait();
	pgsOperand operand = pgsIdent(name).eval(m_vars);
	m_critical.Post();
	return operand;
}

wxString pgsApplication::string(const wxString & name)
{
	return variable(name)->value();
}

bool pgsApplication::variable(const wxString & name, const pgsOperand & value)
{
	m_critical.Wait();
	pgsAssign(name, value->clone()).eval(m_vars);
	m_critical.Post();
	return true;
}

bool pgsApplication::string(const wxString & name, const wxString & value)
{
	return variable(name, pnew pgsString(value));
}

bool pgsApplication::is_connection_valid() const
{
	return (m_connection->GetStatus() == PGCONN_OK);
}
