/*--------------------------------------------------------------------
 * FILE:
 *    conf.c
 *
 * NOTE:
 *
 * Portions Copyright (c) 2003-2009, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#include "postgres.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <netdb.h>
#include <errno.h>
#include <sys/file.h>

#include "libpq-fe.h"
#include "libpq-int.h"
#include "fe-auth.h"

#include "replicate.h"
#include "libpgc/libpgc.h"
#include "pglb/pglb.h"
#include "pgrp/pgreplicate.h"
#include "admin/pgc_admin.h"

int PGRget_Template_Data(char * path);
int PGRget_Admin_Conf_Data(char * path, char * fname);
int PGRinit_Server(char * path);
int PGRinit_Admin(char * path);
int PGRinit_Probe(char * path);
void PGR_Clear_Shm(void);
Physical_Server_Info * PGRget_Physical_Server_Rec(uint16_t physicalServerId);

static int get_pglb_temp(char * path);
static int get_pgrp_temp(char * path);
static int get_cluster_temp(char * path);
static int conf_set_server_table(ConfDataType * conf);
static int conf_set_admin_table(ConfDataType * conf);
static int conf_set_probe_table(ConfDataType * conf);
static int conf_set_pglb_table(ConfDataType * conf);
static int conf_set_cluster_table(ConfDataType * conf);
static int conf_set_pgrp_table(ConfDataType * conf);
static int conf_set_partial_table(ConfDataType * conf);
static int get_server_num(char * dir , char * fname, int type);
static int get_partial_num(char * dir);
static char is_start_tag(char * buf, char * tag);
static int count_server_num(FILE * fp);
static int alloc_server_tbl(void);
static int alloc_partial_tbl(void);
static int shmget_server_tbl(void);
static int shmget_physical_server_tbl(void);
static int shmget_admin_tbl(void);
static int shmget_probe_tbl(void);
static int shmget_pglb_tbl(void);
static int shmget_cluster_tbl(void);
static int shmget_pgrp_tbl(void);

/*--------------------------------------------------------------------
 * SYMBOL
 *    PGRget_Template_Data()
 * NOTES
 *    read template files
 * ARGS
 *    char * path: path of the setup file (I)
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGRget_Template_Data(char * path)
{
	char * func = "PGRget_Template_Data()";

	if (path == NULL)
	{
		path = ".";
	}
	if (get_pglb_temp(path) != STATUS_OK)
	{
		show_error("%s:get_pglb_temp failed",func);
		return STATUS_ERROR;
	}
	if (get_pgrp_temp(path) != STATUS_OK)
	{
		show_error("%s:get_pgrp_temp failed",func);
		return STATUS_ERROR;
	}
	if (get_cluster_temp(path) != STATUS_OK)
	{
		show_error("%s:get_cluster_temp failed",func);
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

static int
get_pglb_temp(char * path)
{
	char * func = "get_pglb_temp()";
	ConfDataType * conf = NULL;
	/*
	 * read pglb template file
	 */
	if (PGR_Get_Conf_Data(path,PGC_PGLB_TEMP_FILE) != STATUS_OK)
	{
		show_error("%s:PGR_Get_Conf_Data failed",func);
		return STATUS_ERROR;
	}
	/*
	 * set each datas into the tables
	 */
	conf = ConfData_Top;
	while (conf != (ConfDataType *)NULL) 
	{
		conf_set_pglb_table(conf);
		conf = (ConfDataType*)conf->next;
	}
	PGR_Free_Conf_Data();
	return STATUS_OK;
}

static int
get_pgrp_temp(char * path)
{
	char * func = "get_pgrp_temp()";
	ConfDataType * conf = NULL;
	/*
	 * read pgrp template file
	 */
	if (PGR_Get_Conf_Data(path,PGC_PGRP_TEMP_FILE) != STATUS_OK)
	{
		show_error("%s:PGR_Get_Conf_Data failed",func);
		return STATUS_ERROR;
	}
	/*
	 * set each datas into the tables
	 */
	conf = ConfData_Top;
	while (conf != (ConfDataType *)NULL) 
	{
		conf_set_pgrp_table(conf);
		conf = (ConfDataType*)conf->next;
	}
	PGR_Free_Conf_Data();
	return STATUS_OK;
}

static int
get_cluster_temp(char * path)
{
	char * func = "get_cluster_temp()";
	ConfDataType * conf = NULL;
	/*
	 * read cluster db template file
	 */
	if (PGR_Get_Conf_Data(path,PGC_DB_TEMP_FILE) != STATUS_OK)
	{
		show_error("%s:PGR_Get_Conf_Data failed",func);
		return STATUS_ERROR;
	}
	/*
	 * set each datas into the tables
	 */
	conf = ConfData_Top;
	while (conf != (ConfDataType *)NULL) 
	{
		if (!STRCMP(conf->table, CLUSTER_SERVER_TAG))
		{
			conf_set_cluster_table(conf);
		}
		else if (!STRCMP(conf->table, NOT_REPLICATE_INFO_TAG))
		{
			conf_set_partial_table(conf);
		}
		conf = (ConfDataType*)conf->next;
	}
	PGR_Free_Conf_Data();
	return STATUS_OK;
}
/*--------------------------------------------------------------------
 * SYMBOL
 *    PGRget_Admin_Conf_Data()
 * NOTES
 *    Initialize mamory and tables
 * ARGS
 *    char * path: path of the setup file (I)
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGRget_Admin_Conf_Data(char * path, char * fname)
{
	char * func = "PGRget_Admin_Conf_Data()";
	ConfDataType * conf = NULL;

	if (path == NULL)
	{
		path = ".";
	}

	/*
	 * read configuration file
	 */
	if (PGR_Get_Conf_Data(path,fname) != STATUS_OK)
	{
		show_error("%s:PGR_Get_Conf_Data failed",func);
		return STATUS_ERROR;
	}

	/*
	 * set each datas into the tables
	 */
	conf = ConfData_Top;
	while (conf != (ConfDataType *)NULL) 
	{
		if (!STRCMP(conf->table, PHYSICAL_SERVER_INFO_TAG))
		{
			conf_set_server_table(conf);
		}
		else if (!STRCMP(conf->table, PGC_ADMIN_SERVER_INFO_TAG))
		{
			conf_set_admin_table(conf);
		}
		else if (!STRCMP(conf->table, PROBE_SERVER_INFO_TAG))
		{
			conf_set_probe_table(conf);
		}
		else if (!STRCMP(conf->table, LOAD_BALANCE_SERVER_TAG))
		{
			conf_set_pglb_table(conf);
		}
		else if (!STRCMP(conf->table, CLUSTER_SERVER_TAG))
		{
			conf_set_cluster_table(conf);
		}
		else if (!STRCMP(conf->table, NOT_REPLICATE_INFO_TAG))
		{
			conf_set_partial_table(conf);
		}
		else if (!STRCMP(conf->table, REPLICATION_SERVER_INFO_TAG))
		{
			conf_set_pgrp_table(conf);
		}
		conf = (ConfDataType*)conf->next;
	}
	PGR_Free_Conf_Data();
	return STATUS_OK;
}

int
PGRinit_Server(char * path)
{
	char * func = "PGRinit_Server()";
	if (get_server_num(path,PGC_SERVER_CONF_FILE, 0) != STATUS_OK)
	{
		show_error("%s:get_server_num failed",func);
		return STATUS_ERROR;
	}
	if (get_partial_num(path) != STATUS_OK)
	{
		show_error("%s:get_partial_num failed",func);
		return STATUS_ERROR;
	}
	if (alloc_server_tbl() != STATUS_OK)
	{
		show_error("%s:alloc_server_tbl failed",func);
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

int
PGRinit_Admin(char * path)
{
	char * func = "PGRinit_Admin()";

	if (get_server_num(path,PGC_ADMIN_CONF_FILE, SERVER_TYPE_ADMIN) != STATUS_OK)
	{
		show_error("%s:get_server_num failed",func);
		return STATUS_ERROR;
	}
	if (alloc_partial_tbl() != STATUS_OK)
	{
		show_error("%s:alloc_partial_tbl failed",func);
		return STATUS_ERROR;
	}
	if (shmget_server_tbl() != STATUS_OK)
	{
		show_error("%s:shmegt_server_tbl failed",func);
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

int
PGRinit_Probe(char * path)
{
	char * func = "PGRinit_Probe()";

	if (get_server_num(path,PGC_PROBE_CONF_FILE, SERVER_TYPE_PROBE) != STATUS_OK)
	{
		show_error("%s:get_server_num failed",func);
		return STATUS_ERROR;
	}
	if (shmget_server_tbl() != STATUS_OK)
	{
		show_error("%s:shmegt_server_tbl failed",func);
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

void
PGR_Clear_Shm(void)
{
	if (PhysicalServerTbl != NULL)
	{
		shmdt(PhysicalServerTbl);
		shmctl(ShmidTbl.PhysicalServerShmid, IPC_RMID,(struct shmid_ds *)NULL);
		PhysicalServerTbl = NULL;
	}
	if (AdminTbl != NULL)
	{
		shmdt(AdminTbl);
		shmctl(ShmidTbl.AdminShmid, IPC_RMID,(struct shmid_ds *)NULL);
		AdminTbl = NULL;
	}
	if (ProbeTbl != NULL)
	{
		shmdt(ProbeTbl);
		shmctl(ShmidTbl.ProbeShmid, IPC_RMID,(struct shmid_ds *)NULL);
		ProbeTbl = NULL;
	}
	if (PglbTbl != NULL)
	{
		shmdt(PglbTbl);
		shmctl(ShmidTbl.PglbShmid, IPC_RMID,(struct shmid_ds *)NULL);
		PglbTbl = NULL;
	}
	if (ClusterDbTbl != NULL)
	{
		shmdt(ClusterDbTbl);
		shmctl(ShmidTbl.ClusterShmid, IPC_RMID,(struct shmid_ds *)NULL);
		ClusterDbTbl = NULL;
	}
	if (PgrpTbl != NULL)
	{
		shmdt(PgrpTbl);
		shmctl(ShmidTbl.PgrpShmid, IPC_RMID,(struct shmid_ds *)NULL);
		PgrpTbl = NULL;
	}
	if (RecordNumTbl != NULL)
	{
		free(RecordNumTbl);
		RecordNumTbl = NULL;
	}
	if (PartialTbl  != NULL)
	{
		free(PartialTbl);
		PartialTbl = NULL;
	}
}

static int
conf_set_server_table(ConfDataType * conf)
{
	int rec_no = 0;
	Physical_Server_Info * server = (Physical_Server_Info *)NULL;
	Physical_Server_Info * end = (Physical_Server_Info *)NULL;

	if ((PhysicalServerTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("PhysicalServerTbl is not initialized");
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	if (rec_no >= RecordNumTbl->serverNum)
	{
		show_error("rec_no[%d] >= serverNum[%d]",rec_no,RecordNumTbl->serverNum);
		return STATUS_ERROR;
	}
	server = PhysicalServerTbl + rec_no;
	if (server == NULL)
	{
		show_error("physical server table is out of range");
		return STATUS_ERROR;
	}
	server->status = DATA_USE;
	end = PhysicalServerTbl + RecordNumTbl->serverNum;
	end->status = DATA_END;
	server->serverNo = rec_no;
	if (!STRCMP(conf->key, SERVER_ID_TAG ))
	{
		server->physicalServerId = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, HOST_NAME_TAG ))
	{
		strncpy(server->hostName,conf->value,sizeof(server->hostName));
		PGR_Set_Host_Name(server->ipAddr,conf->value);
	}
	else if (!STRCMP(conf->key, USER_NAME_TAG ))
	{
		strncpy(server->userName,conf->value,sizeof(server->userName));
	}
	/*
	else if (!STRCMP(conf->key, HOST_KEY_TAG ))
	{
		strncpy(server->hostKey,conf->value,sizeof(server->hostKey));
	}
	else if (!STRCMP(conf->key, PUBLICK_KEY_TAG ))
	{
		strncpy(server->pubKey, conf->value, sizeof(server->pubKey));
	}
	*/
	return STATUS_OK;
}

static int
conf_set_admin_table(ConfDataType * conf)
{
	int rec_no = 0;
	int i = 0;
	char * p1 = NULL;
	char * p2 = NULL;
	SSL_Server_Info * admin = (SSL_Server_Info *)NULL;
	SSL_Server_Info * end = (SSL_Server_Info *)NULL;
	if ((AdminTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("AdminTbl is not initialized");
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	/*
	if (rec_no >= RecordNumTbl->adminNum)
	{
		return STATUS_ERROR;
	}
	*/
	admin = AdminTbl + rec_no;
	if (admin == NULL)
	{
		show_error("admin table is out of range");
		return STATUS_ERROR;
	}
	admin->status = DATA_INIT;
	end = AdminTbl + RecordNumTbl->adminNum;
	end->status = DATA_END;
	admin->serverNo = rec_no;
	if (!STRCMP(conf->key, SERVER_ID_TAG ))
	{
		p1 = conf->value;
		i = 0;
		while ((p2 = strchr(p1,',')) != NULL)
		{
			*p2 = '\0';
			(admin + i) ->physicalServerId = atoi(p1);
			p1 = p2+1;
			i ++;
		}
		(admin + i) ->physicalServerId = atoi(p1);
	}
	else if (!STRCMP(conf->key, CONF_DIR_TAG ))
	{
		strncpy(admin->workPath,conf->value,sizeof(admin->workPath));
		memset(admin->serverCertFile, 0, sizeof(admin->serverCertFile));
		sprintf(admin->serverCertFile,"%s/%s",admin->workPath,SERVER_CERTFILE);
		memset(admin->clientCertFile, 0, sizeof(admin->clientCertFile));
		sprintf(admin->clientCertFile,"%s/%s",admin->workPath,CLIENT_CERTFILE);
	}
	else if (!STRCMP(conf->key, HOST_NAME_TAG ))
	{
		PGR_Set_Host_Name(admin->hostName,conf->value);
	}
	else if (!STRCMP(conf->key, PORT_TAG ))
	{
		admin->portNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, SSL_PATH_TAG ))
	{
		strncpy(admin->sslPath,conf->value,sizeof(admin->sslPath));
	}
	return STATUS_OK;
}

static int
conf_set_probe_table(ConfDataType * conf)
{
	int rec_no = 0;
	int i = 0;
	char * p1 = NULL;
	char * p2 = NULL;

	SSL_Server_Info * probe = (SSL_Server_Info *)NULL;
	SSL_Server_Info * end = (SSL_Server_Info *)NULL;
	if ((ProbeTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("ProbeTbl is not initialized");
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	if (rec_no >= RecordNumTbl->probeNum)
	{
		return STATUS_ERROR;
	}
	probe = ProbeTbl + rec_no;
	if (probe == NULL)
	{
		show_error("probe table is out of range");
		return STATUS_ERROR;
	}
	probe->status = DATA_INIT;
	end = ProbeTbl + RecordNumTbl->probeNum;
	end->status = DATA_END;
	probe->serverNo = rec_no;
	if (!STRCMP(conf->key, SERVER_ID_TAG ))
	{
		p1 = conf->value;
		i = 0;
		while ((p2 = strchr(p1,',')) != NULL)
		{
			*p2 = '\0';
			(probe + i) ->physicalServerId = atoi(p1);
			p1 = p2+1;
			i ++;
		}
		(probe + i) ->physicalServerId = atoi(p1);
	}
	else if (!STRCMP(conf->key, HOST_NAME_TAG ))
	{
		PGR_Set_Host_Name(probe->hostName,conf->value);
	}
	else if (!STRCMP(conf->key, PORT_TAG ))
	{
		probe->portNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, CONF_DIR_TAG ))
	{
		strncpy(probe->workPath,conf->value,sizeof(probe->workPath));
		memset(probe->serverCertFile, 0, sizeof(probe->serverCertFile));
		sprintf(probe->serverCertFile,"%s/%s",probe->workPath,SERVER_CERTFILE);
		memset(probe->clientCertFile, 0, sizeof(probe->clientCertFile));
		sprintf(probe->clientCertFile,"%s/%s",probe->workPath,CLIENT_CERTFILE);
	}
	else if (!STRCMP(conf->key, BIN_DIR_TAG ))
	{
		strncpy(probe->binPath,conf->value,sizeof(probe->binPath));
	}
	else if (!STRCMP(conf->key, SSL_PATH_TAG ))
	{
		strncpy(probe->sslPath,conf->value,sizeof(probe->sslPath));
	}
	return STATUS_OK;
}

static int
conf_set_pglb_table(ConfDataType * conf)
{
	int rec_no = 0;
	Pglb_Info * pglb = (Pglb_Info *)NULL;
	int i = 0;
	char * p1 = NULL;
	char * p2 = NULL;

	if ((PglbTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("PglbTbl is not initialized");
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	if (rec_no >= RecordNumTbl->pglbNum)
	{
		return STATUS_ERROR;
	}
	pglb = PglbTbl + rec_no;
	if (pglb == NULL)
	{
		show_error("pglb table is out of range");
		return STATUS_ERROR;
	}
	pglb->serverNo = rec_no;
	if (!STRCMP(conf->key, SERVER_ID_TAG ))
	{
		p1 = conf->value;
		i = 0;
		while ((p2 = strchr(p1,',')) != NULL)
		{
			*p2 = '\0';
			(pglb + i) ->physicalServerId = atoi(p1);
			p1 = p2+1;
			i ++;
		}
		(pglb + i) ->physicalServerId = atoi(p1);
	}
	else if (!STRCMP(conf->key, HOST_NAME_TAG ))
	{
		PGR_Set_Host_Name(pglb->hostName,conf->value);
	}
	else if (!STRCMP(conf->key, BACKEND_SOCKET_DIR_TAG ))
	{
		strncpy(pglb->backendSocketDir,conf->value,sizeof(pglb->backendSocketDir));
	}
	else if (!STRCMP(conf->key, RECV_PORT_TAG ))
	{
		pglb->receivePortNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, RECOVERY_PORT_TAG ))
	{
		pglb->recoveryPortNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, MAX_CLUSTER_TAG ))
	{
		pglb->maxClusterNum = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, USE_CONNECTION_POOL_TAG ))
	{
		strncpy(pglb->usePool,conf->value,sizeof(pglb->usePool));
	}
	else if (!STRCMP(conf->key, LIFECHECK_TIMEOUT_TAG ))
	{
		strncpy(pglb->lifecheckTimeout,conf->value,sizeof(pglb->lifecheckTimeout));
	}
	else if (!STRCMP(conf->key, LIFECHECK_INTERVAL_TAG ))
	{
		strncpy(pglb->lifecheckInterval,conf->value,sizeof(pglb->lifecheckInterval));
	}
	else if (!STRCMP(conf->key, CONNECTION_LIFE_TIME_TAG ))
	{
		strncpy(pglb->connectionLifetime,conf->value,sizeof(pglb->connectionLifetime));
	}
	else if (!STRCMP(conf->key, LOG_FILE_NAME_TAG ))
	{
		strncpy(pglb->logFileName,conf->value,sizeof(pglb->logFileName));
	}
	else if (!STRCMP(conf->key, LOG_FILE_SIZE_TAG ))
	{
		strncpy(pglb->logFileSize,conf->value,sizeof(pglb->logFileSize));
	}
	else if (!STRCMP(conf->key, LOG_ROTATION_TAG ))
	{
		pglb->logRotate = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, CONF_DIR_TAG ))
	{
		strncpy(pglb->workPath,conf->value,sizeof(pglb->workPath));
	}
	else if (!STRCMP(conf->key, BIN_DIR_TAG ))
	{
		strncpy(pglb->binPath,conf->value,sizeof(pglb->binPath));
	}
	return STATUS_OK;
}

static int
conf_set_cluster_table(ConfDataType * conf)
{
	char * func="conf_set_cluster_table()";
	int rec_no = 0;
	int i = 0;
	char * p1 = NULL;
	char * p2 = NULL;

	Cluster_Info * cluster = (Cluster_Info *)NULL;
	if ((ClusterDbTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("%s:ClusterDbTbl is not initialized",func);
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	if (rec_no >= RecordNumTbl->clusterNum)
	{
		return STATUS_ERROR;
	}
	cluster = ClusterDbTbl + rec_no;
	if (cluster == NULL)
	{
		show_error("%s:cluster table is out of range",func);
		return STATUS_ERROR;
	}
	cluster->serverNo = rec_no;
	if (!STRCMP(conf->key, SERVER_ID_TAG ))
	{
		p1 = conf->value;
		i = 0;
		while ((p2 = strchr(p1,',')) != NULL)
		{
			*p2 = '\0';
			(cluster + i) ->physicalServerId = atoi(p1);
			p1 = p2+1;
			i ++;
		}
		(cluster + i) ->physicalServerId = atoi(p1);
	}
	else if (!STRCMP(conf->key, HOST_NAME_TAG ))
	{
		PGR_Set_Host_Name(cluster->hostName,conf->value);
	}
	else if (!STRCMP(conf->key, PORT_TAG ))
	{
		cluster->portNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, RECOVERY_PORT_TAG ))
	{
		cluster->recoveryPortNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, MAX_CONNECT_TAG ))
	{
		cluster->maxConnections = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, RSYNC_PATH_TAG ))
	{
		strncpy(cluster->rsyncPath,conf->value,sizeof(cluster->rsyncPath));
	}
	else if (!STRCMP(conf->key, RSYNC_OPTION_TAG ))
	{
		strncpy(cluster->rsyncOption,conf->value,sizeof(cluster->rsyncOption));
	}
	else if (!STRCMP(conf->key, RSYNC_COMPRESS_TAG ))
	{
		strncpy(cluster->rsyncCompress,conf->value,sizeof(cluster->rsyncCompress));
	}
	else if (!STRCMP(conf->key, RSYNC_TIMEOUT_TAG ))
	{
		strncpy(cluster->rsyncTimeout,conf->value,sizeof(cluster->rsyncTimeout));
	}
	else if (!STRCMP(conf->key, RSYNC_BWLIMIT_TAG ))
	{
		strncpy(cluster->rsyncBwlimit,conf->value,sizeof(cluster->rsyncBwlimit));
	}
	else if (!STRCMP(conf->key, PG_DUMP_PATH_TAG ))
	{
		strncpy(cluster->pgdumpPath,conf->value,sizeof(cluster->pgdumpPath));
	}
	else if (!STRCMP(conf->key, PING_PATH_TAG ))
	{
		strncpy(cluster->pingPath,conf->value,sizeof(cluster->pingPath));
	}
	else if (!STRCMP(conf->key, STAND_ALONE_TAG ))
	{
		strncpy(cluster->whenStandAlone,conf->value,sizeof(cluster->whenStandAlone));
	}
	else if (!STRCMP(conf->key, TIMEOUT_TAG ))
	{
		strncpy(cluster->replicationTimeout,conf->value,sizeof(cluster->replicationTimeout));
	}
	else if (!STRCMP(conf->key, LIFECHECK_TIMEOUT_TAG ))
	{
		strncpy(cluster->lifecheckTimeout,conf->value,sizeof(cluster->lifecheckTimeout));
	}
	else if (!STRCMP(conf->key, LIFECHECK_INTERVAL_TAG ))
	{
		strncpy(cluster->lifecheckInterval,conf->value,sizeof(cluster->lifecheckInterval));
	}
	else if (!STRCMP(conf->key, CONF_DIR_TAG ))
	{
		strncpy(cluster->workPath,conf->value,sizeof(cluster->workPath));
	}
	else if (!STRCMP(conf->key, BIN_DIR_TAG ))
	{
		strncpy(cluster->binPath,conf->value,sizeof(cluster->binPath));
	}
	return STATUS_OK;
}

static int
conf_set_pgrp_table(ConfDataType * conf)
{
	int rec_no = 0;
	int i = 0;
	char * p1 = NULL;
	char * p2 = NULL;

	Pgrp_Info * pgrp = (Pgrp_Info *)NULL;
	if ((PgrpTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("PgrpTbl is not initialized");
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	if (rec_no >= RecordNumTbl->pgrpNum)
	{
		return STATUS_ERROR;
	}
	pgrp = PgrpTbl + rec_no;
	if (pgrp == NULL)
	{
		show_error("pgrp table is out of range");
		return STATUS_ERROR;
	}
	pgrp->serverNo = rec_no;
	if (!STRCMP(conf->key, SERVER_ID_TAG ))
	{
		p1 = conf->value;
		i = 0;
		while ((p2 = strchr(p1,',')) != NULL)
		{
			*p2 = '\0';
			(pgrp + i) ->physicalServerId = atoi(p1);
			p1 = p2+1;
			i ++;
		}
		(pgrp + i) ->physicalServerId = atoi(p1);
	}
	else if (!STRCMP(conf->key, HOST_NAME_TAG ))
	{
		PGR_Set_Host_Name(pgrp->hostName,conf->value);
	}
	else if (!STRCMP(conf->key, REPLICATE_PORT_TAG ))
	{
		pgrp->replicationPortNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, RECOVERY_PORT_TAG ))
	{
		pgrp->recoveryPortNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, RLOG_PORT_TAG ))
	{
		pgrp->RLogPortNumber = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, USE_REPLICATION_LOG_TAG ))
	{
		strncpy(pgrp->useRlog,conf->value,sizeof(pgrp->useRlog));
	}
	else if (!STRCMP(conf->key, TIMEOUT_TAG ))
	{
		strncpy(pgrp->replicationTimeout,conf->value,sizeof(pgrp->replicationTimeout));
	}
	else if (!STRCMP(conf->key, LIFECHECK_TIMEOUT_TAG ))
	{
		strncpy(pgrp->lifecheckTimeout,conf->value,sizeof(pgrp->lifecheckTimeout));
	}
	else if (!STRCMP(conf->key, LIFECHECK_INTERVAL_TAG ))
	{
		strncpy(pgrp->lifecheckInterval,conf->value,sizeof(pgrp->lifecheckInterval));
	}
	else if (!STRCMP(conf->key, LOG_FILE_NAME_TAG ))
	{
		strncpy(pgrp->logFileName,conf->value,sizeof(pgrp->logFileName));
	}
	else if (!STRCMP(conf->key, LOG_FILE_SIZE_TAG ))
	{
		strncpy(pgrp->logFileSize,conf->value,sizeof(pgrp->logFileSize));
	}
	else if (!STRCMP(conf->key, LOG_ROTATION_TAG ))
	{
		pgrp->logRotate = atoi(conf->value);
	}
	else if (!STRCMP(conf->key, CONF_DIR_TAG ))
	{
		strncpy(pgrp->workPath,conf->value,sizeof(pgrp->workPath));
	}
	else if (!STRCMP(conf->key, BIN_DIR_TAG ))
	{
		strncpy(pgrp->binPath,conf->value,sizeof(pgrp->binPath));
	}
	return STATUS_OK;
}

static int
conf_set_partial_table(ConfDataType * conf)
{
	int rec_no = 0;
	Partial_Info * partial = (Partial_Info *)NULL;
	if ((PartialTbl == NULL) || (RecordNumTbl == NULL))
	{
		show_error("PartialTbl is not initialized");
		return STATUS_ERROR;
	}

	rec_no = conf->rec_no;
	if (rec_no >= RecordNumTbl->partialNum)
	{
		return STATUS_ERROR;
	}
	partial = PartialTbl + rec_no;
	if (partial == NULL)
	{
		show_error("pgrp table is out of range");
		return STATUS_ERROR;
	}
	partial->serverNo = conf->parent_rec_no;
	partial->recNo = rec_no;
	if (!STRCMP(conf->key, DB_NAME_TAG ))
	{
		strncpy(partial->dbName,conf->value,sizeof(partial->dbName));
	}
	else if (!STRCMP(conf->key, TABLE_NAME_TAG ))
	{
		strncpy(partial->tableName,conf->value,sizeof(partial->tableName));
	}
	else if (!STRCMP(conf->key, QUERY_TAG ))
	{
		strncpy(partial->query,conf->value,sizeof(partial->query));
	}
	return STATUS_OK;
}

static int
get_server_num(char * dir , char * fname, int type)
{
	char * func = "get_server_num";
	char * conf_file;
	FILE * fp = NULL;
	char buf[1024];
	char * p = NULL;

	if ((dir == NULL) || ( fname == NULL))
	{
		return STATUS_ERROR;
	}
	conf_file = malloc(strlen(dir) + strlen(fname) + 2);
	if (conf_file == NULL)
	{
		show_error("%s:malloc failed (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	sprintf(conf_file,"%s/%s",dir,fname);
	RecordNumTbl = malloc(sizeof(Record_Num_Info));
	if (RecordNumTbl == NULL)
	{
		free(conf_file);
		show_error("%s:malloc failed (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memset(RecordNumTbl, 0, sizeof(Record_Num_Info));
	
	if ((fp = fopen(conf_file,"r")) == NULL)
	{
		show_error("%s:%s cannnot open",func,conf_file);
		free(conf_file);
		free(RecordNumTbl);
		RecordNumTbl = NULL;
		return STATUS_ERROR;
	}
	memset(buf,0,sizeof(buf));

	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		if ((p = strchr(buf,'#')) != NULL)
		{
			*p = '\0';
		}
		if (is_start_tag(buf, PHYSICAL_SERVER_INFO_TAG))
		{
			RecordNumTbl->serverNum ++;
		}
		if (is_start_tag(buf, PGC_ADMIN_SERVER_INFO_TAG))
		{
			if (type > 0)
				RecordNumTbl->adminNum ++;
			else
				RecordNumTbl->adminNum = count_server_num(fp);
		}
		if (is_start_tag(buf, PROBE_SERVER_INFO_TAG))
		{
			if (type > 0)
				RecordNumTbl->probeNum ++;
			else
				RecordNumTbl->probeNum = count_server_num(fp);
		}
		if (is_start_tag(buf, LOAD_BALANCE_SERVER_TAG))
		{
			if (type > 0)
				RecordNumTbl->pglbNum ++;
			else
				RecordNumTbl->pglbNum  = count_server_num(fp);
		}
		if (is_start_tag(buf, CLUSTER_SERVER_TAG))
		{
			if (type > 0)
				RecordNumTbl->clusterNum ++;
			else
				RecordNumTbl->clusterNum = count_server_num(fp);
		}
		if (is_start_tag(buf, REPLICATION_SERVER_INFO_TAG))
		{
			if (type > 0)
				RecordNumTbl->pgrpNum ++;
			else
				RecordNumTbl->pgrpNum = count_server_num(fp);
		}
		if (is_start_tag(buf, NOT_REPLICATE_INFO_TAG))
		{
			RecordNumTbl->partialNum ++;
		}
		memset(buf,0,sizeof(buf));
	}
	fclose(fp);
	free (conf_file);
	return STATUS_OK;
}


static int
get_partial_num(char * dir)
{
	char * func = "get_partial_num";
	char * conf_file = NULL;
	FILE * fp = NULL;
	char buf[1024];
	char * p = NULL;

	char * fname = PGC_DB_TEMP_FILE ;
	if ((dir == NULL) || ( fname == NULL))
	{
		return STATUS_ERROR;
	}
	conf_file = malloc(strlen(dir) + strlen(fname) + 2);
	if (conf_file == NULL)
	{
		show_error("%s:malloc failed (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	sprintf(conf_file,"%s/%s",dir,fname);
	if (RecordNumTbl == NULL)
	{
		RecordNumTbl = malloc(sizeof(Record_Num_Info));
		if (RecordNumTbl == NULL)
		{
			free(conf_file);
			show_error("%s:malloc failed (%s)",func,strerror(errno));
			return STATUS_ERROR;
		}
		memset(RecordNumTbl, 0, sizeof(Record_Num_Info));
	}
	
	if ((fp = fopen(conf_file,"r")) == NULL)
	{
		show_error("%s:%s cannnot open",func,conf_file);
		free(conf_file);
		return STATUS_ERROR;
	}
	memset(buf,0,sizeof(buf));
	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		if ((p = strchr(buf,'#')) != NULL)
		{
			*p = '\0';
		}
		if (is_start_tag(buf, NOT_REPLICATE_INFO_TAG))
		{
			RecordNumTbl->partialNum ++;
		}
		memset(buf,0,sizeof(buf));
	}
	fclose(fp);
	free (conf_file);
	return STATUS_OK;
}

static char
is_start_tag(char * buf, char * tag)
{
	char start_tag[1024];
	char * p = NULL;

	if ((p = strchr(buf,'<')) == NULL)
		return FALSE;
	memset(start_tag,0,sizeof(start_tag));
	sprintf(start_tag,"<%s>",tag);
	if (!strncasecmp(p,start_tag,strlen(start_tag)))
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

static int
count_server_num(FILE * fp)
{
	int cnt = 0;
	char * p = NULL;
	char buf[1024];

	if (fp == NULL)
		return 0;
	memset(buf,0,sizeof(buf));
	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		if (is_start_tag(buf, SERVER_ID_TAG))
		{
			p = buf;
			while ((p = strchr(p,',')) != NULL)
			{
				cnt ++;
				p += 1;	
			}
			cnt ++;
		}
		else
		{
			break;
		}
		memset(buf,0,sizeof(buf));
	}
	return cnt;	
}

static int
alloc_server_tbl(void)
{
	int size = 0;

	/* Allocate PhysicalServerTbl */
	size = sizeof(Physical_Server_Info) * RecordNumTbl->serverNum +1;
	PhysicalServerTbl = malloc(size);
	memset(PhysicalServerTbl, 0, size);

	/* Allocate AdminTbl */
	size = sizeof(SSL_Server_Info) * RecordNumTbl->adminNum +1;
	AdminTbl = malloc(size);
	memset(AdminTbl, 0, size);

	/* Allocate ProbeTbl */
	size = sizeof(SSL_Server_Info) * RecordNumTbl->probeNum +1;
	ProbeTbl = malloc(size);
	memset(ProbeTbl, 0, size);

	/* Allocate PglbTbl */
	size = sizeof(Pglb_Info) * RecordNumTbl->pglbNum +1;
	PglbTbl = malloc(size);
	memset(PglbTbl, 0, size);

	/* Allocate ClusterDbTbl */
	size = sizeof(Cluster_Info) * RecordNumTbl->clusterNum +1;
	ClusterDbTbl = malloc(size);
	memset(ClusterDbTbl, 0, size);

	/* Allocate ClusterDbTbl */
	size = sizeof(Pgrp_Info) * RecordNumTbl->pgrpNum +1;
	PgrpTbl = malloc(size);
	memset(PgrpTbl, 0, size);

	/* Allocate PartialTBL */
	size = sizeof(Partial_Info) * RecordNumTbl->partialNum +1;
	PartialTbl = malloc(size);
	memset(PartialTbl, 0, size);

	return STATUS_OK;
}

static int
alloc_partial_tbl(void)
{
	int size = 0;

	size = sizeof(Partial_Info) * RecordNumTbl->partialNum;
	if (size > 0)
	{
		PartialTbl = malloc(size);
		memset(PartialTbl, 0, size);
	}

	return STATUS_OK;
}

static int
shmget_server_tbl(void)
{
	if (shmget_physical_server_tbl() != STATUS_OK)
	{
		return STATUS_ERROR;
	}
	if (shmget_admin_tbl() != STATUS_OK)
	{
		return STATUS_ERROR;
	}
	if (shmget_probe_tbl() != STATUS_OK)
	{
		return STATUS_ERROR;
	}
	if (shmget_pgrp_tbl() != STATUS_OK)
	{
		return STATUS_ERROR;
	}
	if (shmget_cluster_tbl() != STATUS_OK)
	{
		return STATUS_ERROR;
	}
	if (shmget_pglb_tbl() != STATUS_OK)
	{
		return STATUS_ERROR;
	}

	return STATUS_OK;
}

static int
shmget_physical_server_tbl(void)
{
	/* char * func = "shmget_physical_server_tbl()";*/
	int size = 0;

	size = sizeof(Physical_Server_Info) * MAX_PROBE_SERVER ;
	ShmidTbl.PhysicalServerShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (ShmidTbl.PhysicalServerShmid < 0)
	{
		show_error("ShmidTbl.PhysicalServerShmid get failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	PhysicalServerTbl = (Physical_Server_Info *)shmat(ShmidTbl.PhysicalServerShmid,0,0);
	if (PhysicalServerTbl == (Physical_Server_Info *)-1)
	{
		show_error("shmat failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(PhysicalServerTbl, 0, size);

	return STATUS_OK;
}

static int
shmget_admin_tbl(void)
{
	int size = 0;

	size = sizeof(SSL_Server_Info) * MAX_ADMIN_SERVER ;
	ShmidTbl.AdminShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (ShmidTbl.AdminShmid < 0)
	{
		show_error("ShmidTbl.AdminShmid get failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	AdminTbl = (SSL_Server_Info *)shmat(ShmidTbl.AdminShmid,0,0);
	if (AdminTbl == (SSL_Server_Info *)-1)
	{
		show_error("shmat failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(AdminTbl, 0, size);

	return STATUS_OK;
}

static int
shmget_probe_tbl(void)
{
	int size = 0;

	size = sizeof(SSL_Server_Info) * MAX_PROBE_SERVER ;
	ShmidTbl.ProbeShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (ShmidTbl.ProbeShmid < 0)
	{
		show_error("ShmidTbl.ProbeShmid get failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	ProbeTbl = (SSL_Server_Info *)shmat(ShmidTbl.ProbeShmid,0,0);
	if (ProbeTbl == (SSL_Server_Info *)-1)
	{
		show_error("shmat failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(ProbeTbl, 0, size);

	return STATUS_OK;
}

static int
shmget_pglb_tbl(void)
{
	int size = 0;

	size = sizeof(Pglb_Info) * MAX_PGLB_SERVER ;
	ShmidTbl.PglbShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (ShmidTbl.PglbShmid < 0)
	{
		show_error("ShmidTbl.PglbShmid get failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	PglbTbl = (Pglb_Info *)shmat(ShmidTbl.PglbShmid,0,0);
	if (PglbTbl == (Pglb_Info *)-1)
	{
		show_error("shmat failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(PglbTbl, 0, size);

	return STATUS_OK;
}

static int
shmget_cluster_tbl(void)
{
	int size = 0;

	size = sizeof(Cluster_Info) * MAX_DB_SERVER ;
	ShmidTbl.ClusterShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (ShmidTbl.ClusterShmid < 0)
	{
		show_error("ShmidTbl.ClusterShmid get failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	ClusterDbTbl = (Cluster_Info *)shmat(ShmidTbl.ClusterShmid,0,0);
	if (ClusterDbTbl == (Cluster_Info *)-1)
	{
		show_error("shmat failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(ClusterDbTbl, 0, size);

	return STATUS_OK;
}

static int
shmget_pgrp_tbl(void)
{
	int size = 0;

	size = sizeof(Pgrp_Info) * MAX_PGLB_SERVER ;
	ShmidTbl.PgrpShmid = shmget(IPC_PRIVATE,size,IPC_CREAT | IPC_EXCL | 0600);
	if (ShmidTbl.PgrpShmid < 0)
	{
		show_error("ShmidTbl.PgrpShmid get failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	PgrpTbl = (Pgrp_Info *)shmat(ShmidTbl.PgrpShmid,0,0);
	if (PgrpTbl == (Pgrp_Info *)-1)
	{
		show_error("shmat failed (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	memset(PgrpTbl, 0, size);

	return STATUS_OK;
}

Physical_Server_Info *
PGRget_Physical_Server_Rec(uint16_t physicalServerId)
{
	int i;
	Physical_Server_Info * server = PhysicalServerTbl;

	if ((server == NULL) || (RecordNumTbl == NULL))
	{
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->serverNum ; i ++)
	{
		if ((server+i)->physicalServerId == physicalServerId)
		{
			return (server+i);
		}
	}
	return NULL;
}
