#include "connection.h"
#include "statement.h"
#include "error.h"



void InitDiag(Diag* pDiag)
{
	pDiag->header.number = 0;
	pDiag->records       = NULL;
}

void FreeDiag(Diag* pDiag)
{
	free(pDiag->records);
	pDiag->header.number = 0;
	pDiag->records       = NULL;
}

/*-----------------------------------------------------------------------------
 * SetError
 *-----------------------------------------------------------------------------
 */
SQLRETURN
SetError(SQLSMALLINT HandleType, SQLHANDLE Handle, ERR Error, const TCHAR* szErrParams, ...)
{
	int       i;
	Diag*     pDiag;
	DIAG_REC* pDIAG_REC;

	/* check input handle */
	switch (HandleType)
	{
		case SQL_HANDLE_ENV:
			pDiag = &((Environment*)Handle)->diag;
			break;
		case SQL_HANDLE_DBC:
			pDiag = &((Connection*)Handle)->diag;
			break;
		case SQL_HANDLE_STMT:
			pDiag = &((Statement*)Handle)->diag;
			break;
		case SQL_HANDLE_DESC:
			pDiag = &((Descriptor*)Handle)->diag;
			break;
		default:
			return SQL_ERROR;
	}

	if (ERR_CLEAR == Error)
	{
		/* clear all errors on handle */
		for (i=0;i < pDiag->header.number;i++)
			free(pDiag->records[i].message_text);
		pDiag->header.number = 0;
		return SQL_SUCCESS;
	}

	/* realloc space (increase it for one more error message)*/
	if (pDiag->header.number >= pDiag->header.allocated)
	{
		if (NULL != (pDIAG_REC = (DIAG_REC*)malloc((pDiag->header.number + 10)*sizeof(DIAG_REC))))
		{
			pDiag->header.allocated = pDiag->header.number + 10;
			memcpy(pDIAG_REC, pDiag->records, pDiag->header.number*sizeof(DIAG_REC));
			free(pDiag->records);
			pDiag->records = pDIAG_REC;
		}
		else
			return SQL_ERROR;
	}

	pDIAG_REC = &pDiag->records[pDiag->header.number++];

	/* set new error */
	if (NULL != szErrParams)
	{
		va_list vaList;
		va_list vaCopy;

		va_start(vaList, szErrParams);
#ifdef WIN32
		vaCopy = vaList;
#else
		va_copy(vaCopy, vaList);
#endif		
		pDIAG_REC->message_text = PrepareText(c_Errors[Error].text, szErrParams, vaCopy);
		va_end(vaList);
	}
	else
	{
		pDIAG_REC->message_text = NULL;
	}

	pDIAG_REC->native = Error;
	
	return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * GetError
 *-----------------------------------------------------------------------------
 */
 
SQLRETURN
GetDiagData(SQLSMALLINT HandleType, SQLHANDLE Handle, SQLSMALLINT RecNumber, SQLSMALLINT DiagIdentifier,
            SQLTCHAR* Sqlstate, SQLINTEGER* NativeError, SQLTCHAR* MessageText,
            SQLSMALLINT BufferLength, SQLSMALLINT* TextLength)
{
	SQLRETURN       nRet       = SQL_SUCCESS;
	const SQLTCHAR* pRetString = NULL;
	Diag*           pDiag;
	TCHAR*          pDSN;

	SQLINTEGER  uInteger;
	
	/* get Diag structure */
	switch (HandleType)
	{
		case SQL_HANDLE_ENV:
			pDiag = &((Environment*)Handle)->diag;
			pDSN = _T("");
			break;
		case SQL_HANDLE_DBC:
			pDiag = &((Connection*)Handle)->diag;
			pDSN = ((Connection*)Handle)->dsn;
			break;
		case SQL_HANDLE_STMT:
			pDiag = &((Statement*)Handle)->diag;
			pDSN = ((Statement*)Handle)->connection->dsn;
			break;
		case SQL_HANDLE_DESC:
			pDiag = &((Descriptor*)Handle)->diag;
			pDSN = ((Descriptor*)Handle)->connection->dsn;
			break;
		default:
			return SQL_ERROR;
	}

	/* RecNumber must be in valid interval */
	if (RecNumber <= 0 ||
	    RecNumber > pDiag->header.number ||
			0 == pDiag->header.number
		 )
		return SQL_NO_DATA;

	RecNumber--;
	switch(DiagIdentifier)
	{
		/* SQLDiagRec was called */
		case 0:                                  /* 0 */
			if (NULL != Sqlstate)
                        {
				_tcsncpy(Sqlstate, c_Errors[pDiag->records[RecNumber].native].sql_state, 5);
                                Sqlstate[5] = '\0';
                        }
			if (NULL != NativeError)
				*NativeError = pDiag->records[RecNumber].native;
		/* WARNING! no break! */

		/* string values */
		case SQL_DIAG_MESSAGE_TEXT:              /* 6 */
			pRetString = (NULL != pDiag->records[RecNumber].message_text) ? pDiag->records[RecNumber].message_text : c_Errors[pDiag->records[RecNumber].native].text;
			break;
		case SQL_DIAG_SQLSTATE:                  /* 4 */
			pRetString = c_Errors[pDiag->records[RecNumber].native].sql_state;
			break;
		case SQL_DIAG_DYNAMIC_FUNCTION:          /* 7 */
			pRetString = pDiag->header.dynamic_function;
			break;
		case SQL_DIAG_CLASS_ORIGIN:              /* 8 */
		case SQL_DIAG_SUBCLASS_ORIGIN:           /* 9 */
			pRetString = (_T('I') == c_Errors[pDiag->records[RecNumber].native].sql_state[0]) ? _T("ODBC 3.0") : _T("ISO 9075");
			break;
		case SQL_DIAG_CONNECTION_NAME:          /* 10 */
			pRetString = (SQL_HANDLE_DBC == HandleType) ? ((Connection*)Handle)->parameters[DATABASE_PARAM] : _T("");
			break;
		case SQL_DIAG_SERVER_NAME:              /* 11 */
			pRetString = pDSN;
			break;

		/* numeric values */
		case SQL_DIAG_NUMBER:                    /* 2 */
			uInteger = pDiag->header.number;
			break;
		case SQL_DIAG_ROW_COUNT:                 /* 3 */
			uInteger = pDiag->header.row_count;
			break;
		case SQL_DIAG_NATIVE:                    /* 5 */
			uInteger = pDiag->records[RecNumber].native;
			break;
		case SQL_DIAG_DYNAMIC_FUNCTION_CODE:    /* 12 */
			uInteger = pDiag->header.bynamic_function_code;
			break;
		case SQL_DIAG_CURSOR_ROW_COUNT:      /* -1249 */
			uInteger = pDiag->header.cursor_row_count;
			break;
		case SQL_DIAG_ROW_NUMBER:            /* -1248 */
			uInteger = pDiag->records[RecNumber].row_number;
			break;
		case SQL_DIAG_COLUMN_NUMBER:         /* -1247 */
			uInteger = pDiag->records[RecNumber].column_number;
			break;
		default:
			return SQL_ERROR;
	}

	/* return string value */
	if (NULL != pRetString)
		nRet = ReturnString(MessageText, BufferLength, TextLength, pRetString, SQL_NTS);
	else
		memcpy(MessageText, &uInteger, sizeof(uInteger));

	return nRet;
}
