#!/usr/sbin/dtrace -qs
/*
 * USAGE      : ./query_time.d
 *
 * DESCRIPTION:
 *   This script shows the followings:
 *     - time taken to execute a query
 *     - read counts by tablespace, database, table/index
 *     - read cache hits and misses and average time for both
 *     - write counts by tablespace, database, table/index and average time
 *
 * EXAMPLE    :
 *   (1) Run psql
 *   (2) Run query_time.d
 *   (3) Run a SQL query from psql
 *
 * AUTHOR     : Robert Lor <robert.lor@sun.com>
 *
 * $Header$
 */

postgresql*:::query-start
{
	self->ts = timestamp;
	self->pid = pid;
}

postgresql*:::buffer-read-start
/self->pid/
{
	self->readts = timestamp;
}

postgresql*:::buffer-read-done
/self->pid && arg7/
{
	/* Buffer cache hit */
	@read_count[arg2, arg3, arg4] = count();
	@read_hit_total["Total buffer cache hits      : "] = count();
	@read_hit_time["Average read time from cache : "] = avg (timestamp - self->readts);
	self->readts = 0;
}

postgresql*:::buffer-read-done
/self->pid && !arg7/
{
	/* Buffer cache miss */
	@read_count[arg2, arg3, arg4] = count();
	@read_miss_total["Total buffer cache misses    : "] = count();
	@read_miss_time["Average read time from disk  : "] = avg (timestamp - self->readts);
	self->readts = 0;
}

postgresql*:::buffer-flush-start
/self->pid/
{
	self->writets = timestamp;
}

postgresql*:::buffer-flush-done
/self->pid/
{
	@write_count[arg2, arg3, arg4] = count();
	@write_time["Average write time to disk   : "] = avg (timestamp - self->writets);
	self->writets = 0;
}

postgresql*:::query-done
/self->ts && self->pid == pid/
{
	printf ("Backend PID    : %d\n", pid);
	printf ("SQL Statement  : %s\n", copyinstr(arg0));
	printf ("Execution time : %d.%03d sec \n", (timestamp - self->ts) / 1000000000, ((timestamp - self->ts) / 1000000) % 1000);

	/*
	 * Note: The printa() lines below will show zeros when too many queries are
	 * executed at the same time on the same backend as the variables are cleared too
	 * quickly. Try running the SQL statemment from psql instead or from a script that
	 * executes SQL queries on different backends and run query_time_wrap.d instead.
	 */
        printf("\n============ Buffer Read Counts ============\n");
        printf("%10s %10s %10s %10s\n","Tablespace", "Database", "Table", "Count");
        printa("%10d %10d %10d %@10d\n",@read_count);

        printf("\n======= Buffer Write Request Counts ========\n");
        printf("%10s %10s %10s %10s\n","Tablespace", "Database", "Table", "Count");
        printa("%10d %10d %10d %@10d\n",@write_count);

        printa("\n%19s %@8d\n",@read_hit_total);
        printa("%19s %@8d\n",@read_miss_total);
        printa("%19s %@8d (ns)\n",@read_hit_time);
        printa("%19s %@8d (ns)\n",@read_miss_time);
        printa("%19s %@8d (ns)\n",@write_time);
        printf("\n\n");

/*
	trunc(@read_count);
	clear(@read_hit_total);
	clear(@read_miss_total);
	clear(@read_hit_time);
	clear(@read_miss_time);
	trunc(@write_count);
	trunc(@write_time);
*/

        self->ts = 0;
        self->pid = 0;
}
