/*
*	replicaManager.c - The replication manager (wb)
*
*	The replication manager runs as separate process and manages the 
*	communication between local transactions, remote transactions and
*	the group communication system. The main components of the rmgr
*	are:-l 
*		RmgrSock - a server socket on which the rmgr listens for
*					connection requests from transactions (local
*					and remote)
*		GCconn - a structure encapsulating the socket to the
*				 group communication process
*		RconnList - list containing one Rconn object for each 
*					active transaction (remote and local). Rconn
*					objects are created whenever a connection
*					request is accepted on the RmgrSock socket.
*					Each Rconn object contains a socket to the
*					associated transaction
*		waitForBknd - a pointer to an Rconn object that is to 
*					  be processed. (see RmgrReceive)
*		handleVector - a 3x3 matrix containing references to
*					   handling procedures used to process
*					   pending data on various sockets. The
*					   matrix is accessed by using the 
*					   source of the data as first index, and
*					   the destination as second index. The 
*					   matrix is not fully used, since for
*					   example there exists no communication
*					   from local to remote txn's
*		active_rsocks - bitmask identifying all the sockets
*						that must be listened on
*						(see manpages on select())
*
*	The basic control flow in the replication manager looks
*	as follows:
*
*	The <RmgrMain> routine runs an endless loop with a select
*	statement at its beginning. It thus blocks until at least
*	one of the sockets reports pending data.
*	If a request is pending on the RmgrSock socket, a new
*	Rconn object is generated and added to the list.
*	If a request is pending on any other socket, the
*	<RmgrReceive> function is called. This function
*	is responsible of finding receiving a whole message
*   When a message has arrived in it entirety, the appropriate
*   message handling procedure is called.
*	After the data has been handled, the control 
*	returns to the <RmgrMain> procedure which will continue to
*	monitor the sockets.
*
*	The handling procedures implement a state machine
*	which acts upon the current state of the connection
*	and the type of message received.
*
*	The rmgr message format used consists of
*		- a 8-byte header containing the message type (4 bytes)
*		  and the length of the message data, header excluded(4 bytes)
*		- a variable amount of data, the length of which is specified
*		  in the header

*	When sent over the group communication system, this message is
*	preceded by an 8-byte header consisting of a 4-byte hostid and
*	a 4 byte xid. Since xid's are not unique in a distributed
*	environment, the unique hostid must be used to identify all
*	postgres backends associated to a replicated transaction 
*	(i.e. one local txn plus its corresponding remote txns)
*
*  Created by: Win Bausch and Bettina Kemme
*
*/


#include <errno.h>

#include "postgres.h"

#include "libpq/libpq.h"
#include "libpq/pqsignal.h"
#include "miscadmin.h"
#include "replication/replication.h"
#include "storage/ipc.h"

#define INVALID_SOCK (-1)

#define MAX_HOSTNAMELEN 64

#define REMOTE_STARTUPS      4
#define DUMMY_HOST_ID  (-1)
#define DUMMY_XID      (-1)

/* This comes from the postgresql.conf file */
int Repl_max_remote_processes;
int Repl_mgr_port;
int Repl_spread_port;
char *Repl_spread_host;
char *Repl_spread_group;
char *Repl_database;

/* DAR HACK these globals used to be in global.c and
      one day maybe they'll go back*/
/* DAR HACK need ifdef here
*    5 globals needed when replicating (wb)
*/
txn_t       txn_type = NOT_REPLICATED;
txn_t       txn_location = LOCAL_TXN;
bool        txn_read_only = false;
bufsockptr  MyRmgrSock = NULL;
bool        may_not_abort_right_now = false;

/*
*	points Rconn to that is waiting for its associated
*	backend to startup (used with remote bknd startup)
*/
static Rconn *waitForBknd = NULL;
/*
*	list of all active connections
*/
static Dllist *RconnList = NULL;

/*
*    list of idle processes ready to execute remote txns
*/
static Dllist *IdleProcessesList = NULL;
/*
*	server socket info
*/
static int ReplicaMgrSock = INVALID_SOCK;
static struct sockaddr_un ReplicaMgrAddr;
/* static short ReplicaMgrPort = 5430; */
/*
*	the machine's host identifier
*/
static uint32 myHostId;

/* failure management */
static int deliveredViewChange = 0;
static uint32 group_failed_hostids[MAX_GROUP_MEMBERS];
static uint32 number_failed_members = 0;


/*
*	connection to group communication process
*/
/* bk now also two */
static DataConn GCconn_total_send;
static DataConn GCconn_basic_send;

/* bk: new two sockets to group communication */
static DataConn GCconn_total;
static DataConn GCconn_basic;
/*
*	handling function type decl
*/
typedef void (*handleReadFunc)(Rconn *conn, bufsockptr socket_ptr);
/*
*	routing matrix
*/
static handleReadFunc handleVector[3][3];
/*
*	active sockets bitmask
*/
static fd_set	active_rsocks;
static int		maxSock = -1;

static int active_remote_processes = 0;
static bool new_remote_txn_waiting = FALSE;

static void handle_special_message(bufsockptr bsock, bool simple);
static bool RmgrIsFailedSite(uint32 hostid);
static void WalkRemoteBackends(void);
static void HandleProtoError(Rconn *conn);

static void enableSocket(int sock);
static void disableSocket(int sock);
static void destroyRconn(Rconn *conn, bool destroysocket);
static Rconn* initRconn(uint32 hid, uint32 xid, bufsockptr bsock);

static void rpmgr_die(SIGNAL_ARGS);
static bool RmgrReceive(DataConn *src);
static bool RmgrAccept(void);
static void RmgrCleanup(int dummy);
static void RmgrStartBackend(int hostid, int xid);
static Rconn* RmgrFindConn(long hostid, long xid, bool remove);

/*
*	connection handling functions handle_<src>2<dst>
*/
static void handle_rtxn2gc(Rconn *conn, bufsockptr socket_dummy);
static void handle_ltxn2gc(Rconn *conn, bufsockptr socket_dummy);
static void handle_gc2ltxn(Rconn *conn, bufsockptr group_socket);
static void handle_gc2rtxn(Rconn *conn, bufsockptr group_socket);

/*
*	RmgrCleanup
*
*	cleans up the rmgr global data structures
*
*	param:
*		dummy - param needed since this function is registered
*			with the on_proc_exit function. on_proc_exit
*			requires the function to have this type
*	return:
*		void
*/
static void
RmgrCleanup(int dummy)
{
  Dlelem 	*curr,
    *next;
  
  /*
   *	free elems in the RconnList list
   */
  curr = DLGetHead(RconnList);
  while(curr){
    Rconn *conn = (Rconn *) DLE_VAL(curr);
    
    next = DLGetSucc(curr);
    if(sockIsConnected(conn->dc.bsock))
      {
	sockPutInt((int) MSG_CLOSING, 4, conn->dc.bsock);
	sockFlush(conn->dc.bsock);
      }
    destroyRconn(conn, TRUE);
    DLRemove(curr);
    DLFreeElem(curr);
    curr = next;
  }
  /*
   *	clean up global sockets
   */
  close(ReplicaMgrSock);
  sockUnlink(ReplicaMgrAddr.sun_path);
  /*sockPutInt((int) MSG_CLOSING, 4, GCconn.bsock);
  sockFlush(GCconn.bsock);
  sockDestroy(GCconn.bsock);*/
  sockPutInt((int) MSG_CLOSING, 4, GCconn_total_send.bsock);
  sockFlush(GCconn_total_send.bsock);
  sockDestroy(GCconn_total_send.bsock);
  sockPutInt((int) MSG_CLOSING, 4, GCconn_basic_send.bsock);
  sockFlush(GCconn_basic_send.bsock);
  sockDestroy(GCconn_basic_send.bsock);
  sockPutInt((int) MSG_CLOSING, 4, GCconn_total.bsock);
  sockFlush(GCconn_total.bsock);
  sockDestroy(GCconn_total.bsock);
  sockPutInt((int) MSG_CLOSING, 4, GCconn_basic.bsock);
  sockFlush(GCconn_basic.bsock);
  sockDestroy(GCconn_basic.bsock);
#ifdef RMGR_STAT
  LogFile_close();
#endif
}

/*
 *	HandleProtoError
 *
 *	Invoked by handling procs if protocol error occurs
 *
 *	param:
 *		conn - pointer to the connection struct reporting 
 *			the error
 *	return:
 *		void	
 */
static void
HandleProtoError(Rconn *conn)
{
  elog(NOTICE, "[%d] reports protocol error!!!", MyProcPid);
  conn->state = CLOSED;
  sockPutInt((int) MSG_PROTO_ERROR, 4, conn->dc.bsock);
  sockFlush(conn->dc.bsock);
  sockClose(conn->dc.bsock);
  proc_exit(200);
}

/*
 *	RmgrMain
 *
 *	Main loop of the replication manager. Listens for requests on 
 *	RmgrSock socket and handles pending data on other sockets.
 *
 *	param:
 *		gcsock_total + basic - socket to the group communication process
 *	return:
 *		int - STATUS_ERROR for failures, STATUS_OK otherwise
 */

int
RmgrMain(int gcsock_total_send, int gcsock_basic_send, int gcsock_total, int gcsock_basic)
{
	fd_set		tmp_rmask;
	Dlelem	   	*curr,
				*next;
	int i;

	elog(NOTICE, "ReplicaMgr pid %d", getpid());
	myHostId = gethostid();

	on_proc_exit(RmgrCleanup, (Datum) 0);
  
	/* register signal handlers. */
	pqsignal(SIGINT, rpmgr_die);
	pqsignal(SIGHUP, rpmgr_die);
	pqsignal(SIGTERM, rpmgr_die);
	pqsignal(SIGPIPE, rpmgr_die);
	pqsignal(SIGUSR1, rpmgr_die);
	pqsignal(SIGUSR2, rpmgr_die);
	pqsignal(SIGFPE, rpmgr_die);
	pqsignal(SIGCHLD, SIG_DFL);
	pqsignal(SIGTTIN, SIG_DFL);
	pqsignal(SIGTTOU, SIG_DFL);
	pqsignal(SIGCONT, SIG_DFL);
	/* OK, let's unblock our signals, all together now... */
	/* sigprocmask(SIG_SETMASK, &oldsigmask, 0); */

	/* initialization segment */
   
	/* DAR HACK i think this is set up in elog now 
	I need to compare with postgres.c

  StrNCpy(OutputFileName, "/usr/local/pgsql/localog\0", MAXPGPATH);
   
  DebugFileOpen();
  */
   
  elog(DEBUG, "I will initialize the GConns...");
   
  /*MemSet(&GCconn, 0, sizeof(GCconn));
  GCconn.recv_state = RECV_IDLE;
  GCconn.datalen = -1;
  GCconn.bsock = sockInit(AF_UNIX_BUFSIZE);
  GCconn.bsock->sock = gcsock;*/
 
  MemSet(&GCconn_total_send, 0, sizeof(GCconn_total_send));
  GCconn_total_send.recv_state = RECV_IDLE;
  GCconn_total_send.datalen = -1;
  GCconn_total_send.bsock = sockInit(AF_UNIX_BUFSIZE);
  GCconn_total_send.bsock->sock = gcsock_total_send;
 
  MemSet(&GCconn_basic_send, 0, sizeof(GCconn_basic_send));
  GCconn_basic_send.recv_state = RECV_IDLE;
  GCconn_basic_send.datalen = -1;
  GCconn_basic_send.bsock = sockInit(AF_UNIX_BUFSIZE);
  GCconn_basic_send.bsock->sock = gcsock_basic_send;
  
  MemSet(&GCconn_total, 0, sizeof(GCconn_total));
  GCconn_total.recv_state = RECV_IDLE;
  GCconn_total.datalen = -1;
  GCconn_total.bsock = sockInit(AF_UNIX_BUFSIZE);
  GCconn_total.bsock->sock = gcsock_total;
 
  MemSet(&GCconn_basic, 0, sizeof(GCconn_basic));
  GCconn_basic.recv_state = RECV_IDLE;
  GCconn_basic.datalen = -1;
  GCconn_basic.bsock = sockInit(AF_UNIX_BUFSIZE);
  GCconn_basic.bsock->sock = gcsock_basic;
 
  RconnList = DLNewList();
  IdleProcessesList = DLNewList();
 
  MemSet(&handleVector, 0, sizeof(handleVector));
  handleVector[REMOTE][GROUPCOMM] = handle_rtxn2gc;
  handleVector[LOCAL_ADDR][GROUPCOMM] = handle_ltxn2gc;
  handleVector[GROUPCOMM][LOCAL_ADDR] = handle_gc2ltxn;
  handleVector[GROUPCOMM][REMOTE] = handle_gc2rtxn;
  
  elog(DEBUG, "Sockets to gcomm set");
 
  /* set up server port */
  if(sockServerPort(NULL, Repl_mgr_port, &ReplicaMgrSock, 
		    (struct sockaddr *) &ReplicaMgrAddr) != STATUS_OK)
    {
      elog(NOTICE, "RmgrMain: unable to get server socket");
      proc_exit(200);
    }
  
  FD_ZERO(&active_rsocks);
  elog(DEBUG, "replication manager port set: %d", Repl_mgr_port);
  enableSocket(ReplicaMgrSock);

  elog(DEBUG, "all sockets enabled");
 
#ifdef RMGR_STAT
  LogFile_init();
#endif
  
  /* bk start up a couple of remote processes */
  for (i = 0; i < REMOTE_STARTUPS; i++) 
  {
	RmgrStartBackend(DUMMY_HOST_ID, DUMMY_XID);

	elog(DEBUG, "replica manager is here");

	memmove((char *) &tmp_rmask, (char *) &active_rsocks, 
	      sizeof(fd_set));

	if(select(maxSock + 1,  &tmp_rmask, (fd_set *) NULL, (fd_set *) NULL,
		(struct timeval *) NULL) < 0)
	{
	  if (errno == EINTR)
	    continue;
	  elog(ERROR, "RmgrMain: select failed: %m");
	  return STATUS_ERROR;
	}

	if (ReplicaMgrSock != INVALID_SOCK &&
	  FD_ISSET(ReplicaMgrSock, &tmp_rmask))
	{
#ifdef RMGR_DEBUG
	  elog(NOTICE,"get a accept");
#endif
	  if(!RmgrAccept())
	    {
	      elog(DEBUG, "Rmgr: accept failed");
	      proc_exit(201);
	    }
	  else {
#ifdef RMGR_DEBUG
		elog(NOTICE,"accepted");
#endif
	  }
	}
	else
	  elog(NOTICE, "wrong message first");
  }

  /*enableSocket(GCconn.bsock->sock);*/

  enableSocket(GCconn_total_send.bsock->sock);
  enableSocket(GCconn_basic_send.bsock->sock);

  enableSocket(GCconn_total.bsock->sock);
  enableSocket(GCconn_basic.bsock->sock);

  elog(DEBUG, "gc set");
 
  for(;;)
    {	
      /*
       *	listen on all active sockets
       */
#ifdef RMGR_DEBUG
      elog(NOTICE, "Rmgr: listening on active sockets");
#endif
      memmove((char *) &tmp_rmask, (char *) &active_rsocks, 
	      sizeof(fd_set));
      
      if(select(maxSock + 1,  &tmp_rmask, (fd_set *) NULL, (fd_set *) NULL,
		(struct timeval *) NULL) < 0)
	{
	  if (errno == EINTR)
	    continue;
	  elog(ERROR, "RmgrMain: select failed: %m");
	  return STATUS_ERROR;
	}
      
      
      /*
       *	Handle messages from the group communication system
       *    bk: simple copy that for now the two sockets
       */
      if (GCconn_basic.bsock->sock != INVALID_SOCK &&
	  FD_ISSET(GCconn_basic.bsock->sock, &tmp_rmask))
	{
	  int nread = 0,
	    ntotal = 0;
	  uint32	hostid = -1,
	    xid = -1;
	  bufsockptr bsock = GCconn_basic.bsock;
	  Rconn *conn = NULL;

	  route_addr_t dst;
	  
	  do{
	    if((nread = sockReadData(bsock, BKND_ID_SIZE - ntotal, TRUE)) < 0){
	      elog(NOTICE, "RMgr: unable to read group comm. msg header");
	      return STATUS_ERROR;
	    }
	    ntotal += nread;
	  }while(ntotal < BKND_ID_SIZE);
	  
	  sockGetInt(&hostid, 4, bsock);
	  sockGetInt(&xid, 4, bsock);
	  sockConsume(bsock);


		/* hostid == 0 indicates that this is a special message: right now
		 * there is only one: sites have left the group */
	  if (hostid == 0)
	  {
	  	
	  	handle_special_message(bsock, TRUE);
	  }
	  else
	  {
	  
	  	if((conn = RmgrFindConn(hostid, xid, FALSE)) == NULL)
	    {
				/*
				 *	generate connection object (always for remote bknd)
				 */
	      if((conn = initRconn(hostid, xid, NULL)) == NULL)
			{	
		  		elog(NOTICE, "RMgr: Rconn memory alloc failed");
		  		return STATUS_ERROR;
			}
#ifdef RMGR_DEBUG
		  elog(NOTICE,"basic conn: new conn created, hostid %x, xid %d",hostid,xid);
#endif
	      DLAddHead(RconnList, DLNewElem(conn));
	      /* bk: in case of basic messages there is no backend created anymore */
	      /* also: waitForBknd may only used by total socket to guarantee that*/
	      /* there is only at most one waiting  */
	      /*waitForBknd = conn;*/
	    }
	  

	  	/*
	   	*	Try to receive the message. If it can't be received
	   	*	entirely, go back to main loop
	   	*/
	  	if(!RmgrReceive(&GCconn_basic))
	    {
	      continue;
	    }
	  
	  	/*
	   	*	The message has now been received entirely
	   */
	  
	  	dst = (hostid == myHostId) ? LOCAL_ADDR : REMOTE;
	  
#ifdef RMGR_DEBUG
	  	elog(NOTICE, "Rmgr: Routing.... src=%d, dst=%d", 
	       (int) GROUPCOMM, (int) dst);
#endif
	 	 /*
	   	*	call the appropriate handling procedure
	   	*/
	  	handleVector[GROUPCOMM][dst](conn, bsock);
	  
#ifdef RMGR_DEBUG
	  	elog(NOTICE, "Rmgr: done... (%d left in buffer)", 
	       sockNumLeft(GCconn_basic.bsock));
#endif
	  
	  }
	}
	
	      /*
       *	Handle messages from the group communication system
       *    bk: simple copy that for now the two sockets
       */
      if (GCconn_total.bsock->sock != INVALID_SOCK &&
	  FD_ISSET(GCconn_total.bsock->sock, &tmp_rmask))
	{
	  int nread = 0,
	    ntotal = 0;
	  uint32	hostid = -1,
	    xid = -1;
	  bufsockptr bsock = GCconn_total.bsock;
	  Rconn *conn = NULL;

	  route_addr_t dst;
	  
	  do{
	    if((nread = sockReadData(bsock, BKND_ID_SIZE - ntotal, TRUE)) < 0){
	      elog(NOTICE, "RMgr: unable to read group comm. msg header");
	      return STATUS_ERROR;
	    }
	    ntotal += nread;
	  }while(ntotal < BKND_ID_SIZE);
	  
	  sockGetInt(&hostid, 4, bsock);
	  sockGetInt(&xid, 4, bsock);
	  sockConsume(bsock);
	  
	  	/* hostid == 0 indicates that this is a special message: right now
		 * there is only one: sites have left the group */
	  if (hostid == 0)
	  {
	  	
	  	handle_special_message(bsock, FALSE);
	  }
	  else
	  {
	  	if((conn = RmgrFindConn(hostid, xid, FALSE)) == NULL)
	    {
				/*
				 *	generate connection object (always for remote bknd)
				 */
	      if((conn = initRconn(hostid, xid, NULL)) == NULL)
			{	
		  		elog(NOTICE, "RMgr: Rconn memory alloc failed");
		  		return STATUS_ERROR;
			}
#ifdef RMGR_DEBUG
		  	elog(NOTICE,"total conn: new conn created, hostid %x, xid %d",hostid,xid);
#endif
	      	DLAddHead(RconnList, DLNewElem(conn));
	      /* bk: I set the waitForBknd now in handle_gc2rtxn */
	      /*waitForBknd = conn;*/
	    }
	  

	  	/*
	   	*	Try to receive the message. If it can't be received
	   	*	entirely, go back to main loop
	   	*/
	  	if(!RmgrReceive(&GCconn_total))
	    {
	      continue;
	    }
	  
	  /*
	   *	The message has now been received entirely
	   */
	  
	  	dst = (hostid == myHostId) ? LOCAL_ADDR : REMOTE;
	  
#ifdef RMGR_DEBUG
	  	elog(NOTICE, "Rmgr: Routing.... src=%d, dst=%d", 
	       (int) GROUPCOMM, (int) dst);
#endif
	  	/*
	   	*	call the appropriate handling procedure
	   	*/
	  	handleVector[GROUPCOMM][dst](conn, bsock);
	  
#ifdef RMGR_DEBUG
	  	elog(NOTICE, "Rmgr: done... (%d left in buffer)", 
	       sockNumLeft(GCconn_total.bsock));
#endif
	  }
	}
      
#ifdef RMGR_DEBUG
      elog(NOTICE,"check connections");
#endif
      /*
       *	Go through the list of open backend connections and
       *	invoke routing fcnt if pending output
       */
      curr = DLGetHead(RconnList);
      while(curr)
      {
		
		Rconn *conn = (Rconn *) DLE_VAL(curr);
		Rconn *conn2 = NULL;
		int sock;
		route_addr_t src;
	
		next = DLGetSucc(curr);
		/* (bk) reorganize: first look if closed -> then close, disable socket if existent
		 * then for non-closed look if there is data on
		*/
	
#ifdef RMGR_DEBUG
		elog(NOTICE, "found conn with hostid %x, xid %d", conn->hostid, conn->xid);   
#endif
		switch(conn->state)
		{
		  case CLOSED:
			  /* remove connection structure, disable socket if necessary,
			   * put in waiting queue if necessary
			  */
			if(conn->dc.bsock != NULL)
			{
#ifdef RMGR_DEBUG
			  elog(NOTICE,"found closed conn");
#endif
			  /* (bk) put socket in the idle process list if remote process */
			  if (conn->hostid != myHostId) 
			  {
#ifdef RMGR_DEBUG
				elog(NOTICE, "remote closed");
#endif
			  
				/*(bk) new check it*/
				if (new_remote_txn_waiting) {
#ifdef RMGR_DEBUG				
				  elog(NOTICE, "remote waiting");
#endif
				  conn2 = waitForBknd;
				  waitForBknd = NULL;
				  conn2->dc.bsock = conn->dc.bsock;
				  conn2->state = BKND_PRESENT;
				  enableSocket(conn2->dc.bsock->sock);
				  new_remote_txn_waiting = FALSE;
				  enableSocket(GCconn_total.bsock->sock);
#ifdef RMGR_DEBUG				  
				  elog(NOTICE,"call from RmgrMain 1");
#endif
				  handle_gc2rtxn(conn2, GCconn_total.bsock);
				}
				else {
#ifdef RMGR_DEBUG				
				  elog(NOTICE,"put process in the wait queue");
#endif
				  DLAddHead(IdleProcessesList, DLNewElem((void *) conn->dc.bsock));
				}
				destroyRconn(conn,FALSE);
			  }
			  else {
				/* bk: now also false, because we do not destroy the socket */
				destroyRconn(conn,FALSE);
			  }
			  DLRemove(curr);
			  DLFreeElem(curr);
			} /* != NULL */
			else 
			{
#ifdef RMGR_DEBUG			
			  elog(NOTICE,"socket is nul");
#endif
			  DLRemove(curr);
			  DLFreeElem(curr);
			  destroyRconn(conn,TRUE);	
			}
			break; /* == CLOSED */
		  case CLOSE_AND_DESTROY:
			elog(NOTICE,"backend wants to disconnect");
			DLRemove(curr);
			DLFreeElem(curr);
			destroyRconn(conn,TRUE);
			break;
		  
		  default:  
#ifdef RMGR_DEBUG		  
			elog(NOTICE,"connection not closed");
#endif
			if (conn->dc.bsock != NULL)
			{
			  sock = conn->dc.bsock->sock;
			  if(FD_ISSET(sock, &tmp_rmask))
			  {
#ifdef RMGR_DEBUG			  
				elog(NOTICE,"there is a message");
#endif
				if(!RmgrReceive(&conn->dc))
				{
				  curr = next;
				  continue;
				}
		    
				src = (conn->hostid == myHostId) ? LOCAL_ADDR : REMOTE;
		    
#ifdef RMGR_DEBUG
				elog(NOTICE, "Rmgr: Routing.... src=%d, dst=%d", 
				 (int) src, (int) GROUPCOMM);
#endif
				/*
				  *	call the appropriate handling procedure
				*/
				handleVector[src][GROUPCOMM](conn, 0);
#ifdef RMGR_DEBUG
				elog(NOTICE, "Rmgr: done... (%d left in buffer)",
					 sockNumLeft(conn->dc.bsock));
#endif
			  }
			} // if bscok != NULL (otherwise there is nothing to do)
			break;	
		} /* switch */
	  curr = next;
	} /* while next connection */
 
      /*
       *	Handle connect request from backends
       */
      
      if (ReplicaMgrSock != INVALID_SOCK &&
	  FD_ISSET(ReplicaMgrSock, &tmp_rmask))
	{
#ifdef RMGR_DEBUG	
	  elog(NOTICE,"get a accept");
#endif
	  if(!RmgrAccept())
	    {
	      elog(DEBUG, "Rmgr: accept failed");
	      proc_exit(201);
	    }
	  else {
#ifdef RMGR_DEBUG	  
		elog(NOTICE,"accepted");
#endif
	  }
	}
      
    }
  
}

/*
 *	RmgrStartBackend
 *
 *	Starts a backend on behalf of the replication manager. The postgres 
 *   postmaster is requested to start a new backend. After the authentication 
 *   phase, the rmgr sends the hostid and xid of the LOCAL transaction 
 *   associated to this remote transaction. Remember that the local txn is not 
 *   running on the same machine as the remote txn, and consequently does not 
 *   necessarily have the same xid.	Hostid and xid are used to identify the 
 *   sets of backends that belong to one single replicated transaction. 
 *   After this has been done, the connection is closed, since the backend 
 *   will now contact the rmgr on the well-known server socket.
 *
*	param:
*		hostid - the hostid identifying the local txn's host
*		xid - the xid identifying the local txn's xid
*	return:
*		void
*
*	NOTE:
*		For now, a full replication approach is used, and the database names
*		are supposed to be repl_<hostname>. This is not a very flexible 
*		scheme and should be modified in the future 
*/			

static void
RmgrStartBackend(int hostid, int xid)
{
  char 			c,
    hostname[MAX_HOSTNAMELEN+1];
  int 			areq = -1;
  int 			read_result = 0;
  StartupPacket 	startpkt;
  bufsockptr		backendsock = sockInit(AF_UNIX_BUFSIZE);

  MemSet(&startpkt, 0, sizeof(startpkt));
  startpkt.protoVersion = (ProtocolVersion) htonl(PG_PROTOCOL_LATEST);
  if(gethostname(hostname, MAX_HOSTNAMELEN) != 0)
      proc_exit(102);
  hostname[MAX_HOSTNAMELEN + 1] = '\0';

  /*
   *	user name is hardcoded :-(
   *	FIXME: replace this with GUC stuff
   */
  strcpy(startpkt.database, Repl_database);
  strcpy(startpkt.user, "postgres\0");

  if (sockClientConnect(NULL, PostPortNumber, backendsock) != STATUS_OK)
    {
		/* DAR HACK */
		elog(NOTICE, "replmgr:sockClientConnect call failed");
      proc_exit(201);
    }

  /*
   *	send start packet and receive authentication request
   *	Authentication is assumed to be implicit for internal
   *	backends
   */
  sockPutInt(4 + sizeof(startpkt), 4, backendsock);
  sockPutnchar((char *)&startpkt, sizeof(startpkt), backendsock);
  sockFlush(backendsock);
  sockWait(TRUE, FALSE, backendsock);
  read_result = sockReadData(backendsock, -1, FALSE);
  sockGetc(&c,backendsock);
  sockGetInt(&areq, 4, backendsock);
  sockConsume(backendsock);
  if(c != 'R' && areq != AUTH_REQ_OK){
    sockDestroy(backendsock);
    elog(NOTICE, "BackendStartup failed with code '%c'", c);
    proc_exit(201);
  }
 
  /*
   *	send hostid and xid to backend
   */
  sockPutInt(hostid, 4, backendsock);
  sockPutInt(xid, 4, backendsock);
  sockFlush(backendsock);
  sockDestroy(backendsock);
 
  /*
   * Disable the GCconn socket. This must be done since we must wait
   * for the remote backend to be started and to process the message that
   * is still in the GCconn buffer, before receiving the next message.
   */
   /* bk: now the total order socket */
  /*disableSocket(GCconn.bsock->sock);*/
  if (hostid != DUMMY_HOST_ID || xid != DUMMY_XID)
	disableSocket(GCconn_total.bsock->sock);
  
  active_remote_processes++;
}

/*
 *	RmgrReceive
 *
*	This function handles a connection which reports pending data.
*	It reads a whole message (not more) into the buffer. 
*
*	If a message does not arrive in one chunk, this function returns
*	control to the RmgrMain routine, which will continue to monitor
*	the sockets. When the next data chunk arrives, this function
*	will read it and check whether the message is complete. If not,
*	it again returns control. Keep in mind that this func may be
*	invoked several times before completing successfully.
*	This implies for example that local vars are not valid across
*	calls....
*	
*	This approach has been adopted so as not to block upon receipt
*	of an incomplete message, which may improve performance under 
*	high load.
*
*	params:
*		src - the connection reporting pending data
*	return:
*		bool - TRUE if the message is complete, FALSE otherwise
*/
static bool
RmgrReceive(DataConn *src)
{	
  char 			header[RMGR_HDR_SIZE];
  int 			nread = 0;
  bufsockptr	bsock = src->bsock;
  
  /*
   * message reception
   *
   * This switch statement is called repeatedly in subsequent
   * calls of this function, until the message has arrived 
   * completely. Only then the code after the switch stat is
   * executed. State is saved in the connection object. Remember
   * that data stored in local vars does not remain valid
   * across subsequent calls of RmgrReceiveAndRoute! 
   *
   */
  switch(src->recv_state)
    {
    case RECV_IDLE:
      src->datalen = RMGR_HDR_SIZE;
      src->recv_state = RECV_HDR;
      /*deliberately NO break*/
      
    case RECV_HDR:
      if((nread = sockReadData(bsock, src->datalen, TRUE)) < 0){
		elog(NOTICE, "RMgr: read error occurred");
		return FALSE;
	  }
#ifdef RMGR_DEBUG
      elog(NOTICE, "Rmgr: Receiving header...(socket=%d, dlen=%d, nread=%d)", 
	   (int)src->bsock->sock, src->datalen, nread);
#endif
      src->datalen -= nread;
      if(src->datalen > 0)
	return FALSE;
      sockPeekHeader(RMGR_HDR_SIZE , bsock, header);
      src->datalen = (int) ntohl(*((uint32 *) (header + RMGR_HDR_SIZE - 4)));
      src->recv_state = RECV_DATA;
      /*deliberately NO break*/
      
    case RECV_DATA:
      if((nread = sockReadData(bsock, src->datalen, TRUE)) < 0)
	  {
	elog(NOTICE, "RMgr: read error occurred");
	return FALSE;
      }
#ifdef RMGR_DEBUG
      elog(NOTICE, "Rmgr: Receiving data...(socket=%d, dlen=%d, nread=%d)", 
	   (int)src->bsock->sock, src->datalen, nread);
#endif
      src->datalen -= nread;
      if(src->datalen > 0)
	return FALSE;
      src->recv_state = RECV_IDLE;
      break;
    }
  return TRUE;
}

/*
*	RmgrAccept
*
*	Responsible for handling RmgrSock requests, i.e. connection requests by
*	transactions. Data transferred by the backend is a hostid and a xid. For
*	remote backends, these ids are matched against the ids stored in 
*	struct pointed to by <waitForBknd>: for each remote bknd, this connection
*	struct must already exist. In every other case, a new connection struct 
*	is generated and inserted into the Rconn list
*	
*	param:
*		void
*	return:
*		bool - TRUE if accept was successful, FALSE otherwise
*/
static bool
RmgrAccept(void)
{
  Rconn 		*conn = NULL;
  rc_msg_t	msg_type = MSG_PROTO_ERROR;
  int 		datalen = -1,
    hostid = -1,
    xid = -1,
    procid = -1;
  bufsockptr 	newsock = NULL; 
  
  /*
   *	accept the connection and read the opening msg
   */
  newsock = sockInit(AF_UNIX_BUFSIZE);
  if (sockServerAccept(ReplicaMgrSock, newsock) != STATUS_OK)
    {
	  elog(NOTICE, "destroying socket!");
      sockDestroy(newsock);
      return FALSE;
    }
  else
    {	
      int nread = 0;
      do
	  {
		sockWait(TRUE, FALSE, newsock);
		nread = sockReadData(newsock, RMGR_HDR_SIZE - nread, TRUE);
		if(nread < 0)
		{
		  sockDestroy(newsock);
		  proc_exit(203);
		}
      } while(nread < RMGR_HDR_SIZE);
      sockGetInt((int *) &msg_type, 4, newsock);
      sockGetInt(&datalen, 4, newsock);
      if(msg_type != MSG_OPENING || datalen != 12)
	  {
		sockDestroy(newsock);
		proc_exit(204);
	  }
      nread = 0;
      do
	  {
		sockWait(TRUE, FALSE, newsock);
		nread = sockReadData(newsock, datalen - nread, TRUE);
		if(nread < 0)
		{
		  sockDestroy(newsock);
		  proc_exit(202);
		}
      } while(nread < datalen);
      sockGetInt(&hostid, 4, newsock);
	  sockGetInt(&procid, 4, newsock);
      sockGetInt(&xid, 4, newsock);
      sockConsume(newsock);
#ifdef RMGR_DEBUG
      elog(NOTICE,"RMgr: connection accepted (sock=%d, msg=%d, len=%d, xid=%d, hid=%x)",
	   newsock->sock, (int) msg_type, datalen, xid, hostid);
#endif	
      /*
       *	generate connection struct if necessary, else use
       *	<waitForBknd> pointer.
       */
      if(hostid == myHostId)
	  {
		if(!(conn = initRconn(hostid, xid, newsock)))
	    {
	      elog(DEBUG, "Rmgr: Rconn initialization failed");
	      return FALSE;
	    }
		/* bk procid only interesting for local backends */
		conn->procid = procid;
		DLAddHead(RconnList, DLNewElem(conn));
		enableSocket(conn->dc.bsock->sock);
	  }
	  else if (hostid == DUMMY_HOST_ID && xid == DUMMY_XID)
	  {
		/* bk this is only at replica manager startup */
		DLAddHead(IdleProcessesList, DLNewElem((void *) newsock));
	  }
      else
	  {
		if(!waitForBknd)
	    {
	      elog(DEBUG, "Rmgr: remote backend connection structure not found");
	      return FALSE;
	    }
		conn = waitForBknd;
		waitForBknd = NULL;
		conn->dc.bsock = newsock;
		conn->state = BKND_PRESENT;
		conn->procid = procid;
	  
		/*
		 * enable GCconn socket (disabled in RmgrStartBackend)
		 */
		 /* now the total order GConn*/
		/* enableSocket(GCconn.bsock->sock);*/
		enableSocket(GCconn_total.bsock->sock);
	  
		/*
		 * handle the message that triggered the startup...
		 * this must be done AFTER enabling the GCconn
		 * socket since handle_gc2rtxn might disable it again...
		 */	
		elog(NOTICE, "call gc2rtxn from RmgrAccept");
		handle_gc2rtxn(conn,GCconn_total.bsock);
		enableSocket(conn->dc.bsock->sock);
	  }

      return TRUE;
  }
}

/*
*	handle_ltxn2gc
*
*	Handles msgs coming from a local txn. First, we peek at the header
*	to see what the message type is. 
*	Action is taken according to he message type and the current state
*	of the connection.
*
*	param:
*		conn - the connection object to be handled
*	return:
*		void
*
*	NOTE: Peeking at a socket buffer leaves the data unprocessed in 
* 		  the buffer, so be sure to either process or ignore it afterwards
*		  Peeking is used to minimize the overhead with socket-to-socket
*		  data transfer.
*/
static void
handle_ltxn2gc(Rconn *conn, bufsockptr dummy_socket)
{
  rc_msg_t 	msg_type;
  char 		header[RMGR_HDR_SIZE];
  char      ids[RMGR_HDR_SIZE+RMGR_ID_SIZES];
  int 		datalen = -1;
  bufsockptr	ltxnsock = conn->dc.bsock;
  Rconn *conn2;
  
  
  sockPeekHeader(RMGR_HDR_SIZE, ltxnsock, header);
  msg_type = (rc_msg_t) ntohl(*((uint32 *)header));
  datalen = (int) ntohl(*((uint32 *) (header + 4)));
  
#ifdef RMGR_DEBUG
    elog(NOTICE, "ltxn2gc: received msg type %d on socket %d (len=%d), state is %d, hostid %x, xid %d",
       (int) msg_type, ltxnsock->sock, datalen, (int) conn->state, conn->hostid, conn->xid);
#endif
  if(msg_type == MSG_PROTO_ERROR){
    sockClose(ltxnsock);
    conn->state = CLOSED;
    sockIgnoreData(RMGR_HDR_SIZE, ltxnsock);
    sockConsume(ltxnsock);
    disableSocket(ltxnsock->sock);
    return;
  }
  
  
  if (conn->state == SEND_PHASE || conn->state == LOCK_PHASE)
  {
	/* bk: this is the last msg for this txn */
#ifdef RMGR_DEBUG	
	elog(NOTICE,"create new conn");
#endif
	/* bk: create new conn */
	if(!(conn2 = initRconn(myHostId, DUMMY_XID, ltxnsock)))
	{
	  elog(DEBUG, "Rmgr: Rconn initialization 2 failed");
	  return;
	}
	conn2->state = BKND_PRESENT;
	conn2->procid = conn->procid;
	DLAddHead(RconnList, DLNewElem(conn2));
	enableSocket(conn2->dc.bsock->sock);
  }
  
  switch(conn->state)
  {
    case READ_PHASE:
      switch(msg_type)
	  {
		case MSG_WRITESET:
#ifdef RMGR_STAT
		  		  LogFile_record(TRUE, conn->xid, "READ->SEND on MSG_WS");
#endif
		//elog(NOTICE, "ltxn2gc: received msg type %d on socket %d (len=%d), state is %d, hostid %x, xid %d",
       //(int) msg_type, ltxnsock->sock, datalen, (int) conn->state, conn->hostid, conn->xid);

			
		  /*write group comm header*/
		  elog(NOTICE,"put info in total send socket");
		  sockPutInt(myHostId, 4, GCconn_total_send.bsock);
		  sockPutInt(conn->xid, 4, GCconn_total_send.bsock);
		  /*transfer write set*/
		  sockTransfer(GCconn_total_send.bsock, ltxnsock, datalen + RMGR_HDR_SIZE);
		  sockFlush(GCconn_total_send.bsock);
		  conn->state = SEND_PHASE;

		  break;
		default:
		  printf("proto 1\n");
		  HandleProtoError(conn);
		  break;
	  }
      break;
    case SEND_PHASE:
      switch(msg_type)
	  {
	  case MSG_ABORT:
#ifdef RMGR_STAT
		LogFile_record(TRUE, conn->xid, "SEND->ABORT on MSG_ABORT");
#endif
		elog(NOTICE, "send msgabort1");
		sockPutInt(myHostId, 4, GCconn_basic_send.bsock);
		sockPutInt(conn->xid, 4, GCconn_basic_send.bsock);
		sockTransfer(GCconn_basic_send.bsock, ltxnsock, 
		       datalen + RMGR_HDR_SIZE);
		sockFlush(GCconn_basic_send.bsock);
		/*disableSocket(ltxnsock->sock); */
		/*
		 * disabling the socket is necessary since the backend
		 * itself may disconnect at any time, which makes the
		 * corresponding socket on rmgr side invalid. This socket will
		 * then report pending data endlessly, without any real 
		 * data being present
		 */
		 /* bk: new: 
		  * the backend does not disconnect anymore, instead it can now
		  * send the next request. hence, no more disabling */
		conn->state = ABORTED;
		break;
	  default:
	  	  printf("proto 2\n");
		HandleProtoError(conn);
		break;
	  }
	  /* bk: set the socket to invalid, it is now to the new conn */
	  conn->dc.bsock = NULL;
      break;
    case LOCK_PHASE:
      switch(msg_type)
	  {
	  case MSG_ABORT:
#ifdef RMGR_STAT
		LogFile_record(TRUE, conn->xid, "LOCK->ABORTED on MSG_ABORT");
#endif
		// sleep for failure testing
		// sleep(5);
		elog(NOTICE,"put msgabort2");
		sockPutInt(myHostId, 4, GCconn_basic_send.bsock);
		sockPutInt(conn->xid, 4, GCconn_basic_send.bsock);
		sockTransfer(GCconn_basic_send.bsock, ltxnsock, datalen + RMGR_HDR_SIZE);
		sockFlush(GCconn_basic_send.bsock);
		/* disableSocket(ltxnsock->sock);*/  /*no more disabling*/
		enableSocket(GCconn_total.bsock->sock);
		conn->state = ABORTED;
		break;
	  case MSG_LOCKED:
#ifdef RMGR_STAT
	   LogFile_record(TRUE, conn->xid, "LOCK->WRITE on MSG_LOCKED");
#endif
	   /*
	   * MSG_LOCKED is interpreted as Commit message
	   */
	    // sleep for failure testing
	   	// sleep(5);
	   	elog(NOTICE,"put commit");
		sockPutInt(myHostId, 4, GCconn_basic_send.bsock);
		sockPutInt(conn->xid, 4, GCconn_basic_send.bsock);
		sockPutInt((int) MSG_COMMIT, 4, GCconn_basic_send.bsock);
		sockPutInt(0, 4, GCconn_basic_send.bsock);
		sockFlush(GCconn_basic_send.bsock);
		/*
		 *	don't listen to the backend socket any more,
		 *	since the backend will disconnect anytime
		 */
		/* disableSocket(ltxnsock->sock);*/  /* bk no more disabling */
		/*
		 *	since lock phase is over, unblock the group 
		 *	communication process
		 */
		  /*enableSocket(GCconn.bsock->sock);*/
		  /* bk: now the total order socket */
		enableSocket(GCconn_total.bsock->sock);
		sockIgnoreData(datalen + RMGR_HDR_SIZE, ltxnsock);
		conn->state = WRITE_PHASE;
		break;
	  default:
	  	  printf("proto 3\n");
		HandleProtoError(conn);
		break;
	  }
	  /* bk: set the socket to invalid, it is now to the new conn */
	  conn->dc.bsock = NULL;
      break;
	case BKND_PRESENT:
#ifdef RMGR_DEBUG	  
	  elog(NOTICE,"i am in bknd_present");
#endif
	  switch(msg_type)
	  {
		case MSG_CLOSING:
		  sockIgnoreData(datalen + RMGR_HDR_SIZE, ltxnsock);
		  sockPutInt((int)MSG_ACK, 4, ltxnsock);
		  sockPutInt(0, 4, ltxnsock);
		  sockFlush(ltxnsock);
		  /*
		   *	socket will disconnect right after receiving this ack
		   */
		  disableSocket(ltxnsock->sock);
		  conn->state = CLOSE_AND_DESTROY;
		  break;
		case MSG_NEW_TXN:
		  /*elog(NOTICE,"receive msg_new_txn");*/
		  /* bk take care: second is now not dummy! */
		   sockPeekHeader(RMGR_HDR_SIZE+RMGR_ID_SIZES, ltxnsock, ids);
		  conn->xid = (uint32) ntohl(*((uint32 *)(ids+RMGR_HDR_SIZE+4)));	
		  sockIgnoreData(RMGR_HDR_SIZE+RMGR_ID_SIZES, ltxnsock);
		  enableSocket(ltxnsock->sock);
		  conn->state = READ_PHASE;
		  break;
		default:
		  printf("proto 1\n");
		  HandleProtoError(conn);
		  break;
		}
      default:
      sockIgnoreData(datalen + RMGR_HDR_SIZE, ltxnsock);
      break;
    }
  sockConsume(ltxnsock);
}

/*
*	handle_gc2ltxn
*
*	Handles msgs coming from the group comm process and intended for a local txn.
*	Action is taken according to he message type and the current state
*	of the connection.
*
*	param:
*		conn - the connection object to be handled
*	return:
*		void
*/
/* bk: new group_socket to see whether it comes total order or not */
static void
handle_gc2ltxn(Rconn *conn, bufsockptr group_socket)
{
  rc_msg_t	msg_type;
  char		header[RMGR_HDR_SIZE];
  int			datalen = 0;

  bufsockptr	ltxnsock;   /* bk don't init immediately, may not be valid anymore  = conn->dc.bsock;*/
  
  /*sockPeekHeader(RMGR_HDR_SIZE ,GCconn.bsock ,header);*/
  sockPeekHeader(RMGR_HDR_SIZE ,group_socket ,header);
  msg_type = (rc_msg_t) ntohl(*((uint32 *)header));
  datalen = (int) ntohl(*((uint32 *) (header + 4)));
#ifdef RMGR_DEBUG
  elog(NOTICE, "gc2ltxn: received msg type %d, len %d, state is %d, hostid %x, xid %d",
       (int) msg_type, datalen, 
       (int) conn->state, conn->hostid, conn->xid);
#endif
  
  
  conn->numDelivered++;
  switch(conn->state)
    {
    case SEND_PHASE:
	  ltxnsock = conn->dc.bsock;
      switch(msg_type)
	  {
		case MSG_WRITESET:
#ifdef RMGR_STAT
		  LogFile_record(TRUE, conn->xid, "SEND->LOCK on MSG_WS");
#endif
		  sockIgnoreData(datalen + RMGR_HDR_SIZE, group_socket);
		  sockPutInt((int)MSG_WS_RECEIVED, 4, ltxnsock);
		  sockPutInt(0, 4, ltxnsock);
		  sockFlush(ltxnsock);
		  /*
		   *	block the group communication socket until 
		   *	the backend reports the end of its lock phase
		   */
		  disableSocket(GCconn_total.bsock->sock);
		  conn->state = LOCK_PHASE;
		  break;
		default:
		  printf("proto 4\n");
		  HandleProtoError(conn);
		  break;
	  }
      break;
    case WRITE_PHASE:
      switch(msg_type)
	  {
		case MSG_COMMIT:
#ifdef RMGR_STAT
		  LogFile_record(TRUE, conn->xid, "WRITE->CLOSED on MSG_COMMIT");
#endif
		  sockIgnoreData(datalen + RMGR_HDR_SIZE, group_socket);
		  conn->state = CLOSED;
		  /* (bk) why enable, i would rather disable */
		  /* bk now: since we keep the socket connection until backend dies
		 * we have to keep it enabled */
		  /* disableSocket(ltxnsock->sock);*/
		  break;
		default:
		  printf("proto 5\n");
		  HandleProtoError(conn);
		  break;
	  }
      break;
    case ABORTED:
      switch(msg_type)
	  {
		case MSG_WRITESET:
		case MSG_ABORT:
		  switch(conn->numDelivered)
		  {
			case 1:
			  sockIgnoreData(datalen + RMGR_HDR_SIZE, 
			     group_socket);
			  /*wait for second message in transit, no state change*/
			  break;
			case 2:
#ifdef RMGR_STAT
			  LogFile_record(conn->hostid == myHostId, 
				   conn->xid, 
				   "ABORT->CLOSED on any MSG_WS | MSG_ABORT");
#endif
			  sockIgnoreData(datalen + RMGR_HDR_SIZE, 
				   group_socket);
			  /* (bk) why enable, i would rather disable */
			  /* bk again socket has to be enabled because connection is not
			   * destroyed */
			  /*disableSocket(ltxnsock->sock);*/
			  conn->state = CLOSED;
			  break;
			default:
			  printf("proto 6\n");
			  HandleProtoError(conn);
			  break;
		  }
		  break;
		default:
	  	  printf("proto 7\n");
		  HandleProtoError(conn);
		  break;
		}
      break;
    default:
      sockIgnoreData(datalen + RMGR_HDR_SIZE, group_socket);
      break;
    }
  sockConsume(group_socket);
}

/*
*	handle_rtxn2gc
*
*	Handles msgs coming from a remote transaction.
*	Action is taken according to he message type and the current state
*	of the connection.
*
*	param:
*		conn - the connection object to be handled
*	return:
*		void
*/
static void
handle_rtxn2gc(Rconn *conn, bufsockptr dummy_socket)
{
  rc_msg_t 	msg_type;
  int 		datalen = -1;
  char		header[RMGR_HDR_SIZE];
  bufsockptr	rtxnsock = conn->dc.bsock;
  
  
  sockPeekHeader(RMGR_HDR_SIZE, rtxnsock, header);
  msg_type = (rc_msg_t) ntohl(*((uint32 *)header));
  datalen = (int) ntohl(*((uint32 *) (header + 4)));
#ifdef RMGR_DEBUG
  elog(NOTICE, "rtxn2gc: received msg type %d on socket %d (len=%d), state is %d, result is %d, hostid %x, xid %d",
       (int) msg_type, rtxnsock->sock, datalen, conn->state, conn->result, conn->hostid, conn->xid);
#endif
  
  if(msg_type == MSG_PROTO_ERROR){
    conn->state = CLOSED;
    sockIgnoreData(RMGR_HDR_SIZE, rtxnsock);
    sockConsume(rtxnsock);
    disableSocket(rtxnsock->sock);
    return;
  }
  
  switch(msg_type)
  {
	case MSG_LOCKED:
	  switch(conn->state)
	  {
		case LOCK_PHASE:
		  conn->state = WRITE_PHASE;
#ifdef RMGR_STAT
		  LogFile_record(FALSE, conn->xid, 
			   "LOCK->WRITE on MSG_LOCKED");
#endif
		  sockIgnoreData(datalen + RMGR_HDR_SIZE, 
			   rtxnsock);
		  /* bk enable now the total order socket */
		  enableSocket(GCconn_total.bsock->sock);
		  break;
		default:
		  printf("proto 8\n");
		  HandleProtoError(conn);
		  break;
	  }
	  break;
	case MSG_READY:
	  switch(conn->state)
	  {
		case WRITE_PHASE:
		  /*elog(NOTICE,"receive msg_ready");*/
		  sockIgnoreData(datalen + RMGR_HDR_SIZE, rtxnsock);
		  conn->state = CLOSED;
		  disableSocket(rtxnsock->sock);
		  break;
		default:
		  printf("proto 10\n");
		  HandleProtoError(conn);
		  break;
	  }
	  break;
    default:
	  printf("protototo\n");
	  HandleProtoError(conn);
      sockIgnoreData(datalen + RMGR_HDR_SIZE, rtxnsock);
      break;
  }
  sockConsume(rtxnsock);
}
/*
*	handle_gc2rtxn
*
*	Handles msgs coming from the group comm process and intended for 
*	a remote txn.
*	Action is taken according to he message type and the current state
*	of the connection.
*
*	param:
*		conn - the connection object to be handled
*	return:
*		void
*/
/* bk: new the socket because we have now two */
static void
handle_gc2rtxn(Rconn *conn, bufsockptr group_socket)
{
  char		header[RMGR_HDR_SIZE];
  rc_msg_t	msg_type;
  int			datalen = 0;
  bufsockptr	rtxnsock = NULL;
  /* (bk) get idle process */
  Dlelem *  idle_process_list_element;
  
  sockPeekHeader(RMGR_HDR_SIZE ,group_socket ,header);
  msg_type = (rc_msg_t) ntohl(*((uint32 *)header));
  datalen = (int) ntohl(*((uint32 *) (header + 4)));
#ifdef RMGR_DEBUG
  elog(NOTICE, "gc2rtxn: received msg type %d (len=%d), state is %d, result %d, hostid %x, xid %d",
       (int) msg_type, datalen, 
       (int) conn->state, conn->result, conn->hostid, conn->xid);
  #endif
  
  
  if(conn->state == NO_BKND && conn->result != TO_ABORT  &&
     /* msg_type != MSG_ABORT)*/
     /* bk: new only when write set arrives */
     msg_type == MSG_WRITESET && !RmgrIsFailedSite(conn->hostid))
    {
#ifdef RMGR_DEBUG
	  elog(NOTICE,"figure out how to get backend");
#endif
      /*
       * the backend is only started if the FIRST message received
       * is NOT an abort message.
       */
#ifdef RMGR_STAT
      LogFile_record(FALSE, conn->xid, "START_BKND_BEGIN");
#endif
	  /* we start a new backend if we have delivered one view change
	   * message because we don't want to block the total order
	   * channel infinitely */
	  idle_process_list_element = DLRemHead(IdleProcessesList);
	  if (idle_process_list_element == 0 && 
	       (active_remote_processes < Repl_max_remote_processes ||
	        deliveredViewChange == 1))
	  {
#ifdef RMGR_DEBUG
  elog(NOTICE, "start new remote backend");
#endif
		waitForBknd = conn;
		/* (bk) we do not have any idle remote process, start a new one */
		RmgrStartBackend(conn->hostid, conn->xid);
		return;
	  }
	  else if (idle_process_list_element == 0)
	  {
		/* (bk) inform that we are waiting */
#ifdef RMGR_DEBUG
  elog(NOTICE, "gc2rtxn: wait for remote backend");
#endif
		new_remote_txn_waiting = TRUE;
		waitForBknd = conn;
		disableSocket(GCconn_total.bsock->sock);
		return;
	  }
	  else
	  { 
		/* (bk) this variable was set when repl manager got first request of 
		*   txn, it points to the same conn which is the parameter of this function 
		*   reset it because no backend startup necessary
		*/
		/* (bk) similar to RmgrAccept */
#ifdef RMGR_DEBUG
  elog(NOTICE, "gc2rtxn: get backend from queue");
#endif 
		waitForBknd = NULL;
		conn->dc.bsock = (bufsockptr) DLE_VAL(idle_process_list_element);
		conn->state = BKND_PRESENT;
		enableSocket(conn->dc.bsock->sock);
		DLFreeElem(idle_process_list_element);
		new_remote_txn_waiting = FALSE;
	  }
	}
    
  
  rtxnsock = conn->dc.bsock;
  conn->numDelivered++;
  switch(conn->state)
    {
    case NO_BKND:
      switch(msg_type)
	  {
		case MSG_ABORT:
#ifdef RMGR_DEBUG
		  elog(NOTICE, "Received early abort!!");
#endif
		  sockIgnoreData(RMGR_HDR_SIZE, group_socket);
		  conn->result = TO_ABORT;
		  break;
		case MSG_WRITESET:
			if (RmgrIsFailedSite(conn->hostid)) 
				elog(NOTICE,"don't even bother with this write set");
			else if (conn->result != TO_ABORT)
			  printf("should have received abort_msg first !!!\n");
#ifdef RMGR_STAT
		  LogFile_record(FALSE, conn->xid, 
			 "NO_BKND->CLOSED on MSG_WS");
#endif
		  sockIgnoreData(RMGR_HDR_SIZE + datalen, 
			 group_socket);
		  conn->state = CLOSED;
		/*(bk): as a dummy without socket connected it will be deleted in rmgr main loop */
		  break;
	  /* bk new: create backend only when write set arrives: better control of backend use */
		case MSG_COMMIT:	
#ifdef RMGR_DEBUG
		  elog(NOTICE, "Received early commit!!");
#endif
#ifdef RMGR_STAT
		  LogFile_record(FALSE, conn->xid, 
			 "BKND_PRESENT->COMMITTABLE on MSG_COMMIT");
#endif
		  conn->result = COMMITTABLE;
		  sockIgnoreData(RMGR_HDR_SIZE, group_socket);
		  break;

		default:
		  printf("proto 11\n");
		  HandleProtoError(conn);
		  break;
	  }
	  break;
    case BKND_PRESENT:
      switch(msg_type)
	  {
		case MSG_WRITESET:
		   //elog(NOTICE, "gc2rtxn: received msg type %d (len=%d), state is %d, result %d, hostid %x, xid %d",
       //(int) msg_type, datalen, 
       //(int) conn->state, conn->result, conn->hostid, conn->xid);
		  switch(conn->result) 
		  {
			case UNKNOWN_RESULT:
			  sockTransfer(rtxnsock, group_socket, 
				 datalen + RMGR_HDR_SIZE);
			  sockFlush(rtxnsock);
			  disableSocket(GCconn_total.bsock->sock);
			  conn->state = LOCK_PHASE;
#ifdef RMGR_STAT
			  LogFile_record(FALSE, conn->xid, "BKND_START_END");
#endif
			  break;
			/* bk: between write-set arrival  backend created the abort msg arrived */
			case TO_ABORT:
			  sockIgnoreData(RMGR_HDR_SIZE + datalen, group_socket);
			  conn->state = CLOSED;
			  break;
			/* bk: before backend creation the commit msg arrived */
			case COMMITTABLE:
			  sockTransfer(rtxnsock, group_socket, 
				 datalen + RMGR_HDR_SIZE);
			  sockPutInt((int) MSG_COMMIT, 4, rtxnsock);
			  sockPutInt(0, 4, rtxnsock);
			  sockFlush(rtxnsock);
			  disableSocket(GCconn_total.bsock->sock);
			  conn->state = LOCK_PHASE;
			  break;
			default:
			  printf("unknown result\n");
			  HandleProtoError(conn);
			  break;
		  }
		  break;
		case MSG_COMMIT:
			/* bk: no more possible */
			/* conn->state = COMMITTABLE; */
			/* sockIgnoreData(RMGR_HDR_SIZE, group_socket); */
			/* break; */
		default:
		  printf("proto 12\n");
		  HandleProtoError(conn);
		  break;
	  }
	  break;
    case COMMITTABLE:
    /* bk: no more possible */
/*      switch(msg_type)
	{
	case MSG_WRITESET:
	  sockTransfer(rtxnsock, group_socket, 
		       datalen + RMGR_HDR_SIZE);
	  sockPutInt((int) MSG_COMMIT, 4, rtxnsock);
	  sockPutInt(0, 4, rtxnsock);
	  sockFlush(rtxnsock);
	  disableSocket(GCconn.bsock->sock);
	  conn->state = LOCK_PHASE;
	  break;
	default:
	  printf("proto 13\n");
	  HandleProtoError(conn);
	  break;
	} */
      break;
    case WRITE_PHASE:
    /* bk: commit/abort may now also arrive in lockphase */
    case LOCK_PHASE:
      switch(msg_type)
	  {
		case MSG_COMMIT:
		case MSG_ABORT:
#ifdef RMGR_STAT
		  LogFile_record(FALSE, conn->xid, 
			 "WRITE->CLOSED on any MSG_COMMIT | MSG_ABORT");
#endif
		  sockTransfer(rtxnsock, group_socket, datalen + RMGR_HDR_SIZE);
		  sockFlush(rtxnsock);
		  /* bk: actually this is no more necessary, result is only important when
		  * commit/abort msg arrives before lock-phase */
		  if (msg_type == MSG_COMMIT)
			conn->result = COMMITTABLE;
		  else
			conn->result = TO_ABORT;
		  break;
		default:
		  printf("proto 14\n");
		  HandleProtoError(conn);
		  break;
	  }
      break;
    default:
	  printf("proto 14\n");
	  HandleProtoError(conn);
	  break;
    }
  sockConsume(group_socket);
}


/*
*	handle_special_message
*
*	Handles msgs coming from the group comm process and intended  
*	for view change information
*	
*
*	param:
*		group_socket - the socket from which to read,
*       simple == true if this message was received from the simple socket
*	return:
*		void
*/

static void
handle_special_message(bufsockptr group_socket, bool simple)
{

  rc_msg_t	msg_type;
  int    msg_type_cast;
  uint32  tmp;
  int			datalen = 0;
  int i,j, nread, ntotal;

  
  /* read the rmgr-header, that is the message type and the length of the rest */
  nread = 0;
  ntotal = 0;
  do{
	if((nread = sockReadData(group_socket, RMGR_HDR_SIZE - ntotal, TRUE)) < 0)
	{
	      elog(ERROR, 
		   "RMgr: unable to read group comm. msg header");
	      
	 }
	 ntotal += nread;
	}while(ntotal < RMGR_HDR_SIZE);
  
  sockGetInt(&msg_type_cast, 4, group_socket);
  sockGetInt(&datalen, 4, group_socket);
  msg_type = (rc_msg_t) msg_type_cast;

  elog(NOTICE, "special: received msg type %d (len=%d)",
       (int) msg_type, datalen);
 
       
   switch(msg_type)
   {
    	case MSG_SITE_HAS_LEFT_GROUP:
    	// first guarantee that the rest of the message is there
    		nread = 0;
    		ntotal = 0;
    		do{
				if((nread = sockReadData(group_socket, datalen - ntotal, TRUE)) < 0)
				{
	      			elog(ERROR, 
		   				"RMgr: unable to read group comm. msg header");
				}
	 			ntotal += nread;
			}while(ntotal < datalen);
			deliveredViewChange++;
			if (deliveredViewChange == 1)
			{
				/* read which members have failed and include them
				 * in the new data structure */
				number_failed_members = datalen / 4;
				elog(NOTICE,"del 1: %d members failed", number_failed_members);
				for (i=0;i<number_failed_members;i++)
				{
					sockGetInt(&group_failed_hostids[i], 4, group_socket);
					elog(NOTICE,"site %u failed", group_failed_hostids[i]);
				}
				
				/* we disable the  socket until we have
				* finished the failure cleanup
				*/
				disableSocket(group_socket->sock);
			/* in the case this is from the simple socket, we
			   check already all existing so to guarantee that
			   the total order will not block for ever in case of
			   not enough remote backends
			*/ 
				if (simple)
				{
					elog(NOTICE,"walk for first time because simple");
					WalkRemoteBackends();
					elog(NOTICE,"end of walk");
				}

				
			}
			else if (deliveredViewChange == 2)
			{
				elog(NOTICE,"del 2");
				j = datalen / 4;
				if (number_failed_members != j)
					elog(NOTICE,"first and second group change have different number of failed members");
				for (i=0;i<j;i++)
				{
					sockGetInt(&tmp, 4, group_socket);
					if (tmp != group_failed_hostids[i])
						elog(NOTICE,"different failed nodes old %u, new %u",
							group_failed_hostids[i], tmp);
				}
				
				disableSocket(group_socket->sock);
				WalkRemoteBackends();
				/* we have finished, the rest will happen automatically,
			       once the remotes have finished and the closed conns
			       are detected */
			    elog(NOTICE,"view change finihsed");
				deliveredViewChange = 0;
				number_failed_members = 0;
				enableSocket(GCconn_total.bsock->sock);
				enableSocket(GCconn_basic.bsock->sock);
			}	
			else 
			{
				elog(NOTICE,"wrong number of delivered view change");	
			}
			break;
		default:
			elog(NOTICE,"so far there shoud not come any other message");
			break;
   }
					
}

/* RmgrIsFailedSite 
*
*  checks whether a given host (given by hostid) is a newly failed site
*  input hostid
*/
static bool
RmgrIsFailedSite(uint32 hostid)
{
	int i;
	
	for (i=0;i<number_failed_members;i++)
	{
		if (hostid == group_failed_hostids[i])
			return TRUE;
	}
	return FALSE;
}


/* walk through all failed remote sites and check what is to
* do
*/

static void
WalkRemoteBackends(void)
{
  Dlelem 	*curr = NULL;
  Dlelem    *next;
  Rconn 	*conn = NULL;
  
  curr = DLGetHead(RconnList);
  
  while(curr){
    conn = (Rconn *) DLE_VAL(curr);
 
    if(RmgrIsFailedSite(conn->hostid)) 
    {
    	elog(NOTICE,"found conn of failed site %u",conn->hostid);
    	switch(conn->state)
    	{
    		case CLOSED:
    			break;
    		case LOCK_PHASE:
    		case WRITE_PHASE:
    			switch(conn->result)
    			{
    				case UNKNOWN_RESULT:
    					elog(NOTICE,"conn in lock/write and unknown");
    					sockPutInt((int) MSG_ABORT, 4, conn->dc.bsock);
			  			sockPutInt(0, 4, conn->dc.bsock);
			  			sockFlush(conn->dc.bsock);
			  			conn->result = TO_ABORT;
			  			break;
			  		case COMMITTABLE:
			  		case TO_ABORT:
			  			elog(NOTICE,"conn in lock/write and nothing to do");
			  			break;
			  		default:
			  			elog(ERROR,"wrong result");
			  			break;
    			}
    			break;
    		case BKND_PRESENT:
    			elog(NOTICE,"BKND_PRESENT cannot really be!");
    			elog(ERROR,"i dy to be on the safe site");
    			break;
    		case NO_BKND:
    			switch(conn->result)
    			{
    				case UNKNOWN_RESULT:
    				case TO_ABORT:
    				/* this means that the decision message has not yet
    				* been delivered and will not be delivered anymore or
    				* that it is to-abort
    				* (this is true because we call this only after the
    				* the view change message is delivered on simple */
 
    				/* we can be here for several reasons */
    				/* first: the write-set can already have been delivered
    			 	* but either we have to wait for a new remote to be created
    			 	* of to be freed */
    			 	// let's take first the blocked one, because this blocks
    			 	// the total order socket and we have to free it
    			 		if (conn == waitForBknd && new_remote_txn_waiting == TRUE)
    			 		{
    			 			elog(NOTICE,"set abort for remote txn waiting for free backend");
    			 			waitForBknd = NULL;
    			 			conn->state = NO_BKND;
    			 			conn->result = TO_ABORT;
    			 			new_remote_txn_waiting = FALSE;
				  			enableSocket(GCconn_total.bsock->sock);
				  			handle_gc2rtxn(conn, GCconn_total.bsock);
    			 		}
    			 	/* here we are were the backend gets currently created;
    			 	*  if the result is still unknown, the decision message
    			 	*  has not arrived. So we set conn->result to TO_ABORT
    			 	*/ 
    			 		else if (conn == waitForBknd)
    			 		{
    			 			elog(NOTICE,"set abort for remote txn waiting for startup");
    			 			conn->result = TO_ABORT;
    			 		}
    			 		else
    			 		{
    			 			/* this can only been if the write set hasn't
    			 			 * arrived yet: if this is already the second
    			 			 * view change delivery, close the connection
    			 			 * because the write set will not arrive anymore
    			 			 */
    			 			if (deliveredViewChange == 2)
    			 			{	
    			 				elog(NOTICE,"close a conn where write set is missing");
    			 				conn->state = CLOSED;
    			 			}
    			 		}
    			 		break;
 
    			 	case COMMITTABLE:
    			 	/* tricky: if the txn waits for a backend to get freed
    			 	 * we might get stuck in the case that for the other
    			 	 * backends the decision messages have not arrivied
    			 	 * before the view change: so start a new backend
    			 	  */
    			 	 if (conn == waitForBknd && new_remote_txn_waiting == TRUE)
    			 		{
    			 			elog(NOTICE,"start a new backend to keep going");
    			 			conn->state = NO_BKND;
    			 			new_remote_txn_waiting = FALSE;
    			 	 		RmgrStartBackend(conn->hostid, conn->xid);
    			 		}
    			 		else if (conn == waitForBknd)
    			 		{
    			 			elog(NOTICE,"do nothing, will commit automatically");
    			 		}
    			 		else
    			 		{
    			 			/* this can only been if the write set hasn't
    			 			 * arrived yet: if this is already the second
    			 			 * view change delivery, close the connection
    			 			 * because the write set will not arrive anymore
    			 			 */
    			 			if (deliveredViewChange == 2)
    			 			{	
    			 				elog(NOTICE,"close a conn where write set is missing");
    			 				conn->state = CLOSED;
    			 			}
    			 		}
    					break;
    				default:
    					elog(NOTICE,"there shouldn't be an default here!!!!");
    			 		elog(ERROR,"exit to be on the safe site");
    			 		break;
    			}
    			break;
    		default:
    			elog(NOTICE,"not in an expected state %d",conn->state);
    			elog(ERROR," dy to be on the safe site");
    			break;
    	}
    }
   	next = DLGetSucc(curr);
    curr = next;
  }
}



/*
*	RmgrFindConn
*
*	Finds (and deletes) an element in the RconnList list by its 
*	host- and xid.
*
*	param:
*		hostid - the hostid to look for
*		xid - the xid to look for
*		remove - bool telling whether to remove the element from
*				the list
*	return:
*		Rconn - pointer to the element, NULL if not found
*/

static	Rconn *
RmgrFindConn(long hostid, long xid, bool remove)
{
  Dlelem 	*curr = NULL;
  Rconn 	*conn = NULL;
  
  
  curr = DLGetHead(RconnList);
  while(curr){
    Rconn *currconn = (Rconn *) DLE_VAL(curr);
    if((currconn->hostid == hostid) && (currconn->xid == xid)){
      conn = currconn;
      if(remove)
	  {
		DLRemove(curr);
		DLFreeElem(curr);
	  }
	  break;
	}
    curr = DLGetSucc(curr);
  }
  return conn;
}




/*
*	destroyRconn
*	
*	frees an Rconn structure, including the associated 
*	buffered socket
*	
*	param:
*		conn - pointer to the connection struct to free
*	return:
*		void
*/

static void
destroyRconn(Rconn *conn, bool destroysocket)
{
#ifdef RMGR_DEBUG
  elog(NOTICE, "destroying Rconn hid = %x, xid = %d", 
       conn->hostid, conn->xid);
#endif
  if (destroysocket)
	sockDestroy(conn->dc.bsock);
  free(conn);
}

/*
*	initRconn
*
*	allocates and initializes an Rconn struct
*
*	param:
*	    hid - the host ID to use
*	    xid - the transaction ID to use
*	    bsock - the socket to the client
*	return:
*		Rconn - the initialized Rconn struct
*/
static Rconn *
initRconn(uint32 hid, uint32 xid, bufsockptr bsock)
{
  Rconn *conn;
  
  if (!(conn = (Rconn *) malloc(sizeof(Rconn))))
    {
      elog(DEBUG, "initRconn: malloc failed");
      return NULL;	
    }
  conn->dc.bsock = bsock;
  conn->dc.recv_state = RECV_IDLE;
  conn->dc.datalen = -1;
  conn->procid = -1;
  conn->hostid = hid;
  conn->xid = xid;
  conn->numDelivered = 0;
  conn->state = (hid == myHostId) ? READ_PHASE : NO_BKND;
  conn->result = UNKNOWN_RESULT;
  
  return conn;
}




/*
*	enableSocket
*	
*	activates a given socket, i.e. this socket will be monitored
*	on the next call of select()
*	
*	param:
*		sock - the socket to activate
*	return:
*		void
*/
static void
enableSocket(int sock)
{
  FD_SET(sock, &active_rsocks);
  if(maxSock < sock)
    maxSock = sock;

  elog(NOTICE, "enabled socket %d (max = %d)", sock, maxSock);
}

/*
*	disableSocket
*	
*	deactivates a given socket, i.e. this socket will not be monitored
*	on the next call of select()
*	
*	param:
*		sock - the socket to activate
*	return:
*		void
*/
static void
disableSocket(int sock)
{
  int i;

#ifdef RMGR_DEBUG
  elog(NOTICE, "disabling socket %d", sock);
#endif

  FD_CLR(sock, &active_rsocks);
  if(sock == maxSock)
    {
      for(i = maxSock-1; i >= 0; i--)
	{
	  if(FD_ISSET(i, &active_rsocks))
	    {
	      maxSock = i;
	      return;
	    }
	}
      
      maxSock = -1;
    }
}

static void
rpmgr_die(SIGNAL_ARGS)
{
  elog(NOTICE, "Replication Manager dying");
  exit(1);
}


#ifdef RMGR_BLAXYZ //these stat fctns are disabled for now!
/* statistic gathering functions*/

static void
_statTimeNow(Rconn *conn, int statSlotIndex)
{
  struct tms dummy;
  
  conn->lastStop = times(&dummy);
  conn->statSlot_ind = statSlotIndex;
}

static void
_statGather(int index, clock_t startTime)
{
  struct tms dummy;
  
  clock_t interval = times(&dummy) - startTime;
  
  if(interval < 0)
    {
      return;
    }
  
  if(tmax[index] == 0)
    {
      tmax[index] = interval;
      tmin[index] = interval;
      tsum[index] += interval;
    }
  else
    {
      if(tmax[index] < interval)
	{
	  tmax[index] = interval;
	}
      
      if(tmin[index] > interval)
	{
	  tmin[index] = interval;
	}
      
      tsum[index] += interval;
    }
}

static void
_statShow(void)
{
	
  printf("-----------------------------------------\n");
  printf("Replication manager statistics (node %x):\n", myHostId);
  printf("-----------------------------------------\n\n");
  
  printf("General stats:\n");
  printf("\tTotal Remote Txns: %d\n", NumRTxns);
  printf("\tTotal Local Txns: %d\n", NumLTxns);
  
  fflush(stdout);
  if(NumLTxns > 0)
    {
      printf("Write set size sent:\n\t\tmin = %d\n\t\tmax = %d\n\t\tavg = %1.3e",
	     WSmin, WSmax, (float)WSsum/(float) NumLTxns);
      
      printf("\nLocal Backends:\n");
      printf("\tSend Phase:\n\t\t min = %1.3e sec.\n\t\tmax = %1.3e sec.\n\t\tavg = %1.3e sec.\n",
	     (float)tmin[SendPhase_ind]/(float)CLK_TCK, 
	     (float)tmax[SendPhase_ind]/(float)CLK_TCK, 
	     (float)tsum[SendPhase_ind]/(float)(NumLTxns * CLK_TCK));
      printf("\tLock Phase:\n\t\tmin = %1.3e sec.\n\t\tmax = %1.3e  sec.\n\t\tavg = %1.3e  sec.\n",
	     (float)tmin[LockPhaseLocal_ind]/(float)CLK_TCK, 
	     (float)tmax[LockPhaseLocal_ind]/(float)CLK_TCK, 
	     (float)tsum[LockPhaseLocal_ind]/(float)(NumLTxns * CLK_TCK));
      printf("\tWrite Phase:\n\t\tmin = %1.3e  sec.\n\t\tmax = %1.3e  sec.\n\t\tavg = %1.3e  sec.\n",
	     (float)tmin[WritePhaseLocal_ind]/(float)CLK_TCK, 
	     (float)tmax[WritePhaseLocal_ind]/(float)CLK_TCK, 
	     (float)tsum[WritePhaseLocal_ind]/(float)(NumLTxns * CLK_TCK));
      fflush(stdout);
    } 
  
  if(NumRTxns > 0)
    {
      printf("\nRemote Backends:\n");
      printf("\tBackend startup:\n\t\tmin= %1.3e  sec.\n\t\tmax= %1.3e  sec.\n\t\tavg= %1.3e  sec.\n",
	     (float)tmin[StartupRemote_ind]/(float)CLK_TCK, 
	     (float)tmax[StartupRemote_ind]/(float)CLK_TCK, 
	     (float)tsum[StartupRemote_ind]/(float)(NumRTxns * CLK_TCK));
      printf("\tLock Phase:\n\t\tmin = %1.3e  sec.\n\t\tmax = %1.3e  sec.\n\t\tavg = %1.3e  sec.\n",
	     (float)tmin[LockPhaseRemote_ind]/(float)CLK_TCK, 
	     (float)tmax[LockPhaseRemote_ind]/(float)CLK_TCK, 
	     (float)tsum[LockPhaseRemote_ind]/(float)(NumRTxns * CLK_TCK));
      printf("\tWrite Phase:\n\t\tmin = %1.3e  sec.\n\t\tmax = %1.3e  sec.\n\t\tavg = %1.3e  sec.\n",
	     (float)tmin[WritePhaseRemote_ind]/(float)CLK_TCK, 
	     (float)tmax[WritePhaseRemote_ind]/(float)CLK_TCK, 
	     (float)tsum[WritePhaseRemote_ind]/(float)(NumRTxns * CLK_TCK));
      fflush(stdout);
    }
}
#endif

