/*
	pgjobs.cpp

	Main entry point of the config tool.
	Parses the parameters and invokes a Config instance.

	Project: pgjobs
	Author: Zlatko Michailov
	Created:  3-Aug-2003
	Updated: 14-Oct-2003
	Updated: 15-Oct-2003
	Updated: 16-Oct-2003
	Updated: 28-Oct-2003
	Updated:  7-May-2004
	Updated: 16-Jun-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003-2004, Zlatko Michailov

*/



//--------------------------------------------------------------------------------

#include <string.h>
#include <iostream>
#include "config.h"
#include "sql.h"


using namespace std;



//--------------------------------------------------------------------------------
// Constants relevant to this file only

static const char OptHost[]					= "-h";
static const char OptPort[]					= "-p";
static const char OptDatabase[]				= "-d";
static const char OptNewHost[]				= "-hh";
static const char OptNewPort[]				= "-pp";
static const char OptNewDatabase[]			= "-dd";
static const char OptLogLevel[]				= "-l";
static const char OptUser[]					= "-u";
static const char OptNotification[]			= "-n";
static const char OptRecipient[]			= "-r";
static const char OptVersion[]				= "-v";



static const char OptionsNone[]				= "";
static const char OptionsDbRead[]			= "[-h HOST] [-p PORT] -d DATABASE";
static const char OptionsSystemWrite[]		= "-v VERSION";
static const char OptionsMasterWrite[]		= "-l LOGLEVEL";
static const char OptionsDbCreate[]			= "[-h HOST] [-p PORT] -d DATABASE "
											  "[-l LOGLEVEL] [-u USER] "
											  "[-n NOTIFICATION] [-r RECIPIENT]";
static const char OptionsDbUpdate[]			= "[-h HOST] [-p PORT] -d DATABASE "
											  "[-hh NEWHOST] [-pp NEWPORT] [-dd NEWDATABASE] "
											  "[-l LOGLEVEL] [-u USER] "
											  "[-n NOTIFICATION] [-r RECIPIENT]";

static const char  Done[]					= "Done.";



//--------------------------------------------------------------------------------
// Structure for input options

struct OptionValue
{
	const char*		Option;
	string&			Value;
	bool			IsMandatory;
	bool			IsForUpdate;
};



//--------------------------------------------------------------------------------
// Forward function declarations

static int EnableDb();
static int DisableDb();
static int DisableAll();
static int ViewSystem();
static int ViewMaster();
static int ViewDb();
static int UpdateSystem();
static int UpdateMaster();
static int UpdateDb();
static int Help();

static int ParseOptions( int n, OptionValue options[] );
static int DisplayConfigRow( Config& config, const string& row, const char** schema, int n );
static int UpdateConfigCols( Config& config, string& row, OptionValue options[], int n );
static int DisableDb( const string& database, const string& host, const string& port );



//--------------------------------------------------------------------------------
// Command line arguments

static int		argc	= 0;
static char**	argv	= 0;



//--------------------------------------------------------------------------------
// Command routing table

struct CommandHandler
{
	const char* Command;
	int ( *Handler )();
	const char* Options;
};



static const CommandHandler CommandRoutingTable[] =
{
	{ "enabledb",		EnableDb,		OptionsDbCreate },
	{ "disabledb",		DisableDb,		OptionsDbRead },
	{ "disableall",		DisableAll,		OptionsNone },
	{ "viewsystem",		ViewSystem,		OptionsNone },
	{ "viewmaster",		ViewMaster,		OptionsNone },
	{ "viewdb",			ViewDb,			OptionsDbRead },
	{ "updatesystem",	UpdateSystem,	OptionsSystemWrite },
	{ "updatemaster",	UpdateMaster,	OptionsMasterWrite },
	{ "updatedb",		UpdateDb,		OptionsDbUpdate },
	{ "--help",			Help,			OptionsNone }
};



//--------------------------------------------------------------------------------

int main( int argc, char** argv )
{
	int err = 0;

	// Save the command line arguments for the command handlers
	::argc = argc;
	::argv = argv;

	cout << endl;

	// Parse command: 0-exe, 1-command, ...
	if ( argc >= 2 )
	{
		int n = sizeof( CommandRoutingTable ) / sizeof( CommandHandler );
		int i;

		// Find handler
		for ( i = 0; i < n; i++ )
		{
			if ( !strcasecmp( argv[ 1 ], CommandRoutingTable[ i ].Command ) )
			{
				// Invoke handler and break out
				err = ( *CommandRoutingTable[ i ].Handler )();
				break;
			}
		}

		// Check if handler was found
		if ( i >= n )
		{
			cout << "Command " << argv[ 1 ] << " is not supported." << endl;
			Help();
			err = 1;
		}
	}
	else
	{
		cout << "No command was supplied." << endl;
		Help();
		err = 2;
	}

	cout << endl;

	return err;
}



//--------------------------------------------------------------------------------
// Command handlers

static int EnableDb()
{
	string		host			= Default;
	string		port			= Default;
	string		database;
	string		logLevel		= LogLevelDefault;
	string		user			= Default;
	string		notification	= None;
	string		recipient		= None;
	int			err				= 0;
	OptionValue	options[]		=
	{
		{ OptHost,			host,			false,	false },
		{ OptPort,			port,			false,	false },
		{ OptDatabase,		database,		true,	false },
		{ OptLogLevel,		logLevel,		false,	false },
		{ OptUser,			user,			false,	false },
		{ OptNotification,	notification,	false,	false },
		{ OptRecipient,		recipient,		false,	false }
	};
	int			sizeOfOptions	= sizeof( options ) / sizeof( OptionValue );

	err = ParseOptions( sizeOfOptions, options );
	if ( !err )
	{
		Config	config( ConfigFilePath );
		DbRef	dbref( database, host, port );
		string	psqlOptions;
		string	row;
		string	error;
		bool	ok;


		// Create database schema
		ok = dbref.ToPsqlOptions( psqlOptions );
		if ( ok )
		{
			SQL		sql( psqlOptions.c_str() );
			string	sqlFile;

			sqlFile  = InstallRoot;
			sqlFile += "/enabledb.sql";

			ok = sql.ExecFile( sqlFile.c_str() );
		}


		// Create config entry
		// Create a default row
		if ( ok )
		{
			ok = config.EnableDb( dbref, error );
		}

		// Read the above default row
		if ( ok )
		{
			ok = config.GetDb( dbref, row, error );
		}

		// Update columns
		if ( ok )
		{
			ok = !UpdateConfigCols( config, row, options,
									sizeof( options ) / sizeof( OptionValue ) );
		}

		// Update row in config file
		if ( ok )
		{
			ok = config.SetDb( dbref, row, error );
		}

		if ( ok )
		{
			// Display new settings
			DisplayConfigRow( config, row, SchemaDb, SizeOfSchemaDb );
		}
		else
		{
			cout << error.c_str() << endl;
			err = 11;
		}
	}

	return err;
}



static int DisableDb()
{
	string		host			= Default;
	string		port			= Default;
	string		database;
	int			err				= 0;
	OptionValue	options[]		=
	{
		{ OptHost,			host,			false,	false },
		{ OptPort,			port,			false,	false },
		{ OptDatabase,		database,		true,	false }
	};
	int			sizeOfOptions	= sizeof( options ) / sizeof( OptionValue );

	err = ParseOptions( sizeOfOptions, options );
	if ( !err )
	{
		err = DisableDb( database, host, port );
		if ( !err )
		{
			cout << Done << endl;
		}
	}

	return err;
}



static int DisableAll()
{
	Config			config( ConfigFilePath );
	StreamPosition	pos	= 0;
	DbInfo			dbInfo;
	string			error;
	bool			ok	= true;

	do
	{
		StreamPosition	posPrev	= pos;
		int				err		= 0;

		error.clear();

		// Read DbConfigs sequentially
		ok = config.SeqGetDb( pos, dbInfo, error );
		if ( ok )
		{
			err = DisableDb( dbInfo.Database, dbInfo.Host, dbInfo.Port );

			// If the row has been deleted, the next row
			// has been shifted to the current position
			if ( !err )
			{
				pos = posPrev;
			}
		}

		if ( !error.empty() )
		{
			cout << error.c_str() << endl;
		}
	}
	while ( ok || !error.empty() );

	cout << Done << endl;

	return 0;
}



static int ViewSystem()
{
	Config	config( ConfigFilePath );
	string	row;
	string	error;
	int		err = 0;

	// Read and display row
	if ( config.GetSystem( row, error ) )
	{
		err = DisplayConfigRow( config, row, SchemaSystem, SizeOfSchemaSystem );
	}
	else
	{
		cout << error.c_str() << endl;
		err = 11;
	}

	return err;
}



static int ViewMaster()
{
	Config	config( ConfigFilePath );
	string	row;
	string	error;
	int		err = 0;

	// Read and display row
	if ( config.GetMaster( row, error ) )
	{
		err = DisplayConfigRow( config, row, SchemaMaster, SizeOfSchemaMaster );
	}
	else
	{
		cout << error.c_str() << endl;
		err = 11;
	}

	return err;
}



static int ViewDb()
{
	string		host			= Default;
	string		port			= Default;
	string		database;
	int			err				= 0;
	OptionValue	options[]		=
	{
		{ OptHost,		host,		false,	false },
		{ OptPort,		port,		false,	false },
		{ OptDatabase,	database,	true,	false }
	};
	int			sizeOfOptions	= sizeof( options ) / sizeof( OptionValue );

	err = ParseOptions( sizeOfOptions, options );
	if ( !err )
	{
		Config	config( ConfigFilePath );
		DbRef	dbref( database, host, port );
		string	row;
		string	error;

		// Read and display row
		if ( config.GetDb( dbref, row, error ) )
		{
			err = DisplayConfigRow( config, row, SchemaDb, SizeOfSchemaDb );
		}
		else
		{
			cout << error.c_str() << endl;
			err = 11;
		}
	}

	return err;
}



static int UpdateSystem()
{
	string		version;
	int			err	= 0;
	OptionValue	options[] =
	{
		{ OptVersion, version, true, false }
	};
	int			sizeOfOptions	= sizeof( options ) / sizeof( OptionValue );

	err = ParseOptions( sizeOfOptions, options );
	if ( !err )
	{
		Config	config( ConfigFilePath );
		string	row;
		string	error;
		bool	ok;

		// Read row
		ok = config.GetSystem( row, error );

		// Update columns
		if ( ok )
		{
			ok = !UpdateConfigCols( config, row, options, sizeOfOptions );
		}

		// Update row in config file
		if ( ok )
		{
			ok = config.SetSystem( row, error );
		}

		if ( ok )
		{
			// Display new settings
			DisplayConfigRow( config, row, SchemaSystem, SizeOfSchemaSystem );
		}
		else
		{
			cout << error.c_str() << endl;
			err = 11;
		}
	}

	return err;
}



static int UpdateMaster()
{
	string		logLevel;
	int			err	= 0;
	OptionValue	options[] =
	{
		{ OptLogLevel, logLevel, true, false }
	};
	int			sizeOfOptions	= sizeof( options ) / sizeof( OptionValue );

	err = ParseOptions( sizeOfOptions, options );
	if ( !err )
	{
		Config	config( ConfigFilePath );
		string	row;
		string	error;
		bool	ok;

		// Read row
		ok = config.GetMaster( row, error );

		// Update columns
		if ( ok )
		{
			ok = !UpdateConfigCols( config, row, options, sizeOfOptions );
		}

		// Update row in config file
		if ( ok )
		{
			ok = config.SetMaster( row, error );
		}

		if ( ok )
		{
			DisplayConfigRow( config, row, SchemaMaster, SizeOfSchemaMaster );
		}
		else
		{
			cout << error.c_str() << endl;
			err = 11;
		}
	}

	return err;
}



static int UpdateDb()
{
	string		host			= Default;
	string		port			= Default;
	string		database;
	string		newHost;
	string		newPort;
	string		newDatabase;
	string		logLevel		= LogLevelDefault;
	string		user			= Default;
	string		notification	= None;
	string		recipient		= None;
	int			err				= 0;
	OptionValue	options[]		=
	{
		{ OptNewHost,		newHost,		false,	false },
		{ OptNewPort,		newPort,		false,	false },
		{ OptNewDatabase,	newDatabase,	false,	false },
		{ OptLogLevel,		logLevel,		false,	false },
		{ OptUser,			user,			false,	false },
		{ OptNotification,	notification,	false,	false },
		{ OptRecipient,		recipient,		false,	false },
		{ OptHost,			host,			false,	false },
		{ OptPort,			port,			false,	false },
		{ OptDatabase,		database,		true,	false }
	};
	int			sizeOfOptions			= sizeof( options ) / sizeof( OptionValue );
	int			sizeOfUpdatableOptions	= sizeOfOptions - 3;

	err = ParseOptions( sizeOfOptions, options );
	if ( !err )
	{
		Config	config( ConfigFilePath );
		DbRef	dbref( database, host, port );
		string	row;
		string	error;
		bool	ok;

		// Read row
		ok = config.GetDb( dbref, row, error );

		// Update columns
		if ( ok )
		{
			ok = !UpdateConfigCols( config, row, options, sizeOfUpdatableOptions );
		}

		// Update row in config file
		if ( ok )
		{
			ok = config.SetDb( dbref, row, error );
		}

		if ( ok )
		{
			DisplayConfigRow( config, row, SchemaDb, SizeOfSchemaDb );
		}
		else
		{
			cout << error.c_str() << endl;
			err = 11;
		}
	}

	return err;
}



static int Help()
{
	int n = sizeof( CommandRoutingTable ) / sizeof( CommandHandler );
	int i;

	cout << endl << "pgjobs COMMAND [OPTIONS ...]" << endl << endl;

	// Traverse all handlers
	for ( i = 0; i < n; i++ )
	{
		cout << "    " << CommandRoutingTable[ i ].Command << " "
			<< CommandRoutingTable[ i ].Options << endl;
	}

	cout << endl;

	return 0;
}



//--------------------------------------------------------------------------------
// Internal support

static int ParseOptions( int n, OptionValue options[] )
{
	int	err	= 0;
	int	i;
	int	j;

	// Read all arguments
	for ( i = 2; i < argc; i++ )
	{
		// Try to find the current argument in the provided list
		for ( j = 0; j < n; j++ )
		{
			if ( !strcmp( argv[ i ], options[ j ].Option ) )
			{
				// A value may follow
				if ( ++i < argc )
				{
					options[ j ].Value			= argv[ i ];
					options[ j ].IsForUpdate	= true;
				}
				break;
			}
		}

		// Check for unexpected options and show a warning
		if ( j >= n )
		{
			cout << "Option " << argv[ i ] << " is irrelevant. Continue." << endl;
		}
	}

	// Check for unsupplied mandatory arguments
	for ( j = 0; j < n; j++ )
	{
		if ( options[ j ].IsMandatory && !options[ j ].IsForUpdate )
		{
			cout << "Option " << options[ j ].Option << " was not supplied." << endl;
			err = 21;
		}
	}

	return err;
}



static int DisplayConfigRow( Config& config, const string& row, const char** schema, int n )
{
	// Columns:
	//    Name = Value
	for ( int i = 0 ; i < n; i++ )
	{
		string col;

		cout << Indent << schema[ i ] << " = ";
		if ( config.GetCol( row, i + 1, col ) )
		{
			cout << col.c_str();
		}
		cout << endl;
	}

	return 0;
}



static int UpdateConfigCols( Config& config, string& row, OptionValue options[], int n )
{
	// Columns
	for ( int i = 0 ; i < n; i++ )
	{
		if ( options[ i ].IsForUpdate )
		{
			string col = options[ i ].Value;

			if ( !config.SetCol( row, i + 1, col ) )
			{
				cout << "Could not persist option " << options[ i ].Option
					 << " " << options[ i ].Value << ". Continue." << endl;
			}
		}
	}

	return 0;
}



static int DisableDb( const string& database, const string& host, const string& port )
{
	int		err	= 0;
	Config	config( ConfigFilePath );
	DbRef	dbref( database, host, port );
	string	psqlOptions;
	string	error;
	bool	ok;

	// Remove the db row
	ok = config.DisableDb( dbref, error );


	// Drop database schema
	if ( ok )
	{
		ok = dbref.ToPsqlOptions( psqlOptions );
	}

	if ( ok )
	{
		SQL		sql( psqlOptions.c_str() );
		string	sqlFile;

		sqlFile  = InstallRoot;
		sqlFile += "/disabledb.sql";

		sql.ExecFile( sqlFile.c_str() );
	}


	if ( !ok )
	{
		cout << error.c_str() << endl;
		err = 11;
	}

	return err;
}



