################################################################################
##$Namespace: logger
##$Version: 1.0.0
##$Author: Michael E Allen
##$Copyright: Copyright(C)2001 Michael E Allen
##$License: GNU General Public License
##$Create Date: April 23, 2001
##$Description: Provides components for calculating checksums
##$Bugs: 
##+	None
##$To Do: 
##+	Nothing
##$Changes:
##+	None
##$Disclaimer:
##+	This program is free software; you can redistribute it and/or modify
##+	it under the terms of the GNU General Public License as published by
##+	the Free Software Foundation; either version 2 of the License, or
##+	(at your option) any later version.
##+	This program is distributed in the hope that it will be useful,
##+	but WITHOUT ANY WARRANTY; without even the implied warranty of
##+	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##+	GNU General Public License for more details.
##+	
##+	You should have received a copy of the GNU General Public License
##+	along with this program; if not, write to the Free Software
##+	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
################################################################################
namespace eval ::logger {
	variable FileHandle ""
	variable FilePath ""
	variable FileName ""
	variable FileNameTimeStampFormat "%Y%m%d"
	variable LogTimeStampFormat "%m/%d/%Y %I:%M:%S%p %Z"
	variable LogBreakLine "=========================================="
	variable LogIndentationLevel 0

	######################################################################
	##Open
	##  Description:
	##    Prepares the log by writing a standard message
	##    in the form of:
	##      LOG FOR my_file.tcl
	##      HOST:my_host
	##	    PACKAGE:xyz
	##
	##    PACKAGE is a list of a REQUIRED packages for this script
	##    
	##  Inputs:
	##    pPath
	##      The path of the file to write the messages
	##
	##      Uses the following format for the filename
	##        path/filename_timestamp.log
	##      Where filename is the name of the file used to invoke the
	##      main script and timestamp is defined by the variable 
	##      FileNameTimeStamp
	########################################################################
	proc Open {{pPath "."}} { 
		variable FileHandle
		variable FilePath
		variable FileName
		variable FileNameTimeStampFormat
		variable LogTimeStampFormat
		variable LogBreakLine
		variable LogIndentationLevel
	
		global argv0
		
		if {($FileHandle != "")} {
			Close $FileHandle
		}
	
		set FilePath $pPath
	
		set lFileName [file tail $argv0]
		set lRootFileName [file rootname $lFileName]
		set lNow [::datetime::Now $FileNameTimeStampFormat]
	
		set FileName "$pPath/$lRootFileName\_$lNow.log"
		
		try {
			if [catch {set FileHandle [open $FileName a]} lCatch] {
				file mkdir $pPath
				set FileHandle [open $FileName a]
			}
		} catch lErrorMessage {
			Error {
				throw LoggerException $lErrorMessage
			}
		}
		
		##puts $FileHandle $LogBreakLine
		##puts $FileHandle "LOG FOR $lFileName"
		##puts $FileHandle "HOST:[info hostname]"
	
		##foreach lPackageName [lsort -dictionary [package names]] {
			##if {($lPackageName != "Tcl")} {
				##puts $FileHandle "PACKAGE:$lPackageName"
			##}
		##}
	
		##puts $FileHandle $LogBreakLine
	
		puts $FileHandle \
			"\[[::datetime::Now $LogTimeStampFormat]]\[[format %6s [pid]]]:OPENED"
	
		flush $FileHandle
	
		set LogIndentationLevel 1
		return
	}
	
	#######################################################################
	##RollOverFileName
	##  Description:
	##    Checks the timestamp of the current filename and if different
	##    from the current date, closes the log and opens a new one
	#######################################################################
	proc RollOverFileName {} {
		variable FilePath
		variable FileName
		variable FileNameTimeStampFormat
		
		global argv0
		
		set lFileName [file tail $argv0]
		set lRootFileName [file rootname $lFileName]
		set lNow [::datetime::Now $FileNameTimeStampFormat]
		
		set lCurrentFileName "$FilePath/$lRootFileName\_$lNow.log"
	
		if {($lCurrentFileName != $FileName)} {
			Close
			Open $FilePath
		}
	}
	
	#######################################################################
	##Close
	##  Description:
	##    Closes the log by writing a standard message to the file
	##    and closing the file 
	#######################################################################
	proc Close {} {
		variable FileHandle
		variable FileName
		variable LogTimeStampFormat
		##variable LogBreakLine
		variable LogIndentationLevel
	
		if {($FileHandle == "")} {
			return
		}
	
		set lNow [::datetime::Now $LogTimeStampFormat]
		puts $FileHandle "\[$lNow]\[[format %6s [pid]]]:CLOSED"
		##puts $FileHandle $LogBreakLine
	
		flush $FileHandle
	
		close $FileHandle
		
		set FileHandle ""
		set FileName ""
		set LogIndentationLevel 0
	}
	
	#######################################################################
	##Print
	##  Description:
	##    Output a line of text to the log file
	##    All text is indented based on the current indentation level
	##
	##  Input:
	##    pLine
	##      A Line of text to be output
	#######################################################################
	proc Print {{pLine ""}} {
		variable FileHandle
		variable LogTimeStampFormat
	
		if {($FileHandle == "")} {
			return
		}
	
		RollOverFileName	
	
		set lLine "[Indent]$pLine"
		set lNow [::datetime::Now $LogTimeStampFormat]
		puts $FileHandle "\[$lNow]\[[format %6s [pid]]]:$lLine"
		
		flush $FileHandle
	}
	
	#######################################################################
	##EnterProc
	##  Description:
	##    Outputs a line of text with the name of the current procedure
	##    and increases the indentation level by 1
	#######################################################################
	proc EnterProc {} {
		variable LogIndentationLevel
	
		set lProcName [lindex [info level 0] 0]
	
		global errorInfo
		global errorCode
	
		if {([info level] < 2)} {
			return
		}
		
		Print "ENTERED:[lindex [info level -1] 0]"
	
		incr LogIndentationLevel
	}
	
	#######################################################################
	##ExitProc:
	##  Description:
	##    Outputs a line of text with the name of the current procedure
	##    and decreases the indentation level by 1
	#######################################################################
	proc ExitProc {} {
		variable LogIndentationLevel
	
		set lProcName [lindex [info level 0] 0]
	
		global errorInfo
		global errorCode
	
		if {([info level] < 2)} {
			return
		}
	
		incr LogIndentationLevel -1
		
		Print "EXIT:[lindex [info level -1] 0]"
	}
	
	#######################################################################
	##Indent:
	##  Description:
	##    Outputs spaces:  2 * the current indentation level
	#######################################################################
	proc Indent {} {
		variable LogIndentationLevel
		
		set lProcName [lindex [info level 0] 0]
	
		global errorInfo
		global errorCode
	
		set lIndentation [expr ($LogIndentationLevel * 2)]
		set lIndentation "%$lIndentation"
		append lIndentation "s"
		return [format $lIndentation ""]
	}
}