/* -*-pgsql-c-*- */
/*
 * $Header: /home/t-ishii/repository/pgpool/pool_auth.c,v 1.5 2004/04/28 13:52:38 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003, 2004	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * pool_auth.c: authenticaton stuff
 *
*/

#include "pool.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_PARAM_H
#include <param.h>
#endif
#include <errno.h>
#include <string.h>

static int do_clear_text_password(POOL_CONNECTION *backend, POOL_CONNECTION *frontend);
static int do_crypt(POOL_CONNECTION *backend, POOL_CONNECTION *frontend);
static int do_md5(POOL_CONNECTION *backend, POOL_CONNECTION *frontend);

/*
* do authentication against backend. if success return 0 otherwise non 0.
*/
POOL_STATUS pool_do_auth(POOL_CONNECTION *frontend,
						 POOL_CONNECTION_POOL *cp)
{
	int status;
	char kind, kind1;
	int pid, pid1;
	int key, key1;

	status = pool_read(MASTER_CONNECTION(cp)->con, &kind, 1);
	if (status < 0)
	{
		pool_error("pool_do_auth: error while reading \"R\"", strerror(errno));
		return -1;
	}

	if (pool_config.replication_enabled)
	{
		status = pool_read(SECONDARY_CONNECTION(cp)->con, &kind1, 1);
		if (status < 0)
		{
			pool_error("pool_do_auth: error while reading \"R\" from secondary backend",
					   strerror(errno));
			return -1;
		}

		if (kind != kind1)
		{
			pool_error("pool_do_auth: kind is not match between backends master(%c) secondary(%c)",
					   kind, kind1);
			return -1;
		}
	}

	/* Error response? */
	if (kind == 'E')
	{
		ErrorResponse(frontend, cp);
		return -1;
	}
	else if (kind != 'R')
	{
		pool_error("pool_do_auth: expect \"R\" got %c", kind);
		return -1;
	}

	/*
	 * read authentication request kind.
	 *
	 * 0: authenticatoon ok
	 * 1: kerberos v4
	 * 2: kerberos v5
	 * 3: clear text password
	 * 4: crypt password
	 * 5: MD5 password
	 * 6: SCM credential
	 *
	 * in replication mode, we only supports  kind = 0, 3. this is because to "salt"
	 * cannot be replicated among master and secondary.
	 * in non replication mode, we supports  kind = 0, 3, 4, 5
	 */

	status = pool_read(MASTER_CONNECTION(cp)->con, &pid, sizeof(pid));
	if (status < 0)
	{
		pool_error("pool_do_auth: read authentication kind failed");
		return -1;
	}

	if (pool_config.replication_enabled)
	{
		status = pool_read(SECONDARY_CONNECTION(cp)->con, &pid1, sizeof(pid1));

		if (status < 0)
		{
			pool_error("pool_do_auth: read authentication kind from secondary failed");
			return -1;
		}
	}

	pid = ntohl(pid);

	/* clear text password authentication? */
	if (pid == 3)
	{
		pool_debug("trying clear text password authentication");

		pid = do_clear_text_password(MASTER_CONNECTION(cp)->con, frontend);

		if (pid >= 0 && pool_config.replication_enabled)
		{
			pid = do_clear_text_password(SECONDARY_CONNECTION(cp)->con, frontend);
		}
	}

	/* crypt authentication? */
	else if (pid == 4)
	{
		pool_debug("trying crypt authentication");

		pid = do_crypt(MASTER_CONNECTION(cp)->con, frontend);

		if (pid >= 0 && pool_config.replication_enabled)
		{
			pid = do_crypt(SECONDARY_CONNECTION(cp)->con, frontend);
		}
	}

	/* MD5 authentication? */
	else if (pid == 5)
	{
		pool_debug("trying MD5 authentication");

		pid = do_md5(MASTER_CONNECTION(cp)->con, frontend);

		if (pid >= 0 && pool_config.replication_enabled)
		{
			pid = do_md5(SECONDARY_CONNECTION(cp)->con, frontend);
		}
	}

	if (pid != 0)
	{
		pool_error("pool_do_auth: backend does not return Authenticaton Ok");
		return -1;
	} 

	/*
	 * Authentication OK. Now read pid and secret key from the
	 * backend
	 */
	status = pool_read(MASTER_CONNECTION(cp)->con, &kind, 1);
	if (status < 0)
	{
		pool_error("pool_do_auth: error while reading \"W\"");
		return -1;
	}
	if (pool_config.replication_enabled)
	{
		status = pool_read(SECONDARY_CONNECTION(cp)->con, &kind1, 1);
		if (status < 0)
		{
			pool_error("pool_do_auth: error while reading \"W\" from secondary backend",
					   strerror(errno));
			return -1;
		}

		if (kind != kind1)
		{
			pool_error("pool_do_auth: kind is not match between backends master(%c) secondary(%c)",
					   kind, kind1);
			return -1;
		}
	}

	/* Error response? */
	if (kind == 'E')
	{
		ErrorResponse(frontend, cp);
		return -1;
	}
	else if (kind != 'K')
	{
		pool_error("pool_do_auth: expect \"K\" got %c", kind);
		return -1;
	}

	/*
	 * OK, read pid and secret key
	 */
	/* pid */
	pool_read(MASTER_CONNECTION(cp)->con, &pid, sizeof(pid));
	MASTER_CONNECTION(cp)->pid = pid;

	/* key */
	pool_read(MASTER_CONNECTION(cp)->con, &key, sizeof(key));
	MASTER_CONNECTION(cp)->key = key;

	if (pool_config.replication_enabled)
	{
		pool_read(SECONDARY_CONNECTION(cp)->con, &pid1, sizeof(pid1));
		SECONDARY_CONNECTION(cp)->pid = pid;

		/* key */
		pool_read(SECONDARY_CONNECTION(cp)->con, &key1, sizeof(key1));
		SECONDARY_CONNECTION(cp)->key = key;
	}

	return (pool_send_auth_ok(frontend, pid, key));
}


/*
* send authentication ok to frontend. if success return 0 otherwise non 0.
*/
POOL_STATUS pool_send_auth_ok(POOL_CONNECTION *frontend, int pid, int key)
{
	char kind;
	char buff[128];
	int len;

	/* return "Authentication OK" to the frontend */
	kind = 'R';
	pool_write(frontend, &kind, 1);
	len = htonl(0);
	memcpy(buff, &len, sizeof(len));
	if (pool_write_and_flush(frontend, buff, sizeof(len)) < 0)
	{
		return -1;
	}

	/* Backend Key Data */
	kind = 'K';
	pool_write(frontend, &kind, 1);
	pool_write(frontend, &pid, sizeof(pid));
	if (pool_write_and_flush(frontend, &key, sizeof(key)) < 0)
	{
		return -1;
	}

	return 0;
}

/*
 * perform clear text password authetication
 */
static int do_clear_text_password(POOL_CONNECTION *backend, POOL_CONNECTION *frontend)
{
	static int size;
	static char password[1024];
	char response;
	int kind;

	/* master? */
	if (!backend->issecondary_backend)
	{
		pool_write(frontend, "R", 1);	/* authenticaton */
		kind = htonl(3);		/* clear text password authentication */
		pool_write_and_flush(frontend, &kind, sizeof(kind));	/* indicating clear text password authentication */

		/* read password packet */
		if (pool_read(frontend, &size, sizeof(size)))
		{
			pool_error("do_clear_text_password: failed to read password packet size");
			return -1;
		}

		if (pool_read(frontend, password, ntohl(size) - 4))
		{
			pool_error("do_clear_text_password: failed to read password (size: %d)", ntohl(size) - 4);
			return -1;
		}
	}

	/* send password packet to backend */
	pool_write(backend, &size, sizeof(size));
	pool_write_and_flush(backend, password, ntohl(size) -4);
	if (pool_read(backend, &response, sizeof(response)))
	{
		pool_error("do_clear_text_password: failed to read authentication response");
		return -1;
	}

	if (response != 'R')
	{
		pool_debug("do_clear_text_password: backend does not return R while processing clear text password authentication");
		return -1;
	}

	/* expect to read "Authentication OK" response. kind should be 0... */
	if (pool_read(backend, &kind, sizeof(kind)))
	{
		pool_debug("do_clear_text_password: failed to read Authentication OK response");
		return -1;
	}

	return kind;
}

/*
 * perform crypt authetication
 */
static int do_crypt(POOL_CONNECTION *backend, POOL_CONNECTION *frontend)
{
	char salt[2];
	static int size;
	static char password[1024];
	char response;
	int kind;

	/* read salt */
	if (pool_read(backend, salt, sizeof(salt)))
	{
		pool_error("do_crypt: failed to read salt");
		return -1;
	}

	/* master? */
	if (!backend->issecondary_backend)
	{
		pool_write(frontend, "R", 1);	/* authenticaton */
		kind = htonl(4);		/* crypt authentication */
		pool_write(frontend, &kind, sizeof(kind));	/* indicating crypt authentication */
		pool_write_and_flush(frontend, salt, sizeof(salt));		/* salt */

		/* read password packet */
		if (pool_read(frontend, &size, sizeof(size)))
		{
			pool_error("do_crypt: failed to read password packet size");
			return -1;
		}

		if (pool_read(frontend, password, ntohl(size) - 4))
		{
			pool_error("do_crypt: failed to read password (size: %d)", ntohl(size) - 4);
			return -1;
		}
	}

	/* send password packet to backend */
	pool_write(backend, &size, sizeof(size));
	pool_write_and_flush(backend, password, ntohl(size) -4);
	if (pool_read(backend, &response, sizeof(response)))
	{
		pool_error("do_crypt: failed to read authentication response");
		return -1;
	}

	if (response != 'R')
	{
		pool_debug("do_crypt: backend does not return R while processing crypt authentication(%02x) secondary: %d", response, backend->issecondary_backend);
		return -1;
	}

	/* expect to read "Authentication OK" response. kind should be 0... */
	if (pool_read(backend, &kind, sizeof(kind)))
	{
		pool_debug("do_crypt: failed to read Authentication OK response");
		return -1;
	}

	return kind;
}

/*
 * perform MD5 authetication
 */
static int do_md5(POOL_CONNECTION *backend, POOL_CONNECTION *frontend)
{
	char salt[4];
	static int size;
	static char password[1024];
	char response;
	int kind;

	/* read salt */
	if (pool_read(backend, salt, sizeof(salt)))
	{
		pool_error("do_md5: failed to read salt");
		return -1;
	}

	/* master? */
	if (!backend->issecondary_backend)
	{
		pool_write(frontend, "R", 1);	/* authenticaton */
		kind = htonl(5);
		pool_write(frontend, &kind, sizeof(kind));	/* indicating MD5 */
		pool_write_and_flush(frontend, salt, sizeof(salt));		/* salt */

		/* read password packet */
		if (pool_read(frontend, &size, sizeof(size)))
		{
			pool_error("do_md5: failed to read password packet size");
			return -1;
		}

		if (pool_read(frontend, password, ntohl(size) - 4))
		{
			pool_error("do_md5: failed to read password (size: %d)", ntohl(size) - 4);
			return -1;
		}
	}

	/* send password packet to backend */
	pool_write(backend, &size, sizeof(size));
	pool_write_and_flush(backend, password, ntohl(size) -4);
	if (pool_read(backend, &response, sizeof(response)))
	{
		pool_error("do_md5: failed to read authentication response");
		return -1;
	}

	if (response != 'R')
	{
		pool_debug("do_md5: backend does not return R while processing MD5 authentication");
		return -1;
	}

	/* expect to read "Authentication OK" response. kind should be 0... */
	if (pool_read(backend, &kind, sizeof(kind)))
	{
		pool_debug("do_md5: failed to read Authentication OK response");
		return -1;
	}

	return kind;
}
