/*
 * $Header: /home/t-ishii/repository/exttable/exttable.c,v 1.4 2003/12/15 07:11:27 t-ishii Exp $
 *
 * Copyright (c) 2002-2003	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "postgres.h"
#include "funcapi.h"
#include "access/heapam.h"
#include "access/tupdesc.h"		/* TupleDesc */
#include "catalog/pg_type.h"
#include "nodes/execnodes.h"	/* ReturnSetInfo */
#include "utils/builtins.h"		/* textout() */
#include "utils/syscache.h"		/* SearchSysCache() */

PG_FUNCTION_INFO_V1(exttable);

extern Datum exttable(PG_FUNCTION_ARGS);

/* ----------
 * exttable
 * read an external tables and returns it
 *
 * C FUNCTION definition
 * exttable(text, char) returns setof record
 *
 * Calling sequence
 * SELECT exttable('/full/path/to/file', '\t') AS (f1 int, f2 text, ....);
 * ----------
 */

typedef struct
{
	FILE *fd;	/* file descriptor */
	char *path;	/* path */
	char delimiter;		/* delimiter */
} exttable_status;

static char **readfile(exttable_status *status, int natts);

Datum
exttable(PG_FUNCTION_ARGS)
{
	FuncCallContext *funcctx;
	exttable_status *mystatus;
	ReturnSetInfo	*returnset_info;
	TupleDesc	desc;	/* tuple descriptor expected by caller */
	char **cols;

	returnset_info = (ReturnSetInfo *)fcinfo->resultinfo;
	desc = returnset_info->expectedDesc;

	if (SRF_IS_FIRSTCALL())
	{
		MemoryContext oldcontext;
		TupleDesc	tupdesc;	/* result tuple descriptor */
		BpChar *p;

		/* create a function context for cross-call persistence */
		funcctx = SRF_FIRSTCALL_INIT();

		/*
		 * switch to memory context appropriate for multiple function
		 * calls
		 */
		oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

		/* build tupdesc for result tuples from expected desc */
		tupdesc = CreateTupleDesc(desc->natts, false, desc->attrs);
		funcctx->slot = TupleDescGetSlot(tupdesc);

		mystatus = (exttable_status *) palloc(sizeof(*mystatus));
		funcctx->user_fctx = (void *) mystatus;

		/* read the file and set actuaul file contents */
		mystatus->path = DatumGetCString(DirectFunctionCall1(textout, (Datum)PG_GETARG_TEXT_P(0)));
		p = PG_GETARG_BPCHAR_P(1);
		mystatus->delimiter = (char)*(VARDATA(p));
		mystatus->fd = NULL;

		MemoryContextSwitchTo(oldcontext);
	}

	funcctx = SRF_PERCALL_SETUP();
	mystatus = (exttable_status *) funcctx->user_fctx;

	if ((cols = readfile(mystatus, desc->natts)))
	{
		Datum		*values;
		char		*nulls;
		HeapTuple	tuple;
		Datum		result;
		int		i;

		/*
		 * Form tuple with appropriate data.
		 */
		values = (Datum *)palloc(sizeof(Datum)*desc->natts);
		nulls = (char *)palloc(sizeof(char)*desc->natts);
		MemSet(values, 0, sizeof(Datum)*desc->natts);
		MemSet(nulls, ' ', sizeof(char)*desc->natts);

		/* loop for number of attributes specified by
		   as t1(i int, j int, k text) etc. */
		for (i=0;i < desc->natts;i++)
		{
			HeapTuple	tuple;
			Form_pg_type typetuple;
			Oid typeoid;

			/* get pg_type tuple oid */
			typeoid = desc->attrs[i]->atttypid;

			/* get a pg_type tuple by oid */
			tuple = SearchSysCache(TYPEOID, 
								   ObjectIdGetDatum(typeoid),
								   0, 0, 0);
			if (!HeapTupleIsValid(tuple))
				elog(ERROR, "Cannot find type tuple");

			/* call type input function according to it's data type */
			typetuple = (Form_pg_type)GETSTRUCT(tuple);
			values[i] = OidFunctionCall1(typetuple->typinput,
										 CStringGetDatum(cols[i]));
			ReleaseSysCache(tuple);
		}

		tuple = heap_formtuple(funcctx->slot->ttc_tupleDescriptor,
							   values, nulls);
		result = TupleGetDatum(funcctx->slot, tuple);

		pfree(values);
		pfree(nulls);

		for (i=0;i < desc->natts;i++)
			pfree(cols[i]);
		pfree((char *)cols);

		SRF_RETURN_NEXT(funcctx, result);
	}
	fclose(mystatus->fd);
	SRF_RETURN_DONE(funcctx);
}

#include "lib/stringinfo.h"

static char **readfile(exttable_status *status, int natts)
{
	char **cols;
	char p;
	StringInfo token;
	int c;
	int i;

	if (status->fd == NULL)
	{
		status->fd = fopen(status->path, "r");
		if (!status->fd)
		{
			elog(ERROR, "exttable: Could not open %s", status->path);
		}
	}

	cols = palloc(sizeof(char *)*natts);

	for (i=0;i<natts;i++)
	{
		token = makeStringInfo();
		while ((c = getc(status->fd)) != EOF && c != status->delimiter && c != '\n')
		{
			p = c;
			appendStringInfoChar(token, p);
		}
		cols[i] = token->data;
		if (c != status->delimiter)
			break;
	}

	/* ignore a line having too few tokens (e.g. newline alone) */
	if (i < natts)
		return NULL;

	if (i != natts && c == EOF)
		return NULL;

	return cols;
}
