/*
 *   IPC package for CygWin32
 *
 *   Copyright (C) 1997 Philippe CHAPUY
 *   Copyright (C) 1998 Ludovic LANGE
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   HISTORY:
 *   --------
 *
 *   13/05/1998 : Version 1.00 released
 *                First public release
 *                adress any comments to llange@capgemini.fr
 *
 */



/*
 * Philippe Chapuy, le 20/05/97
 */

#define EXTERN
#include "IpcNtExt.h"
#include <unistd.h>
#include <stdlib.h>
#include "strace.h"

static int findkey (key_t key);
static int newseg (key_t key, int shmflg, int size);
static void killseg (int id);


static struct shmid_ds **shm_segs;
static int shm_rss = 0; /* number of shared memory pages that are in memory */
static int shm_swp = 0; /* number of shared memory pages that are in swap */
static int max_shmid = 0; /* every used id is <= max_shmid */

static unsigned short shm_seq = 0; /* incremented, for recognizing stale ids */

static CYGWIN32_IPCNT_SHMSTR *shareadrshm         ;		/*PCPC*/
static struct shmid_ds *ptrshm;

static int		  GFirstShm	 = 0;		/*PCPC*/
static int		  GFdShm	    ;		/*PCPC*/

/************************************************************************/
/* Demande d'acces a la zone partagee de gestion des shm		*/
/************************************************************************/

static int IsGSemShmExist()
{

 GSemShm = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, CYGWIN32_IPCNT_SEMSHM) ;
 if( GSemShm != NULL )
 {
  return (1) ;
 }
 else
 {
  return (0) ;
 }

}

static int shm_connect(void)
{
 int LRet ;

 if( GFirstShm == 0 )
 {
  if( IsGSemShmExist() )
  {
   GFirstShm = 1 ;
   GFdShm  = open(CYGWIN32_IPCNT_FILESHM, O_RDWR) ;
   shareadrshm = (CYGWIN32_IPCNT_SHMSTR *)
   		mmap(0, sizeof(CYGWIN32_IPCNT_SHMSTR), PROT_WRITE|PROT_READ,
			MAP_SHARED, GFdShm, 0) ;
   if( shareadrshm == (CYGWIN32_IPCNT_SHMSTR *) -1 )
   {
    close (GFdShm) ;
    return (0) ;
   }
   ptrshm = &(shareadrshm->shm[0]) ;
   shm_segs = &(shareadrshm->shm_segs[0]);
  }
  else
  {
   /* RAF unmap */
   return(0) ;
  }
 }


 LRet = WaitForSingleObject(GSemShm, INFINITE) ;


 shm_seq = shareadrshm->shm_seq ;
 shm_rss = shareadrshm->shm_rss ;
 shm_swp = shareadrshm->shm_swp ;
 max_shmid = shareadrshm->max_shmid ;

 return(1) ;
}

/************************************************************************/
/*PCPC									*/
/************************************************************************/
static void sem_deconnect()
{
 long LPrevious ;


 shareadrshm->shm_seq = shm_seq ;
 shareadrshm->shm_rss = shm_rss ;
 shareadrshm->shm_swp = shm_swp ;
 shareadrshm->max_shmid = max_shmid ;

 ReleaseSemaphore(GSemShm, (LONG) 1, &LPrevious) ;

 if( LPrevious != 0 )
 {
  printf ("Error, shared memory semaphore not equal 0\n") ;
 }
}


static int findkey (key_t key)
{
	int id;
	struct shmid_ds *shp;
debug_printf("findkey : key=%p\n",key);

	for (id = 0; id <= max_shmid; id++) {
		if (shm_segs[id] == IPC_UNUSED)
		{
			continue;
		}
		shp = (struct shmid_ds *) &(ptrshm[id]);
		if (key == shp->shm_perm.key)
		{
debug_printf("findkey : return %X\n",id);
			CYGWIN32_IPCNT_RETURN (id) ;
		}
	}
debug_printf("findkey : return -1\n");
	CYGWIN32_IPCNT_RETURN (-1);
}

/*
 * allocate new shmid_ds and pgtable. protected by shm_segs[id] = NOID.
 */
static int newseg (key_t key, int shmflg, int size)
{
	struct shmid_ds *shp;
	int id, i, LFd;
	char LBuff[100] ;
	char *LTempo ;

debug_printf("newseg : key=%p shmflg=%X size=%X\n",key,shmflg,size);
	if (size < 1)
	{
debug_printf("newseg : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN ( -EINVAL ) ;
	}

	for (id = 0; id < SHMMNI; id++)
		if (!shm_segs[id] || shm_segs[id] == IPC_UNUSED) {
			shm_segs[id] = (struct shmid_ds *) IPC_NOID;
			goto found;
		}
debug_printf("newseg : return -ENOSPC\n");
	CYGWIN32_IPCNT_RETURN_DECONNECT ( -ENOSPC ) ;

found:
	shp = (struct shmid_ds *) &(ptrshm[id]) ;

	LTempo = (char *) malloc (size) ;
	if (!LTempo) {
		shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
debug_printf("newseg : return -ENOMEM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -ENOMEM ) ;
	}
 	strcpy(LBuff, "/tmp/") ;
 	itoa(id, LBuff+strlen(LBuff)) ;
	LFd = open(LBuff, O_WRONLY|O_CREAT|O_TRUNC) ;
	if (LFd == -1) {
		shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
debug_printf("newseg : return -ENOMEM (2)\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -ENOMEM ) ;
	}
	memset (LTempo, 0x00, size) ;
	i = write (LFd, LTempo, size) ;
	if( i != size )
	{
		close (LFd) ;
		free (LTempo) ;
		shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
debug_printf("newseg : return -ENOMEM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -ENOMEM ) ;
	}
	close (LFd) ;
	free (LTempo) ;

	shp->shm_perm.key = key;
	shp->shm_perm.mode = 0 ;
	shp->shm_perm.seq = shm_seq;
	shp->shm_segsz = size;
	shp->shm_lpid = shp->shm_nattch = 0;
	shp->shm_atime = shp->shm_dtime = 0;

	if (id > max_shmid)
		max_shmid = id;
	shm_segs[id] = (struct shmid_ds *)
			((char *) shp - (int) shareadrshm);
debug_printf("newseg : return %p\n",(shp->shm_perm.seq) * SHMMNI + id );
	CYGWIN32_IPCNT_RETURN_DECONNECT((shp->shm_perm.seq) * SHMMNI + id );
}

int shmget (key_t key, int size, int shmflg)
{
	struct shmid_ds *shp;
	int id = 0;

debug_printf("shmget : key=%p size=%X shmflg=%X\n",key,size,shmflg);	
	if (shm_connect() == 0)
	{
debug_printf("shmget : return -EACCES\n");
	    CYGWIN32_IPCNT_RETURN (-EACCES) ;
	}

	if (size < 0 )
	{
debug_printf("shmget : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
	}
	if (key == IPC_PRIVATE)
	{
debug_printf("shmget : -> newseg\n");
		return newseg(key, shmflg, size);
	}
	if ((id = findkey (key)) == -1) {
		if (!(shmflg & IPC_CREAT))
		{
debug_printf("shmget : return -ENOENT\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT ( -ENOENT ) ;
		}
		return newseg(key, shmflg, size);
	}
	if ((shmflg & IPC_CREAT) && (shmflg & IPC_EXCL))
	{
debug_printf("shmget : return -EEXIST\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EEXIST ) ;
	}
	shp = (struct shmid_ds *)
			((char *) shm_segs[id] + (int) shm_segs);
	if (shp->shm_perm.mode & SHM_DEST)
	{
debug_printf("shmget : return -EIDRM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EIDRM ) ;
	}
	if (size > shp->shm_segsz)
	{
debug_printf("shmget : return -EINVAL (2)\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
	}
debug_printf("shmget : return %p\n",(shp->shm_perm.seq) * SHMMNI + id);
	CYGWIN32_IPCNT_RETURN_DECONNECT ((shp->shm_perm.seq) * SHMMNI + id) ;
}

/*
 * Only called after testing nattch and SHM_DEST.
 * Here pages, pgtable and shmid_ds are freed.
 */
static void killseg (int id)
{
	struct shmid_ds *shp;

	shp = (struct shmid_ds *)
			((char *) shm_segs[id] + (int) shm_segs);
	if (shm_segs[id] == IPC_NOID || shm_segs[id] == IPC_UNUSED)
	{
		printf ("shm: killseg called on unused seg id=%d\n", id);
		return;
	}
	shp->shm_perm.seq++;     /* for shmat */
	shm_seq = (shm_seq+1) % ((unsigned)(1<<31)/SHMMNI); /* increment, but avoid overflow */
	shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
	if (id == max_shmid)
		while (max_shmid && (shm_segs[--max_shmid] == IPC_UNUSED));
	return;
}

int shmctl (int shmid, int cmd, struct shmid_ds *buf)
{
	struct shmid_ds tbuf;
	struct shmid_ds *shp;
	struct ipc_perm *ipcp;
	int id;

debug_printf("shmctl : shmid=%X cmd=0x%02X shmid_ds=%p\n",shmid,cmd,buf);	
	if (cmd < 0 || shmid < 0)
	{
debug_printf("shmctl : return -EINVAL\n");	
		CYGWIN32_IPCNT_RETURN (-EINVAL) ;
	}

	if (shm_connect() == 0)
	{
debug_printf("shmctl : return -EACCES\n");	
	 CYGWIN32_IPCNT_RETURN (-EACCES) ;
	}

	id = (unsigned int) shmid % SHMMNI;

	switch (cmd) { /* replace with proc interface ? */
	case IPC_INFO:
	{
		struct shminfo shminfo;
		if (!buf)
		{
debug_printf("shmctl : IPC_INFO return -EFAULT\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EFAULT) ;
		}
		shminfo.shmmni = 0;
		shminfo.shmmax = 0;
		shminfo.shmmin = 0;
		shminfo.shmall = 0;
		shminfo.shmseg = 0;
		memcpy (buf, &shminfo, sizeof(struct shminfo));
debug_printf("shmctl : IPC_INFO return %X\n",max_shmid);	
		CYGWIN32_IPCNT_RETURN_DECONNECT(max_shmid) ;
	}
	case SHM_INFO:
	{
		struct shm_info shm_info;
		if (!buf)
		{
debug_printf("shmctl : SHM_INFO return -EFAULT\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT(-EFAULT) ;
		}
		shm_info.shm_rss = shm_rss;
		shm_info.shm_swp = shm_swp;
		memcpy (buf, &shm_info, sizeof(shm_info));
debug_printf("shmctl : SHM_INFO return %X\n",max_shmid);	
		CYGWIN32_IPCNT_RETURN_DECONNECT(max_shmid) ;
	}
	case SHM_STAT:
		if (!buf)
		{
debug_printf("shmctl : SHM_STAT return -EFAULT\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT ( -EFAULT ) ;
		}

		if (shmid > max_shmid)
		{
debug_printf("shmctl : SHM_STAT return -EINVAL\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
		}

		shp = (struct shmid_ds *)
			((char *) shm_segs[id] + (int) shm_segs);

		if (shm_segs[id] == IPC_UNUSED || shm_segs[id] == IPC_NOID)
		{
debug_printf("shmctl : SHM_STAT return -EINVAL\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
		}
		id = (unsigned int) shp->shm_perm.seq * SHMMNI + shmid;
		memcpy( &(tbuf.shm_perm), &(shp->shm_perm),
					sizeof (struct ipc_perm));
		tbuf.shm_segsz  = shp->shm_segsz;
		tbuf.shm_atime  = shp->shm_atime;
		tbuf.shm_dtime  = shp->shm_dtime;
		tbuf.shm_ctime  = shp->shm_ctime;
		tbuf.shm_cpid   = shp->shm_cpid;
		tbuf.shm_lpid   = shp->shm_lpid;
		tbuf.shm_nattch = shp->shm_nattch;
		memcpy (buf, &tbuf, sizeof(struct shmid_ds));
debug_printf("shmctl : SHM_STAT return %X\n",id);	
		CYGWIN32_IPCNT_RETURN_DECONNECT ( id ) ;
	}

	shp = (struct shmid_ds *)
			((char *) shm_segs[id] + (int) shm_segs);
	if (shp == IPC_UNUSED || shp == IPC_NOID)
	{
debug_printf("shmctl : return -EINVAL\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
	}
	if (shp->shm_perm.seq != (unsigned int) shmid / SHMMNI)
	{
debug_printf("shmctl : return -EIDRM\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EIDRM ) ;
	}
	ipcp = &shp->shm_perm;

	switch (cmd) {
	case SHM_UNLOCK:
		if (!(ipcp->mode & SHM_LOCKED))
		{
debug_printf("shmctl : SHM_UNLOCK return -EINVAL\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL) ;
		}
		ipcp->mode &= ~SHM_LOCKED;
		break;
	case SHM_LOCK:
		if (ipcp->mode & SHM_LOCKED)
		{
debug_printf("shmctl : SHM_LOCK return -EINVAL\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
		}
		ipcp->mode |= SHM_LOCKED;
		break;
	case IPC_STAT:
		if (!buf)
		{
debug_printf("shmctl : IPC_STAT return -EFAULT\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EFAULT) ;
		}
		memcpy( &(tbuf.shm_perm), &(shp->shm_perm),
					sizeof (struct ipc_perm));
		tbuf.shm_segsz  = shp->shm_segsz;
		tbuf.shm_atime  = shp->shm_atime;
		tbuf.shm_dtime  = shp->shm_dtime;
		tbuf.shm_ctime  = shp->shm_ctime;
		tbuf.shm_cpid   = shp->shm_cpid;
		tbuf.shm_lpid   = shp->shm_lpid;
		tbuf.shm_nattch = shp->shm_nattch;
		memcpy (buf, &tbuf, sizeof(struct shmid_ds));
		break;
	case IPC_SET:
debug_printf("shmctl : IPC_SET return -EPERM\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EPERM ) ;
	case IPC_RMID:
			shp->shm_perm.mode |= SHM_DEST;
			if (shp->shm_nattch <= 0)
			{
				killseg (id);
			}
			break;
	default:
debug_printf("shmctl : return -EINVAL\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
	}
debug_printf("shmctl : return 0\n");	
	CYGWIN32_IPCNT_RETURN_DECONNECT (0) ;
}

/*
 * Fix shmaddr, allocate descriptor, map shm, add attach descriptor to lists.
 */
#if SHMAT_LINUX
int shmat (int shmid, char *shmaddr, int shmflg, char **adr)
{
#else
char *shmat (int shmid, char *shmaddr, int shmflg)
{
	char *adr=(char *)-1;
#endif
	struct shmid_ds *shp;
	struct vm_area_struct *shmd;
	int i, LPid ;
	unsigned int id;
	char *addr;
	unsigned long len;
	char LBuff[100] ;

	LPid = getpid() ;

debug_printf("shmat : shmid=%X shmaddr=%p shmflag=%X\n",shmid,shmaddr,shmflg);	
	if (shm_connect() == 0)
	{
debug_printf("shmat : return -EACCESS\n");	
#if SHMAT_LINUX
	 CYGWIN32_IPCNT_RETURN (-EACCES) ;
#else
	 errno=EACCES;
	 return (char *)-1;
#endif
	}

	if (shmid < 0) {
debug_printf("shmat : return -EINVAL\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -EINVAL ) ;
	}

	id = (unsigned int) shmid % SHMMNI;
	shp = (struct shmid_ds *)
			((char *) shm_segs[id] + (int) shm_segs);
	if (shm_segs[id] == IPC_UNUSED || shm_segs[id] == IPC_NOID)
	{
debug_printf("shmat : return -EINVAL\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -EINVAL ) ;
	}

	if (!(addr = (char *) shmaddr)) {
		if (shmflg & SHM_REMAP)
		{
debug_printf("shmat : return -EINVAL\n");	
			CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -EINVAL ) ;
		}
	}
	/*
	 * Check if addr exceeds TASK_SIZE (from do_mmap)
	 */
	if (shp->shm_perm.seq != (unsigned int) shmid / SHMMNI)
	{
debug_printf("shmat : return -EIDRM\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -EIDRM ) ;
	}

	for (i=0; i<SHMMNI; i++)
	{
	 if (shp->attaches[i].pid == LPid)
	 {
debug_printf("shmat : return -EINVAL\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -EINVAL ) ;
	 }
	}

	for (i=0; i<SHMMNI; i++)
	{
	 if (shp->attaches[i].pid == 0)
	 {
	  break ;
	 }
	}

	if( i == SHMMNI )
	{
debug_printf("shmat : return -ENOMEM\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -ENOMEM ) ;
	}

	if (((struct shmid_ds *) ((char *) shp-(int)shm_segs) != shm_segs[id]) || (shp->shm_perm.seq != (unsigned int) shmid / SHMMNI)) {
debug_printf("shmat : return -EIDRM\n");	
		CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( -EIDRM ) ;
	}

	shmd = &(shp->attaches[i]) ;

	shmd->pid = LPid ;
 	strcpy(LBuff, "/tmp/") ;
 	itoa(id, LBuff+strlen(LBuff)) ;
	shmd->fd = open(LBuff, O_RDWR) ;
	len = lseek( shmd->fd, 0, SEEK_END) ;
	lseek( shmd->fd, 0, SEEK_SET) ;

	shp->shm_nattch++;            /* prevent destruction */

#if SHMAT_LINUX
	*adr = shmd->adr =
#else
	adr = shmd->adr =
#endif
	    mmap (shmaddr, len, (PROT_READ|PROT_WRITE), MAP_SHARED | ( shmaddr != 0 ? MAP_FIXED : 0 ), shmd->fd, 0) ;

debug_printf("shmat : return %p\n",adr);	

	CYGWIN32_IPCNT_RETURN_DECONNECT_SHMAT ( 0 ) ;
	
}

/*
 * detach and kill segment if marked destroyed.
 */
int shmdt (char *shmaddr)
{
	struct shmid_ds *shp;
	struct vm_area_struct *shmd ;
	int LPid, LId, LIndex ;
	int LFound = 0;

debug_printf("shmdt : shmaddr=%p\n",shmaddr);	
	LPid = getpid() ;

	if (shm_connect() == 0)
	{
debug_printf("shmdt : return -EACCES\n");	
	    CYGWIN32_IPCNT_RETURN (-EACCES) ;
	}

	for (LId = 0; LId < SHMMNI; LId++)
	{
	    shp = (struct shmid_ds *)
		((char *) shm_segs[LId] + (int) shm_segs);
	    if (shm_segs[LId] != IPC_UNUSED)
	    {
		for(LIndex = 0; LIndex < SHMMNI; LIndex++ )
		{
		    shmd = &(shp->attaches[LIndex]) ;
		    if( shmd->adr == shmaddr )
		    {
			if ( shmd->pid == LPid )
			{
			    munmap( shmaddr, shp->shm_segsz ) ;
			    shp->shm_nattch--;            /* for destruction */
			    close (shmd->fd) ;
			    shmd->pid = 0 ;
			    shmd->adr = 0 ;
			    shmd->fd  = 0 ;
			    break ;
			} else {
debug_printf("shmdt : return -EINVAL\n");	
	     CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
	    }
	   }
	  }
	  /* No destruction of SHM
	  if( shp->shm_nattch <= 0 )
	  {
	   killseg(LId) ;
	  }
	  */
	 }
	 if( LFound == 1 ) break ;
	}

	if( LFound == 0 )
	{
debug_printf("shmdt : return -EINVAL\n");	
	 CYGWIN32_IPCNT_RETURN_DECONNECT ( -EINVAL ) ;
	}

debug_printf("shmdt : return 0\n");	
	CYGWIN32_IPCNT_RETURN_DECONNECT (0) ;
}

